/*
 * Copyright (C) 2005-2006 Junjiro Okajima
 *
 * This program, aufs is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/* $Id: misc.c,v 1.17 2006/10/23 13:00:54 sfjro Exp $ */

//#include <linux/fs.h>
//#include <linux/namei.h>
//#include <linux/mm.h>
#include <asm/uaccess.h>
#include "aufs.h"

/*
 * @hidden_dir: must be locked.
 * @hidden_dentry: target dentry.
 * @parent: aufs dentry, can be NULL.
 */
// todo: remove @parent.
int safe_unlink(struct inode *hidden_dir, struct dentry *hidden_dentry)
{
	int err;
	struct inode *hidden_inode;

	LKTRTrace("%.*s\n", DLNPair(hidden_dentry));
	IMustLock(hidden_dir);

	dget(hidden_dentry);
	hidden_inode = hidden_dentry->d_inode;
	if (hidden_inode)
		atomic_inc(&hidden_inode->i_count);
#if 0 // partial testing
	{struct qstr *name = &hidden_dentry->d_name;
	if (name->len == sizeof(AUFS_XINO_FNAME)-1
	    && !strncmp(name->name, AUFS_XINO_FNAME, name->len))
		err = vfs_unlink(hidden_dir, hidden_dentry);
	else err = -1;}
#else
	// vfs_unlink() locks inode
	err = vfs_unlink(hidden_dir, hidden_dentry);
#endif
	dput(hidden_dentry);
	if (hidden_inode)
		iput(hidden_inode);

	TraceErr(err);
	return err;
}

int hidden_notify_change(struct dentry *hidden_dentry, struct iattr *ia,
			 struct dentry *dentry)
{
	int err;
	struct inode *hidden_inode;

	LKTRTrace("%.*s, ia_valid 0x%x, dentry %p\n",
		  DLNPair(hidden_dentry), ia->ia_valid, dentry);
	hidden_inode = hidden_dentry->d_inode;
	IMustLock(hidden_inode);

	err = -EPERM;
	if (!IS_IMMUTABLE(hidden_inode) && !IS_APPEND(hidden_inode)) {
		err = notify_change(hidden_dentry, ia);
		//err = -1;
#if 1 // todo: remove this
		if (unlikely(!err && dentry
			     && (ia->ia_valid & ~(ATTR_ATIME | ATTR_ATIME_SET))
			     && IS_MS(dentry->d_sb, MS_UDBA_INOTIFY))) {
			//dump_stack();
			direval_dec(dentry);
			if (!IS_ROOT(dentry))
				direval_dec(dentry->d_parent);
		}
#endif
	}
	TraceErr(err);
	return err;
}

void *kzrealloc(void *p, int nused, int new_sz)
{
	void *q;

	LKTRTrace("p %p, nused %d, sz %d, ksize %d\n",
		  p, nused, new_sz, ksize(p));
	DEBUG_ON(new_sz <= 0);
	if (new_sz <= nused)
		return p;
	if (new_sz <= ksize(p)) {
		memset(p + nused, 0, new_sz - nused);
		return p;
	}

	q = kmalloc(new_sz, GFP_KERNEL);
	//q = NULL;
	if (unlikely(!q))
		return NULL;
	memcpy(q, p, nused);
	memset(q + nused, 0, new_sz - nused);
	//smp_mb();
	kfree(p);
	return q;
}

/* ---------------------------------------------------------------------- */

readf_t find_readf(struct file *hidden_file)
{
	const struct file_operations *op = hidden_file->f_op;

	if (op) {
		if (op->read)
			return op->read;
		if (op->aio_read)
			return do_sync_read;
	}
	return ERR_PTR(-ENOSYS);
}

writef_t find_writef(struct file *hidden_file)
{
	const struct file_operations *op = hidden_file->f_op;

	if (op) {
		if (op->write)
			return op->write;
		if (op->aio_write)
			return do_sync_write;
	}
	return ERR_PTR(-ENOSYS);
}

ssize_t aufs_fread(readf_t func, struct file *file, void *buf, size_t size,
		   loff_t *pos)
{
	ssize_t err;
	mm_segment_t oldfs;

	LKTRTrace("%.*s, sz %d, *pos %Ld\n",
		  DLNPair(file->f_dentry), size, *pos);

	oldfs = get_fs();
	set_fs(KERNEL_DS);
	do {
		err = func(file, (char __user*)buf, size, pos);
	} while (err == -EAGAIN || err == -EINTR);
	set_fs(oldfs);
	//smp_mb();
	TraceErr(err);
	return err;
}

static ssize_t do_fwrite(writef_t func, struct file *file, void *buf,
			 size_t size, loff_t *pos)
{
	ssize_t err;
	mm_segment_t oldfs;

	//smp_mb();
	oldfs = get_fs();
	set_fs(KERNEL_DS);
	do {
		err = func(file, (const char __user*)buf, size, pos);
	} while (err == -EAGAIN || err == -EINTR);
	set_fs(oldfs);
	return err;
}

ssize_t aufs_fwrite(writef_t func, struct file *file, void *buf, size_t size,
		    loff_t *pos)
{
	ssize_t err;

	LKTRTrace("%.*s, sz %d, *pos %Ld\n",
		  DLNPair(file->f_dentry), size, *pos);

	// signal block and no wkq?
	/*
	 * it breaks RLIMIT_FSIZE and normal user's limit,
	 * users should care about quota and real 'filesystem full.'
	 */
	if (!is_kthread(current)) {
		void f(void *completion) {
			err = do_fwrite(func, file, buf, size, pos);
			complete(completion);
		}
		wkq_wait(f);
	} else
		err = do_fwrite(func, file, buf, size, pos);

	TraceErr(err);
	return err;
}

/* ---------------------------------------------------------------------- */

#if 1
struct nameidata *fake_dm(struct nameidata *fake_nd, struct nameidata *nd,
			  struct super_block *sb, aufs_bindex_t bindex)
{
	if (!nd)
		return NULL;

#ifdef CONFIG_AUFS_FAKE_DM
	fake_nd->dentry = NULL;
	fake_nd->mnt = NULL;
#else
	fake_nd->dentry = dtohd_index(nd->dentry, bindex);
	DEBUG_ON(!fake_nd->dentry);
	dget(fake_nd->dentry);
	DEBUG_ON(!sb);
	fake_nd->mnt = sbr_mnt(sb, bindex);
	DEBUG_ON(!fake_nd->mnt);
	mntget(fake_nd->mnt);
#endif
	return fake_nd;
}

void fake_dm_release(struct nameidata *fake_nd)
{
#ifndef CONFIG_AUFS_FAKE_DM
	if (fake_nd) {
		mntput(fake_nd->mnt);
		dput(fake_nd->dentry);
	}
#endif
}
#endif

/* ---------------------------------------------------------------------- */

int copy_file(struct file *dst, struct file *src, loff_t len)
{
	int err;
	unsigned long blksize;
	char *buf;
	readf_t rfunc;
	writef_t wfunc;

	LKTRTrace("%.*s, %.*s\n",
		  DLNPair(dst->f_dentry), DLNPair(src->f_dentry));

	err = -ENOMEM;
	blksize = dst->f_dentry->d_sb->s_blocksize;
	if (!blksize || PAGE_SIZE < blksize)
		blksize = PAGE_SIZE;
	LKTRTrace("blksize %lu\n", blksize);
	buf = kmalloc(blksize, GFP_KERNEL);
	//buf = NULL;
	if (unlikely(!buf))
		goto out;

	rfunc = find_readf(src);
	err = PTR_ERR(rfunc);
	if (IS_ERR(rfunc))
		goto out_free;
	wfunc = find_writef(dst);
	err = PTR_ERR(wfunc);
	if (IS_ERR(wfunc))
		goto out_free;

	dst->f_pos = src->f_pos = 0;
	//smp_mb();
	while (len) {
		size_t sz, rbytes, wbytes;
		char *p;
		int all_zero, i;

		LKTRTrace("len %lld\n", len);
		sz = blksize;
		if (len < blksize)
			sz = len;

		rbytes = 0;
		while (!rbytes)
			err = rbytes = aufs_fread(rfunc, src, buf, sz,
						  &src->f_pos);
		if (unlikely(err < 0))
			break;

		all_zero = 0;
		if (len > rbytes && rbytes == blksize) {
			all_zero = 1;
			p = buf;
			for (i = 0; all_zero && i < rbytes; i++)
				all_zero = !*p++;
		}
		if (!all_zero) {
			wbytes = rbytes;
			p = buf;
			while (wbytes) {
				size_t b;
				err = b = aufs_fwrite(wfunc, dst, (void*)p,
						      wbytes, &dst->f_pos);
				if (unlikely(err < 0))
					break;
				wbytes -= b;
				p += b;
			}
		} else {
			loff_t res;
			LKTRLabel(hole);
			err = res = vfs_llseek(dst, rbytes, SEEK_CUR);
			if (unlikely(res < 0))
				break;
		}
		len -= rbytes;
		err = 0;
	}

 out_free:
	kfree(buf);
 out:
	TraceErr(err);
	return err;
}

struct file *xcreate_unlink(struct super_block *sb, char *fname, int silent,
			    struct dentry *parent)
{
	struct file *file;
	int err;
	struct dentry *hidden_parent;
	struct inode *hidden_dir;
	const int udba = IS_MS(sb, MS_UDBA_INOTIFY);

	LKTRTrace("%s\n", fname);

	file = filp_open(fname, O_RDWR | O_CREAT | O_EXCL | O_LARGEFILE,
			 S_IRUGO | S_IWUGO);
	//file = ERR_PTR(-1);
	if (IS_ERR(file)) {
		if (!silent)
			Err("open %s(%ld)\n", fname, PTR_ERR(file));
		return file;
	}
	if (unlikely(udba && parent))
		direval_dec(parent);

	/* keep file count */
	hidden_parent = dget_parent(file->f_dentry);
	hidden_dir = hidden_parent->d_inode;
	i_lock(hidden_dir);
	err = safe_unlink(hidden_dir, file->f_dentry);
	if (unlikely(udba && parent))
		direval_dec(parent);
	i_unlock(hidden_dir);
	dput(hidden_parent);
	if (unlikely(err)) {
		if (!silent)
			Err("unlink %s(%d)\n", fname, err);
		goto out;
	}
	if (sb != file->f_dentry->d_sb)
		return file; /* success */

	if (!silent)
		Err("%s must be outside\n", fname);
	err = -EINVAL;

 out:
	fput(file);
	file = ERR_PTR(err);
	return file;
}

/* ---------------------------------------------------------------------- */

int test_ro(struct super_block *sb, aufs_bindex_t bindex, struct inode *inode)
{
	int err;

	err = br_rdonly(stobr(sb, bindex));
	if (!err && inode) {
		struct inode *hi = itohi_index(inode, bindex);
		if (hi)
			err = IS_IMMUTABLE(hi) ? -EROFS : 0;
	}
	return err;
}

int test_perm(struct inode *hidden_inode, int mask)
{
	if (!current->fsuid)
		return 0;
	if (unlikely(SB_NFS(hidden_inode->i_sb)
		     && (mask & MAY_WRITE)
		     && S_ISDIR(hidden_inode->i_mode)))
		mask |= MAY_READ; /* force permission check */
	return permission(hidden_inode, mask, NULL);
}
