/*
 * Copyright (C) 2005-2006 Junjiro Okajima
 *
 * This program, aufs is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/* $Id: sbinfo.c,v 1.17 2006/11/20 03:30:30 sfjro Exp $ */

#include "aufs.h"

struct aufs_sbinfo *stopd(struct super_block *sb)
{
	struct aufs_sbinfo *sbinfo;
	sbinfo = sb->s_fs_info;
	//DEBUG_ON(sbinfo->si_bend < 0);
	return sbinfo;
}

aufs_bindex_t sbend(struct super_block *sb)
{
	SiMustAnyLock(sb);
	return stopd(sb)->si_bend;
}

struct aufs_branch *stobr(struct super_block *sb, aufs_bindex_t bindex)
{
	SiMustAnyLock(sb);
	DEBUG_ON(bindex < 0 || sbend(sb) < bindex
		 || !stopd(sb)->si_branch[0 + bindex]);
	return stopd(sb)->si_branch[0 + bindex];
}

int sigen(struct super_block *sb)
{
	SiMustAnyLock(sb);
	return stopd(sb)->si_generation;
}

int sigen_inc(struct super_block *sb)
{
	int gen;

	SiMustWriteLock(sb);
	gen = ++stopd(sb)->si_generation;
	update_digen(sb->s_root);
	sb->s_root->d_inode->i_version++;
	//smp_mb();
	return gen;
}

int find_bindex(struct super_block *sb, struct aufs_branch *br)
{
	aufs_bindex_t bindex, bend;

	bend = sbend(sb);
	for (bindex = 0; bindex <= bend; bindex++)
		if (stobr(sb, bindex) == br)
			return bindex;
	return -1;
}

/* ---------------------------------------------------------------------- */

/* dentry and super_block lock. call at entry point */
void aufs_read_lock(struct dentry *dentry, int flags)
{
	si_read_lock(dentry->d_sb);
	if (flags & AUFS_D_WLOCK)
		di_write_lock(dentry);
	else
		di_read_lock(dentry, flags);
}

void aufs_read_unlock(struct dentry *dentry, int flags)
{
	if (flags & AUFS_D_WLOCK)
		di_write_unlock(dentry);
	else
		di_read_unlock(dentry, flags);
	si_read_unlock(dentry->d_sb);
}

void aufs_write_lock(struct dentry *dentry)
{
	si_write_lock(dentry->d_sb);
	di_write_lock(dentry);
}

void aufs_write_unlock(struct dentry *dentry)
{
	di_write_unlock(dentry);
	si_write_unlock(dentry->d_sb);
}

void aufs_read_and_write_lock2(struct dentry *d1, struct dentry *d2, int isdir)
{
	DEBUG_ON(d1 == d2 || d1->d_sb != d2->d_sb);
	si_read_lock(d1->d_sb);
	di_write_lock2(d1, d2, isdir);
}

void aufs_read_and_write_unlock2(struct dentry *d1, struct dentry *d2)
{
	DEBUG_ON(d1 == d2 || d1->d_sb != d2->d_sb);
	di_write_unlock2(d1, d2);
	si_read_unlock(d1->d_sb);
}

/* ---------------------------------------------------------------------- */

struct aufs_pseudo_link {
	struct list_head list;
	struct inode *inode;
};

int is_plinked(struct super_block *sb, struct inode *inode)
{
	int found;
	struct aufs_sbinfo *sbinfo;
	struct list_head *plink_list;
	struct aufs_pseudo_link *plink;

	LKTRTrace("i%lu\n", inode->i_ino);
	SiMustAnyLock(sb);
	DEBUG_ON(!IS_MS(sb, MS_PLINK));

	found = 0;
	sbinfo = stopd(sb);
	plink_list = &sbinfo->si_plink;
	spin_lock(&sbinfo->si_plink_lock);
	list_for_each_entry(plink, plink_list, list)
		if (plink->inode == inode) {
			found = 1;
			break;
		}
	spin_unlock(&sbinfo->si_plink_lock);
	return found;
}

void append_plink(struct super_block *sb, struct inode *inode)
{
	struct aufs_sbinfo *sbinfo;
	struct list_head *plink_list;
	struct aufs_pseudo_link *plink;
	int found, do_warn, cnt;

	LKTRTrace("i%lu\n", inode->i_ino);
	SiMustAnyLock(sb);
	DEBUG_ON(!IS_MS(sb, MS_PLINK));

	cnt = 0;
	found = 0;
	sbinfo = stopd(sb);
	plink_list = &sbinfo->si_plink;
	spin_lock(&sbinfo->si_plink_lock);
	list_for_each_entry(plink, plink_list, list) {
		cnt++;
		if (plink->inode == inode) {
			found = 1;
			break;
		}
	}

	do_warn = 0;
	if (!found) {
		struct aufs_pseudo_link *plink;

		plink = kmalloc(sizeof(*plink), GFP_ATOMIC);
		if (plink) {
			plink->inode = igrab(inode);
			list_add(&plink->list, plink_list);
			cnt++;
		} else
			do_warn = 1;
	}
	spin_unlock(&sbinfo->si_plink_lock);

	if (unlikely(cnt > 100))
		Warn1("unexpectedly many pseudo links, %d\n", cnt);
	if (unlikely(do_warn))
		Warn("err %d, damaged pseudo link\n", -ENOMEM);
}

static void do_put_plink(struct aufs_pseudo_link *plink, int do_del)
{
	TraceEnter();

	iput(plink->inode);
	if (do_del)
		list_del(&plink->list);
	kfree(plink);
}

void put_plink(struct super_block *sb)
{
	struct aufs_sbinfo *sbinfo;
	struct list_head *plink_list;
	struct aufs_pseudo_link *plink, *tmp;

	TraceEnter();
	SiMustWriteLock(sb);
	DEBUG_ON(!IS_MS(sb, MS_PLINK));

	sbinfo = stopd(sb);
	plink_list = &sbinfo->si_plink;
	//spin_lock(&sbinfo->si_plink_lock);
	list_for_each_entry_safe(plink, tmp, plink_list, list)
		do_put_plink(plink, 0);
	INIT_LIST_HEAD(plink_list);
	//spin_unlock(&sbinfo->si_plink_lock);
}

void half_refresh_plink(struct super_block *sb, unsigned int id)
{
	struct aufs_sbinfo *sbinfo;
	struct list_head *plink_list;
	struct aufs_pseudo_link *plink, *tmp;
	struct inode *inode;
	aufs_bindex_t bstart, bend, bindex;
	int do_put;

	TraceEnter();
	SiMustWriteLock(sb);
	DEBUG_ON(!IS_MS(sb, MS_PLINK));

	sbinfo = stopd(sb);
	plink_list = &sbinfo->si_plink;
	//spin_lock(&sbinfo->si_plink_lock);
	list_for_each_entry_safe(plink, tmp, plink_list, list) {
		do_put = 0;
		inode = igrab(plink->inode);
		ii_write_lock(inode);
		bstart = ibstart(inode);
		bend = ibend(inode);
		if (bstart >= 0) {
			for (bindex = bstart; bindex <= bend; bindex++) {
				if (!itohi_index(inode, bindex)
				    || itoid_index(inode, bindex) != id)
					continue;
				set_itohi_index(inode, bindex, NULL, 0);
				do_put = 1;
				break;
			}
		} else
			do_put_plink(plink, 1);

		if (do_put) {
			for (bindex = bstart; bindex <= bend; bindex++)
				if (itohi_index(inode, bindex)) {
					do_put = 0;
					break;
				}
			if (do_put)
				do_put_plink(plink, 1);
		}
		ii_write_unlock(inode);
		iput(inode);
	}
	//spin_unlock(&sbinfo->si_plink_lock);
}

/* ---------------------------------------------------------------------- */

unsigned int new_br_id(struct super_block *sb)
{
	unsigned int id;

	TraceEnter();
	SiMustWriteLock(sb);

	while (1) {
		id = ++stopd(sb)->si_last_br_id;
		if (id && find_brindex(sb, id) < 0)
			return id;
	}
}
