/*
 * Copyright (C) 2005-2006 Junjiro Okajima
 *
 * This program, aufs is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/* $Id: branch.c,v 1.17 2006/08/07 14:49:56 sfjro Exp $ */

//#include <linux/fs.h>
//#include <linux/namei.h>
#include "aufs.h"

static void free_branch(struct aufs_branch *br)
{
	if (br->br_xino)
		fput(br->br_xino);
	if (br->br_wh) {
		BUG(); // not yet
		dput(br->br_wh);
	}
	rw_destroy(&br->br_wh_rwsem);
	mntput(br->br_mnt);
	DEBUG_ON(br_count(br));
	kfree(br);
}

void free_branches(struct aufs_sbinfo *sbinfo)
{
	aufs_bindex_t bmax;
	struct aufs_branch **br;

	TraceEnter();
	bmax = sbinfo->si_bend+1;
	br = sbinfo->si_branch;
	while (bmax--)
		free_branch(*br++);
}

int br_rdonly(struct aufs_branch *br)
{
	return ((br->br_sb->s_flags & MS_RDONLY)
		|| !(br->br_perm & MAY_WRITE))
		?-EROFS:0;
}

/* returns writable bindex, otherwise an error */
static int find_rw_parent(struct dentry *dentry, aufs_bindex_t bend)
{
	aufs_bindex_t bindex;
	struct super_block *sb;
	struct dentry *parent;

	sb = dentry->d_sb;
	parent = dentry->d_parent; // dget_parent()
	for (bindex = dbstart(parent); bindex <= bend; bindex++)
		if (dtohd_index(parent, bindex)
		    && !br_rdonly(stobr(sb, bindex)))
			return bindex;
	return -EROFS;
}

int find_rw_br(struct super_block *sb, aufs_bindex_t bend)
{
	aufs_bindex_t bindex;

	for (bindex = bend; bindex >= 0; bindex--)
		if (!br_rdonly(stobr(sb, bindex)))
			return bindex;
	return -EROFS;
}

int find_rw_parent_br(struct dentry *dentry, aufs_bindex_t bend)
{
	int err;

	err = find_rw_parent(dentry, bend);
	if (err >= 0)
		return err;
	return find_rw_br(dentry->d_sb, bend);
}

/* ---------------------------------------------------------------------- */

/* checks if two hidden_dentries have overlapping branches */
static int do_is_overlap(struct super_block *sb, struct dentry *hidden_d1,
			 struct dentry *hidden_d2)
{
	struct dentry *d;

	d = hidden_d1;
	do {
		if (d == hidden_d2)
			return 1;
		d = d->d_parent; // dget_parent()
	} while (!IS_ROOT(d));

	return 0;
}

#ifdef CONFIG_BLK_DEV_LOOP
#include <linux/loop.h>
static int is_overlap_loopback(struct super_block *sb, struct dentry *hidden_d1,
			       struct dentry *hidden_d2)
{
	struct inode *hidden_inode;
	struct loop_device *l;

	hidden_inode = hidden_d1->d_inode;
	if (MAJOR(hidden_inode->i_sb->s_dev) != LOOP_MAJOR)
		return 0;

	l = hidden_inode->i_sb->s_bdev->bd_disk->private_data;
	hidden_d1 = l->lo_backing_file->f_dentry;
	if (hidden_d1->d_sb == sb)
		return 1;
	return do_is_overlap(sb, hidden_d1, hidden_d2);
}
#else
#define is_overlap_loopback(sb, hidden_d1, hidden_d2) 0
#endif

static int is_overlap(struct super_block *sb, struct dentry *hidden_d1,
		      struct dentry *hidden_d2)
{
	LKTRTrace("d1 %.*s, d2 %.*s\n",
		  DLNPair(hidden_d1), DLNPair(hidden_d2));
	if (hidden_d1 == hidden_d2)
		return 1;
	return do_is_overlap(sb, hidden_d1, hidden_d2)
		|| do_is_overlap(sb, hidden_d2, hidden_d1)
		|| is_overlap_loopback(sb, hidden_d1, hidden_d2)
		|| is_overlap_loopback(sb, hidden_d2, hidden_d1);
}

/* ---------------------------------------------------------------------- */

static struct aufs_branch *alloc_addbr(struct super_block *sb, int new_nbranch)
{
	struct aufs_branch **branchp, *add_branch;
	int sz;
	void *p;
	struct dentry *root, **dentryp;
	struct inode *inode, **inodep;

	LKTRTrace("new_nbranch %d\n", new_nbranch);
	SiMustWriteLock(sb);
	root = sb->s_root;
	DiMustWriteLock(root);
	inode = root->d_inode;
	IiMustWriteLock(inode);

	add_branch = kmalloc(sizeof(*add_branch), GFP_KERNEL);
	//add_branch = NULL;
	if (!add_branch)
		goto out;

	sz = sizeof(*branchp)*(new_nbranch-1);
	p = stopd(sb)->si_branch;
	branchp = kzrealloc(p, sz, sz+sizeof(*branchp));
	if (!branchp)
		goto out;
	DEBUG_ON(sizeof(*branchp) != sizeof(*dentryp)
		 || sizeof(*branchp) != sizeof(*inodep));
	if (branchp == p)
		// re-use the old ones.
		return add_branch; /* success */
	stopd(sb)->si_branch = branchp;

	p = dtopd(root)->di_dentry;
	dentryp = kzrealloc(p, sz, sz+sizeof(*dentryp));
	if (!p)
		goto out;
	dtopd(root)->di_dentry = dentryp;

	p = itopd(inode)->ii_inode;
	inodep = kzrealloc(p, sz, sz+sizeof(*inodep));
	if (!p)
		goto out;
	itopd(inode)->ii_inode = inodep;
	return add_branch; /* success */

 out:
	kfree(add_branch);
	TraceErr(-ENOMEM);
	return ERR_PTR(-ENOMEM);
}

static int test_br(struct super_block *sb, struct inode *inode,
		   unsigned int perm, char *path)
{
	int err;

	err = 0;
	if ((perm & MAY_WRITE) && IS_RDONLY(inode)) {
		Err("writable flag for readonly fs or inode, %s\n", path);
		err = -EINVAL;
	}

	TraceErr(err);
	return err;
}

/* retunrs,,,
 * 0: success, the caller will add it
 * plus: success, it is already unified, the caller should ignore it
 * minus: error
 */
static int test_add(struct super_block *sb, struct opt_add *add, int remount)
{
	int err;
	struct dentry *root;
	struct inode *inode, *hidden_inode;
	aufs_bindex_t bend, bindex;

	LKTRTrace("%s, remo%d\n", add->path, remount);

	root = sb->s_root;
	if (find_dbindex(root, add->nd.dentry) != -1) {
		err = 1;
		if (!remount) {
			err = -EINVAL;
			Err("%s duplicated\n", add->path);
		}
		goto out;
	}

	err = -ENOSPC;
	bend = sbend(sb);
	//bend = AUFS_BRANCH_MAX;
	if (AUFS_BRANCH_MAX <= bend) {
		Err("number of branches exceeded %s\n", add->path);
		goto out;
	}

	err = -EDOM;
	if (add->bindex < 0 || bend+1 < add->bindex) {
		Err("bad index %d\n", add->bindex);
		goto out;
	}

	err = -ENOENT;
	inode = add->nd.dentry->d_inode;
	if (!inode || !inode->i_nlink) {
		Err("no existence %s\n", add->path);
		goto out;
	}

	err = -ENOTDIR;
	if (!S_ISDIR(inode->i_mode)) {
		Err("not dir %s\n", add->path);
		goto out;
	}

	err = -EINVAL;
	if (inode->i_sb == sb) {
		Err("%s must be outside\n", add->path);
		goto out;
	}

	if (!strcmp(sbt(inode->i_sb), AUFS_FSTYPE)
#ifdef CONFIG_AUFS_COMPAT
	    || !strcmp(sbt(inode->i_sb), AUFS_NAME)
#endif
		) {
		Err("nested " AUFS_NAME " %s\n", add->path);
		goto out;
	}

	if (!add->bindex && !(add->perm & MAY_WRITE)
	    && !(sb->s_flags & MS_RDONLY))
		Warn("first branch should be rw, %s\n", add->path);

	err = test_br(sb, add->nd.dentry->d_inode, add->perm, add->path);
	if (err)
		goto out;

	if (bend == -1)
		return 0; /* success */

	hidden_inode = dtohd(root)->d_inode;
	if ((hidden_inode->i_mode & S_IALLUGO) != (inode->i_mode & S_IALLUGO)
	    || hidden_inode->i_uid != inode->i_uid
	    || hidden_inode->i_gid != inode->i_gid)
		Warn("different uid/gid/permission, %s\n", add->path);

	err = -EINVAL;
	for (bindex = 0; bindex <= bend; bindex++)
		if (is_overlap(sb, add->nd.dentry, dtohd_index(root, bindex))) {
			Err("%s is overlapped\n", add->path);
			goto out;
		}
	err = 0;

 out:
	TraceErr(err);
	return err;
}

int br_add(struct super_block *sb, struct opt_add *add, int remount)
{
	int err, sz;
	aufs_bindex_t bend, add_bindex;
	struct dentry *root;
	struct aufs_iinfo *iinfo;
	struct aufs_sbinfo *sbinfo;
	struct aufs_dinfo *dinfo;
	struct inode *root_inode;
	unsigned long long maxb;
	struct aufs_branch **branchp, *add_branch;
	struct dentry **dentryp;
	struct inode **inodep;

	LKTRTrace("b%d, %s, 0x%x, %.*s\n", add->bindex, add->path,
		  add->perm, DLNPair(add->nd.dentry));
	SiMustWriteLock(sb);
	root = sb->s_root;
	DiMustWriteLock(root);
	root_inode = root->d_inode;
	IiMustWriteLock(root_inode);

	err = test_add(sb, add, remount);
	if (err < 0)
		goto out;
	if (err)
		return 0; /* success */

	bend = sbend(sb);
	add_branch = alloc_addbr(sb, bend+2);
	err = PTR_ERR(add_branch);
	if (IS_ERR(add_branch))
		goto out;
	if (!(add->perm & MAY_WRITE)) {
		rw_init_nolock(&add_branch->br_wh_rwsem);
		add_branch->br_wh = NULL;
	} else {
		i_lock(add->nd.dentry->d_inode);
		rw_init_wlock(&add_branch->br_wh_rwsem);
		err = init_wh(add->nd.dentry, add_branch);
		rw_write_unlock(&add_branch->br_wh_rwsem);
		i_unlock(add->nd.dentry->d_inode);
		if (err) {
			kfree(add_branch);
			goto out;
		}
	}
	add_branch->br_sb = add->nd.dentry->d_sb;
	add_branch->br_mnt = mntget(add->nd.mnt);
	atomic_set(&add_branch->br_count, 0);
	add_branch->br_perm = add->perm;
	add_branch->br_xino = NULL;
	err = 0;

	sbinfo = stopd(sb);
	dinfo = dtopd(root);
	iinfo = itopd(root_inode);

	add_bindex = add->bindex;
	sz = sizeof(*(sbinfo->si_branch))*(bend+1-add_bindex);
	branchp = sbinfo->si_branch;
	memmove(branchp+add_bindex+1, branchp+add_bindex, sz);
	dentryp = dinfo->di_dentry;
	memmove(dentryp+add_bindex+1, dentryp+add_bindex, sz);
	inodep = iinfo->ii_inode;
	memmove(inodep+add_bindex+1, inodep+add_bindex, sz);

	branchp[0+add_bindex] = add_branch;
	dentryp[0+add_bindex] = NULL;
	inodep[0+add_bindex] = NULL;

	sbinfo->si_bend++;
	dinfo->di_bend++;
	iinfo->ii_bend++;
	if (bend == -1) {
		dinfo->di_bstart = 0;
		iinfo->ii_bstart = 0;
	}
	set_dtohd_index(root, add_bindex, dget(add->nd.dentry));
	set_itohi_index(root_inode, add_bindex, igrab(add->nd.dentry->d_inode));
	if (!add_bindex)
		cpup_attr_all(root_inode);

	maxb = add->nd.dentry->d_sb->s_maxbytes;
	if (sb->s_maxbytes < maxb)
		sb->s_maxbytes = maxb;
	sigen_inc(sb);

	if (IS_MS(sb, MS_XINO)) {
		struct file *base_file = stobr(sb, 0)->br_xino;
		if (!add_bindex)
			base_file = stobr(sb, 1)->br_xino;
		err = xino_init(sb, add_bindex, base_file, /*do_test*/1);
		if (err) {
			DEBUG_ON(add_branch->br_xino);
			Err("xino err %d\n", err);
			// ignore this error?
		}
	}

 out:
	TraceErr(err);
	return err;
}

/* ---------------------------------------------------------------------- */

/* copied from fs/dcachs.c d_genocide() */
static int test_children(struct dentry *root, aufs_bindex_t bindex)
{
	int ret = 0;
	struct dentry *this_parent = root;
	struct list_head *next;

	// sbinfo is locked
	//spin_lock(&dcache_lock);
 repeat:
	next = this_parent->d_subdirs.next;
 resume:
	while (next != &this_parent->d_subdirs) {
		struct list_head *tmp = next;
		struct dentry *dentry = list_entry(tmp, struct dentry, d_u.d_child);

		next = tmp->next;
		di_read_lock(dentry, AUFS_I_RLOCK);
		if (!dtohd_index(dentry, bindex)
		    || d_unhashed(dentry)
		    || !dentry->d_inode) {
			di_read_unlock(dentry, AUFS_I_RLOCK);
			continue;
		}
		if (!list_empty(&dentry->d_subdirs)) {
			this_parent = dentry;
			di_read_unlock(dentry, AUFS_I_RLOCK);
			goto repeat;
		}
		if (!atomic_read(&dentry->d_count)
		    || (S_ISDIR(dentry->d_inode->i_mode)
			&& dbstart(dentry) != dbend(dentry))) {
			di_read_unlock(dentry, AUFS_I_RLOCK);
			continue;
		}
		//atomic_dec(&dentry->d_count);
		//LKTRTrace("%.*s\n", DLNPair(dentry));
		ret = 1;
		di_read_unlock(dentry, AUFS_I_RLOCK);
		goto out;
	}
	if (this_parent != root) {
		next = this_parent->d_u.d_child.next;
		di_read_lock(this_parent, AUFS_I_RLOCK);
		//atomic_dec(&this_parent->d_count);
		if (dtohd_index(this_parent, bindex)
		    && atomic_read(&this_parent->d_count)
		    && (!S_ISDIR(this_parent->d_inode->i_mode)
			|| dbstart(this_parent) == dbend(this_parent))) {
			ret = 1;
			di_read_unlock(this_parent, AUFS_I_RLOCK);
			//LKTRTrace("%.*s\n", DLNPair(this_parent));
			goto out;
		}
		di_read_unlock(this_parent, AUFS_I_RLOCK);
		this_parent = this_parent->d_parent;
		goto resume;
	}

 out:
	//spin_unlock(&dcache_lock);
	TraceErr(ret);
	return ret;
}

int br_del(struct super_block *sb, struct opt_del *del, int remount)
{
	int err, do_wh, rerr;
	struct dentry *root;
	struct inode *inode, *hidden_dir;
	aufs_bindex_t bindex, bend;
	struct aufs_sbinfo *sbinfo;
	struct aufs_dinfo *dinfo;
	struct aufs_iinfo *iinfo;
	struct aufs_branch *br;

	LKTRTrace("%s, %.*s\n", del->path, DLNPair(del->hidden_root));
	SiMustWriteLock(sb);
	root = sb->s_root;
	DiMustWriteLock(root);
	inode = root->d_inode;
	IiMustWriteLock(inode);

	bindex = find_dbindex(root, del->hidden_root);
	if (bindex < 0) {
		if (remount)
			return 0; /* success */
		err = -ENOENT;
		Err("%s no such branch\n", del->path);
		goto out;
	}
	LKTRTrace("bindex b%d\n", bindex);

	err = -EBUSY;
	bend = sbend(sb);
	br = stobr(sb, bindex);
	if (!bend || br_count(br))
		goto out;

	do_wh = 0;
	if (br->br_wh) {
		BUG(); // not yet
		//no need to lock br_wh_rwsem ??
		dput(br->br_wh);
		br->br_wh = NULL;
		do_wh = 1;
	}

	if (test_children(root, bindex)) {
		if (do_wh)
			goto out_wh;
		goto out;
	}
	err = 0;

	if (sb->s_maxbytes == del->hidden_root->d_sb->s_maxbytes) {
		aufs_bindex_t bi;
		sb->s_maxbytes = 0;
		for (bi = 0; bi <= bend; bi++) {
			struct super_block *hsb;
			if (bi == bindex)
				continue;
			hsb = dtohd_index(root, bi)->d_sb;
			if (sb->s_maxbytes < hsb->s_maxbytes)
				sb->s_maxbytes = hsb->s_maxbytes;
		}
	}

	sbinfo = stopd(sb);
	dinfo = dtopd(root);
	iinfo = itopd(inode);

	dput(dtohd_index(root, bindex));
	//truncate_inode_pages(itohi_index(inode, bindex)->i_mapping, 0);
	iput(itohi_index(inode, bindex));
	free_branch(sbinfo->si_branch[bindex+0]);

	//todo: realloc and shrink memeory
	if (bindex < bend) {
		const aufs_bindex_t n = bend-bindex;
		memmove(sbinfo->si_branch+bindex, sbinfo->si_branch+bindex+1,
			sizeof(*sbinfo->si_branch)*n);
		memmove(dinfo->di_dentry+bindex, dinfo->di_dentry+bindex+1,
			sizeof(*dinfo->di_dentry)*n);
		memmove(iinfo->ii_inode+bindex, iinfo->ii_inode+bindex+1,
			sizeof(*iinfo->ii_inode)*n);
	}
	sbinfo->si_branch[0+bend] = NULL;
	dinfo->di_dentry[0+bend] = NULL;
	iinfo->ii_inode[0+bend] = NULL;

	sbinfo->si_bend--;
	dinfo->di_bend--;
	iinfo->ii_bend--;
	sigen_inc(sb);
	if (!bindex)
		cpup_attr_all(inode);
	goto out; /* success */

 out_wh:
	/* revert */
	BUG(); // not yet
	hidden_dir = del->hidden_root->d_inode;
	i_lock(hidden_dir);
	rw_write_lock(&br->br_wh_rwsem);
	rerr = init_wh(del->hidden_root, br);
	rw_write_unlock(&br->br_wh_rwsem);
	i_unlock(hidden_dir);
	if (rerr) {
		IOErr("failed re-creating base whiteout, %s. "
		      "forcing readonly. (%d)\n", del->path, rerr);
		br->br_perm &= ~MAY_WRITE;
		LKTRTrace("perm 0x%x\n", br->br_perm);
	}
 out:
	TraceErr(err);
	return err;
}

int br_mod(struct super_block *sb, struct opt_mod *mod, int remount)
{
	int err;
	struct dentry *root;
	aufs_bindex_t bindex;
	struct aufs_branch *br;

	LKTRTrace("%s, %.*s, 0x%x\n",
		  mod->path, DLNPair(mod->hidden_root), mod->perm);
	SiMustWriteLock(sb);
	root = sb->s_root;
	DiMustWriteLock(root);
	IiMustWriteLock(root->d_inode);

	bindex = find_dbindex(root, mod->hidden_root);
	if (bindex < 0) {
		if (remount)
			return 0; /* success */
		err = -ENOENT;
		Err("%s no such branch\n", mod->path);
		goto out;
	}
	LKTRTrace("bindex b%d\n", bindex);

	err = test_br(sb, mod->hidden_root->d_inode, mod->perm, mod->path);
	if (err)
		goto out;

	br = stobr(sb, bindex);
	if (br->br_perm == mod->perm)
		return 0; /* success */

	if (br->br_perm & MAY_WRITE) {
		if (!(mod->perm & MAY_WRITE)) {
			/* rw --> ro, file might be mmapped */
			struct file *file, *hf;

			if (br->br_wh) {
				BUG(); // not yet
				rw_write_lock(&br->br_wh_rwsem);
				DEBUG_ON(!br->br_wh);
				dput(br->br_wh);
				br->br_wh = NULL;
				rw_write_unlock(&br->br_wh_rwsem);
			}

#if 1
			// no need file_list_lock() since sbinfo is locked
			list_for_each_entry(file, &sb->s_files, f_u.fu_list) {
				LKTRTrace("%.*s\n", DLNPair(file->f_dentry));
				fi_read_lock(file);
				if (!S_ISREG(file->f_dentry->d_inode->i_mode)
				    || !(file->f_mode & FMODE_WRITE)
				    || fbstart(file) != bindex) {
					fi_read_unlock(file);
					continue;
				}

				hf = ftohf(file);
				hf->f_flags = file_roflags(hf->f_flags);
				hf->f_mode &= ~FMODE_WRITE;
				fi_read_unlock(file);
			}
#endif
		}
	} else if (mod->perm & MAY_WRITE && !br->br_wh) {
		i_lock(mod->hidden_root->d_inode);
		rw_write_lock(&br->br_wh_rwsem);
		err = init_wh(mod->hidden_root, br);
		rw_write_unlock(&br->br_wh_rwsem);
		i_unlock(mod->hidden_root->d_inode);
		if (err)
			goto out;
	}

	set_sbr_perm(sb, bindex, mod->perm);
	// no need to inc?
	//sigen_inc(sb);
	return 0; /* success */

 out:
	TraceErr(err);
	return err;
}
