/*
 * Copyright (C) 2005-2006 Junjiro Okajima
 *
 * This program, aufs is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/* $Id: inode.h,v 1.6 2006/07/24 06:38:16 sfjro Exp $ */

#ifndef __AUFS_INODE_H__
#define __AUFS_INODE_H__

#include <linux/fs.h>
#include <linux/aufs_type.h>
#include "misc.h"

#ifdef __KERNEL__

struct aufs_vdir;
struct aufs_iinfo {
	struct aufs_rwsem	ii_rwsem;
	struct inode		**ii_inode;
	aufs_bindex_t		ii_bstart, ii_bend;
	struct aufs_vdir	*ii_vdir; // i_mapping and page cache?
};

struct aufs_icntnr {
	struct aufs_iinfo iinfo;
	struct inode vfs_inode;
};

#define i_lock(i)		mutex_lock(&(i)->i_mutex)
#define i_unlock(i)		mutex_unlock(&(i)->i_mutex)
#define IMustLock(i)		DEBUG_ON(!mutex_is_locked(&(i)->i_mutex))

#define ii_read_lock(i)		rw_read_lock(&itopd(i)->ii_rwsem)
#define ii_read_unlock(i)	rw_read_unlock(&itopd(i)->ii_rwsem)
#define ii_downgrade_lock(i)	rw_dgrade_lock(&itopd(i)->ii_rwsem)
#define ii_write_lock(i)	rw_write_lock(&itopd(i)->ii_rwsem)
#define ii_write_unlock(i)	rw_write_unlock(&itopd(i)->ii_rwsem)
#define IiMustReadLock(i)	do{ \
	SiMustAnyLock((i)->i_sb); \
	RwMustReadLock(&itopd(i)->ii_rwsem); \
	}while(0)
#define IiMustWriteLock(i)	do{ \
	SiMustAnyLock((i)->i_sb); \
	RwMustWriteLock(&itopd(i)->ii_rwsem); \
	}while(0)
#define IiMustAnyLock(i)	do{ \
	SiMustAnyLock((i)->i_sb); \
	RwMustAnyLock(&itopd(i)->ii_rwsem); \
	}while(0)

/* ---------------------------------------------------------------------- */

//inode.c
int refresh_inode(struct dentry *dentry);
struct inode *aufs_new_inode(struct dentry *dentry);

//i_op.c
extern struct inode_operations aufs_iop;
extern struct inode_operations aufs_symlink_iop;
extern struct inode_operations aufs_dir_iop;
int wr_dir(struct dentry *dentry, int negative, struct dentry *src_dentry,
	   aufs_bindex_t force_btgt, int do_lock_srcdir);

//i_op_add.c
int aufs_mknod(struct inode *dir, struct dentry *dentry, int mode, dev_t dev);
int aufs_symlink(struct inode *dir, struct dentry *dentry, const char *symname);
int aufs_create(struct inode *dir, struct dentry *dentry, int mode,
		struct nameidata *nd);
int aufs_link(struct dentry *src_dentry, struct inode *dir,
	      struct dentry *dentry);
int aufs_mkdir(struct inode *dir, struct dentry *dentry, int mode);

//i_op_del.c
int wr_dir_need_wh(struct dentry *dentry, int isdir, aufs_bindex_t *bcpup,
		   struct dentry *locked);
int aufs_unlink(struct inode *dir, struct dentry *dentry);
int aufs_rmdir(struct inode *dir, struct dentry *dentry);

// i_op_ren.c
int aufs_rename(struct inode *src_dir, struct dentry *src_dentry,
		struct inode *dir, struct dentry *dentry);

//iinfo.c
struct aufs_iinfo *itopd(struct inode *inode);
aufs_bindex_t ibstart(struct inode *inode);
aufs_bindex_t ibend(struct inode *inode);
struct aufs_vdir *ivdir(struct inode *inode);
struct inode *itohi_index(struct inode *inode, aufs_bindex_t bindex);
struct inode *itohi(struct inode *inode);

void set_ibstart(struct inode *inode, aufs_bindex_t bindex);
void set_ibend(struct inode *inode, aufs_bindex_t bindex);
void set_ivdir(struct inode *inode, struct aufs_vdir *vdir);
void set_itohi_index(struct inode *inode, aufs_bindex_t bindex,
		     struct inode *val);

int iinfo_init(struct inode *inode);
void iinfo_fin(struct inode *inode);

#endif /* __KERNEL__ */
#endif /* __AUFS_INODE_H__ */
