/*
 * Copyright (C) 2005-2006 Junjiro Okajima
 *
 * This program, aufs is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/* $Id: super.h,v 1.12 2006/08/07 14:51:58 sfjro Exp $ */

#ifndef __AUFS_SUPER_H__
#define __AUFS_SUPER_H__

#include <linux/fs.h>
#include <linux/aufs_type.h>
#include "misc.h"

#define AUFS_ROOT_INO		2
#define AUFS_FIRST_INO		11
#define AUFS_XINO_FNAME		".aufs.xino"
#define AUFS_XINO_DEFPATH	"/tmp/" AUFS_XINO_FNAME
#define AUFS_DIRWH		3
#define AUFS_RDCACHE		10 // seconds

#ifdef __KERNEL__
#include "branch.h"

struct aufs_sbinfo {
	struct aufs_rwsem	si_rwsem;
	struct aufs_branch	**si_branch;
	aufs_bindex_t		si_bend;

	/* set true when refresh_dirs() at remount time failed.
	 * then try refreshing dirs at access time again.
	 * if it is false, refreshing dirs at access time is unnecesary
	 */
	unsigned int		si_failed_refresh_dirs:1;

	unsigned int		si_flags;
	int			si_generation; // try root inode i_version

	/* external inode number table */
	struct mutex		si_xino_mutex;
	ino_t			si_xino;	// time bomb
	//struct file		*si_xino_bmap;

	/*
	 * if si_dirwh is,
	 * zero or minus: remove all whiteouts and rmdir in rmdir(2).
	 * plus: if the number of whiteouts are larger than dirwh, leave all of
	 *	 them after rename_whtmp to reduce the cost of rmdir(2).
	 *	 future fsck.aufs or kernel thread will remove them later.
	 */
	int			si_dirwh;

	/* readdir cache time, max, in HZ */
	unsigned long		si_rdcache;

	char			*si_dupopt;
};

#define sbt(sb)		({(sb)->s_type->name;})
#define SB_NFS(sb)	(!strcmp(sbt(sb), "nfs"))
#define SB_AUFS(sb)	(!strcmp(sbt(sb), AUFS_FSTYPE))

/* see linux/include/linux/jiffies.h */
#define AufsGenYounger(a,b)	((b) - (a) < 0)
#define AufsGenOlder(a,b)	AufsGenYounger(b,a)

/* flags for aufs_read_lock()/di_read_lock() */
#define AUFS_D_WLOCK		1
#define AUFS_I_RLOCK		2
#define AUFS_I_WLOCK		4

/* lock superblock. mainly for entry point functions */
#define si_read_lock(sb)	rw_read_lock(&stopd(sb)->si_rwsem)
#define si_read_unlock(sb)	rw_read_unlock(&stopd(sb)->si_rwsem)
#define si_write_lock(sb)	rw_write_lock(&stopd(sb)->si_rwsem)
#define si_write_unlock(sb)	rw_write_unlock(&stopd(sb)->si_rwsem)
#define SiMustReadLock(sb)	RwMustReadLock(&stopd(sb)->si_rwsem)
#define SiMustWriteLock(sb)	RwMustWriteLock(&stopd(sb)->si_rwsem)
#define SiMustAnyLock(sb)	RwMustAnyLock(&stopd(sb)->si_rwsem)

/* Mount time flags */
#define MS_XINO			1
#define MS_ZXINO		(1<<1)
#define MS_SET(sb, flg)		do{stopd(sb)->si_flags |= flg;}while(0)
#define MS_CLR(sb, flg)		do{stopd(sb)->si_flags &= ~(flg);}while(0)
#define IS_MS(sb, flg)		({stopd(sb)->si_flags & flg;})

/* ---------------------------------------------------------------------- */

/* Superblock to branch */
#define sbr_sb(sb, bindex)	({stobr(sb, bindex)->br_sb;})
#define sbr_mnt(sb, bindex)	({stobr(sb, bindex)->br_mnt;})
#define sbr_count(sb, bindex)	br_count(stobr(sb, bindex))
#define sbr_get(sb, bindex)	br_get(stobr(sb, bindex))
#define sbr_put(sb, bindex)	br_put(stobr(sb, bindex))
#define sbr_perm(sb, bindex) 	({stobr(sb, bindex)->br_perm;})
#define set_sbr_perm(sb, bindex, perm) \
				do{stobr(sb, bindex)->br_perm = perm;}while(0)
#define sbr_perm_str(sb, bindex, str, len) \
				br_perm_str(str, len, stobr(sb, bindex)->br_perm)

/* ---------------------------------------------------------------------- */

/* kmem cache */
enum {AUFS_CACHE_DINFO, AUFS_CACHE_ICNTNR, AUFS_CACHE_FINFO, AUFS_CACHE_VDIR,
      AUFS_CACHE_DEHSTR, _AufsCacheLast};
extern kmem_cache_t *aufs_cachep[];
#define do_cache_alloc(i)	kmem_cache_alloc(aufs_cachep[i], SLAB_KERNEL)
#define cache_alloc_dinfo() 	do_cache_alloc(AUFS_CACHE_DINFO)
#define cache_alloc_icntnr() 	do_cache_alloc(AUFS_CACHE_ICNTNR)
#define cache_alloc_finfo() 	do_cache_alloc(AUFS_CACHE_FINFO)
#define cache_alloc_vdir() 	do_cache_alloc(AUFS_CACHE_VDIR)
#define cache_alloc_dehstr() 	do_cache_alloc(AUFS_CACHE_DEHSTR)
#define do_cache_free(i, p)	kmem_cache_free(aufs_cachep[i], p)
#define cache_free_dinfo(p)	do_cache_free(AUFS_CACHE_DINFO, p)
#define cache_free_icntnr(p)	do_cache_free(AUFS_CACHE_ICNTNR, p)
#define cache_free_finfo(p)	do_cache_free(AUFS_CACHE_FINFO, p)
#define cache_free_vdir(p)	do_cache_free(AUFS_CACHE_VDIR, p)
#define cache_free_dehstr(p)	do_cache_free(AUFS_CACHE_DEHSTR, p)

/* ---------------------------------------------------------------------- */

//xino.c
int xino_write(struct super_block *sb, aufs_bindex_t bindex,
	       ino_t hidden_ino, ino_t ino);
int xino_read(struct super_block *sb, aufs_bindex_t bindex,
	      ino_t hidden_ino, ino_t *ino, int force);
int xino_init(struct super_block *sb, aufs_bindex_t bindex,
	      struct file *base_file, int do_test);
int xino_set(struct super_block *sb, struct opt_xino *xino, int remount);
int xino_clr(struct super_block *sb);
struct file *xino_def(struct super_block *sb);

//sbinfo.c
struct aufs_sbinfo *stopd(struct super_block *sb);
aufs_bindex_t sbend(struct super_block *sb);
struct aufs_branch *stobr(struct super_block *sb, aufs_bindex_t bindex);
int sigen(struct super_block *sb);
int sigen_inc(struct super_block *sb);
aufs_bindex_t find_bindex(struct super_block *sb, struct aufs_branch *br);

void aufs_read_lock(struct dentry *dentry, int flags);
void aufs_read_unlock(struct dentry *dentry, int flags);
void aufs_write_lock(struct dentry *dentry);
void aufs_write_unlock(struct dentry *dentry);
void aufs_read_and_write_lock2(struct dentry *d1, struct dentry *d2, int isdir);
void aufs_read_and_write_unlock2(struct dentry *d1, struct dentry *d2);

#endif /* __KERNEL__ */
#endif /* __AUFS_SUPER_H__ */
