/*
 * Copyright (C) 2005-2006 Junjiro Okajima
 *
 * This program, aufs is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/* $Id: xino.c,v 1.1 2006/07/17 02:30:32 sfjro Exp $ */

#include "aufs.h"

int xino_write(struct super_block *sb, aufs_bindex_t bindex, ino_t hidden_ino,
	       ino_t ino)
{
	struct aufs_branch *br;
	loff_t pos;
	ssize_t sz;

	LKTRTrace("b%d, hi%lu, i%lu\n", bindex, hidden_ino, ino);
	DEBUG_ON(!ino);
	if (!IS_MS(sb, MS_XINO))
		return 0;

	br = stobr(sb, bindex);
	DEBUG_ON(!br || !br->br_xino);
	pos = hidden_ino*sizeof(hidden_ino);
	sz = aufs_fwrite(br->br_xino, &ino, sizeof(ino), &pos);
	//if (LktrCond) sz = 1;
	if (sz == sizeof(ino))
		return 0; /* success */

	IOErr("write failed (%d)\n", sz);
	return -EIO;
}

int xino_read(struct super_block *sb, aufs_bindex_t bindex, ino_t hidden_ino,
	      ino_t *ino, int force)
{
	struct file *file;
	loff_t pos, pos2;
	ssize_t sz;
	struct aufs_sbinfo *si;

	LKTRTrace("b%d, hi%lu, force %d\n", bindex, hidden_ino, force);
	if (!IS_MS(sb, MS_XINO)) {
		*ino = iunique(sb, AUFS_FIRST_INO);
		return 0;
	}

	file = stobr(sb, bindex)->br_xino;
	DEBUG_ON(!file);
	sz = 0;
	pos2 = pos = hidden_ino*sizeof(hidden_ino);
	if (i_size_read(file->f_dentry->d_inode) >= pos+sizeof(*ino)) {
		sz = aufs_fread(file, ino, sizeof(*ino), &pos);
		//if (LktrCond) sz = 1;
		if (sz == sizeof(*ino) && *ino)
			return 0; /* success */
	}
	if (!force)
		goto out;

	if (!sz || sz == sizeof(*ino)) {
		si = stopd(sb);
		mutex_lock(&si->si_xino_mutex);
		*ino = si->si_xino++;
		mutex_unlock(&si->si_xino_mutex);
		if (*ino >= AUFS_FIRST_INO) {
			sz = aufs_fwrite(file, ino, sizeof(*ino), &pos2);
			//if (LktrCond) sz = 1;
			if (sz == sizeof(ino))
				return 0; /* success */
			*ino = 0;
			IOErr("write failed (%d)\n", sz);
			return -EIO;
		} else {
			*ino = 0;
			IOErr("inode number overflow\n");
			return -EIO;
		}
	}
	IOErr("read failed (%d)\n", sz);
 out:
	return -EIO;
}

static aufs_bindex_t is_sb_shared(struct super_block *sb, aufs_bindex_t btgt,
				  aufs_bindex_t bend)
{
	aufs_bindex_t bindex;
	struct super_block *tgt_sb = sbr_sb(sb, btgt);

	for (bindex = 0; bindex <= bend; bindex++)
		if (btgt != bindex && tgt_sb == sbr_sb(sb, bindex))
			return bindex;
	return -1;
}

static struct file *xino_create2(struct file *base_file)
{
	struct file *file;
	int err;
	struct dentry *base, *dentry, *parent;
	struct inode *dir;
	struct qstr *name;

	base = base_file->f_dentry;
	LKTRTrace("%.*s\n", DLNPair(base));
	parent = base->d_parent;
	dir = parent->d_inode;
	IMustLock(dir);

	name = &base->d_name;
	dentry = lookup_one_len(name->name, parent, name->len);
	//dentry = ERR_PTR(-1);
	if (IS_ERR(dentry)) {
		file = (void*)dentry;
		Err("%.*s lookup err %ld\n", LNPair(name), PTR_ERR(dentry));
		goto out;
	}
	err = vfs_create(dir, dentry, S_IRUGO|S_IWUGO, NULL);
	//err = -1;
	if (err) {
		file = ERR_PTR(err);
		Err("%.*s create err %d\n", LNPair(name), err);
		goto out_dput;
	}
	file = dentry_open(dget(dentry), mntget(base_file->f_vfsmnt),
			   O_RDWR|O_CREAT|O_EXCL|O_LARGEFILE);
	//file = ERR_PTR(-1);
	if (IS_ERR(file)) {
		Err("%.*s open err %ld\n", LNPair(name), PTR_ERR(file));
		goto out_dput;
	}
	err = safe_unlink(dir, dentry, NULL);
	//err = -1;
	if (!err)
		goto out_dput; /* success */

	Err("%.*s unlink err %d\n", LNPair(name), err);
	fput(file);
	file = ERR_PTR(err);

 out_dput:
	dput(dentry);
 out:
	TraceErrPtr(file);
	return file;
}

int xino_init(struct super_block *sb, aufs_bindex_t bindex,
	      struct file *base_file, int do_test)
{
	int err;
	struct aufs_branch *br;
	aufs_bindex_t bshared, bend;
	struct file *file;
	struct inode *inode, *hidden_inode;

	LKTRTrace("b%d, base_file %p, do_test %d\n",
		  bindex, base_file, do_test);
	SiMustWriteLock(sb);
	DEBUG_ON(!IS_MS(sb, MS_XINO));
	br = stobr(sb, bindex);
	DEBUG_ON(br->br_xino);

	bend = sbend(sb);
	bshared = -1;
	if (do_test)
		bshared = is_sb_shared(sb, bindex, bend);
	if (bshared != -1) {
		struct aufs_branch *shared_br;
		shared_br = stobr(sb, bshared);
		DEBUG_ON(!shared_br || !shared_br->br_xino);
		file = shared_br->br_xino;
		get_file(file);
	} else {
		struct dentry *parent = dget_parent(base_file->f_dentry);
		i_lock(parent->d_inode);
		file = xino_create2(base_file);
		//file = ERR_PTR(-1);
		i_unlock(parent->d_inode);
		dput(parent);
		err = PTR_ERR(file);
		if (IS_ERR(file))
			goto out;
	}
	br->br_xino = file;
	inode = sb->s_root->d_inode;
	hidden_inode = itohi_index(inode, bindex);
	err = xino_write(sb, bindex, hidden_inode->i_ino, inode->i_ino);
	//err = -1;
	if (!err)
		return 0; /* success */

	fput(file);
	br->br_xino = NULL;

 out:
	TraceErr(err);
	return err;
}

int xino_set(struct super_block *sb, struct opt_xino *xino, int remount)
{
	int err;
	aufs_bindex_t bindex, bend;
	struct aufs_branch *br;
	struct dentry *parent;
	struct qstr *name;
	struct vfsmount *mnt;

	LKTRTrace("%s\n", xino->path);

	err = 0;
	mnt = xino->file->f_vfsmnt;
	name = &xino->file->f_dentry->d_name;
	parent = dget_parent(xino->file->f_dentry);
	if (remount
	    && stobr(sb, 0)->br_xino
	    && stobr(sb, 0)->br_xino->f_dentry->d_parent == parent)
		goto out;

	MS_SET(sb, MS_XINO);
	bend = sbend(sb);
	for (bindex = 0; bindex <= bend; bindex++) {
		struct file *file;
		struct inode *inode;

		br = stobr(sb, bindex);
		if (!br->br_xino)
			continue;

		if (file_count(br->br_xino) > 1) {
			fput(br->br_xino);
			br->br_xino = NULL;
			continue;
		}

		DEBUG_ON(file_count(br->br_xino) != 1);
		i_lock(parent->d_inode);
		file = xino_create2(xino->file);
		//file = ERR_PTR(-1);
		i_unlock(parent->d_inode);
		err = PTR_ERR(file);
		if (IS_ERR(file))
			break;
		inode = br->br_xino->f_dentry->d_inode;
		err = copy_file(file, br->br_xino, i_size_read(inode));
		//err = -1;
		if (err) {
			fput(file);
			break;
		}
		fput(br->br_xino);
		br->br_xino = file;
	}

	for (bindex = 0; bindex <= bend; bindex++)
		if (!stobr(sb, bindex)->br_xino) {
			err = xino_init(sb, bindex, xino->file, /*do_test*/1);
			//err = -1;
			if (!err)
				continue;
			IOErr("creating xino for branch %d(%d)\n", bindex, err);
			err = -EIO;
		}
 out:
	dput(parent);
	TraceErr(err);
	return err;
}

int xino_clr(struct super_block *sb)
{
	aufs_bindex_t bindex, bend;

	TraceEnter();
	SiMustWriteLock(sb);

	bend = sbend(sb);
	for (bindex = 0; bindex <= bend; bindex++) {
		struct aufs_branch *br;
		br = stobr(sb, bindex);
		if (br->br_xino) {
			fput(br->br_xino);
			br->br_xino = NULL;
		}
	}

	MS_CLR(sb, MS_XINO);
	return 0;
}

struct file *xino_def(struct super_block *sb)
{
	struct file *file;
	aufs_bindex_t bend, bindex, bwr;
	char *page, *p;

	bend = sbend(sb);
	bwr = -1;
	for (bindex = 0; bwr == -1 && bindex <= bend; bindex++)
		if (sbr_perm(sb, bindex) & MAY_WRITE)
			bwr = bindex;

	if (bwr != -1) {
		file = ERR_PTR(-ENOMEM);
		page = __getname();
		//page = NULL;
		if (!page)
			goto out;
		p = d_path(dtohd(sb->s_root), sbr_mnt(sb, 0), page,
			   PATH_MAX-sizeof(AUFS_XINO_FNAME));
		//p = ERR_PTR(-1);
		file = (void*)p;
		if (p && !IS_ERR(p)){
			strcat(p, "/" AUFS_XINO_FNAME);
			LKTRTrace("%s\n", p);
			file = xcreate_unlink(sb, p, /*silent*/0);
			//file = ERR_PTR(-1);
		}
		__putname(page);
	} else
		file = xcreate_unlink(sb, AUFS_XINO_DEFPATH, /*silent*/0);
	//file = ERR_PTR(-1);

 out:
	TraceErrPtr(file);
	return file;
}
