/*
 * Copyright (C) 2005-2006 Junjiro Okajima
 *
 * This program, aufs is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/* $Id: cpup.c,v 1.11 2006/08/07 14:50:32 sfjro Exp $ */

#include <asm/uaccess.h>
#include "aufs.h"

/* violent cpup_attr_*() functions don't care inode lock */
void cpup_attr_timesizes(struct inode *inode)
{
	struct inode *hidden_inode;

	LKTRTrace("i%lu\n", inode->i_ino);
	//IMustLock(inode);
	hidden_inode = itohi(inode);
	DEBUG_ON(!hidden_inode);

	inode->i_atime = hidden_inode->i_atime;
	inode->i_mtime = hidden_inode->i_mtime;
	inode->i_ctime = hidden_inode->i_ctime;
	spin_lock(&inode->i_lock);
	i_size_write(inode, i_size_read(hidden_inode));
	inode->i_blocks = hidden_inode->i_blocks;
	spin_unlock(&inode->i_lock);
}

void cpup_attr_nlink(struct inode *inode)
{
	LKTRTrace("i%lu\n", inode->i_ino);
	//IMustLock(inode);
	DEBUG_ON(!inode->i_mode);
	inode->i_nlink = itohi(inode)->i_nlink;
}

void cpup_attr_changable(struct inode *inode)
{
	struct inode *hidden_inode;

	LKTRTrace("i%lu\n", inode->i_ino);
	//IMustLock(inode);
	hidden_inode = itohi(inode);
	DEBUG_ON(!hidden_inode);

	inode->i_mode = hidden_inode->i_mode;
	inode->i_uid = hidden_inode->i_uid;
	inode->i_gid = hidden_inode->i_gid;
	cpup_attr_timesizes(inode);

	//??
	inode->i_flags = hidden_inode->i_flags;
}

void cpup_attr_all(struct inode *inode)
{
	struct inode *hidden_inode;

	LKTRTrace("i%lu\n", inode->i_ino);
	//IMustLock(inode);
	hidden_inode = itohi(inode);
	DEBUG_ON(!hidden_inode);

	cpup_attr_changable(inode);
	if (inode->i_nlink > 0)
		cpup_attr_nlink(inode);

	switch (inode->i_mode & S_IFMT) {
	case S_IFBLK:
	case S_IFCHR:
		inode->i_rdev = hidden_inode->i_rdev;
	}
	inode->i_blkbits = hidden_inode->i_blkbits;
	inode->i_blksize = hidden_inode->i_blksize;
}

/* ---------------------------------------------------------------------- */

/* Note: dt_dentry and dt_hidden_dentry are not dget/dput-ed */

/* keep the timestamps of the parent dir when cpup */
void dtime_store(struct dtime *dt, struct dentry *dentry,
		 struct dentry *hidden_dentry)
{
	struct inode *inode;

	TraceEnter();

	dt->dt_dentry = dentry; /* can be NULL */
	dt->dt_hidden_dentry = hidden_dentry;
	if (!hidden_dentry)
		return;
	inode = hidden_dentry->d_inode;
	if (!inode)
		return;
	dt->dt_atime = inode->i_atime;
	dt->dt_mtime = inode->i_mtime;
}

void dtime_revert(struct dtime *dt)
{
	struct iattr attr;
	int err;

	TraceEnter();

	if (!dt->dt_hidden_dentry)
		return;

	attr.ia_atime = dt->dt_atime;
	attr.ia_mtime = dt->dt_mtime;
	attr.ia_valid = ATTR_FORCE | ATTR_MTIME | ATTR_MTIME_SET
		| ATTR_ATIME | ATTR_ATIME_SET;
	err = hidden_notify_change(dt->dt_hidden_dentry, &attr);
	if (err)
		Warn("restoring timestamps failed(%d). ignored\n", err);
#if 0 //??
	if (dt->dt_dentry)
		cpup_attr_timesizes(dt->dt_dentry->d_inode,
				    dt->dt_hidden_dentry->d_inode);
#endif
}

/* ---------------------------------------------------------------------- */

static int cpup_iattr(struct dentry *hidden_dst, struct dentry *hidden_src)
{
	int err;
	struct iattr ia;
	struct inode *hidden_isrc, *hidden_idst;

	LKTRTrace("%.*s\n", DLNPair(hidden_dst));
	hidden_idst = hidden_dst->d_inode;
	//IMustLock(hidden_idst);
	hidden_isrc = hidden_src->d_inode;
	//IMustLock(hidden_isrc);

	ia.ia_valid = ATTR_FORCE | ATTR_MODE | ATTR_UID | ATTR_GID
		| ATTR_ATIME | ATTR_MTIME
		| ATTR_ATIME_SET | ATTR_MTIME_SET;
	ia.ia_mode = hidden_isrc->i_mode;
	ia.ia_uid = hidden_isrc->i_uid;
	ia.ia_gid = hidden_isrc->i_gid;
	ia.ia_atime = hidden_isrc->i_atime;
	ia.ia_mtime = hidden_isrc->i_mtime;
	err = hidden_notify_change(hidden_dst, &ia);
	//err = -1;
	if (!err)
		hidden_idst->i_flags = hidden_isrc->i_flags; //??

	TraceErr(err);
	return err;
}

/*
 * to support a sparse file which is opened with O_APPEND,
 * we need to close the file.
 */
static int cpup_regular(struct dentry *dentry, aufs_bindex_t bdst,
			aufs_bindex_t bsrc, loff_t len)
{
	int err, i;
	struct super_block *sb;
	struct inode *hidden_inode;
	enum {SRC, DST};
	struct {
		aufs_bindex_t bindex;
		unsigned int flags;
		struct dentry *dentry;
		struct file *file;
		void *label, *label_file;
	} *h, hidden[] = {
		{.bindex = bsrc, .flags = O_RDONLY|O_NOATIME, .file = NULL,
		 .label = &&out, .label_file = &&out_src_file},
		{.bindex = bdst, .flags = O_WRONLY|O_NOATIME, .file = NULL,
		 .label = &&out_src_file, .label_file = &&out_dst_file},
	};

	LKTRTrace("dentry %.*s, bdst %d, bsrc %d, len %lld\n",
		  DLNPair(dentry), bdst, bsrc, len);
	DEBUG_ON(bsrc <= bdst);
	DEBUG_ON(!len);
	sb = dentry->d_sb;
	DEBUG_ON(test_ro(sb, bdst, dentry->d_inode));
	// bsrc branch can be ro/rw.

	h = hidden;
	for (i = 0; i < 2; i++, h++) {
		h->dentry = dtohd_index(dentry, h->bindex);
		DEBUG_ON(!h->dentry);
		hidden_inode = h->dentry->d_inode;
		DEBUG_ON(!hidden_inode
			 || !S_ISREG(hidden_inode->i_mode));
		h->file = hidden_dentry_open(dentry, h->bindex, h->flags);
		//h->file = ERR_PTR(-1);
		err = PTR_ERR(h->file);
		if (IS_ERR(h->file))
			goto *h->label;
		err = -EINVAL;
		if (!h->file->f_op)
			goto *h->label_file;
	}

	/* stop updating while we copyup */
	IMustLock(hidden[SRC].dentry->d_inode);
	err = copy_file(hidden[DST].file, hidden[SRC].file, len);

 out_dst_file:
	fput(hidden[DST].file);
	sbr_put(sb, hidden[DST].bindex);
 out_src_file:
	fput(hidden[SRC].file);
	sbr_put(sb, hidden[SRC].bindex);
 out:
	TraceErr(err);
	return err;
}

/* return with hidden dst inode is locked */
static int cpup_entry(struct dentry *dentry, aufs_bindex_t bdst,
		      aufs_bindex_t bsrc, loff_t len, int do_dt)
{
	int err, do_superio, isdir;
	struct dentry *hidden_src, *hidden_dst, *hidden_parent;
	struct inode *hidden_inode, *hidden_dir, *dir;
	struct dtime dt;
	struct superio sio;
	umode_t mode;

	LKTRTrace("%.*s, i%lu, bdst %d, bsrc %d, len %Ld\n",
		  DLNPair(dentry), dentry->d_inode->i_ino, bdst, bsrc, len);
	DEBUG_ON(bdst >= bsrc
		 || test_ro(dentry->d_sb, bdst, NULL));
	// bsrc branch can be ro/rw.

	hidden_src = dtohd_index(dentry, bsrc);
	DEBUG_ON(!hidden_src);
	hidden_inode = hidden_src->d_inode;
	DEBUG_ON(!hidden_inode);

	/* stop refrencing while we are creating */
	dir = dentry->d_parent->d_inode;
	hidden_dst = dtohd_index(dentry, bdst);
	DEBUG_ON(hidden_dst && hidden_dst->d_inode);
	hidden_parent = hidden_dst->d_parent;
	hidden_dir = hidden_parent->d_inode;
	IMustLock(hidden_dir);

	if (do_dt)
		dtime_store(&dt, NULL, hidden_parent);
	do_superio = superio_test(hidden_dir, MAY_WRITE);
	if (do_superio)
		superio_store(&sio);

	isdir = 0;
	mode = hidden_inode->i_mode;
	switch (mode & S_IFMT) {
	case S_IFREG:
		/* stop updating while we are referencing */
		IMustLock(hidden_inode);
		err = vfs_create(hidden_dir, hidden_dst, mode, NULL);
		//err = -1;
		if (!err) {
			loff_t l = i_size_read(hidden_inode);
			if (len == -1 || l < len)
				len = l;
			if (len)
				err = cpup_regular(dentry, bdst, bsrc, len);
			if (err) {
				int rerr;
				rerr = safe_unlink(hidden_dir, hidden_dst, NULL);
				if (rerr) {
					IOErr("failed unlinking cpup-ed %.*s"
					      "(%d, %d)\n",
					      DLNPair(hidden_dst), err, rerr);
					err = -EIO;
				}
			}
		}
		break;
	case S_IFDIR:
		isdir = 1;
		err = vfs_mkdir(hidden_dir, hidden_dst, mode);
		//err = -1;
		if (!err) {
			/* setattr case: dir is not locked */
			if (ibstart(dir) == bdst)
				cpup_attr_nlink(dir);
			cpup_attr_nlink(dentry->d_inode);
		}
		break;
	case S_IFLNK: {
		int symlen;
		char *sym;
		mm_segment_t old_fs;

		err = -ENOMEM;
		sym = __getname();
		if (!sym)
			break;
		old_fs = get_fs();
		set_fs(KERNEL_DS);
		err = symlen = hidden_inode->i_op->readlink
			(hidden_src, (char __user *)sym, PATH_MAX);
		set_fs(old_fs);
		if (symlen > 0) {
			sym[symlen] = 0;
			err = vfs_symlink(hidden_dir, hidden_dst, sym, mode);
		}
		__putname(sym);
	}
		break;
	case S_IFCHR:
	case S_IFBLK: {
		int do_sio2;
		struct superio sio2;

		do_sio2 = !capable(CAP_MKNOD);
		if (do_sio2)
			superio_store(&sio2);
		err = vfs_mknod(hidden_dir, hidden_dst, mode, hidden_inode->i_rdev);
		if (do_sio2)
			superio_revert(&sio2);
	}
		break;
	case S_IFIFO:
	case S_IFSOCK:
		err = vfs_mknod(hidden_dir, hidden_dst, mode, hidden_inode->i_rdev);
		break;
	default:
		IOErr("Unknown inode type 0%o\n", mode);
		err = -EIO;
	}

	if (do_superio)
		superio_revert(&sio);
	if (do_dt)
		dtime_revert(&dt);
	return err;
}

int cpup_single(struct dentry *dentry, aufs_bindex_t bdst, aufs_bindex_t bsrc,
		loff_t len, int do_dt)
{
	int err, rerr;
	struct dentry *hidden_src, *hidden_dst;
	struct inode *dst_inode, *hidden_dir, *inode;
	struct super_block *sb;
	aufs_bindex_t old_ibstart;
	struct dtime dt;

	LKTRTrace("%.*s, i%lu, bdst %d, bsrc %d, len %Ld\n",
		  DLNPair(dentry), dentry->d_inode->i_ino, bdst, bsrc, len);
	sb = dentry->d_sb;
	DEBUG_ON(bsrc <= bdst);
	hidden_dst = dtohd_index(dentry, bdst);
	DEBUG_ON(!hidden_dst || hidden_dst->d_inode);
	hidden_dir = hidden_dst->d_parent->d_inode;
	hidden_src = dtohd_index(dentry, bsrc);
	DEBUG_ON(!hidden_src || !hidden_src->d_inode);
	IMustLock(hidden_dir);
	inode = dentry->d_inode;
	DEBUG_ON(itohi_index(inode, bdst));
	IiMustWriteLock(inode);

	old_ibstart = ibstart(inode);
	err = cpup_entry(dentry, bdst, bsrc, len, do_dt);
	//err = -1;
	if (err)
		goto out;
	dst_inode = hidden_dst->d_inode;
	i_lock(dst_inode);

	err = xino_write(sb, bdst, dst_inode->i_ino, dentry->d_inode->i_ino);
	//err = -1;
	if (!err)
		err = cpup_iattr(hidden_dst, hidden_src);
	//err = -1;
#if 0
	if (0 && !err)
		err = cpup_xattrs(hidden_src, hidden_dst);
#endif
	if (!err) {
		if (bdst < old_ibstart)
			set_ibstart(inode, bdst);
		set_itohi_index(inode, bdst, igrab(dst_inode));
		i_unlock(dst_inode);
		return 0; /* success */
	}

	/* revert */
	i_unlock(dst_inode);
	dtime_store(&dt, NULL, hidden_dst->d_parent);
	if (!S_ISDIR(dst_inode->i_mode))
		rerr = safe_unlink(hidden_dir, hidden_dst, NULL);
	else
		rerr = vfs_rmdir(hidden_dir, hidden_dst);
	//rerr = -1;
	dtime_revert(&dt);
	if (rerr) {
		IOErr("failed removing broken entry(%d, %d)\n", err, rerr);
		err = -EIO;
	}

 out:
	TraceErr(err);
	return err;
}

int cpup_simple(struct dentry *dentry, aufs_bindex_t bdst, loff_t len,
		int do_dt)
{
	int err, do_superio;
	struct inode *inode;
	aufs_bindex_t bsrc, bend;
	struct superio sio;

	LKTRTrace("%.*s, bdst %d, len %Ld\n",
		  DLNPair(dentry), bdst, len);
	inode = dentry->d_inode;
	DEBUG_ON(!S_ISDIR(inode->i_mode) && dbstart(dentry) < bdst);

	bend = dbend(dentry);
	for (bsrc = bdst+1; bsrc <= bend; bsrc++)
		if (dtohd_index(dentry, bsrc))
			break;
	BUG_ON(!dtohd_index(dentry, bsrc));

	// test here
	//do_superio = superio_test(dtohd_index(dentry->d_parent, bdst)->d_inode,
	//		  MAY_EXEC|MAY_WRITE);
	do_superio = 0;
	if (do_superio)
		superio_store(&sio);

	err = lookup_negative(dentry, bdst);
	//err = -1;
	if (!err) {
		err = cpup_single(dentry, bdst, bsrc, len, do_dt);
		//err = -1;
		if (!err)
			goto out; /* success */

		/* revert */
		set_dtohd_index(dentry, bdst, NULL);
		set_dbstart(dentry, bsrc);
	}

 out:
	if (do_superio)
		superio_revert(&sio);
	TraceErr(err);
	return err;
}

/* cf. revalidate function in file.c */
int cpup_dirs(struct dentry *dentry, aufs_bindex_t bdst, struct dentry *locked)
{
	int err;
	struct super_block *sb;
	struct dentry *d, *parent, *hidden_parent;

	LKTRTrace("%.*s, b%d, parent i%lu, locked %p\n",
		  DLNPair(dentry), bdst, dentry->d_parent->d_inode->i_ino,
		  locked);
	sb = dentry->d_sb;
	DEBUG_ON(test_ro(sb, bdst, NULL));
	parent = dentry->d_parent;
	IiMustWriteLock(parent->d_inode);
	if (IS_ROOT(parent))
		return 0;
	if (locked) {
		DiMustAnyLock(locked);
		IiMustAnyLock(locked->d_inode);
	}

	/* slow loop, keep it simple and stupid */
	err = 0;
	while (1) {
		parent = dentry->d_parent; // dget_parent()
		hidden_parent = dtohd_index(parent, bdst);
		if (hidden_parent)
			return 0; /* success */

		/* find top dir which is needed to cpup */
		do {
			d = parent;
			parent = d->d_parent; // dget_parent()
			if (parent != locked)
				di_read_lock(parent, 0);
			hidden_parent = dtohd_index(parent, bdst);
			if (parent != locked)
				di_read_unlock(parent, 0);
		} while (!hidden_parent);

		if (d != dentry->d_parent)
			di_write_lock(d);
		if (parent != locked)
			di_read_lock(parent, AUFS_I_RLOCK);

		/* somebody else might create while we were sleeping */
		if (!dtohd_index(d, bdst) || !dtohd_index(d, bdst)->d_inode) {
			if (dtohd_index(d, bdst))
				update_bstart(d);
			//DEBUG_ON(dbstart(d) <= bdst);
			i_lock(hidden_parent->d_inode);
			err = cpup_simple(d, bdst, -1, 1);
			//err = -1;
			i_unlock(hidden_parent->d_inode);
		}

		if (parent != locked)
			di_read_unlock(parent, AUFS_I_RLOCK);
		if (d != dentry->d_parent)
			di_write_unlock(d);
		if (err)
			break;
	}

	TraceErr(err);
	return err;
}

int test_and_cpup_dirs(struct dentry *dentry, aufs_bindex_t bdst,
		       struct dentry *locked, struct inode *relock)
{
	int err;
	struct dentry *parent;
	struct inode *dir;

	parent = dentry->d_parent;
	dir = parent->d_inode;
	LKTRTrace("%.*s, b%d, parent i%lu, locked %p, relock %p\n",
		  DLNPair(dentry), bdst, dir->i_ino, locked, relock);
	DiMustReadLock(parent);
	IiMustReadLock(dir);
	if (relock)
		IMustLock(relock);

	if (itohi_index(dir, bdst))
		return 0;

	err = 0;
	di_read_unlock(parent, AUFS_I_RLOCK);
	di_write_lock(parent);
	if (itohi_index(dir, bdst))
		goto out;

	if (relock)
		i_unlock(relock);
	err = cpup_dirs(dentry, bdst, locked);
	//err = -1;
	if (relock)
		i_lock(relock);

 out:
	di_downgrade_lock(parent, AUFS_I_RLOCK);
	if (relock)
		IMustLock(relock);
	TraceErr(err);
	return err;
}
