/*
 * Copyright (C) 2005-2006 Junjiro Okajima
 *
 * This program, aufs is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/* $Id: iinfo.c,v 1.11 2006/07/31 03:47:13 sfjro Exp $ */

//#include <linux/mm.h>
#include "aufs.h"

struct aufs_iinfo *itopd(struct inode *inode)
{
	struct aufs_iinfo *iinfo;

 	iinfo = &(container_of(inode, struct aufs_icntnr, vfs_inode)->iinfo);
	/* bad_inode case */
	if (!iinfo->ii_inode)
		return NULL;
	DEBUG_ON(!iinfo->ii_inode
		 /* || stopd(inode->i_sb)->si_bend < iinfo->ii_bend */
		 || iinfo->ii_bend < iinfo->ii_bstart
		);
	return iinfo;
}

aufs_bindex_t ibstart(struct inode *inode)
{
	IiMustAnyLock(inode);
	return itopd(inode)->ii_bstart;
}

aufs_bindex_t ibend(struct inode *inode)
{
	IiMustAnyLock(inode);
	return itopd(inode)->ii_bend;
}

struct aufs_vdir *ivdir(struct inode *inode)
{
	IiMustAnyLock(inode);
	DEBUG_ON(!S_ISDIR(inode->i_mode));
	return itopd(inode)->ii_vdir;
}

struct inode *itohi_index(struct inode *inode, aufs_bindex_t bindex)
{
	struct inode *hidden_inode;

	IiMustAnyLock(inode);
	DEBUG_ON(bindex < 0 || ibend(inode) < bindex);
	hidden_inode = itopd(inode)->ii_inode[0+bindex];
	DEBUG_ON(hidden_inode && atomic_read(&hidden_inode->i_count) <= 0);
	return hidden_inode;
}

struct inode *itohi(struct inode *inode)
{
	return itohi_index(inode, ibstart(inode));
}

// hard/soft set
void set_ibstart(struct inode *inode, aufs_bindex_t bindex)
{
	IiMustWriteLock(inode);
	DEBUG_ON(sbend(inode->i_sb) < bindex);
	/* */
	itopd(inode)->ii_bstart = bindex;
}

void set_ibend(struct inode *inode, aufs_bindex_t bindex)
{
	IiMustWriteLock(inode);
	DEBUG_ON(sbend(inode->i_sb) < bindex);
	DEBUG_ON(bindex < ibstart(inode));
	itopd(inode)->ii_bend = bindex;
}

void set_ivdir(struct inode *inode, struct aufs_vdir *vdir)
{
	IiMustWriteLock(inode);
	DEBUG_ON(!S_ISDIR(inode->i_mode)
		|| (itopd(inode)->ii_vdir && vdir));
	itopd(inode)->ii_vdir = vdir;
}

void set_itohi_index(struct inode *inode, aufs_bindex_t bindex,
		     struct inode *val)
{
	struct inode *hi = itopd(inode)->ii_inode[0+bindex];
	IiMustWriteLock(inode);
	DEBUG_ON(bindex < ibstart(inode) || ibend(inode) < bindex
		 || (val && atomic_read(&val->i_count) <= 0)
		 || (val && hi));
	if (!val && hi) {
		//truncate_inode_pages(hi->i_mapping, 0);
		iput(hi);
	}
	itopd(inode)->ii_inode[0+bindex] = val;
}

/* ---------------------------------------------------------------------- */

int iinfo_init(struct inode *inode)
{
	struct aufs_iinfo *iinfo;
	struct super_block *sb;
	int size;

	sb = inode->i_sb;
	DEBUG_ON(!sb);
	iinfo = &(container_of(inode, struct aufs_icntnr, vfs_inode)->iinfo);
	DEBUG_ON(!iinfo);
	size = sizeof(*iinfo->ii_inode)*(sbend(sb)+1);
	iinfo->ii_inode = kzalloc(size, GFP_KERNEL);
	//iinfo->ii_inode = NULL;
	if (iinfo->ii_inode) {
		rw_init_nolock(&iinfo->ii_rwsem);
		iinfo->ii_bstart = -1;
		iinfo->ii_bend = -1;
		iinfo->ii_vdir = NULL;
		return 0;
	}
	return -ENOMEM;
}

void iinfo_fin(struct inode *inode)
{
	struct aufs_iinfo *iinfo;
	struct inode **i;

	iinfo = itopd(inode);
	/* bad_inode case */
	if (!iinfo)
		return;

	if (iinfo->ii_vdir)
		free_vdir(iinfo->ii_vdir);

	if (iinfo->ii_bstart >= 0) {
		i = iinfo->ii_inode+iinfo->ii_bstart;
		while (iinfo->ii_bstart++ <= iinfo->ii_bend) {
			if (*i) {
				//truncate_inode_pages((*i)->i_mapping, 0);
				iput(*i);
			}
			i++;
		}
		//iinfo->ii_bstart = iinfo->ii_bend = -1;
	}

	rw_destroy(&iinfo->ii_rwsem);
	kfree(iinfo->ii_inode);
	//iinfo->ii_inode = NULL;
}
