/*
 * Copyright (C) 2005-2006 Junjiro Okajima
 *
 * This program, aufs is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/* $Id: dentry.c,v 1.21 2006/10/09 00:24:13 sfjro Exp $ */

//#include <linux/fs.h>
//#include <linux/namei.h>
#include "aufs.h"

#if 0 // not yet
struct dentry *aufs_lookup_one_len(const char *name, struct dentry *parent,
				   int len, struct open_intent *intent,
				   struct aufs_branch *br,
				   struct dentry *parent2)
{
	struct dentry *dentry;
	struct inode *dir;
	char *o, *p;
	struct nameidata nd;
	int l, err;

	LKTRTrace();
	dir = parent->d_inode;
	IMustLock(dir);

	if (!(br->br_perm & AUFS_MUST_FOLLOW)) {
		dentry = lookup_one_len(name, parent, len);
		goto out;
	}

	dentry = ERR_PTR(-ENOMEM);
	o = __getname();
	if (!o)
		goto out;

	l = PATH_MAX - len - 1;
	if (!parent2)
		i_unlock(dir);
	else
		unlock_rename(parent, parent2);
	p = d_path(parent, br->br_mnt, o, l);
	DEBUG_ON(o[l - 1]);
	o[l - 1] = '/';
	strncpy(o + l, name, len);
	o[PATH_MAX - 1] = 0;
	LKTRTrace("%s\n", p);
	err = path_lookup(p, LOOKUP_FOLLOW, &nd);
	if (!parent2)
		i_lock(dir);
	else
		lock_rename(parent, parent2);
	__putname(o);
	if (!err) {
		dentry = dget(nd.dentry);
		path_release(&nd);
	} else
		dentry = ERR_PTR(err);
 out:
	TraceErrPtr(dentry);
	return dentry;
}
#endif

/*
 * returns positive/negative dentry, NULL or an error.
 * NULL means whiteout-ed or not-found.
 */
static struct dentry *do_lookup(struct dentry *hidden_parent,
				struct dentry *dentry, aufs_bindex_t bindex,
				struct qstr *wh_name, int allow_neg,
				mode_t type)
{
	struct dentry *hidden_dentry;
	int wh_found, wh_able, opq;
	struct inode *hidden_dir, *hidden_inode;
	struct qstr *name;

	LKTRTrace("%.*s/%.*s, b%d, allow_neg %d, type 0%o\n",
		  DLNPair(hidden_parent), DLNPair(dentry), bindex, allow_neg,
		  type);
	DEBUG_ON(IS_ROOT(dentry));
	hidden_dir = hidden_parent->d_inode;
	IMustLock(hidden_dir);

	wh_found = 0;
	wh_able = (sbr_perm(dentry->d_sb, bindex) & (MAY_WRITE | AUFS_MAY_WH));
	if (/* unlikely */(wh_able))
		wh_found = is_wh(hidden_parent, wh_name, /*try_sio*/0);
	//if (LktrCond) wh_found = -1;
	hidden_dentry = ERR_PTR(wh_found);
	if (!wh_found)
		goto real_lookup;
	if (unlikely(wh_found < 0))
		goto out;

	/* We found a whiteout */
	//set_dbend(dentry, bindex);
	set_dbwh(dentry, bindex);
	if (!allow_neg)
		return NULL; /* success */

 real_lookup:
	// do not superio.
	name = &dentry->d_name;
	hidden_dentry = lookup_one_len(name->name, hidden_parent, name->len);
	//hidden_dentry = ERR_PTR(-1);
	if (IS_ERR(hidden_dentry))
		goto out;
	DEBUG_ON(d_unhashed(hidden_dentry));
	hidden_inode = hidden_dentry->d_inode;
	if (!hidden_inode) {
		if (!allow_neg)
			goto out_neg;
	} else if (wh_found
		   || (type && type != (hidden_inode->i_mode & S_IFMT)))
		goto out_neg;

	if (dbend(dentry) <= bindex)
		set_dbend(dentry, bindex);
	if (dbstart(dentry) == -1 || bindex < dbstart(dentry))
		set_dbstart(dentry, bindex);
	set_dtohd_index(dentry, bindex, hidden_dentry);

	if (!hidden_inode || !S_ISDIR(hidden_inode->i_mode) || !wh_able)
		return hidden_dentry; /* success */

	i_lock(hidden_inode);
	opq = is_diropq(hidden_dentry);
	//opq = -1;
	i_unlock(hidden_inode);
	if (opq > 0)
		set_dbdiropq(dentry, bindex);
	else if (unlikely(opq < 0)) {
		set_dtohd_index(dentry, bindex, NULL);
		hidden_dentry = ERR_PTR(opq);
	}
	goto out;

 out_neg:
	dput(hidden_dentry);
	hidden_dentry = NULL;
 out:
	TraceErrPtr(hidden_dentry);
	return hidden_dentry;
}

/*
 * returns the number of hidden positive dentries,
 * otherwise an error.
 */
int lookup_dentry(struct dentry *dentry, aufs_bindex_t bstart, mode_t type)
{
	int npositive, err, allow_neg;
	struct dentry *parent;
	aufs_bindex_t bindex, btail;
	const struct qstr *name = &dentry->d_name;
	struct qstr whname;

	LKTRTrace("%.*s, b%d, type 0%o\n", LNPair(name), bstart, type);
	DEBUG_ON(bstart < 0 || IS_ROOT(dentry));
	parent = dentry->d_parent;

	/* No dentries should get created for possible whiteout names. */
	err = -EPERM;
	if (unlikely(!strncmp(name->name, WHPFX, WHLEN)))
		goto out;

	err = alloc_whname(name->name, name->len, &whname);
	if (unlikely(err))
		goto out;
	allow_neg = !type;
	npositive = 0;
	btail = dbtaildir(parent);
	for (bindex = bstart; bindex <= btail; bindex++) {
		struct dentry *hidden_parent, *hidden_dentry;
		struct inode *hidden_inode;
		struct inode *hidden_dir;

		hidden_dentry = dtohd_index(dentry, bindex);
		if (hidden_dentry) {
			if (hidden_dentry->d_inode)
				npositive++;
			if (type != S_IFDIR)
				break;
			continue;
		}
		hidden_parent = dtohd_index(parent, bindex);
		if (!hidden_parent)
			continue;
		hidden_dir = hidden_parent->d_inode;
		if (!hidden_dir || !S_ISDIR(hidden_dir->i_mode))
			continue;

		i_lock(hidden_dir);
		hidden_dentry = do_lookup(hidden_parent, dentry, bindex,
					  &whname, allow_neg, type);
		//hidden_dentry = ERR_PTR(-1);
		i_unlock(hidden_dir);
		err = PTR_ERR(hidden_dentry);
		if (IS_ERR(hidden_dentry))
			goto out_wh;
		allow_neg = 0;

		if (dbwh(dentry) != -1)
			break;
		if (!hidden_dentry)
			continue;
		hidden_inode = hidden_dentry->d_inode;
		if (!hidden_inode)
			continue;
		npositive++;
		if (!type)
			type = hidden_inode->i_mode & S_IFMT;
		if (type != S_IFDIR)
			break;
		else if (dbdiropq(dentry) != -1)
			break;
	}

	if (npositive) {
		LKTRLabel(positive);
		update_dbstart(dentry);
	}
	err = npositive;

 out_wh:
	free_whname(&whname);
 out:
	TraceErr(err);
	return err;
}

struct dentry *sio_lookup_one_len(const char *name, struct dentry *parent,
				  int len)
{
	struct dentry *dentry;

	LKTRTrace("%.*s/%.*s\n", DLNPair(parent), len, name);
	IMustLock(parent->d_inode);

	if (!test_perm(parent->d_inode, MAY_EXEC))
		dentry = lookup_one_len(name, parent, len);
	else {
		void f(void *completion) {
			dentry = lookup_one_len(name, parent, len);
			complete(completion);
		}
		wkq_wait(f);
	}

	TraceErrPtr(dentry);
	return dentry;
}

int lookup_negative(struct dentry *dentry, aufs_bindex_t bindex)
{
	int err;
	struct dentry *parent, *hidden_parent, *hidden_dentry;
	struct inode *hidden_dir;

	LKTRTrace("%.*s, b%d\n", DLNPair(dentry), bindex);
	parent = dentry->d_parent;
	DEBUG_ON(!parent || !parent->d_inode
		 || !S_ISDIR(parent->d_inode->i_mode));
	hidden_parent = dtohd_index(parent, bindex);
	DEBUG_ON(!hidden_parent);
	hidden_dir = hidden_parent->d_inode;
	DEBUG_ON(!hidden_dir || !S_ISDIR(hidden_dir->i_mode));
	IMustLock(hidden_dir);

	hidden_dentry = sio_lookup_one_len(dentry->d_name.name,
					   hidden_parent, dentry->d_name.len);
	//hidden_dentry = ERR_PTR(-1);
	err = PTR_ERR(hidden_dentry);
	if (IS_ERR(hidden_dentry))
		goto out;
	if (unlikely(hidden_dentry->d_inode)) {
		err = -EIO;
		IOErr("b%d %.*s should be negative\n",
		      bindex, DLNPair(hidden_dentry));
		dput(hidden_dentry);
		goto out;
	}

	if (bindex < dbstart(dentry))
		set_dbstart(dentry, bindex);
	if (dbend(dentry) < bindex)
		set_dbend(dentry, bindex);
	set_dtohd_index(dentry, bindex, hidden_dentry);
	err = 0;

 out:
	TraceErr(err);
	return err;
}

/*
 * returns the number of found hidden positive dentries,
 * otherwise an error.
 */
int refresh_dentry(struct dentry *dentry, mode_t type)
{
	int npositive, pgen, new_sz, sgen, dgen;
	struct aufs_dinfo *dinfo;
	struct super_block *sb;
	struct dentry *parent, **p;
	aufs_bindex_t bindex, parent_bend, parent_bstart, bwh, bdiropq, bend;

	LKTRTrace("%.*s, type 0%o\n", DLNPair(dentry), type);
	DiMustWriteLock(dentry);
	sb = dentry->d_sb;
	DEBUG_ON(IS_ROOT(dentry));
	parent = dentry->d_parent; // dget_parent()
	pgen = digen(parent);
	sgen = sigen(sb);
	dgen = digen(dentry);
	DEBUG_ON(pgen != sgen
		 || AufsGenOlder(sgen, dgen)
		 || AufsGenOlder(pgen, dgen));

	npositive = -ENOMEM;
	new_sz = sizeof(*dinfo->di_dentry) * (sbend(sb) + 1);
	dinfo = dtopd(dentry);
	p = kzrealloc(dinfo->di_dentry, sizeof(*p) * (dinfo->di_bend + 1),
		      new_sz);
	//p = NULL;
	if (unlikely(!p))
		goto out;
	dinfo->di_dentry = p;
	//smp_mb();

	bend = dinfo->di_bend;
	bwh = dinfo->di_bwh;
	bdiropq = dinfo->di_bdiropq;
	p += dinfo->di_bstart;
	for (bindex = dinfo->di_bstart; bindex <= bend; bindex++, p++) {
		struct dentry *hd, *tmp, **q;
		aufs_bindex_t new_bindex;

		hd = *p;
		if (!hd)
			continue;
		DEBUG_ON(!hd->d_inode);
		if (hd->d_parent == dtohd_index(parent, bindex)) // dget_parent()
			continue;

		new_bindex = find_dbindex(parent, hd->d_parent); // dget_parent()
		DEBUG_ON(new_bindex == bindex);
		if (dinfo->di_bwh == bindex)
			bwh = new_bindex;
		if (dinfo->di_bdiropq == bindex)
			bdiropq = new_bindex;
		if (new_bindex < 0) { // test here
			dput(hd);
			*p = NULL;
			continue;
		}
		/* swap two hidden dentries, and loop again */
		q = dinfo->di_dentry + new_bindex;
		tmp = *q;
		*p = tmp;
		*q = hd;
		if (tmp) {
			bindex--;
			p--;
		}
	}

	dinfo->di_bwh = bwh;
	dinfo->di_bdiropq = bdiropq;
	parent_bend = dbend(parent);
	p = dinfo->di_dentry;
	for (bindex = 0; bindex <= parent_bend; bindex++, p++)
		if (*p) {
			dinfo->di_bstart = bindex;
			break;
		}
	p = dinfo->di_dentry + parent_bend;
	for (bindex = parent_bend; bindex >= 0; bindex--, p--)
		if (*p) {
			dinfo->di_bend = bindex;
			break;
		}
	//smp_mb();

	npositive = 0;
	parent_bstart = dbstart(parent);
	if (type != S_IFDIR && dinfo->di_bstart == parent_bstart)
		goto out_dgen; /* success */

	npositive = lookup_dentry(dentry, parent_bstart, type);
	//npositive = -1;
	if (npositive < 0)
		goto out;

 out_dgen:
	update_digen(dentry);
 out:
	TraceErr(npositive);
	return npositive;
}

static int hidden_d_revalidate(struct dentry *dentry, struct nameidata *nd)
{
	int err, positive;
	struct nameidata fake_nd, *p;
	aufs_bindex_t bindex, btail, bstart;
	struct super_block *sb;
	struct inode *inode;
	struct dentry *hidden_dentry, *parent;
	umode_t mode;
	struct qstr *name;

	LKTRTrace("%.*s\n", DLNPair(dentry));

	err = 0;
	sb = dentry->d_sb;
	parent = dentry->d_parent;
	name = &dentry->d_name;
	positive = 1;
	inode = dentry->d_inode;
	if (!inode || !inode->i_nlink)
		positive = 0;
	mode = 0;
	if (positive)
		mode = inode->i_mode & S_IFMT;
	btail = bstart = dbstart(dentry);
	if (positive && S_ISDIR(inode->i_mode))
		btail = dbtaildir(dentry);
	p = NULL;
	if (nd)
		fake_nd = *nd;
	//di_read_lock(parent, 0);
	for (bindex = bstart; bindex <= btail; bindex++) {
#ifdef CONFIG_AUFS_DBA
		int hidden_positive;
#endif
		hidden_dentry = dtohd_index(dentry, bindex);
		if (!hidden_dentry)
			continue;

#if 0 //def CONFIG_AUFS_DBA
		if (unlikely(hidden_dentry->d_parent
			     != dtohd_index(parent, bindex)
			     || name->len != hidden_dentry->d_name.len
			     || memcmp(name->name, hidden_dentry->d_name.name, name->len)
			    )) {
			err = -EINVAL;
			break;
		}
#endif

		if (unlikely(hidden_dentry->d_op
			     && hidden_dentry->d_op->d_revalidate)) {
			p = fake_dm(&fake_nd, nd, sb, bindex);
			if (!hidden_dentry->d_op->d_revalidate(hidden_dentry,
							       p))
				err = -EINVAL;
			fake_dm_release(p);
			if (err)
				break;
		}
#ifdef CONFIG_AUFS_DBA
		/* someone might delete/create it on the branch directly */
		hidden_positive = 1;
		if (!hidden_dentry->d_inode || !hidden_dentry->d_inode->i_nlink)
			hidden_positive = 0;
		if (unlikely(positive != hidden_positive
			     || (hidden_positive
				 && (hidden_dentry->d_inode->i_mode & S_IFMT)
				 != mode))) {
			err = -EINVAL;
			break;
		}
#endif
	}
	//di_read_unlock(parent, 0);

#ifdef CONFIG_AUFS_DBA
	if (!err && inode) {
		struct inode *hi = itohi(inode);
		if (!timespec_equal(&inode->i_ctime, &hi->i_ctime))
			cpup_attr_all(inode);
	}

	if (0 && unlikely(err)) {
		di_read_unlock(dentry, AUFS_I_RLOCK);
		di_write_lock(dentry);
		set_dtohd_index(dentry, bindex, NULL);
		set_itohi_index(inode, bindex, NULL, 0);
		di_downgrade_lock(dentry, AUFS_I_RLOCK);
	}
#endif

	TraceErr(err);
	return err;
}

static int simple_reval_dpath(struct dentry *dentry, int sgen)
{
	int err;
	mode_t type;
	struct dentry *parent;

	LKTRTrace("%.*s, sgen %d\n", DLNPair(dentry), sgen);
	SiMustAnyLock(dentry->d_sb);
	DiMustWriteLock(dentry);
	DEBUG_ON(!dentry->d_inode);

	if (!AufsGenOlder(digen(dentry), sgen))
		return 0;

	parent = dget_parent(dentry);
	di_read_lock(parent, AUFS_I_RLOCK);
	DEBUG_ON(digen(parent) != sgen);
#ifdef CONFIG_AUFS_DEBUG
	{
		struct dentry *d = parent;
		while (!IS_ROOT(d)) {
			DEBUG_ON(digen(d) != sgen);
			d = d->d_parent;
		}
	}
#endif
	type = dentry->d_inode->i_mode & S_IFMT;
	/* returns a number of positive dentries */
	err = refresh_dentry(dentry, type);
	if (err >= 0)
		err = refresh_inode(dentry);
	di_read_unlock(parent, AUFS_I_RLOCK);
	dput(parent);
	TraceErr(err);
	return err;
}

int reval_dpath(struct dentry *dentry, int sgen)
{
	int err;
	struct dentry *d, *parent;

	LKTRTrace("%.*s, sgen %d\n", DLNPair(dentry), sgen);
	DEBUG_ON(!dentry->d_inode);
	DiMustWriteLock(dentry);

	if (!stopd(dentry->d_sb)->si_failed_refresh_dirs)
		return simple_reval_dpath(dentry, sgen);

	/* slow loop, keep it simple and stupid */
	/* cf: cpup_dirs() */
	err = 0;
	while (digen(dentry) != sgen) {
		d = dentry;
		while (1) {
			parent = d->d_parent; // dget_parent()
			if (digen(parent) == sgen)
				break;
			d = parent;
		}

		if (d != dentry)
			di_write_lock(d);

		/* someone might update our dentry while we were sleeping */
		if (AufsGenOlder(digen(d), sgen)) {
			di_read_lock(parent, AUFS_I_RLOCK);
			/* returns a number of positive dentries */
			err = refresh_dentry(d, d->d_inode->i_mode & S_IFMT);
			//err = -1;
			if (err >= 0)
				err = refresh_inode(d);
			//err = -1;
			di_read_unlock(parent, AUFS_I_RLOCK);
		}

		if (d != dentry)
			di_write_unlock(d);
		if (unlikely(err))
			break;
	}

	TraceErr(err);
	return err;
}

/* THIS IS A BOOLEAN FUNCTION: returns 1 if valid, 0 otherwise */
static int aufs_d_revalidate(struct dentry *dentry, struct nameidata *nd)
{
	int valid, sgen, err;
	struct super_block *sb;

	LKTRTrace("dentry %.*s, nd %.*s\n",
		  DLNPair(dentry), DLNPair(nd->dentry));
	//dir case: DEBUG_ON(dentry->d_parent != nd->dentry);
	//remove failure case: DEBUG_ON(!IS_ROOT(dentry) && d_unhashed(dentry));
	DEBUG_ON(!dentry->d_fsdata);

	sb = dentry->d_sb;
	si_read_lock(sb);
	sgen = sigen(dentry->d_sb);
	if (digen(dentry) == sgen)
		di_read_lock(dentry, AUFS_I_RLOCK);
	else {
		DEBUG_ON(!dentry->d_inode);
		di_write_lock(dentry);
		err = reval_dpath(dentry, sgen);
		//err = -1;
		di_downgrade_lock(dentry, AUFS_I_RLOCK);
		if (unlikely(err))
			goto out;
	}

#if 0 // fix it
	/* parent dir i_nlink is not updated in the case of setattr */
	if (S_ISDIR(inode->i_mode)) {
		i_lock(inode);
		ii_write_lock(inode);
		cpup_attr_nlink(inode);
		ii_write_unlock(inode);
		i_unlock(inode);
	}
#endif
	err = hidden_d_revalidate(dentry, nd);
	//err = -1;

 out:
	aufs_read_unlock(dentry, AUFS_I_RLOCK);
	TraceErr(err);
	valid = !err;
	if (!valid)
		LKTRTrace("%.*s invalid\n", DLNPair(dentry));
	return valid;
}

static void aufs_d_release(struct dentry *dentry)
{
	struct aufs_dinfo *dinfo;
	aufs_bindex_t bend, bindex;

	LKTRTrace("%.*s\n", DLNPair(dentry));
	DEBUG_ON(!d_unhashed(dentry));

	dinfo = dentry->d_fsdata;
	if (unlikely(!dinfo))
		return;

	/* dentry may not be revalidated */
	bindex = dinfo->di_bstart;
	if (bindex >= 0) {
		struct dentry **p;
		bend = dinfo->di_bend;
		DEBUG_ON(bend < bindex);
		p = dinfo->di_dentry + bindex;
		while (bindex++ <= bend) {
			if (*p)
				dput(*p);
			p++;
		}
	}
	rw_destroy(&dinfo->di_rwsem);
	kfree(dinfo->di_dentry);
	cache_free_dinfo(dinfo);
}

struct dentry_operations aufs_dop = {
	.d_revalidate	= aufs_d_revalidate,
	.d_release	= aufs_d_release,
};
