/*
 * Copyright (C) 2005-2006 Junjiro Okajima
 *
 * This program, aufs is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/* $Id: file.c,v 1.21 2006/10/09 00:28:13 sfjro Exp $ */

//#include <linux/fsnotify.h>
#include <linux/pagemap.h>
//#include <linux/poll.h>
//#include <linux/security.h>
#include "aufs.h"

/* drop flags for writing */
unsigned int file_roflags(unsigned int flags)
{
	flags &= ~(O_WRONLY | O_RDWR | O_APPEND | O_CREAT | O_TRUNC);
	flags |= O_RDONLY | O_NOATIME;
	return flags;
}

/* common functions to regular file and dir */
static struct file *hidden_filp_open(struct dentry *dentry,
				     aufs_bindex_t bindex, int flags,
				     int locked)
{
	struct file *hidden_file;
	char *o, *p;
	struct dentry *hidden_dentry;
	struct inode *hidden_dir;

	LKTRTrace("%.*s, b%d, flags 0%o, l%d\n",
		  DLNPair(dentry), bindex, flags, locked);

	hidden_file = ERR_PTR(-ENOMEM);
	o = __getname();
	if (unlikely(!o))
		goto out;

	hidden_dentry = dtohd_index(dentry, bindex);
	p = d_path(hidden_dentry, sbr_mnt(dentry->d_sb, bindex), o, PATH_MAX);
	hidden_file = (void*)p;
	if (IS_ERR(p))
		goto out_name;

	DEBUG_ON(!SB_NFS(hidden_dentry->d_sb));
	hidden_dir = NULL;
	if (locked) {
		hidden_dir = hidden_dentry->d_parent->d_inode;
		IMustLock(hidden_dir);
		i_unlock(hidden_dir);
	}
	hidden_file = filp_open(p, flags, 0);
	if (locked)
		i_lock(hidden_dir);
	if (unlikely(!IS_ERR(hidden_file)
		     && hidden_file->f_dentry != hidden_dentry)) {
		fput(hidden_file);
		hidden_file = ERR_PTR(-ESTALE);
	}

 out_name:
	__putname(o);
 out:
	TraceErrPtr(hidden_file);
	return hidden_file;
}

struct file *hidden_open(struct dentry *dentry, aufs_bindex_t bindex, int flags,
			 int locked)
{
	struct dentry *hidden_dentry;
	struct inode *hidden_inode;
	struct super_block *sb;
	struct vfsmount *hidden_mnt;
	struct file *hidden_file;
	struct aufs_branch *br;

	LKTRTrace("%.*s, b%d, flags 0%o\n", DLNPair(dentry), bindex, flags);
	DEBUG_ON(!dentry);
	hidden_dentry = dtohd_index(dentry, bindex);
	DEBUG_ON(!hidden_dentry);
	hidden_inode = hidden_dentry->d_inode;
	DEBUG_ON(!hidden_inode);

#ifdef CONFIG_AUFS_DBA
	hidden_file = ERR_PTR(-ENOENT);
	if (unlikely(!hidden_inode->i_nlink))
		goto out;
#endif

	sb = dentry->d_sb;
	br = stobr(sb, bindex);
	br_get(br);
	if (test_ro(sb, bindex, dentry->d_inode))
		/* drop flags for writing */
		flags = file_roflags(flags);
	flags &= ~O_CREAT;

#if 0 //debugging
	{
		static DECLARE_WAIT_QUEUE_HEAD(wq);
		wait_event_timeout(wq, 0, 3 * HZ);
	}
#endif

	// a linux nfs maintainer said that never call dentry_open() to NFS.
	if (!SB_NFS(hidden_dentry->d_sb)) {
		dget(hidden_dentry);
		hidden_mnt = mntget(br->br_mnt);
		hidden_file = dentry_open(hidden_dentry, hidden_mnt, flags);
		//hidden_file = ERR_PTR(-1); mntput(hidden_mnt); dput(hidden_dentry);
	} else
		hidden_file = hidden_filp_open(dentry, bindex, flags, locked);
	if (!IS_ERR(hidden_file))
		return hidden_file;

	br_put(br);

#ifdef CONFIG_AUFS_DBA
 out:
#endif
	TraceErrPtr(hidden_file);
	return hidden_file;
}

int do_open(struct inode *inode, struct file *file,
	    int (*open)(struct file *file, int flags))
{
	int err;
	struct dentry *dentry;

	dentry = file->f_dentry;
	LKTRTrace("i%lu, %.*s\n", inode->i_ino, DLNPair(dentry));

	aufs_read_lock(dentry, AUFS_I_RLOCK);
	err = init_finfo(file);
	//err = -1;
	if (!err) {
		err = open(file, file->f_flags);
		fi_write_unlock(file);
		if (unlikely(err))
			fin_finfo(file);
	}
	//DbgFile(file);
	aufs_read_unlock(dentry, AUFS_I_RLOCK);
	TraceErr(err);
	return err;
}

int reopen_nondir(struct file *file)
{
	int err;
	struct dentry *dentry;
	aufs_bindex_t bstart, bindex, bend;
	struct file *hidden_file;

	dentry = file->f_dentry;
	LKTRTrace("%.*s\n", DLNPair(dentry));
	DEBUG_ON(S_ISDIR(dentry->d_inode->i_mode)
		 || !dtohd(dentry)->d_inode);
	bstart = dbstart(dentry);

	if (fbstart(file) == bstart)
		return 0; /* success */
	DEBUG_ON(fbstart(file) <= bstart
		 || ftopd(file)->fi_hfile[0 + bstart].hf_file);

	hidden_file = hidden_open(dentry, bstart, file->f_flags & ~O_TRUNC,
				  /*locked*/0);
	//hidden_file = ERR_PTR(-1);
	err = PTR_ERR(hidden_file);
	if (IS_ERR(hidden_file))
		goto out; // close all?
	err = 0;
	//cpup_file_flags(hidden_file, file);
	set_fbstart(file, bstart);
	set_ftohf_index(file, bstart, hidden_file);
	memcpy(&hidden_file->f_ra, &file->f_ra, sizeof(file->f_ra)); //??

	/* close lower files */
	bend = fbend(file);
	for (bindex = bstart + 1; bindex <= bend; bindex++)
		set_ftohf_index(file, bindex, NULL);
	set_fbend(file, bstart);

 out:
	TraceErr(err);
	return err;
}

static int cpup_wh_file(struct file *file, aufs_bindex_t bdst, loff_t len)
{
	int err;
	struct dentry *dentry, *parent, *hidden_parent, *tmp_dentry;
	struct dentry *hidden_dentry_bstart, *hidden_dentry_bdst;
	struct inode *hidden_dir;
	aufs_bindex_t bstart;
	struct aufs_dinfo *dinfo;
	struct dtime dt;

	dentry = file->f_dentry;
	LKTRTrace("%.*s, bdst %d, len %Lu\n", DLNPair(dentry), bdst, len);
	DEBUG_ON(S_ISDIR(dentry->d_inode->i_mode)
		 || !(file->f_mode & FMODE_WRITE));
	DiMustWriteLock(dentry);
	parent = dentry->d_parent;
	IiMustAnyLock(parent->d_inode);
	hidden_parent = dtohd_index(parent, bdst);
	DEBUG_ON(!hidden_parent);
	hidden_dir = hidden_parent->d_inode;
	DEBUG_ON(!hidden_dir);
	IMustLock(hidden_dir);

	tmp_dentry = lookup_whtmp(hidden_parent, &dentry->d_name);
	//tmp_dentry = ERR_PTR(-1);
	err = PTR_ERR(tmp_dentry);
	if (IS_ERR(tmp_dentry))
		goto out;

	dtime_store(&dt, NULL, hidden_parent);
	dinfo = dtopd(dentry);
	bstart = dinfo->di_bstart;
	hidden_dentry_bdst = dinfo->di_dentry[0 + bdst];
	hidden_dentry_bstart = dinfo->di_dentry[0 + bstart];
	dinfo->di_bstart = bdst;
	dinfo->di_dentry[0 + bdst] = tmp_dentry;
	dinfo->di_dentry[0 + bstart] = ftohf(file)->f_dentry;
	//smp_mb();
	err = cpup_single(dentry, bdst, bstart, len, 0);
	//err = -1;
	if (!err)
		err = reopen_nondir(file);
		//err = -1;
#if 0 // todo
	dinfo->di_dentry[0 + bstart] = hidden_dentry_bstart;
	dinfo->di_dentry[0 + bdst] = hidden_dentry_bdst;
	dinfo->di_bstart = bstart;
	//smp_mb();
#endif
	if (unlikely(err))
		goto out_tmp;

	DEBUG_ON(!d_unhashed(dentry));
	err = safe_unlink(hidden_dir, tmp_dentry, NULL);
	//err = -1;
	if (unlikely(err)) {
		IOErr("failed remove copied-up tmp file %.*s(%d)\n",
		      DLNPair(tmp_dentry), err);
		err = -EIO;
	}
	dtime_revert(&dt);

 out_tmp:
	dput(tmp_dentry);
 out:
	TraceErr(err);
	return err;
}

int ready_to_write(struct file *file, loff_t len)
{
	int err;
	struct dentry *dentry, *parent, *hidden_dentry, *hidden_parent;
	struct inode *hidden_inode, *hidden_dir, *inode;
	struct super_block *sb;
	aufs_bindex_t bstart, bcpup;

	dentry = file->f_dentry;
	LKTRTrace("%.*s, len %Ld\n", DLNPair(dentry), len);
	DEBUG_ON(!(file->f_mode & FMODE_WRITE));
	FiMustWriteLock(file);

	sb = dentry->d_sb;
	bstart = fbstart(file);
	DEBUG_ON(ftobr_index(file, bstart) != stobr(sb, bstart));

	inode = dentry->d_inode;
	ii_read_lock(inode);
	LKTRTrace("rdonly %d, bstart %d\n",
		  test_ro(sb, bstart, inode), bstart);
	err = test_ro(sb, bstart, inode);
	ii_read_unlock(inode);
	if (!err && (ftohf(file)->f_mode & FMODE_WRITE))
		return 0;

	/* need to cpup */
	parent = dentry->d_parent; // dget_parent()
	di_write_lock(dentry);
	di_write_lock(parent);
	bcpup = err = find_rw_parent_br(dentry, bstart);
	//err = -1;
	if (unlikely(err < 0))
		goto out_unlock;
	err = 0;

	hidden_parent = dtohd_index(parent, bcpup);
	if (!hidden_parent) {
		err = cpup_dirs(dentry, bcpup, NULL);
		//err = -1;
		if (unlikely(err))
			goto out_unlock;
		hidden_parent = dtohd_index(parent, bcpup);
	}

	hidden_dir = hidden_parent->d_inode;
	hidden_dentry = ftohf(file)->f_dentry;
	hidden_inode = hidden_dentry->d_inode;
	i_lock(hidden_dir);
	i_lock(hidden_inode);
	if (d_unhashed(dentry) || d_unhashed(hidden_dentry)
	    /* || !hidden_inode->i_nlink */) {
		if (!test_perm(hidden_dir, MAY_EXEC | MAY_WRITE))
			err = cpup_wh_file(file, bcpup, len);
		else {
			void f(void *completion) {
				err = cpup_wh_file(file, bcpup, len);
				complete(completion);
			}
			wkq_wait(f);
		}
		//err = -1;
		TraceErr(err);
	} else {
		if (!dtohd_index(dentry, bcpup))
			err = sio_cpup_simple(dentry, bcpup, len, 1);
		//err = -1;
		TraceErr(err);
		if (!err)
			err = reopen_nondir(file);
			//err = -1;
		TraceErr(err);
	}
	i_unlock(hidden_inode);
	i_unlock(hidden_dir);

 out_unlock:
	di_write_unlock(parent);
	di_write_unlock(dentry);
// out:
	TraceErr(err);
	return err;

}

/* ---------------------------------------------------------------------- */

static int cpup_pseudo_link(struct dentry *dentry, aufs_bindex_t bdst)
{
	int err;
	struct super_block *sb;
	struct dentry *parent, *hidden_src;
	struct inode *inode, *hidden_inode, *hidden_dir;
	aufs_bindex_t bstart;

	LKTRTrace("%.*s, b%d\n", DLNPair(dentry), bdst);
	DiMustWriteLock(dentry);
	sb = dentry->d_sb;
	DEBUG_ON(test_ro(sb, bdst, NULL));
	inode = dentry->d_inode;
	DEBUG_ON(ibstart(inode) < bdst);

	err = -ENOMEM;
	hidden_inode = itohi(inode);
	igrab(hidden_inode);
	hidden_src = d_alloc_anon(hidden_inode);
	//hidden_src = NULL;
	if (unlikely(!hidden_src)) {
		iput(hidden_inode);
		goto out;
	}

	parent = dentry->d_parent; // dget_parent()
	di_read_lock(parent, AUFS_I_RLOCK);
	err = test_and_cpup_dirs(dentry, bdst, NULL, NULL);
	//err = -1;
	if (unlikely(err))
		goto out_parent;

	hidden_dir = dtohd_index(parent, bdst)->d_inode;
	i_lock(hidden_dir);
	if (!dtohd_index(dentry, bdst)) {
		err = lookup_negative(dentry, bdst);
		//err = -1;
		if (unlikely(err))
			goto out_unlock;
	}

	bstart = ibstart(inode);
	if (bdst == bstart) {
		// vfs_link() does lock the inode
		err = vfs_link(hidden_src, hidden_dir,
			       dtohd_index(dentry, bdst));
		//err = -1;
		if (unlikely(err
			     && (hidden_src->d_flags & DCACHE_DISCONNECTED)))
			Err("please report me this operation\n");
	} else {
		struct aufs_dinfo *dinfo = dtopd(dentry);
		struct dentry *hd = dinfo->di_dentry[0 + bstart];
		aufs_bindex_t old_bend = dinfo->di_bend;
		if (dinfo->di_bend < bstart)
			dinfo->di_bend = bstart;
		dinfo->di_dentry[0 + bstart] = hidden_src;
		//smp_mb();
		err = cpup_single(dentry, bdst, bstart, -1, 1);
		dinfo->di_dentry[0 + bstart] = hd;
		dinfo->di_bend = old_bend;
	}

 out_unlock:
	i_unlock(hidden_dir);
 out_parent:
	di_read_unlock(parent, AUFS_I_RLOCK);
 out:
	if (hidden_src)
		dput(hidden_src);
	TraceErr(err);
	return err;
}

static int refresh_file(struct file *file, int (*reopen)(struct file *file))
{
	int err, new_sz;
	struct dentry *dentry;
	aufs_bindex_t bend, bindex, bstart;
	struct aufs_hfile *p;
	struct aufs_finfo *finfo;
	struct super_block *sb;
	struct inode *inode;
	struct file *hidden_file;

	dentry = file->f_dentry;
	LKTRTrace("%.*s\n", DLNPair(dentry));
	FiMustWriteLock(file);
	DiMustReadLock(dentry);
	inode = dentry->d_inode;
	IiMustReadLock(inode);

	err = -ENOMEM;
	sb = dentry->d_sb;
	finfo = ftopd(file);
	bstart = finfo->fi_bstart;
	bend = finfo->fi_bstart;
	new_sz = sizeof(*finfo->fi_hfile) * (sbend(sb) + 1);
	p = kzrealloc(finfo->fi_hfile, sizeof(*p) * (finfo->fi_bend + 1),
		      new_sz);
	//p = NULL;
	if (unlikely(!p))
		goto out;
	finfo->fi_hfile = p;
	//smp_mb();
	hidden_file = p[0 + bstart].hf_file;

	p = finfo->fi_hfile + finfo->fi_bstart;
	bend = finfo->fi_bend;
	for (bindex = finfo->fi_bstart; bindex <= bend; bindex++, p++) {
		struct aufs_hfile tmp, *q;
		aufs_bindex_t new_bindex;

		if (!p->hf_file)
			continue;
		new_bindex = find_bindex(sb, p->hf_br);
		if (new_bindex == bindex)
			continue;
		if (new_bindex < 0) { // test here
			set_ftohf_index(file, bindex, NULL);
			continue;
		}

		/* swap two hidden inode, and loop again */
		q = finfo->fi_hfile + new_bindex;
		tmp = *q;
		*q = *p;
		*p = tmp;
		if (tmp.hf_file) {
			bindex--;
			p--;
		}
	}

	p = finfo->fi_hfile;
	bend = dbend(dentry);
	for (finfo->fi_bstart = 0; finfo->fi_bstart <= bend;
	     finfo->fi_bstart++, p++)
		if (p->hf_file)
			break;
	p = finfo->fi_hfile + bend;
	for (finfo->fi_bend = bend; finfo->fi_bend >= 0; finfo->fi_bend--, p--)
		if (p->hf_file)
			break;
	//smp_mb();
	DEBUG_ON(finfo->fi_bend < finfo->fi_bstart);

	err = 0;
#if 0 // todo:
	if (!dtohd(dentry)->d_inode) {
		update_figen(file);
		goto out; /* success */
	}
#endif

 again:
	bstart = ibstart(inode);
	if (bstart < finfo->fi_bstart) {
		if (test_ro(sb, bstart, inode)) {
			struct dentry *parent = dentry->d_parent; // dget_parent()
			di_read_lock(parent, 0);
			bstart = err = find_rw_parent_br(dentry, bstart);
			di_read_unlock(parent, 0);
			//todo: err = -1;
			if (unlikely(err < 0))
				goto out;
		}
		di_read_unlock(dentry, AUFS_I_RLOCK);
		di_write_lock(dentry);
		if (bstart != ibstart(inode)) { // todo
			/* someone changed our inode while we were sleeping */
			di_downgrade_lock(dentry, AUFS_I_RLOCK);
			goto again;
		}

		// always superio.
		if (!is_kthread(current)) {
			void f(void *completion) {
				err = cpup_pseudo_link(dentry, bstart);
				complete(completion);
			}
			wkq_wait(f);
		} else
			err = cpup_pseudo_link(dentry, bstart);
		//err = -1;
		di_downgrade_lock(dentry, AUFS_I_RLOCK);
		if (unlikely(err))
			goto out;
	}

	err = reopen(file);
	//err = -1;
	if (!err) {
		update_figen(file);
		return 0; /* success */
	}

	/* error, close all hidden files */
	bend = fbend(file);
	for (bindex = fbstart(file); bindex <= bend; bindex++)
		set_ftohf_index(file, bindex, NULL);

 out:
	TraceErr(err);
	return err;
}

/* common function to regular file and dir */
int reval_and_lock_finfo(struct file *file, int (*reopen)(struct file *file),
			 int wlock, int locked)
{
	int err, sgen, fgen, pseudo_link;
	struct dentry *dentry;
	struct super_block *sb;

	dentry = file->f_dentry;
	LKTRTrace("%.*s, w %d, l %d\n", DLNPair(dentry), wlock, locked);
	sb = dentry->d_sb;
	SiMustAnyLock(sb);

	err = 0;
	if (IS_ROOT(dentry) || is_mmapped(file))
		goto out_lock;

	sgen = sigen(sb);
	di_read_lock(dentry, AUFS_I_RLOCK);
	fgen = figen(file);
	pseudo_link = (dbstart(dentry) != ibstart(dentry->d_inode));
	di_read_unlock(dentry, AUFS_I_RLOCK);
	if (sgen == fgen && !pseudo_link)
		goto out_lock;

	LKTRTrace("sgen %d, fgen %d\n", sgen, fgen);
	if (sgen != digen(dentry)) {
		/*
		 * d_path() and path_lookup() is a simple and good approach
		 * to revalidate. but si_rwsem in DEBUG_RWSEM will cause a
		 * deadlock. removed the code.
		 */
		di_write_lock(dentry);
		err = reval_dpath(dentry, sgen);
		//err = -1;
		di_write_unlock(dentry);
		if (unlikely(err < 0))
			goto out;
		DEBUG_ON(digen(dentry) != sgen);
	}

	fi_write_lock(file);
	di_read_lock(dentry, AUFS_I_RLOCK);
	err = refresh_file(file, reopen);
	//err = -1;
	di_read_unlock(dentry, AUFS_I_RLOCK);
	if (!err) {
		if (!wlock)
			fi_downgrade_lock(file);
	} else
		fi_write_unlock(file);
	goto out;

 out_lock:
	if (!wlock)
		fi_read_lock(file);
	else
		fi_write_lock(file);
 out:
	TraceErr(err);
	return err;
}

/* ---------------------------------------------------------------------- */

// cf. aufs_nopage()
// for madvise(2)
static int aufs_readpage(struct file *file, struct page *page)
{
	TraceEnter();
	unlock_page(page);
	return 0;
}

// they will never be called.
#ifdef CONFIG_AUFS_DEBUG
static int aufs_prepare_write(struct file *file, struct page *page,
			      unsigned from, unsigned to)
{BUG();return 0;}
static int aufs_commit_write(struct file *file, struct page *page,
			     unsigned from, unsigned to)
{BUG();return 0;}
static int aufs_writepage(struct page *page, struct writeback_control *wbc)
{BUG();return 0;}
static void aufs_sync_page(struct page *page)
{BUG();}

#if 0 // comment
static int aufs_writepages(struct address_space *mapping,
			   struct writeback_control *wbc)
{BUG();return 0;}
static int aufs_readpages(struct file *filp, struct address_space *mapping,
			  struct list_head *pages, unsigned nr_pages)
{BUG();return 0;}
static sector_t aufs_bmap(struct address_space *mapping, sector_t block)
{BUG();return 0;}
#endif

static int aufs_set_page_dirty(struct page *page)
{BUG();return 0;}
static void aufs_invalidatepage (struct page *page, unsigned long offset)
{BUG();}
static int aufs_releasepage (struct page *page, gfp_t gfp)
{BUG();return 0;}
static ssize_t aufs_direct_IO(int rw, struct kiocb *iocb,
			      const struct iovec *iov, loff_t offset,
			      unsigned long nr_segs)
{BUG();return 0;}
static struct page* aufs_get_xip_page(struct address_space *mapping,
				      sector_t offset, int create)
{BUG();return NULL;}
//static int aufs_migratepage (struct page *newpage, struct page *page)
//{BUG();return 0;}
#endif

#if 0 // comment
struct address_space {
	struct inode		*host;		/* owner: inode, block_device */
	struct radix_tree_root	page_tree;	/* radix tree of all pages */
	rwlock_t		tree_lock;	/* and rwlock protecting it */
	unsigned int		i_mmap_writable;/* count VM_SHARED mappings */
	struct prio_tree_root	i_mmap;		/* tree of private and shared mappings */
	struct list_head	i_mmap_nonlinear;/*list VM_NONLINEAR mappings */
	spinlock_t		i_mmap_lock;	/* protect tree, count, list */
	unsigned int		truncate_count;	/* Cover race condition with truncate */
	unsigned long		nrpages;	/* number of total pages */
	pgoff_t			writeback_index;/* writeback starts here */
	struct address_space_operations *a_ops;	/* methods */
	unsigned long		flags;		/* error bits/gfp mask */
	struct backing_dev_info *backing_dev_info; /* device readahead, etc */
	spinlock_t		private_lock;	/* for use by the address_space */
	struct list_head	private_list;	/* ditto */
	struct address_space	*assoc_mapping;	/* ditto */
} __attribute__((aligned(sizeof(long))));

struct address_space_operations {
	int (*writepage)(struct page *page, struct writeback_control *wbc);
	int (*readpage)(struct file *, struct page *);
	void (*sync_page)(struct page *);

	/* Write back some dirty pages from this mapping. */
	int (*writepages)(struct address_space *, struct writeback_control *);

	/* Set a page dirty.  Return true if this dirtied it */
	int (*set_page_dirty)(struct page *page);

	int (*readpages)(struct file *filp, struct address_space *mapping,
			struct list_head *pages, unsigned nr_pages);

	/*
	 * ext3 requires that a successful prepare_write() call be followed
	 * by a commit_write() call - they must be balanced
	 */
	int (*prepare_write)(struct file *, struct page *, unsigned, unsigned);
	int (*commit_write)(struct file *, struct page *, unsigned, unsigned);
	/* Unfortunately this kludge is needed for FIBMAP. Don't use it */
	sector_t (*bmap)(struct address_space *, sector_t);
	void (*invalidatepage) (struct page *, unsigned long);
	int (*releasepage) (struct page *, gfp_t);
	ssize_t (*direct_IO)(int, struct kiocb *, const struct iovec *iov,
			loff_t offset, unsigned long nr_segs);
	struct page* (*get_xip_page)(struct address_space *, sector_t,
			int);
	/* migrate the contents of a page to the specified target */
	int (*migratepage) (struct page *, struct page *);
};
#endif

struct address_space_operations aufs_aop = {
	.readpage	= aufs_readpage,
#ifdef CONFIG_AUFS_DEBUG
	.writepage	= aufs_writepage,
	.sync_page	= aufs_sync_page,
	//.writepages	= aufs_writepages,
	.set_page_dirty	= aufs_set_page_dirty,
	//.readpages	= aufs_readpages,
	.prepare_write	= aufs_prepare_write,
	.commit_write	= aufs_commit_write,
	//.bmap		= aufs_bmap,
	.invalidatepage	= aufs_invalidatepage,
	.releasepage	= aufs_releasepage,
	.direct_IO	= aufs_direct_IO,
	.get_xip_page	= aufs_get_xip_page,
	//.migratepage	= aufs_migratepage
#endif
};
