/*
 * Copyright (C) 2005-2006 Junjiro Okajima
 *
 * This program, aufs is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/* $Id: i_op_ren.c,v 1.18 2006/10/09 00:24:13 sfjro Exp $ */

//#include <linux/fs.h>
//#include <linux/namei.h>
#include "aufs.h"

enum {SRC, DST};
struct rename_args {
	struct dentry *hidden_dentry[2], *parent[2], *hidden_parent[2];
	struct aufs_nhash whlist;
	aufs_bindex_t btgt, bstart[2];

	unsigned int isdir:1;
	unsigned int issamedir:1;
	unsigned int whsrc:1;
	unsigned int whdst:1;
};

static int do_rename(struct inode *src_dir, struct dentry *src_dentry,
		     struct inode *dir, struct dentry *dentry,
		     struct rename_args *a)
{
	int err, need_diropq, bycpup, rerr;
	struct rmdir_whtmp_arg *tharg;
	struct dentry *wh_dentry[2], *hidden_dst;
	struct inode *hidden_dir[2];
	aufs_bindex_t bindex, bend;

	LKTRTrace("%.*s/%.*s, %.*s/%.*s, "
		  "hd{%p, %p}, hp{%p, %p}, wh %p, btgt %d, bstart{%d, %d}, "
		  "flags{%d, %d, %d, %d}\n",
		  DLNPair(a->parent[SRC]), DLNPair(src_dentry),
		  DLNPair(a->parent[DST]), DLNPair(dentry),
		  a->hidden_dentry[SRC], a->hidden_dentry[DST],
		  a->hidden_parent[SRC], a->hidden_parent[DST],
		  &a->whlist, a->btgt,
		  a->bstart[SRC], a->bstart[DST],
		  a->isdir, a->issamedir, a->whsrc, a->whdst);
	hidden_dir[SRC] = a->hidden_parent[SRC]->d_inode;
	hidden_dir[DST] = a->hidden_parent[DST]->d_inode;
	IMustLock(hidden_dir[SRC]);
	IMustLock(hidden_dir[DST]);

	/* prepare workqueue arg */
	hidden_dst = NULL;
	tharg = NULL;
	if (a->isdir && a->hidden_dentry[DST]->d_inode) {
		err = -ENOMEM;
		tharg = kmalloc(sizeof(*tharg), GFP_KERNEL);
		//tharg = NULL;
		if (unlikely(!tharg))
			goto out;
		hidden_dst = dget(a->hidden_dentry[DST]);
	}

	wh_dentry[SRC] = wh_dentry[DST] = NULL;
	/* create whiteout for src_dentry */
	if (a->whsrc) {
		wh_dentry[SRC] = simple_create_wh(src_dentry, a->btgt,
						  a->hidden_parent[SRC]);
		//wh_dentry[SRC] = ERR_PTR(-1);
		err = PTR_ERR(wh_dentry[SRC]);
		if (IS_ERR(wh_dentry[SRC]))
			goto out_tharg;
	}

	/* lookup whiteout for dentry */
	if (a->whdst) {
		struct dentry *d;
		d = lookup_wh(a->hidden_parent[DST], &dentry->d_name);
		//d = ERR_PTR(-1);
		err = PTR_ERR(d);
		if (IS_ERR(d))
			goto out_whsrc;
		if (!d->d_inode)
			dput(d);
		else
			wh_dentry[DST] = d;
	}

	/* rename dentry to tmpwh */
	if (tharg) {
		err = rename_whtmp(a->hidden_parent[DST],
				   a->hidden_dentry[DST]);
		//err = -1;
		if (unlikely(err))
			goto out_whdst;
		set_dtohd_index(dentry, a->btgt, NULL);
		err = lookup_negative(dentry, a->btgt);
		//err = -1;
		if (unlikely(err))
			goto out_whtmp;
		a->hidden_dentry[DST] = dtohd_index(dentry, a->btgt);
	}

	/* cpup src */
	if (a->hidden_dentry[DST]->d_inode && a->bstart[SRC] != a->btgt) {
		i_lock(a->hidden_dentry[SRC]->d_inode);
		err = sio_cpup_simple(src_dentry, a->btgt, -1, 0);
		//err = -1; // untested dir
		i_unlock(a->hidden_dentry[SRC]->d_inode);
		if (unlikely(err))
			goto out_whtmp;
	}

	/* rename by vfs_rename or cpup */
	need_diropq = a->isdir;
	bycpup = 0;
	if (dbstart(src_dentry) == a->btgt) {
		if (a->isdir && dbdiropq(src_dentry) == a->btgt)
			need_diropq = 0;
		err = vfs_rename(hidden_dir[SRC], dtohd(src_dentry),
				 hidden_dir[DST], a->hidden_dentry[DST]);
		//err = -1;
	} else {
		bycpup = 1;
		i_lock(a->hidden_dentry[SRC]->d_inode);
		set_dbstart(src_dentry, a->btgt);
		set_dtohd_index(src_dentry, a->btgt,
				dget(a->hidden_dentry[DST]));
		err = sio_cpup_single(src_dentry, a->btgt, a->bstart[SRC], -1,
				      /*do_dt*/0);
		//err = -1; // untested dir
		if (unlikely(err)) {
			set_dtohd_index(src_dentry, a->btgt, NULL);
			set_dbstart(src_dentry, a->bstart[SRC]);
		}
		i_unlock(a->hidden_dentry[SRC]->d_inode);
	}
	if (unlikely(err))
		goto out_whtmp;

	/* make dir opaque */
	if (need_diropq) {
		struct dentry *diropq;
		struct inode *hi = dtohd_index(src_dentry, a->btgt)->d_inode;

		i_lock(hi);
		diropq = create_diropq(src_dentry, a->btgt);
		//diropq = ERR_PTR(-1);
		i_unlock(hi);
		err = PTR_ERR(diropq);
		if (IS_ERR(diropq))
			goto out_rename;
		dput(diropq);
	}

	/* remove whiteout for dentry */
	if (wh_dentry[DST]) {
		err = unlink_wh_dentry(hidden_dir[DST], wh_dentry[DST], dentry);
		//err = -1;
		if (unlikely(err))
			goto out_diropq;
	}

	/* remove whtmp */
	if (tharg) {
		if (SB_NFS(hidden_dst->d_sb)
		    || !is_longer_wh(&a->whlist, a->btgt,
				     stopd(dentry->d_sb)->si_dirwh)) {
			err = rmdir_whtmp(hidden_dst, &a->whlist, a->btgt, dir);
			if (unlikely(err))
				Warn("failed removing whtmp dir %.*s (%d), "
				     "ignored.\n", DLNPair(hidden_dst), err);
		} else {
			kick_rmdir_whtmp(hidden_dst, &a->whlist, a->btgt, dir,
					 tharg);
			dput(hidden_dst);
			tharg = NULL;
		}
	}
	err = 0;
	goto out_success;

#define RevertFailure(fmt, args...) { \
		IOErrWhck("revert failure: " fmt " (%d, %d)\n", \
			##args, err, rerr); \
		err = -EIO; \
	}

 out_diropq:
	if (need_diropq) {
		struct inode *hi = dtohd_index(src_dentry, a->btgt)->d_inode;
		i_lock(hi);
		rerr = remove_diropq(src_dentry, a->btgt);
		//rerr = -1;
		i_unlock(hi);
		if (rerr)
			RevertFailure("remove diropq %.*s",
				      DLNPair(src_dentry));
	}
 out_rename:
	if (!bycpup) {
		struct dentry *d;
		struct qstr *name = &src_dentry->d_name;
		d = lookup_one_len(name->name, a->hidden_parent[SRC],
				   name->len);
		//d = ERR_PTR(-1);
		rerr = PTR_ERR(d);
		if (IS_ERR(d)) {
			RevertFailure("lookup_one_len %.*s",
				      DLNPair(src_dentry));
			goto out_whtmp;
		}
		rerr = vfs_rename
			(hidden_dir[DST], dtohd_index(src_dentry, a->btgt),
			 hidden_dir[SRC], d);
		//rerr = -1;
		d_drop(d);
		dput(d);
		//set_dtohd_index(src_dentry, a->btgt, NULL);
		if (rerr)
			RevertFailure("rename %.*s", DLNPair(src_dentry));
	} else {
		rerr = safe_unlink(hidden_dir[DST], a->hidden_dentry[DST],
				   NULL);
		//rerr = -1;
		set_dtohd_index(src_dentry, a->btgt, NULL);
		set_dbstart(src_dentry, a->bstart[SRC]);
		if (rerr)
			RevertFailure("unlink %.*s",
				      DLNPair(a->hidden_dentry[DST]));
	}
 out_whtmp:
	if (tharg) {
		struct dentry *d;
		struct qstr *name = &dentry->d_name;
		LKTRLabel(here);
		d = lookup_one_len(name->name, a->hidden_parent[DST],
				   name->len);
		//d = ERR_PTR(-1);
		rerr = PTR_ERR(d);
		if (IS_ERR(d)) {
			RevertFailure("lookup %.*s", LNPair(name));
			goto out_whdst;
		}
		if (d->d_inode) {
			d_drop(d);
			dput(d);
			goto out_whdst;
		}
		rerr = vfs_rename(hidden_dir[DST], hidden_dst,
				  hidden_dir[DST], d);
		//rerr = -1;
		d_drop(d);
		dput(d);
		if (rerr) {
			RevertFailure("rename %.*s", DLNPair(hidden_dst));
			goto out_whdst;
		}
		set_dtohd_index(dentry, a->btgt, NULL);
		set_dtohd_index(dentry, a->btgt, dget(hidden_dst));
	}
 out_whdst:
	if (wh_dentry[DST]) {
		dput(wh_dentry[DST]);
		wh_dentry[DST] = NULL;
	}
 out_whsrc:
	if (wh_dentry[SRC]) {
		LKTRLabel(here);
		rerr = unlink_wh_dentry(hidden_dir[SRC], wh_dentry[SRC],
					src_dentry);
		//rerr = -1;
		if (rerr)
			RevertFailure("unlink %.*s", DLNPair(wh_dentry[SRC]));
	}
#undef RevertFailure
	d_drop(src_dentry);
	bend = dbend(src_dentry);
	for (bindex = dbstart(src_dentry); bindex <= bend; bindex++) {
		struct dentry *hd;
		hd = dtohd_index(src_dentry, bindex);
		if (hd)
			d_drop(hd);
	}
	d_drop(dentry);
	bend = dbend(dentry);
	for (bindex = dbstart(dentry); bindex <= bend; bindex++) {
		struct dentry *hd;
		hd = dtohd_index(dentry, bindex);
		if (hd)
			d_drop(hd);
	}
	update_dbstart(dentry);
	if (tharg)
		d_drop(hidden_dst);
 out_success:
	if (wh_dentry[SRC])
		dput(wh_dentry[SRC]);
	if (wh_dentry[DST])
		dput(wh_dentry[DST]);
 out_tharg:
	if (tharg) {
		dput(hidden_dst);
		kfree(tharg);
	}
 out:
	TraceErr(err);
	return err;
}

/*
 * test if @dentry dir can be rename destination or not.
 * success means, it is a logically empty dir.
 */
static int may_rename_dstdir(struct dentry *dentry, aufs_bindex_t btgt,
			     struct aufs_nhash *whlist)
{
	LKTRTrace("%.*s\n", DLNPair(dentry));

	return test_empty(dentry, whlist);
}

/*
 * test if @dentry dir can be rename source or not.
 * if it can, return 0 and @children is filled.
 * success means,
 * - or, it is a logically empty dir.
 * - or, it exists on writable branch and has no children including whiteouts
 *       on the lower branch.
 */
static int may_rename_srcdir(struct dentry *dentry, aufs_bindex_t btgt)
{
	int err;
	aufs_bindex_t bstart;

	LKTRTrace("%.*s\n", DLNPair(dentry));

	bstart = dbstart(dentry);
	if (bstart != btgt) {
		struct aufs_nhash whlist;

		init_nhash(&whlist);
		err = test_empty(dentry, &whlist);
		free_nhash(&whlist);
		goto out;
	}

	if (bstart == dbtaildir(dentry))
		return 0; /* success */

	err = test_empty_lower(dentry);

 out:
	if (/* unlikely */(err == -ENOTEMPTY))
		err = -EXDEV;
	TraceErr(err);
	return err;
}

int aufs_rename(struct inode *src_dir, struct dentry *src_dentry,
		struct inode *dir, struct dentry *dentry)
{
	int err;
	aufs_bindex_t bend, bindex;
	struct inode *inode;
	enum {PARENT, CHILD};
	struct dtime dt[2][2];
	struct rename_args a;

	LKTRTrace("i%lu, %.*s, i%lu, %.*s\n",
		  src_dir->i_ino, DLNPair(src_dentry),
		  dir->i_ino, DLNPair(dentry));
	IMustLock(src_dir);
	IMustLock(dir);
	if (dentry->d_inode)
		IMustLock(dentry->d_inode);

	inode = src_dentry->d_inode;
	a.isdir = S_ISDIR(inode->i_mode);
	if (/* unlikely */(a.isdir && dentry->d_inode
			   && !S_ISDIR(dentry->d_inode->i_mode)))
		return -ENOTDIR;

	aufs_read_and_write_lock2(dentry, src_dentry, a.isdir);
	a.parent[SRC] = a.parent[DST] = dentry->d_parent;
	a.issamedir = (src_dir == dir);
	if (a.issamedir)
		di_write_lock(a.parent[DST]);
	else {
		a.parent[SRC] = src_dentry->d_parent;
		di_write_lock2(a.parent[SRC], a.parent[DST], /*isdir*/1);
	}

	/* which branch we process */
	a.bstart[DST] = dbstart(dentry);
	a.btgt = err = wr_dir(dentry, 1, src_dentry, -1, 0);
	if (unlikely(err < 0))
		goto out;

	/* are they available to be renamed */
	err = 0;
	init_nhash(&a.whlist);
	if (a.isdir && dentry->d_inode) {
		set_dbstart(dentry, a.bstart[DST]);
		err = may_rename_dstdir(dentry, a.btgt, &a.whlist);
		set_dbstart(dentry, a.btgt);
	}
	a.hidden_dentry[DST] = dtohd(dentry);
	if (unlikely(err))
		goto out;
	a.bstart[SRC] = dbstart(src_dentry);
	a.hidden_dentry[SRC] = dtohd(src_dentry);
	if (a.isdir) {
		err = may_rename_srcdir(src_dentry, a.btgt);
		if (unlikely(err))
			goto out_children;
	}

	/* prepare the writable parent dir on the same branch */
	a.whsrc = err = wr_dir_need_wh(src_dentry, a.isdir, &a.btgt,
				       a.issamedir ? NULL : a.parent[DST]);
	if (unlikely(err < 0))
		goto out_children;
	a.whdst = (a.bstart[DST] == a.btgt);
	if (!a.whdst) {
		err = cpup_dirs(dentry, a.btgt,
				a.issamedir ? NULL : a.parent[SRC]);
		if (unlikely(err))
			goto out_children;
	}

	a.hidden_parent[SRC] = dtohd_index(a.parent[SRC], a.btgt);
	a.hidden_parent[DST] = dtohd_index(a.parent[DST], a.btgt);
	lock_rename(a.hidden_parent[SRC], a.hidden_parent[DST]);

	/* store timestamps to be revertible */
	dtime_store(dt[PARENT] + SRC, a.parent[SRC], a.hidden_parent[SRC]);
	dtime_store(dt[PARENT] + DST, NULL, NULL);
	if (!a.issamedir)
		dtime_store(dt[PARENT] + DST, a.parent[DST],
			    a.hidden_parent[DST]);
	if (a.isdir) {
		dtime_store(dt[CHILD] + SRC, src_dentry, a.hidden_dentry[SRC]);
		dtime_store(dt[CHILD] + DST, NULL, NULL);
		if (a.hidden_dentry[DST]->d_inode)
			dtime_store(dt[CHILD] + DST, dentry,
				    a.hidden_dentry[DST]);
	}

	//smp_mb();
	err = do_rename(src_dir, src_dentry, dir, dentry, &a);
	if (unlikely(err))
		goto out_dt;
	unlock_rename(a.hidden_parent[SRC], a.hidden_parent[DST]);

	/* update dir attributes */
	if (ibstart(dir) == a.btgt) {
		cpup_attr_timesizes(dir);
		if (a.isdir)
			cpup_attr_nlink(dir);
		dir->i_version++;
	}
	if (!a.issamedir && ibstart(src_dir) == a.btgt) {
		cpup_attr_timesizes(src_dir);
		if (a.isdir)
			cpup_attr_nlink(src_dir);
		src_dir->i_version++;
	}
	// is this POSIX?
	if (a.isdir) {
		//i_lock(inode);
		cpup_attr_timesizes(inode);
		//i_unlock(inode);
	}

	/* dput/iput all lower dentries */
	set_dbwh(src_dentry, -1);
	bend = dbend(src_dentry);
	for (bindex = a.btgt + 1; bindex <= bend; bindex++) {
		struct dentry *hd;
		hd = dtohd_index(src_dentry, bindex);
		if (hd)
			set_dtohd_index(src_dentry, bindex, NULL);
	}
	set_dbend(src_dentry, a.btgt);

	bend = ibend(inode);
	for (bindex = a.btgt + 1; bindex <= bend; bindex++) {
		struct inode *hi;
		hi = itohi_index(inode, bindex);
		if (hi)
			set_itohi_index(inode, bindex, NULL, 0);
	}
	set_ibend(inode, a.btgt);
	goto out_children; /* success */

 out_dt:
	dtime_revert(dt[PARENT] + SRC);
	if (!a.issamedir)
		dtime_revert(dt[PARENT] + DST);
	if (a.isdir && err != -EIO) {
		int i;
		for (i = 0; i < 2; i++) {
			struct dentry *hd;
			hd = dt[CHILD][i].dt_hidden_dentry;
			if (!hd || !hd->d_inode)
				continue;
			i_lock(hd->d_inode);
			dtime_revert(dt[CHILD] + i);
			i_unlock(hd->d_inode);
		}
	}
	unlock_rename(a.hidden_parent[SRC], a.hidden_parent[DST]);
 out_children:
	free_nhash(&a.whlist);
 out:
	if (unlikely(err && a.isdir)) {
		update_dbstart(dentry);
		d_drop(dentry);
	}
	if (a.issamedir)
		di_write_unlock(a.parent[DST]);
	else
		di_write_unlock2(a.parent[SRC], a.parent[DST]);
	aufs_read_and_write_unlock2(dentry, src_dentry);
	TraceErr(err);
	return err;
}
