/*
 * Copyright (C) 2005-2006 Junjiro Okajima
 *
 * This program, aufs is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/* $Id: super.c,v 1.25 2006/10/09 00:26:10 sfjro Exp $ */

#include <linux/init.h>
#include <linux/module.h>
#include <linux/seq_file.h>
#include <linux/statfs.h>
#include <linux/version.h>
#include "aufs.h"

/*
 * super_operations
 */
static struct inode *aufs_alloc_inode(struct super_block *sb)
{
	struct aufs_icntnr *c;

	TraceEnter();

	c = cache_alloc_icntnr();
	//c = NULL;
	if (c) {
		inode_init_once(&c->vfs_inode);
		c->vfs_inode.i_version = 1; //sigen(sb);
		c->iinfo.ii_hinode = NULL;
		return &c->vfs_inode;
	}
	return NULL;
}

static void aufs_destroy_inode(struct inode *inode)
{
	LKTRTrace("i%lu\n", inode->i_ino);
	iinfo_fin(inode);
	cache_free_icntnr(container_of(inode, struct aufs_icntnr, vfs_inode));
}

static void aufs_read_inode(struct inode *inode)
{
	int err;

	LKTRTrace("i%lu\n", inode->i_ino);

	err = iinfo_init(inode);
	//if (LktrCond) err = -1;else err = iinfo_init(inode);
	if (!err) {
		inode->i_version++;
		inode->i_op = &aufs_iop;
		inode->i_fop = &aufs_file_fop;
		inode->i_mapping->a_ops = &aufs_aop;
		return; /* success */
	}

	LKTRTrace("intializing inode info failed(%d)\n", err);
	make_bad_inode(inode);
}

static int aufs_show_options(struct seq_file *m, struct vfsmount *mnt)
{
	int err, n;
	aufs_bindex_t bindex, bend;
	struct super_block *sb;
	struct aufs_sbinfo *sbinfo;
	char *hidden_path, *page;
	struct dentry *root;

	TraceEnter();

	err = -ENOMEM;
	page = __getname();
	//page = NULL;
	if (unlikely(!page))
		goto out;

	sb = mnt->mnt_sb;
	root = sb->s_root;
	aufs_read_lock(root, 0);
	sbinfo = stopd(sb);
	if (IS_MS(sb, MS_XINO)) {
		struct aufs_branch *br;
		struct file *f;
		char *p, *q;

		br = stobr(sb, 0);
		f = br->br_xino;
		p = d_path(f->f_dentry, f->f_vfsmnt, page, PATH_MAX);
		//p = ERR_PTR(-1);
		err = PTR_ERR(p);
		if (IS_ERR(p))
			goto out_unlock;
		q = strchr(p, ' ');
		DEBUG_ON(!q);
		*q = 0;
		seq_printf(m, ",xino=%s", p);
	} else
		seq_printf(m, ",noxino");
	if (unlikely(!IS_MS(sb, MS_PLINK)))
		seq_printf(m, ",noplink");
	n = sbinfo->si_udba;
	if (unlikely(n != AUFS_UDBA_DEF))
		seq_printf(m, ",udba=%s", udba_str(n));
	n = sbinfo->si_dirwh;
	if (unlikely(n != AUFS_DIRWH))
		seq_printf(m, ",dirwh=%d", n);
	n = sbinfo->si_rdcache / HZ;
	if (unlikely(n != AUFS_RDCACHE))
		seq_printf(m, ",rdcache=%d", n);

#ifdef CONFIG_AUFS_COMPAT
	seq_printf(m, ",dirs=");
#else
	seq_printf(m, ",br:");
#endif
	err = 0;
	bend = sbend(sb);
	for (bindex = 0; bindex <= bend; bindex++) {
		char a[16];
		hidden_path = d_path(dtohd_index(root, bindex),
				     sbr_mnt(sb, bindex), page, PATH_MAX);
		//hidden_path = ERR_PTR(-1);
		err = PTR_ERR(hidden_path);
		if (IS_ERR(hidden_path))
			break;
		err = sbr_perm_str(sb, bindex, a, sizeof(a));
		//err = -1;
		if (unlikely(err))
			break;
		seq_printf(m, "%s=%s", hidden_path, a);
		if (bindex != bend)
			seq_printf(m, ":");
	}

 out_unlock:
	aufs_read_unlock(root, 0);
	__putname(page);
 out:
	TraceErr(err);
	return err;
}

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,18)
#define StatfsSb(d)	(d)->d_sb
#define StatfsArg(s)	(sbr_sb(s, 0)->s_root)
static int aufs_statfs(struct dentry *arg, struct kstatfs *buf)
#else
#define StatfsSb(s)	s
#define StatfsArg(s)	(sbr_sb(s, 0))
static int aufs_statfs(struct super_block *arg, struct kstatfs *buf)
#endif
{
	int err;
	struct super_block *sb = StatfsSb(arg);

	TraceEnter();

	si_read_lock(sb);
	err = vfs_statfs(StatfsArg(sb), buf);
	si_read_unlock(sb);
	//err = -1;
	if (!err) {
		//buf->f_type = AUFS_SUPER_MAGIC;
		buf->f_type = 0;
		buf->f_namelen -= WHLEN;
	}
	//buf->f_bsize = buf->f_blocks = buf->f_bfree = buf->f_bavail = -1;

	TraceErr(err);
	return err;
}
#undef StatfsSb
#undef StatfsArg

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,18)
#define UmountBeginSb(mnt)	(mnt)->mnt_sb
static void aufs_umount_begin(struct vfsmount *arg, int flags)
#else
#define UmountBeginSb(sb)	(sb)
static void aufs_umount_begin(struct super_block *arg)
#endif
{
	struct super_block *sb = UmountBeginSb(arg);

	if (IS_MS(sb, MS_PLINK)) {
		si_write_lock(sb);
		put_plink(sb);
		//kobj_umount(stopd(sb));
		si_write_unlock(sb);
	}
}
#undef UmountBeginSb

static void free_sbinfo(struct aufs_sbinfo *sbinfo)
{
	TraceEnter();
	DEBUG_ON(!sbinfo);
	DEBUG_ON(!list_empty(&sbinfo->si_plink));

	rw_destroy(&sbinfo->si_rwsem);
	free_branches(sbinfo);
	kfree(sbinfo->si_branch);
	kfree(sbinfo->si_dupopt);
	kfree(sbinfo);
}

/* final actions when unmounting a file system */
static void aufs_put_super(struct super_block *sb)
{
	struct aufs_sbinfo *sbinfo;

	TraceEnter();

	sbinfo = stopd(sb);
	if (sbinfo)
		free_sbinfo(sbinfo);
}

/* ---------------------------------------------------------------------- */

/*
 * refresh directories at remount time.
 */
static int do_refresh_dir(struct dentry *dentry)
{
	int err;
	struct dentry *parent;

	LKTRTrace("%.*s\n", DLNPair(dentry));
	DEBUG_ON(!dentry->d_inode || !S_ISDIR(dentry->d_inode->i_mode));

	di_write_lock(dentry);
	parent = dentry->d_parent;
	if (!IS_ROOT(parent))
		di_read_lock(parent, AUFS_I_RLOCK);
	else
		DiMustAnyLock(parent);
	err = refresh_dentry(dentry, S_IFDIR);
	if (err >= 0)
		err = refresh_inode(dentry);
	if (!IS_ROOT(parent))
		di_read_unlock(parent, AUFS_I_RLOCK);
	di_write_unlock(dentry);

	TraceErr(err);
	return err;
}

static int refresh_dir(struct dentry *root, int sgen)
{
	int err;
	struct dentry *this_parent = root;
	struct list_head *next;

	LKTRTrace("sgen %d\n", sgen);
	SiMustWriteLock(root->d_sb);
	DEBUG_ON(digen(root) != sgen);
	DiMustWriteLock(root);

	err = 0;
	//spin_lock(&dcache_lock);
 repeat:
	next = this_parent->d_subdirs.next;
 resume:
	if (!IS_ROOT(this_parent)
	    && this_parent->d_inode
	    && S_ISDIR(this_parent->d_inode->i_mode)
	    && digen(this_parent) != sgen) {
		err = do_refresh_dir(this_parent);
		if (unlikely(err))
			goto out;
	}

	while (next != &this_parent->d_subdirs) {
		struct list_head *tmp = next;
		struct dentry *dentry = list_entry(tmp, struct dentry,
						   d_u.d_child);
		next = tmp->next;
		if (unlikely(d_unhashed(dentry) || !dentry->d_inode))
			continue;
		if (!list_empty(&dentry->d_subdirs)) {
			this_parent = dentry;
			goto repeat;
		}
		if (dentry->d_inode && S_ISDIR(dentry->d_inode->i_mode)) {
			DEBUG_ON(IS_ROOT(dentry) || digen(dentry) == sgen);
			err = do_refresh_dir(dentry);
			if (unlikely(err))
				goto out;
		}
	}
	if (this_parent != root) {
		next = this_parent->d_u.d_child.next;
		this_parent = this_parent->d_parent;
		goto resume;
	}

 out:
	//spin_unlock(&dcache_lock);
	TraceErr(err);
	return err;
}

// stop extra interpretation of errno in mount(8), and strange error messages.
static int cvt_err(int err)
{
	TraceErr(err);

	switch (err) {
	case -ENOENT:
	case -ENOTDIR:
	case -EEXIST:
		err = -EINVAL;
	}
	return err;
}

/* protected by s_umount */
static int aufs_remount_fs(struct super_block *sb, int *flags, char *data)
{
	int err, len, sgen;
	struct dentry *root;
	struct inode *inode;
	struct aufs_sbinfo *sbinfo;
	char *dupopt;
	struct opts opts;
	char *p;

	LKTRTrace("flags 0x%x, data %s\n", *flags, data ? data : "NULL");

	if (unlikely(!data || !*data))
		return 0; /* success */

	err = -ENOMEM;
	opts.max_opt = PAGE_SIZE / sizeof(*opts.opt);
	memset(&opts.xino, 0, sizeof(opts.xino));
	opts.opt = (void*)__get_free_page(GFP_KERNEL);
	//opts.opt = ERR_PTR(-1);
	if (unlikely(!opts.opt))
		goto out;

	p = kstrdup(data, GFP_KERNEL);
	//p = NULL;
	if (unlikely(!p))
		goto out_opts;
	sbinfo = stopd(sb);
	dupopt = sbinfo->si_dupopt;
	DEBUG_ON(!dupopt);
	len = strlen(dupopt);
	if (!strncmp(dupopt, data, len)
	    && (!data[len] || data[len] == ','))
		data += len + 1;
	err = -EINVAL;
	if (unlikely(!data || !*data))
		goto out_p;

	/* parse it before aufs lock */
	err = parse_opts(sb, data, &opts, /*remount*/1);
	//err = -1;
	if (unlikely(err))
		goto out_p;

	root = sb->s_root;
	inode = root->d_inode;
	i_lock(inode);
	aufs_write_lock(root);

	sgen = sigen(sb);
	err = do_opts(sb, &opts, /*remount*/1);
	//err = -1;
	free_opts(&opts);

	/* do_opts() may return an error */
	if (sgen != sigen(sb)) {
		int rerr;
		sbinfo->si_failed_refresh_dirs = 0;
		rerr = refresh_dir(root, sigen(sb));
		if (unlikely(rerr)) {
			sbinfo->si_failed_refresh_dirs = 1;
			Warn("Refreshing directories failed, ignores (%d)\n",
			     rerr);
		}
	}

	aufs_write_unlock(root);
	i_unlock(inode);

	if (!err) {
		kfree(dupopt);
		sbinfo->si_dupopt = p;
		p = NULL;
	}

 out_p:
	if (unlikely(p))
		kfree(p);
 out_opts:
	free_page((unsigned long)opts.opt);
 out:
	if (!err)
		return 0;
	err = cvt_err(err);
	TraceErr(err);
	return err;
}

static struct super_operations aufs_sop = {
	.alloc_inode	= aufs_alloc_inode,
	.destroy_inode	= aufs_destroy_inode,
	.read_inode	= aufs_read_inode,
	//.dirty_inode	= aufs_dirty_inode,
	//.write_inode	= aufs_write_inode,
	//void (*put_inode) (struct inode *);
	.drop_inode	= generic_delete_inode,
	//.delete_inode	= aufs_delete_inode,
	//.clear_inode	= aufs_clear_inode,

	.show_options	= aufs_show_options,
	.statfs		= aufs_statfs,

	.put_super	= aufs_put_super,
	//void (*write_super) (struct super_block *);
	//int (*sync_fs)(struct super_block *sb, int wait);
	//void (*write_super_lockfs) (struct super_block *);
	//void (*unlockfs) (struct super_block *);
	.remount_fs	= aufs_remount_fs,
	.umount_begin	= aufs_umount_begin
};

/* ---------------------------------------------------------------------- */

/*
 * at first mount time.
 */

static int alloc_sbinfo(struct super_block *sb)
{
	struct aufs_sbinfo *sbinfo;

	TraceEnter();

	sbinfo = kmalloc(sizeof(*sbinfo), GFP_KERNEL);
	//sbinfo = NULL;
	if (unlikely(!sbinfo))
		goto out;
	sbinfo->si_branch = kzalloc(sizeof(*sbinfo->si_branch), GFP_KERNEL);
	if (unlikely(!sbinfo->si_branch)) {
		kfree(sbinfo);
		goto out;
	}
	rw_init_wlock(&sbinfo->si_rwsem);
	sbinfo->si_bend = -1;
	atomic_long_set(&sbinfo->si_xino, AUFS_FIRST_INO);
	spin_lock_init(&sbinfo->si_plink_lock);
	INIT_LIST_HEAD(&sbinfo->si_plink);
	sbinfo->si_generation = 1;
	sbinfo->si_last_br_id = 0;
	sbinfo->si_failed_refresh_dirs = 0;
	sbinfo->si_dupopt = NULL;
	sbinfo->si_flags = 0;
	sbinfo->si_dirwh = AUFS_DIRWH;
	sbinfo->si_rdcache = AUFS_RDCACHE * HZ;
	sbinfo->si_udba = AUFS_UDBA_DEF;
	//memset(&sbinfo->si_kobj, 0, sizeof(sbinfo->si_kobj));
	sb->s_fs_info = sbinfo;
	MS_SET(sb, MS_XINO | MS_PLINK);
	return 0; /* success */

 out:
	TraceErr(-ENOMEM);
	return -ENOMEM;
}

static int alloc_root(struct super_block *sb)
{
	int err;
	struct inode *inode;
	struct dentry *root;

	TraceEnter();

	err = -ENOMEM;
	inode = iget(sb, AUFS_ROOT_INO);
	//inode = NULL; inode = ERR_PTR(-1);
	if (unlikely(!inode))
		goto out;
	err = PTR_ERR(inode);
	if (IS_ERR(inode))
		goto out;
	err = -ENOMEM;
	if (unlikely(is_bad_inode(inode)))
		goto out_iput;

	root = d_alloc_root(inode);
	//root = NULL; root = ERR_PTR(-1);
	if (unlikely(!root))
		goto out_iput;
	if (!IS_ERR(root)) {
		err = alloc_dinfo(root);
		//err = -1;
		if (!err) {
			sb->s_root = root;
			return 0; /* success */
		}
		dput(root);
		goto out;
	}
	err = PTR_ERR(root);

 out_iput:
	iput(inode);
 out:
	TraceErr(err);
	return err;

}

static int aufs_fill_super(struct super_block *sb, void *raw_data, int silent)
{
	int err, need_xino;
	struct dentry *root;
	struct inode *inode;
	struct opts opts;
	struct opt_xino xino;
	aufs_bindex_t bend, bindex;
	char *p, *arg = raw_data;

	if (unlikely(!arg || !*arg)) {
		err = -EINVAL;
		Err("no arg\n");
		goto out;
	}
	LKTRTrace("%s, silent %d\n", arg, silent);

	err = -ENOMEM;
	p = kstrdup(arg, GFP_KERNEL);
	//p = NULL;
	if (unlikely(!p))
		goto out;
	opts.opt = (void*)__get_free_page(GFP_KERNEL);
	//opts.opt = NULL;
	if (unlikely(!opts.opt))
		goto out_p;
	opts.max_opt = PAGE_SIZE / sizeof(opts);
	memset(&opts.xino, 0, sizeof(opts.xino));

	err = alloc_sbinfo(sb);
	//err = -1;
	if (unlikely(err))
		goto out_opts;
	SiMustWriteLock(sb);
	/* all timestamps always follow the ones on the branch */
	sb->s_flags |= MS_NOATIME | MS_NODIRATIME;
	sb->s_op = &aufs_sop;
	//err = kobj_mount(stopd(sb));
	//if (err)
	//goto out_info;

	err = alloc_root(sb);
	//err = -1; sb->s_root = NULL;
	if (unlikely(err)) {
		DEBUG_ON(sb->s_root);
		si_write_unlock(sb);
		goto out_info;
	}
	root = sb->s_root;
	DiMustWriteLock(root);
	inode = root->d_inode;
	ii_write_lock(inode);

	// lock vfs_inode first, then aufs.
	aufs_write_unlock(root);

	/*
	 * actually we can parse options regardless aufs lock here.
	 * but at remount time, parsing must be done before aufs lock.
	 * so we follow the same rule.
	 */
	err = parse_opts(sb, arg, &opts, /*remount*/0);
	//err = -1;
	if (unlikely(err))
		goto out_root;

	i_lock(inode);
	inode->i_op = &aufs_dir_iop;
	inode->i_fop = &aufs_dir_fop;
	aufs_write_lock(root);

	/* handle options except xino */
	sb->s_maxbytes = 0;
	xino = opts.xino;
	memset(&opts.xino, 0, sizeof(opts.xino));
	need_xino = IS_MS(sb, MS_XINO);
	MS_CLR(sb, MS_XINO);
	err = do_opts(sb, &opts, /*remount*/0);
	//err = -1;
	free_opts(&opts);
	if (unlikely(err))
		goto out_unlock;

	bend = sbend(sb);
	if (unlikely(bend < 0)) {
		err = -EINVAL;
		Err("no branches\n");
		goto out_unlock;
	}
	DEBUG_ON(!sb->s_maxbytes);

	/* post-process options, xino only */
	if (need_xino) {
		MS_SET(sb, MS_XINO);
		if (!xino.file) {
			xino.file = xino_def(sb);
			//xino.file = ERR_PTR(-1);
			err = PTR_ERR(xino.file);
			if (IS_ERR(xino.file))
				goto out_unlock;
			err = 0;
		}

		for (bindex = 0; !err && bindex <= bend; bindex++)
			err = xino_init(sb, bindex, xino.file,
					/*do_test*/bindex);
		//err = -1;
		fput(xino.file);
		if (unlikely(err))
			goto out_unlock;
	}

	stopd(sb)->si_dupopt = p;
	//DbgDentry(root);
	aufs_write_unlock(root);
	i_unlock(inode);
	//DbgSb(sb);
	return 0; /* success */

 out_unlock:
	aufs_write_unlock(root);
	i_unlock(inode);
 out_root:
	dput(root);
	sb->s_root = NULL;
 out_info:
	free_sbinfo(stopd(sb));
	sb->s_fs_info = NULL;
 out_opts:
	free_page((unsigned long)opts.opt);
 out_p:
	kfree(p);
 out:
	err = cvt_err(err);
	TraceErr(err);
	return err;
}

/* ---------------------------------------------------------------------- */

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,18)
static int aufs_get_sb(struct file_system_type *fs_type, int flags,
		       const char *dev_name, void *raw_data,
		       struct vfsmount *mnt)
{
	/* all timestamps always follow the ones on the branch */
	//mnt->mnt_flags |= MNT_NOATIME | MNT_NODIRATIME;
	return get_sb_nodev(fs_type, flags, raw_data, aufs_fill_super, mnt);
}
#else
static struct super_block *aufs_get_sb(struct file_system_type *fs_type,
				       int flags, const char *dev_name,
				       void *raw_data)
{
	return get_sb_nodev(fs_type, flags, raw_data, aufs_fill_super);
}
#endif

static struct file_system_type aufs_fs_type = {
	.name		= AUFS_FSTYPE,
	//.fs_flags	= FS_REVAL_DOT,
	.get_sb		= aufs_get_sb,
	.kill_sb	= generic_shutdown_super,
	//no need to __module_get() and module_put().
	.owner		= THIS_MODULE,
};

/* ---------------------------------------------------------------------- */

/*
 * aufs caches
 */

kmem_cache_t *aufs_cachep[_AufsCacheLast];
static int __init create_cache(void)
{
#define Cache(type) \
	kmem_cache_create(#type, sizeof(struct type), 0, \
			  SLAB_RECLAIM_ACCOUNT, NULL, NULL)

	if ((aufs_cachep[AUFS_CACHE_DINFO] = Cache(aufs_dinfo))
	    && (aufs_cachep[AUFS_CACHE_ICNTNR] = Cache(aufs_icntnr))
	    && (aufs_cachep[AUFS_CACHE_FINFO] = Cache(aufs_finfo))
	    //&& (aufs_cachep[AUFS_CACHE_FINFO] = NULL)
	    && (aufs_cachep[AUFS_CACHE_VDIR] = Cache(aufs_vdir))
	    && (aufs_cachep[AUFS_CACHE_DEHSTR] = Cache(aufs_dehstr))
	    && (aufs_cachep[AUFS_CACHE_HINOTIFY] = Cache(aufs_hinotify)))
		return 0;
	return -ENOMEM;

#undef Cache
}

static void __exit destroy_cache(void)
{
	int i;
	for (i = 0; i < _AufsCacheLast; i++)
		if (aufs_cachep[i])
			kmem_cache_destroy(aufs_cachep[i]);
}

/* ---------------------------------------------------------------------- */

/*
 * functions for module interface.
 */

MODULE_LICENSE("GPL");
MODULE_AUTHOR("Junjiro Okajima");
MODULE_DESCRIPTION(AUFS_NAME " -- Another unionfs");
MODULE_VERSION(AUFS_VERSION);

unsigned char aufs_nwkq = AUFS_NWKQ;
MODULE_PARM_DESC(nwkq, "the number of workqueue thread, " AUFS_WKQ_NAME);
module_param_named(nwkq, aufs_nwkq, byte, 0444);

static int __init aufs_init(void)
{
	int err;

#ifdef CONFIG_AUFS_DEBUG
	//sbinfo->si_xino is atomic_long_t
	DEBUG_ON(sizeof(ino_t) != sizeof(long));
	{
		struct aufs_destr destr;
		destr.len = -1;
		DEBUG_ON(destr.len < NAME_MAX);
	}
#ifdef CONFIG_4KSTACKS
	printk("CONFIG_4KSTACKS is defined.\n");
#endif
#if 0 // verbose debug
	{
		union {
			struct aufs_branch br;
			struct aufs_dinfo di;
			struct aufs_finfo fi;
			struct aufs_iinfo ii;
			struct aufs_hinode hi;
			struct aufs_sbinfo si;
			struct aufs_destr destr;
			struct aufs_de de;
			struct aufs_wh wh;
			struct aufs_vdir vd;
		} u;

		printk("br{"
		       "xino %d, readf %d, writef %d, "
		       "perm %d, sb %d, mnt %d, count %d, id %d, "
		       "wh_sem %d, wh %d, run %d} %d\n",
		       offsetof(typeof(u.br), br_xino),
		       offsetof(typeof(u.br), br_xino_read),
		       offsetof(typeof(u.br), br_xino_write),
		       offsetof(typeof(u.br), br_perm),
		       offsetof(typeof(u.br), br_sb),
		       offsetof(typeof(u.br), br_mnt),
		       offsetof(typeof(u.br), br_count),
		       offsetof(typeof(u.br), br_id),
		       offsetof(typeof(u.br), br_wh_rwsem),
		       offsetof(typeof(u.br), br_wh),
		       offsetof(typeof(u.br), br_wh_running),
		       sizeof(u.br));
		printk("di{rwsem %d, dentry %d, bstart %d, bend %d, bwh %d, "
		       "bdiropq %d, gen %d} %d\n",
		       offsetof(typeof(u.di), di_rwsem),
		       offsetof(typeof(u.di), di_dentry),
		       offsetof(typeof(u.di), di_bstart),
		       offsetof(typeof(u.di), di_bend),
		       offsetof(typeof(u.di), di_bwh),
		       offsetof(typeof(u.di), di_bdiropq),
		       offsetof(typeof(u.di), di_generation),
		       sizeof(u.di));
		printk("fi{rwsem %d, hfile %d, bstart %d, bend %d, "
		       "mapcnt %d, vm_ops %d, vdir_cach %d, gen %d} %d\n",
		       offsetof(typeof(u.fi), fi_rwsem),
		       offsetof(typeof(u.fi), fi_hfile),
		       offsetof(typeof(u.fi), fi_bstart),
		       offsetof(typeof(u.fi), fi_bend),
		       offsetof(typeof(u.fi), fi_mapcnt),
		       offsetof(typeof(u.fi), fi_hidden_vm_ops),
		       offsetof(typeof(u.fi), fi_vdir_cache),
		       offsetof(typeof(u.fi), fi_generation),
		       sizeof(u.fi));
		printk("ii{rwsem %d, hinode %d, bstart %d, bend %d, vdir %d} "
		       "%d\n",
		       offsetof(typeof(u.ii), ii_rwsem),
		       offsetof(typeof(u.ii), ii_hinode),
		       offsetof(typeof(u.ii), ii_bstart),
		       offsetof(typeof(u.ii), ii_bend),
		       offsetof(typeof(u.ii), ii_vdir),
		       sizeof(u.ii));
		printk("hi{inode %d, id %d} %d\n",
		       offsetof(typeof(u.hi), hi_inode),
		       offsetof(typeof(u.hi), hi_id),
		       sizeof(u.hi));
		printk("si{rwsem %d, br %d, bend %d, xino %d, "
		       "pl_lock %d, pl %d, "
		       "last id %d, gen %d, "
		       "failed_refresh %d, "
		       "dupopt %d, flags %d, dirwh %d, rdcache %d, "
		       "udba %d, kobj %d"
		       "} %d\n",
		       offsetof(typeof(u.si), si_rwsem),
		       offsetof(typeof(u.si), si_branch),
		       offsetof(typeof(u.si), si_bend),
		       offsetof(typeof(u.si), si_xino),
		       offsetof(typeof(u.si), si_plink_lock),
		       offsetof(typeof(u.si), si_plink),
		       offsetof(typeof(u.si), si_last_br_id),
		       offsetof(typeof(u.si), si_generation),
		       -1,//offsetof(typeof(u.si), si_failed_refresh_dirs),
		       offsetof(typeof(u.si), si_dupopt),
		       offsetof(typeof(u.si), si_flags),
		       offsetof(typeof(u.si), si_dirwh),
		       offsetof(typeof(u.si), si_rdcache),
		       offsetof(typeof(u.si), si_udba),
		       offsetof(typeof(u.si), si_kobj),
		       sizeof(u.si));
		printk("destr{len %d, name %d} %d\n",
		       offsetof(typeof(u.destr), len),
		       offsetof(typeof(u.destr), name),
		       sizeof(u.destr));
		printk("de{ino %d, type %d, str %d} %d\n",
		       offsetof(typeof(u.de), de_ino),
		       offsetof(typeof(u.de), de_type),
		       offsetof(typeof(u.de), de_str),
		       sizeof(u.de));
		printk("wh{hash %d, bindex %d, str %d} %d\n",
		       offsetof(typeof(u.wh), wh_hash),
		       offsetof(typeof(u.wh), wh_bindex),
		       offsetof(typeof(u.wh), wh_str),
		       sizeof(u.wh));
		printk("vd{deblk %d, nblk %d, last %d, ver %d, jiffy %d} %d\n",
		       offsetof(typeof(u.vd), vd_deblk),
		       offsetof(typeof(u.vd), vd_nblk),
		       offsetof(typeof(u.vd), vd_last),
		       offsetof(typeof(u.vd), vd_version),
		       offsetof(typeof(u.vd), vd_jiffy),
		       sizeof(u.vd));
	}
#endif
#endif

	err = -EINVAL;
	if (aufs_nwkq <= 0)
		goto out;
	err = create_cache();
	if (err)
		goto out;
	err = init_wkq();
	if (err)
		goto out_cache;
	//err = init_kobj();
	//if (err)
	//goto out_wkq;
	err = aufs_inotify_init();
	if (err)
		goto out_kobj;
	err = register_filesystem(&aufs_fs_type);
	if (err)
		goto out_inotify;
	printk(AUFS_NAME " " AUFS_VERSION "\n");
	return 0; /* success */

 out_inotify:
	aufs_inotify_exit();
 out_kobj:
	//fin_kobj();
// out_wkq:
	fin_wkq();
 out_cache:
	destroy_cache();
 out:
	TraceErr(err);
	return err;
}

static void __exit aufs_exit(void)
{
	unregister_filesystem(&aufs_fs_type);
	aufs_inotify_exit();
	//fin_kobj();
	fin_wkq();
	destroy_cache();
}

module_init(aufs_init);
module_exit(aufs_exit);

// fake Kconfig
#if 1
#if defined(CONFIG_AUFS_NO_KSIZE) && !defined(CONFIG_AUFS_MODULE)
#error mis-configuration. disable CONFIG_AUFS_NO_KSIZE when CONFIG_AUFS_MODULE is not enabled.
#endif

#if defined(CONFIG_AUFS_DBA) && !defined(CONFIG_INOTIFY)
#error mis-configuration. enable CONFIG_INOTIFY to use CONFIG_AUFS_DBA.
#endif

#ifdef CONFIG_PROVE_LOCKING
#warning CONFIG_PROVE_LOCKING will produce msgs to innocent locks.
#endif
#endif
