/* Creates a compressed image, given a file as an argument.
 * (c)1999 Paul `Rusty' Russell.  GPL.
 *
 * CHANGELOG:
 * * Sat Deb 14 22:49:20 CEST 2004 Christian Leber
 * - Changed to work with the advancecomp packages, so that it's
 *   better algorithms may be used
 * * Sun Okt 26 01:05:29 CEST 2003 Klaus Knopper
 * - Changed format of index pointers to network byte order
 * * Sat Sep 29 2001 Klaus Knopper <knopper@knopper.net>
 * - changed compression to Z_BEST_COMPRESSION,
 * * Sat Jun 17 2000 Klaus Knopper <knopper@knopper.net>
 * - Support for reading file from stdin,
 * - Changed Preamble.
 * * Sat Jul 28 2001 Klaus Knopper <knopper@knopper.net>
 * - cleanup and gcc 2.96 / glibc checking
 */

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <zlib.h>
#include "compressed_loop.h"
#include "portable.h"
#include "pngex.h"
#include "utility.h"
#include "compress.h"
#include "siglock.h"
#define __OPTIMIZE__
#include <netinet/in.h>
#include <asm/byteorder.h>

#include "lib/mng.h"
#include "lib/endianrw.h"

#include <iostream>
#include <iomanip>

#include <cstdio>

#define MAX_KMALLOC_SIZE 2L<<17

#define CLOOP_PREAMBLE "#!/bin/sh\n" "#V2.0 Format\n" "modprobe cloop file=$0 && mount -r -t iso9660 /dev/cloop $1\n" "exit $?\n"

struct cb_list
{
	struct cb_list *next;
	size_t size;
	char data[0];
};

void free_cb_list(struct cb_list *cbl)
{
 if(cbl->next) free_cb_list(cbl->next);
 free(cbl);
}

/* Now using the goto style because it is quicker to read */
static struct cb_list *create_compressed_blocks(int handle, unsigned long
                          blocksize, unsigned long *numblocks, int method)
/* method==0: Use "gzip -9", method==-1: Try best (very slow). */
{
 struct cb_list *cbl,**cbp=&cbl;
 unsigned long i=0,j;
 const int maxalg=11;
 unsigned int last, took[maxalg];
 unsigned long long total_uncompressed=0,total_compressed=0;
 unsigned long maxlen = blocksize + blocksize/1000 + 12;
 char *compressed[maxalg], *uncompressed;
 if((uncompressed=(char *)malloc(blocksize))==NULL)
  {
   fprintf(stderr, "*** Can't malloc(%ld).\n",blocksize);
   return NULL;
  }
 for(i=0; i<maxalg; i++) compressed[i] = (char *)malloc(maxlen), took[i]=0;

 for(i=0,last=0; !last; i++)
  {
   int z_error;
   unsigned long total=0;
   uLong len[maxalg];
   unsigned int best;
   //memset(compressed,0,len); memset(uncompressed,0,blocksize);
   for(j=0; j<maxalg; j++) memset(compressed[j],0,maxlen), len[j]=maxlen;
   memset(uncompressed,0,blocksize);
   while(total<blocksize) /* Read a complete block */
    {
     ssize_t r=read(handle, uncompressed+total, blocksize-total);
     if(r<=0) { last=1; break; }
     total+=r;
    }
   total_uncompressed += total;
   if (total != blocksize)
    {
     last=1;
     fprintf(stderr, "Partial read (%lu bytes of %lu), padding with zeros.\n",
					total, blocksize);
    }
   if(method==-1)
    {
     for(j=0; j<maxalg-1; j++) /* Try ZLIB compression first. */
      {
       if((z_error=compress2((Bytef*)compressed[j], (uLongf*)&len[j], (Bytef*)uncompressed, blocksize, j)) != Z_OK)
        {
         fprintf(stderr, "*** Error %d compressing block %lu! (compressed=%p, len=%lu, uncompressed=%p, blocksize=%lu)\n", z_error, i, compressed,len,uncompressed,blocksize);
         goto error_free_cb_list;
        }
      }

     /* Try 7ZIP compression now. */
     unsigned zlib_length=(unsigned int)len[maxalg-1];
     if(!compress_zlib(shrink_extreme, (unsigned char *)compressed[maxalg-1], zlib_length, (unsigned char *)uncompressed, blocksize))
      {
       fprintf(stderr, "*** Error %d compressing block %lu! (compressed=%p, len=%lu, uncompressed=%p, blocksize=%lu)\n", z_error, i, compressed,len,uncompressed,blocksize);
       goto error_free_cb_list;
      }
  
     for(best=0,j=1; j<maxalg; j++)
      {
       if(len[best] >= len[j]) best=j;
      }
    } /* COMPRESSION_BEST */
   else
    {
     best=Z_BEST_COMPRESSION;
     if((z_error=compress2((Bytef*)compressed[best], (uLongf*)&len[best], (Bytef*)uncompressed, blocksize, best)) != Z_OK)
      {
       fprintf(stderr, "*** Error %d compressing block %lu! (compressed=%p, len=%lu, uncompressed=%p, blocksize=%lu)\n", z_error, i, compressed,len,uncompressed,blocksize);
       goto error_free_cb_list;
      }
    }
   ++took[best];  
    
   if((*cbp = (cb_list *)malloc(sizeof(struct cb_list)+len[best]))==NULL) /* get another block */
    {
     fprintf(stderr, "*** Out of memory allocating block ptrs (virtual memory exhausted).\n");
     goto error_free_cb_list;
    }
   total_compressed+=len[best];
   /* Print status */
   fprintf(stderr, "[%2d] Block# %5lu size %6lu -> %6lu [compression ratio %3lu%%, overall: %3Lu%%]\n", best, i, total, len[best], total>0?((len[best]*100)/total):100,total_uncompressed>0?((total_compressed*100)/total_uncompressed):100);
   (*cbp)->size = len[best];
   memcpy((*cbp)->data, compressed[best], len[best]);
   (*cbp)->next=NULL;
   cbp=&((*cbp)->next);
  } /* for */
 goto free_compressed;

 error_free_cb_list:
    if(cbl) { free_cb_list(cbl); cbl=NULL; i=0; }

 free_compressed:
    for(j=0; j<maxalg; j++) free(compressed[j]);
 free_uncompressed:
    free(uncompressed);
 
 *numblocks=i;
 fprintf(stderr,"\nStatistics:\n");
 for(j=0; j<maxalg-1; j++) fprintf(stderr,"gzip(%d): %5d (%5.2g%%)\n", j, took[j], 100.0F*(float)took[j]/(float)i);
 fprintf(stderr,"7zip: %5d (%5.2g%%)\n\n", took[maxalg-1], 100.0F*(float)took[maxalg-1]/(float)i);
 return cbl;
}

int usage(char *progname)
{
 fprintf(stderr, "Usage: %s [-b|--best] filename blocksize(bytes).\n",progname);
 fprintf(stderr, "Use '-' as filename for stdin.\n");
 return 1;
}

int main(int argc, char **argv)
{
 int in=-1;
 unsigned long blocksize=0,b;
 struct cloop_head head;
 unsigned long numblocks;
 unsigned long long bytes_so_far;
 unsigned long i;
 int method=0;
 struct cb_list *compressed_blocks,*cbp;

 if(argc<3) return usage(argv[0]);

 for(i=1;i<argc;i++)
  {
   if(!(strcmp(argv[i],"-b")&&strcmp(argv[i],"--best"))) method=-1;
   else if((b=atoi(argv[i]))>0) blocksize=b;
   else if(in<0) in=strcmp(argv[i],"-")==0?dup(fileno(stdin)):open(argv[i], O_RDONLY);
   else
    {
     return usage(argv[0]);
    }
  }

 if (blocksize == 0 || blocksize % 512 != 0)
  {
   fprintf(stderr, "*** Blocksize must be a multiple of 512.\n");
   return 1;
  }

 if (blocksize > MAX_KMALLOC_SIZE)
  {
   fprintf(stderr, "WARNING: Blocksize %lu may be too big for a kmalloc() (%lu max).\n",blocksize,MAX_KMALLOC_SIZE);
   sleep(2);
  }

 if (sizeof(CLOOP_PREAMBLE) > CLOOP_HEADROOM)
  {
   fprintf(stderr, "*** Preamble (%u chars) > headroom (%u)\n",
			sizeof(CLOOP_PREAMBLE), CLOOP_HEADROOM);
   return 1;
  }
		
 if (in < 0)
  {
   perror("Opening input");
   return 1;
  }

 compressed_blocks = create_compressed_blocks(in, blocksize, &numblocks, method);

 close(in);

 memset(head.preamble, 0, sizeof(head.preamble));
 memcpy(head.preamble, CLOOP_PREAMBLE, sizeof(CLOOP_PREAMBLE));
 head.block_size = htonl(blocksize);
 head.num_blocks = htonl(numblocks);

 fprintf(stderr, "Block size %lu, number of blocks %lu.\n",
         blocksize, numblocks);

 bytes_so_far = sizeof(head) + sizeof(loff_t) * (numblocks + 1);

 /* Write out head... */
 write(STDOUT_FILENO, &head, sizeof(head));

 if (!compressed_blocks) return 1;

 /* Write offsets */
 for (i=0,cbp=compressed_blocks; i < numblocks+1; i++)
  {
   loff_t tmp;
   tmp = __cpu_to_be64(bytes_so_far);
   write(STDOUT_FILENO, &tmp, sizeof(tmp));
   if(cbp) { bytes_so_far += cbp->size; cbp=cbp->next; }
  }

 /* Now write blocks and free them. */
 for (i = 0, cbp=compressed_blocks; cbp && i < numblocks; i++)
  {
   if (write(STDOUT_FILENO, cbp->data, cbp->size) != cbp->size)
    {
     perror("writing block");
     free_cb_list(compressed_blocks);
     return 1;
    }
   cbp=cbp->next;
   free(compressed_blocks); compressed_blocks=cbp;
  }
 fprintf(stderr,"Done.\n");
 return 0;
}
