
/************************************************************************
 *
 *  erase.c
 *
 *  Shell erase command
 *
 *  erase [-s|-e|<address> <size>]
 *
 *  We differentiate between "monitor flash", "environment flash"  and 
 *  "system flash".
 *
 *  Monitor flash is the flash memory used for holding YAMON.
 *  Environment flash is the flash memory used for environment variables.
 *  System flash is the flash available for the user.
 *
 *  -s ( "Erase system flash" ) option is only available on boards 
 *  having system flash.
 *
 * ######################################################################
 *
 * mips_start_of_legal_notice
 * 
 * Copyright (c) 2003 MIPS Technologies, Inc. All rights reserved.
 *
 *
 * Unpublished rights (if any) reserved under the copyright laws of the
 * United States of America and other countries.
 *
 * This code is proprietary to MIPS Technologies, Inc. ("MIPS
 * Technologies"). Any copying, reproducing, modifying or use of this code
 * (in whole or in part) that is not expressly permitted in writing by MIPS
 * Technologies or an authorized third party is strictly prohibited. At a
 * minimum, this code is protected under unfair competition and copyright
 * laws. Violations thereof may result in criminal penalties and fines.
 *
 * MIPS Technologies reserves the right to change this code to improve
 * function, design or otherwise. MIPS Technologies does not assume any
 * liability arising out of the application or use of this code, or of any
 * error or omission in such code. Any warranties, whether express,
 * statutory, implied or otherwise, including but not limited to the implied
 * warranties of merchantability or fitness for a particular purpose, are
 * excluded. Except as expressly provided in any written license agreement
 * from MIPS Technologies or an authorized third party, the furnishing of
 * this code does not give recipient any license to any intellectual
 * property rights, including any patent rights, that cover this code.
 *
 * This code shall not be exported or transferred for the purpose of
 * reexporting in violation of any U.S. or non-U.S. regulation, treaty,
 * Executive Order, law, statute, amendment or supplement thereto.
 *
 * This code constitutes one or more of the following: commercial computer
 * software, commercial computer software documentation or other commercial
 * items. If the user of this code, or any related documentation of any
 * kind, including related technical data or manuals, is an agency,
 * department, or other entity of the United States government
 * ("Government"), the use, duplication, reproduction, release,
 * modification, disclosure, or transfer of this code, or any related
 * documentation of any kind, is restricted in accordance with Federal
 * Acquisition Regulation 12.212 for civilian agencies and Defense Federal
 * Acquisition Regulation Supplement 227.7202 for military agencies. The use
 * of this code by the Government is further restricted in accordance with
 * the terms of the license agreement(s) and/or applicable contract terms
 * and conditions covering this code from MIPS Technologies or an authorized
 * third party.
 *
 * 
 * mips_end_of_legal_notice
 * 
 *
 ************************************************************************/


/************************************************************************
 *  Include files
 ************************************************************************/

#include <sysdefs.h>
#include <shell_api.h>
#include <syscon_api.h>
#include <flash_api.h>
#include <sysdev.h>
#include <io_api.h>
#include <ctype.h>
#include <stdio.h>
#include <string.h>
#include <sysenv_api.h>
#include <sys_api.h>
#include <env_api.h>
#include <shell.h>

#include <project_config.h>
#include <extern_param.h>
/************************************************************************
 *  Definitions
 ************************************************************************/

/************************************************************************
 *  Public variables
 ************************************************************************/
extern t_extern_param *ext_para_ptr;
/************************************************************************
 *  Static variables
 ************************************************************************/

/* OPTIONS */
static t_cmd_option options[] =
{ 
#define OPTION_ENV  0
    { "e",  "Erase and reinitialise entire environment area." },
#define OPTION_MGC  1
    { "m",  "Erase magic number." },  
};
#define OPTION_COUNT    (sizeof(options)/sizeof(t_cmd_option))

static char erase_msg[]         = "Erasing...";
static char *syntax_sysflash    = "erase [-s|-e|<address> <size>]";
static char *syntax_no_sysflash = "erase -e|-m| <address> <size>";

static UINT32 default_start;
static UINT32 default_size;

static bool   sysflash_avail;

/************************************************************************
 *  Static function prototypes
 ************************************************************************/

static UINT32 
get_options(
    UINT32 argc,
    char   **argv,

    bool   *env,
    bool   *mgc,
    UINT32 *start,
    UINT32 *size );



static UINT32
do_erase(

    bool   env,
    bool   mgc,
    UINT32 start,
    UINT32 size );

/************************************************************************
 *  Implementation : Static functions
 ************************************************************************/

/************************************************************************
 *                          erase
 ************************************************************************/
static MON_FUNC(erase)
{
    UINT32 start;
    UINT32 size;
    bool   env;
    bool   mgc;
    UINT32 rc;
    
    rc = get_options( argc, argv,  &env, &mgc, &start, &size );

    if( rc == OK )
        rc = do_erase( env, mgc, start, size );

    return rc;
}


/************************************************************************
 *                          do_erase
 ************************************************************************/
static UINT32
do_erase(

    bool   env,
    bool   mgc,
    UINT32 start,
    UINT32 size )
{
    UINT32 rc = OK;
    t_FLASH_ctrl_descriptor   flash_ctrl;
    t_FLASH_write_descriptor  flash_write ;
    char              msg[120];
    char              ch;
    bool              cancelled = FALSE;

    if( env )
    {
        if(SHELL_PUTS( erase_msg )) return OK;

        /* copy magic # area to ram */   
        flash_write.adr    = 0x81000000;    
        flash_write.length = 0x100 ; 
        flash_write.buffer = (UINT8 *)(ext_para_ptr->env_param_saddr + Config_Param_Size - 0x100) ; 
        if( (rc = IO_write( SYS_MAJOR_FLASH, 0, &flash_write )) != OK )
        return rc;  
     
        /* Erase entire file flash */
      
        flash_ctrl.command = FLASH_CTRL_ERASE_FILEFLASH;
        rc = IO_ctrl( SYS_MAJOR_FLASH, 0, (UINT8 *)(&flash_ctrl) );
    
        if(rc != OK)
        {
            SHELL_PUTC( '\n' );
            return rc;
        }
   
        /* restore magic #  to flash */   
        flash_write.adr    = (UINT8 *)(ext_para_ptr->env_param_saddr + Config_Param_Size - 0x100) ; 
        flash_write.length = 0x100 ; 
        flash_write.buffer = (UINT8 *)0x81000000; 
        if( (rc = IO_write( SYS_MAJOR_FLASH, 0, &flash_write )) != OK )
            return rc;  
   
   
        /* Reinit sysenv (handling of environment records in flash) */
        SYSENV_init();
        
    }
    else if(mgc) 
    {
        /* start and size of file flash */
        void  *SYSENV_file_flash_start ;
        UINT32 SYSENV_file_flash_size  ;
        INT32  rcode;
        /* get start of file flash */
        rcode = SYSCON_read( SYSCON_BOARD_FILEFLASH_BASE_ID,
                         &SYSENV_file_flash_start,
                         sizeof(SYSENV_file_flash_start) ) ;
      
        if (rcode != OK)
        {   
           return(rcode) ;
        }
        else
           SYSENV_file_flash_start = (void *)KSEG1(SYSENV_file_flash_start);
   
        /* get size for file flash  */
        rcode = SYSCON_read( SYSCON_BOARD_FILEFLASH_SIZE_ID,
                             &SYSENV_file_flash_size,
                             sizeof(SYSENV_file_flash_size) ) ;
        
        if (rcode != OK)
        {   
            return(rcode) ;
        }
   
        
        if(SHELL_PUTS( erase_msg )) return OK;

        /* copy env. records area to ram */   
        flash_write.adr    = 0x81000000;    
        flash_write.length = SYSENV_file_flash_size ; 
        flash_write.buffer = (UINT8 *)SYSENV_file_flash_start ; 
        if( (rc = IO_write( SYS_MAJOR_FLASH, 0, &flash_write )) != OK )
            return rc;  
     
            /* Erase entire file flash */
      
        flash_ctrl.command = FLASH_CTRL_ERASE_FILEFLASH;
        rc = IO_ctrl( SYS_MAJOR_FLASH, 0, (UINT8 *)(&flash_ctrl) );
    
        if(rc != OK)
        {
            SHELL_PUTC( '\n' );
            return rc;
        }
   
        /* restore env. records  to flash */   
        flash_write.adr    = SYSENV_file_flash_start;   
        flash_write.length = SYSENV_file_flash_size ; 
        flash_write.buffer = (UINT8 *)0x81000000; 
        if( (rc = IO_write( SYS_MAJOR_FLASH, 0, &flash_write )) != OK )
            return rc;  
        
    }   
    else
    {
        /* Inquire what will actually be erased */

        
        flash_ctrl.user_physadr = PHYS(start);
        flash_ctrl.user_length  = size;
        

        sprintf( msg,
            "The following area will be erased:\n"
            "Start address = 0x%08x\n"
            "Size          = 0x%08x\n"
            "Confirm ? (y/n) ",
            flash_ctrl.user_physadr, flash_ctrl.user_length);

        if(SHELL_PUTS( msg )) 
        {
            SHELL_PUTC( '\n' );
            return OK;
        }

        /* Wait for user to type any key */
        do
        {
            while( !GETCHAR( DEFAULT_PORT, &ch ) );
        }
        while( (tolower(ch) != 'y') &&
               (tolower(ch) != 'n') &&
               (ch          != CTRL_C ) );

        if( ch == CTRL_C )
            rc = SHELL_ERROR_CONTROL_C_DETECTED;
        else
        {
            if(SHELL_PUTC( ch )) 
            {
                SHELL_PUTC( '\n' );
                return OK;
            }
        }

        if( tolower(ch) == 'y' )
        {
            SHELL_PUTC( '\n' );
      
            if(SHELL_PUTS( erase_msg )) 
            {
                SHELL_PUTC( '\n' );
                return OK;
            }

            flash_ctrl.command = FLASH_CTRL_ERASE_FLASH_AREA;
            rc = IO_ctrl( SYS_MAJOR_FLASH, 0, (UINT8 *)(&flash_ctrl) );
            if (rc != OK)
            {
                SHELL_PUTC( '\n' );
                return rc;
            }
        }
        else
            cancelled = TRUE;
    }

    SHELL_PUTS( cancelled ? "\nCancelled\n" : "Done\n" );

    if( !cancelled )
    {
        /* Reinit environment variables */
        env_init();
        if( env_check() )
        {
            SHELL_PUTC( '\n' );
        }
    }

    return rc;
}


/************************************************************************
 *                          get_options
 ************************************************************************/
static UINT32 
get_options(
    UINT32 argc,
    char   **argv,
    bool   *env,
    bool   *mgc,
    UINT32 *start,
    UINT32 *size )
{
    t_shell_option decode;
    UINT32     type;
    UINT32     int_count = 0;
    bool       ok = TRUE;
    UINT32     i;
    UINT32     arg;
    UINT32     error = SHELL_ERROR_SYNTAX;

    /* Setup default */
  
    *env    = FALSE;
    *start  = default_start;
    *size   = default_size;

    for( arg = 1; 
         ok && 
         (arg < argc) && 
         shell_decode_token( argv[arg], &type, &decode );
         arg++ )
    {
        switch( type )
        {
            case SHELL_TOKEN_OPTION :
                /* Find match */
                for(i=0; 
                (i<OPTION_COUNT) &&
                (strcmp(decode.option, options[i].option) != 0);
                i++) ;

                if( i == OPTION_COUNT )
                {
                    error        = SHELL_ERROR_OPTION;
                    shell_error_data = argv[arg];
                    ok       = FALSE;
                    break;            
                }
                else
                {
                    switch(i)
                    {
                        case OPTION_ENV :
                            *env    = TRUE;
         
                            break;
            
                        case OPTION_MGC :
                            *mgc    = TRUE;
         
                        break;  
                        default : /* Should not happen */
                    }
                }

                break;
            case SHELL_TOKEN_NUMBER :
                if( int_count == 0 )
                    *start = decode.number;
                else if( int_count == 1 )
                    *size  = decode.number;
                else
                    ok = FALSE;

                int_count++;
                break;
            default :
                ok = FALSE;
                break;
        }
    }

    if( *env || *mgc)
    {
        if( int_count != 0 )
            ok = FALSE;
    }
    else
    {
        /* 0 or 2 numbers is OK unless there is no system flash, in
         * which case 0 numbers is NOT ok (since there is not default
         * range) 
         */

        if( ! ( (int_count == 2) ||
            ((int_count == 0) && sysflash_avail)
            ) )
        {
            ok = FALSE;
        }
    }

    return ok ? OK : error;
}


/* Command definition for help */
static t_cmd cmd_def =
{
    "erase",
     erase,
     NULL,

    "Erase flash memory.\n"
    "\n"
    "An option may be applied specifying which flash region to erase.\n"
    "If no such option is applied, the address range to be erased\n"
    "is specified by the <address> and <size> parameters.\n"
    "If no such range is specified either, the range corresponding to the\n"
    "default option is assumed (if there is a default option, this is\n"
    "platform specific).\n"
    "\n"
    "If (and only if) the -e option (erase environment flash) is\n"
    "applied, the system environment variables are reinitialised to\n"
    "factory default values.\n"
    "\n"
    "If a range is specified, all flash blocks touched by the range\n"
    "are cleared. The block size depends on the flash memory type used by the\n"
    "board. The blocks to be cleared are displayed, and the user is asked\n"
    "for confirmation before the operation is performed.\n"
    "\n"
    "Erasing a large flash area takes time. It can easily take several\n"
    "minutes to erase a 32 MByte area.\n"
    "\n"
    "Any set flash sector lock bits will be cleared before the sector is\n"
    "erased. If they cannot be cleared (e.g. due to hardware protection of the\n"
    "lock bits), the command will fail.",

    options,
    0,
    FALSE
};



/************************************************************************
 *  Implementation : Public functions
 ************************************************************************/

/************************************************************************
 *
 *                          shell_erase_init
 *  Description :
 *  -------------
 *
 *  Initialise command
 *
 *  Return values :
 *  ---------------
 *
 *  void
 *
 ************************************************************************/
t_cmd *
shell_erase_init( void )
{
    cmd_def.syntax = syntax_no_sysflash;

    cmd_def.option_count = OPTION_COUNT;

    return &cmd_def;
}

