/************************************************************************
 *
 *      LAN_8139CPLUS.c
 *
 *      The 'LAN_8139CPLUS' module implements the LAN_8139CPLUS 
 *      device driver as an IO device with following services:
 *
 *        1) init  serial device:  configure and initialize LAN 
 *                                 RTL8139CPLUS driver
 *        2) open  serial device:  register receive handler
 *        3) close serial device:  not used
 *        4) read  serial device:  poll for received frame
 *        5) write serial device:  request frame to be transmitted
 *        6) ctrl  serial device:  display diagnostics 
 *
 *
 * ######################################################################
 *
 * mips_start_of_legal_notice
 * 
 * Copyright (c) 2003 MIPS Technologies, Inc. All rights reserved.
 *
 *
 * Unpublished rights (if any) reserved under the copyright laws of the
 * United States of America and other countries.
 *
 * This code is proprietary to MIPS Technologies, Inc. ("MIPS
 * Technologies"). Any copying, reproducing, modifying or use of this code
 * (in whole or in part) that is not expressly permitted in writing by MIPS
 * Technologies or an authorized third party is strictly prohibited. At a
 * minimum, this code is protected under unfair competition and copyright
 * laws. Violations thereof may result in criminal penalties and fines.
 *
 * MIPS Technologies reserves the right to change this code to improve
 * function, design or otherwise. MIPS Technologies does not assume any
 * liability arising out of the application or use of this code, or of any
 * error or omission in such code. Any warranties, whether express,
 * statutory, implied or otherwise, including but not limited to the implied
 * warranties of merchantability or fitness for a particular purpose, are
 * excluded. Except as expressly provided in any written license agreement
 * from MIPS Technologies or an authorized third party, the furnishing of
 * this code does not give recipient any license to any intellectual
 * property rights, including any patent rights, that cover this code.
 *
 * This code shall not be exported or transferred for the purpose of
 * reexporting in violation of any U.S. or non-U.S. regulation, treaty,
 * Executive Order, law, statute, amendment or supplement thereto.
 *
 * This code constitutes one or more of the following: commercial computer
 * software, commercial computer software documentation or other commercial
 * items. If the user of this code, or any related documentation of any
 * kind, including related technical data or manuals, is an agency,
 * department, or other entity of the United States government
 * ("Government"), the use, duplication, reproduction, release,
 * modification, disclosure, or transfer of this code, or any related
 * documentation of any kind, is restricted in accordance with Federal
 * Acquisition Regulation 12.212 for civilian agencies and Defense Federal
 * Acquisition Regulation Supplement 227.7202 for military agencies. The use
 * of this code by the Government is further restricted in accordance with
 * the terms of the license agreement(s) and/or applicable contract terms
 * and conditions covering this code from MIPS Technologies or an authorized
 * third party.
 *
 * 
 * mips_end_of_legal_notice
 * 
 *
 ************************************************************************/




/************************************************************************
 *      Include files
 ************************************************************************/

#include <string.h>
#include <stdio.h>
#include <sysdefs.h>
#include <syserror.h>
#include <sysdev.h>
#include <io_api.h>
#include <syscon_api.h>
#include <excep_api.h>
#include <sys_api.h>
#include <lan_api.h>
#include <lan_rtl8139cplus_api.h>

/************************************************************************
*************************************************************************/
/*
 *  data structure
 */
typedef volatile struct _nic_frame_descriptor_
        {
               unsigned long    StsLen;
               unsigned long    DataPtr;
               unsigned long    VLan;
               unsigned long    Reserved;
        } NIC_FD_T;


/*
 *  general definitions
 */
 
/*================================== Declaration ==================================*/

/* Symbolic offsets to registers. */
#define VPint32   *(volatile unsigned int *)
#define VPint16	  *(volatile unsigned short *)
#define VPint8    *(volatile unsigned char *)
#define MIS_PSEL2_REG             0x1801b034
#define MIS_GP1DIR_REG            0x1801b104
#define MIS_GP1DATO_REG           0x1801b10c

#define REG_BASE	0xb8016000

#define NIC_ID0				(VPint32(REG_BASE+0x0000))		/* ID Register */
#define NIC_ID1				(VPint32(REG_BASE+0x0004))		/* ID Register */
#define MAR1                            (VPint32(REG_BASE+0x0008))
#define MAR2                            (VPint32(REG_BASE+0x000C))
#define TRSR                            (VPint32(REG_BASE+0x0034))
#define CMDR				(VPint8(REG_BASE+0x003B))		/* Command register */
#define EIMR				(VPint16(REG_BASE+0x003C))		/* Interrupt Mask Register */
#define EISR				(VPint16(REG_BASE+0x003E))		/* Interrupt Status Register */
#define TCR				(VPint32(REG_BASE+0x0040))		/* Tx Configuration Register */
#define RCR				(VPint32(REG_BASE+0x0044))		/* Rx Configuration Register  */
#define MSR				(VPint8(REG_BASE+0x0058))		/* Media Status Register */
#define MIIAR				(VPint32(REG_BASE+0x005C))
#define TxFDP1				(VPint32(REG_BASE+0x0100))		/* Tx First Desc. Pointer */
#define TxCDO1				(VPint16(REG_BASE+0x0104))		/* Tx Current Desc. Offset */
#define RxFDP				(VPint32(REG_BASE+0x01F0))		/* Rx First Desc. Pointer */
#define RxCDO				(VPint16(REG_BASE+0x01F4))		/* Rx Current Desc. Offset */
#define RxRingSize			(VPint8(REG_BASE+0x01F6))		/* as Symbol*/
#define EthrntRxCPU_Des_Num	        (VPint8(REG_BASE+0x0230))               /* Desc.# CPU has finished processing */
#define Rx_Pse_Des_Thres	        (VPint8(REG_BASE+0x0232))		/* Difference ...*/
#define IO_CMD				(VPint32(REG_BASE+0x0234))		/* Ethernet IO Command */





/* Descriptor setting */
#define TX_DESC_NUM				32			// must be exponent of 2
#define RX_DESC_NUM				32			// must be exponent of 2
#define TX_MAX_SIZE				1518
#define MAX_PKT_SIZE			        2048
#define RX_DESC_BUFFER_SIZE		        0x800
#define SHIFT_RX				((RX_DESC_NUM-1)<<2)
#define OWN_BIT					0x80000000
#define EOR_BIT					0x40000000
#define FS_BIT					0x20000000
#define LS_BIT					0x10000000
#define TES_BIT					0x00800000
#define IPCS					0x4
#define TCPCS					0x5
#define UDPCS					0x6
#define CS_SH					16
#define VLANTAG					0x1234
#define PROTO_IP_0				0x08
#define PROTO_IP_1				0x0
#define PROTO_TCP				0x6
#define PROTO_UDP				0x11


/* Misc */

#define UNCACHE_MASK			0x20000000

enum {
	/* RCR */
 	AcceptErr			= 0x20,	     /* Accept packets with CRC errors */
	AcceptRunt			= 0x10,	     /* Accept runt (<64 bytes) packets */
	AcceptBroadcast		= 0x08,	     /* Accept broadcast packets */
	AcceptMulticast		= 0x04,	     /* Accept multicast packets */
	AcceptMyPhys		= 0x02,	     /* Accept pkts with our MAC as dest */
	AcceptAllPhys		= 0x01,	     /* Accept all pkts w/ physical dest */
	AcceptAll 			= AcceptBroadcast | AcceptMulticast | AcceptMyPhys |  AcceptAllPhys | AcceptErr | AcceptRunt,
	AcceptNoBroad 		= AcceptMulticast |AcceptMyPhys |  AcceptAllPhys | AcceptErr | AcceptRunt,
	AcceptNoMulti 		= AcceptMyPhys |  AcceptAllPhys | AcceptErr | AcceptRunt,
	NoErrAccept 		= AcceptBroadcast | AcceptMulticast | AcceptMyPhys,
	NoErrPromiscAccept 	= AcceptBroadcast | AcceptMulticast | AcceptMyPhys |  AcceptAllPhys,

	/* TCR */
	TCR_IFG			= 0x3,
	TCR_IFG_OFFSET	        = 10,
	TCR_NORMAL		= 0,
	TCR_LOOPBACK		= 3,
	TCR_MODE_OFFSET	        = 8,

	/* Rx flow control descriptor threshold (0~31) */
	RX_FC_THRESHOLD	        = 8,

	/* MSR */
	
	RXFCE			= 1<<6,
	TXFCE			= 1<<5,
	SP100			= 1<<3,
	LINK			= 1<<2,
	TXPF			= 1<<1,
	RXPF			= 1<<0,
	FORCE_TX 		= 1<<5,

	/* Ethernet IO CMD */
	RX_MIT 				= 3,
	RX_TIMER 			= 1,
	RX_FIFO 			= 0,
	TX_FIFO 			= 0,
	TX_MIT				= 7,	
	TE				= 1,
	RE				= 1,
	CMD_CONFIG 		        = RE << 3  | TE << 2  | RX_MIT << 8 | RX_FIFO << 11 |  RX_TIMER << 13 | TX_MIT << 16 | TX_FIFO<<19,
	RX_DISALBE 			= TE << 2  | RX_MIT << 8 | RX_FIFO << 11 |  RX_TIMER << 13 | TX_MIT << 16 | TX_FIFO<<19,
};



//unsigned char descriptor_tx[TX_DESC_NUM*16+256];
//unsigned char descriptor_rx[RX_DESC_NUM*16+256];
NIC_FD_T *pTxBDPtr;
NIC_FD_T *pRxBDPtr;

//NIC_FD_T pTxBDPtr[TX_DESC_NUM];
//NIC_FD_T pRxBDPtr[RX_DESC_NUM];
unsigned int	txBDhead = 0;	// index for system to release buffer
unsigned int	txBDtail = 0;	// index for system to set buf to BD
unsigned int	rxBDtail = 0;	// index for system to set buf to BD

/************************************************************************
 *  Constant Definitions
*************************************************************************/


/* Maximum cache line size */
#define CACHE_LINE_SIZE 0x20

typedef enum
{
    LAN_MINOR_RTL8139CPLUS_DEVICE_1 = 0,    /* The one and only RTL8139CPLUS
		                                  LAN controller               */

	/******* ADD NEW MINOR DEVICES JUST BEFORE THIS LINE ONLY ********/

    LAN_MINOR_RTL8139CPLUS_DEVICES
} t_LAN_MINOR_RTL8139CPLUS_devices ;




/* Init Block: 
   Size of Init Block                                                  */
#define LAN_8139CPLUS_INITBLOCK_SIZE    28

/* Init Block: 
   'TLEN' Number of Transmit Descriptor Ring Entries, (encoded)        */
#define LAN_8139CPLUS_INITBLOCK_TLEN    4

/* Init Block: 
   'RLEN' Number of Receive Descriptor Ring Entries, (encoded)         */
#define LAN_8139CPLUS_INITBLOCK_RLEN    4

/* 
   Number of descriptors in transmit ring, 
   derived from the TLEN field                                         */
#define LAN_8139CPLUS_TDRE_COUNT        (1 << LAN_8139CPLUS_INITBLOCK_TLEN)
#define LAN_8139CPLUS_TDRE_SIZE         16

/* 
   Number of descriptors in receive ring, 
   derived from the RLEN field                                         */
#define LAN_8139CPLUS_RDRE_COUNT        (1 << LAN_8139CPLUS_INITBLOCK_RLEN)
#define LAN_8139CPLUS_RDRE_SIZE         16

/* TX and RX data buffer size: fixed to 1536 bytes,                    */
#define LAN_8139CPLUS_BUFFER_SIZE       0x600
#define LAN_8139CPLUS_FCS_SIZE          0x4

/* 
   Number of TX data buffers,
   derived from the TLEN field                                         */
#define LAN_8139CPLUS_TX_BUFFERS        (1 << LAN_8139CPLUS_INITBLOCK_TLEN)

/* 
   Number of RX data buffers,
   derived from the RLEN field                                         */
#define LAN_8139CPLUS_RX_BUFFERS        (1 << LAN_8139CPLUS_INITBLOCK_RLEN)


/************************************************************************
 *  Macro Definitions
*************************************************************************/



#define IF_ERROR( completion, function )  \
{ \
  completion = function ; \
  if ( completion != OK )  \
{ \
    return( completion ) ; \
} \
}

/************************************************************************
 *  Type Definitions
*************************************************************************/

/* protoype for int handler */
typedef void    (*t_inthandler)(void *data) ;


/*
 *      Network device statistics. 
 */
typedef struct net_device_stats
{
        UINT32   rx_packets;             /* total packets received       */
        UINT32   tx_packets;             /* total packets transmitted    */
        UINT32   rx_bytes;               /* total bytes received         */
        UINT32   tx_bytes;               /* total bytes transmitted      */
        UINT32   rx_errors;              /* bad packets received         */
        UINT32   tx_errors;              /* packet transmit problems     */
        UINT32   multicast;              /* multicast packets received   */
        UINT32   collisions;
        UINT32   interrupts ;            /* total number of interrupts   */

        /* detailed rx_errors: */
        UINT32   rx_zero_length_errors;
        UINT32   rx_buffer_length_errors;
        UINT32   rx_over_errors;         /* recved pkt with overflow     */
        UINT32   rx_crc_errors;          /* recved pkt with crc error    */
        UINT32   rx_frame_errors;        /* recv'd frame alignment error */
        UINT32   rx_fifo_errors;         /* recv'r fifo overrun          */
        UINT32   rx_no_resource_errors;  /* recv'r no resource errors    */

        /* detailed tx_errors */
        UINT32   tx_aborted_errors;
        UINT32   tx_carrier_errors;
        UINT32   tx_fifo_errors;
        UINT32   tx_heartbeat_errors;
        UINT32   tx_window_errors;
        UINT32   tx_timeout_errors;
        UINT32   tx_bus_parity_errors;

} t_net_device_stats ;


/*
   Device context for a RTL8139CPLUS LAN controller 
                                                                        */
typedef struct LAN_8139CPLUS_device
{
	/* pointer for the RTL8139CPLUS LAN controller, reg. base address   */
	void                       *p79C973Regs;

        /* Init Block, (pointer is in CPU address space)  */
        void                       *pInitBlock ;

        /* TX Descriptor Ring, (pointers are in CPU address space) */
	UINT32                     TXDRE[LAN_8139CPLUS_TDRE_COUNT] ;

        /* RX Descriptor Ring, (pointers are in CPU address space) */
	UINT32                     RXDRE[LAN_8139CPLUS_TDRE_COUNT] ;

        /* TX Buffers, (pointers are in CPU address space) */
	UINT32                     TxBuffer[LAN_8139CPLUS_TX_BUFFERS];

        /* RX Buffers, (pointers are in CPU address space) */
	UINT32                     RxBuffer[LAN_8139CPLUS_RX_BUFFERS];

        /* TDRE index */
        UINT8                      NextTDREIndex ;

        /* RDRE index */
        UINT8                      NextRDREIndex ;

        /* network statistics */
        t_net_device_stats         status ;

} t_LAN_8139CPLUS_device ;

/* Global driver states */
#define LAN_8139CPLUS_DRIVER_IS_STOPPED   0x42
#define LAN_8139CPLUS_DRIVER_IS_STARTED   0x44

/************************************************************************
 *  LAN RTL8139CPLUS: Shared memory register field encodings
*************************************************************************/



/************************************************************************
 *      Public variables
 ************************************************************************/



/************************************************************************
 *      Static variables
 ************************************************************************/
static char msg[160] ;

/* Global driver state */
static UINT32 LAN_8139CPLUS_state = LAN_8139CPLUS_DRIVER_IS_STOPPED ;

/* MAC broadcast address */
static t_mac_addr mac_broadcast_adr = { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff } ;

/* User registered receive handler */
static UINT32 (*usr_receive)( UINT32 length, UINT8 *data ) = NULL ;

/* Device context per minor device for this driver */
static t_LAN_8139CPLUS_device   minor_device[LAN_MINOR_RTL8139CPLUS_DEVICES] ;

/* Dynamic allocated memory state */
static UINT32 first_time_init  = 0 ;

t_LAN_8139CPLUS_device *pdevice ; 

/************************************************************************
 *      Static function prototypes, local helper functions
 ************************************************************************/
#if 0 
static          
void LAN_8139CPLUS_test_tx();

static          
void LAN_8139CPLUS_test_rxtx();
#endif

static 
void Reset_HW();
/************************************************************************
 *
 *                          LAN_8139CPLUS_allocate_memory
 *  Description :
 *  -------------
 *     This routine allocates memory for:
 *
 *     - Init Block,
 *     - TX and RX descriptor ring elements,
 *     - Tx data buffers,
 *     - Rx data buffers and
 *
 *    initializes the allocated structures.
 *  
 *
 *  Parameters :
 *  ------------
 *
 *  'pdevice',     IN,    reference for this device  context
 *
 *
 *  Return values :
 *  ---------------
 *
 *  'OK'(=0)
 *
 *
 *
 ************************************************************************/
static
INT32 LAN_8139CPLUS_allocate_memory( t_LAN_8139CPLUS_device *pdevice )  ;




/************************************************************************
 *
 *                          LAN_8139CPLUS_init_ring_descriptors
 *  Description :
 *  -------------
 *   This routine initializes the RTL8139CPLUS ring descriptor
 *   data structures.
 * 
 *
 *  Parameters :
 *  ------------
 *
 *  'pdevice',     IN,    reference for this device  context
 *
 *
 *  Return values :
 *  ---------------
 *
 *  'OK'(=0)
 *
 *
 *
 ************************************************************************/
static
INT32 LAN_8139CPLUS_init_ring_descriptors( t_LAN_8139CPLUS_device *pdevice ) ;


/************************************************************************
 *
 *                          LAN_8139CPLUS_stop
 *  Description :
 *  -------------
 *   This routine stops the RTL8139CPLUS LAN controller
 *   by stopping  DMA transfer and resetting the chip.
 *  
 *
 *  Parameters :
 *  ------------
 *
 *  'pdevice',     IN,    reference for this device  context
 *
 *
 *  Return values :
 *  ---------------
 *
 *  'OK'(=0)
 *
 *
 *
 ************************************************************************/
static
INT32 LAN_8139CPLUS_stop( t_LAN_8139CPLUS_device *pdevice ) ;            

  


/************************************************************************
 *
 *                          LAN_8139CPLUS_dump_regs
 *  Description :
 *  -------------
 *    Dump all RTL8139CPLUS LAN controller registers
 * 
 *
 *  Parameters :
 *  ------------
 *
 *  'pdevice',     IN,    reference for this device  context
 *
 *
 *  Return values :
 *  ---------------
 *
 *  'OK'(=0)
 *
 *
 *
 ************************************************************************/
#ifdef ETH_DEBUG
static
INT32 LAN_8139CPLUS_dump_regs( t_LAN_8139CPLUS_device *pdevice ) ;
#endif


/************************************************************************
 *
 *                          LAN_8139CPLUS_dump_status
 *  Description :
 *  -------------
 *    Dump all RTL8139CPLUS LAN controller statistics
 * 
 *
 *  Parameters :
 *  ------------
 *
 *  'pdevice',     IN,    reference for this device  context
 *
 *
 *  Return values :
 *  ---------------
 *
 *  'OK'(=0)
 *
 *
 *
 ************************************************************************/
static
void LAN_8139CPLUS_dump_status( t_LAN_8139CPLUS_device *pdevice ) ;


/************************************************************************
 *
 *                         LAN_8139CPLUS_dump_descriptors 
 *  Description :
 *  -------------
 *    Dump all RTL8139CPLUS LAN controller TX and RX ring descriptors
 * 
 *
 *  Parameters :
 *  ------------
 *
 *  'pdevice',     IN,    reference for this device  context
 *
 *
 *  Return values :
 *  ---------------
 *
 *  'OK'(=0)
 *
 *
 *
 ************************************************************************/
#ifdef ETH_DEBUG
static
INT32 LAN_8139CPLUS_dump_descriptors( t_LAN_8139CPLUS_device *pdevice ) ;
#endif


/************************************************************************
 *
 *                         LAN_8139CPLUS_receive
 *  Description :
 *  -------------
 *    Receive a packet 
 * 
 *
 *  Parameters :
 *  ------------
 *
 *  'pdevice',     IN,    reference for this device  context
 *
 *
 *  Return values :
 *  ---------------
 *
 *  'OK'(=0)
 *
 *
 *
 ************************************************************************/
static
INT32 LAN_8139CPLUS_receive( t_LAN_8139CPLUS_device *pdevice ) ;


/************************************************************************
 *
 *                          LAN_8139CPLUS_dump_device
 *  Description :
 *  -------------
 *    Dump all RTL8139CPLUS LAN controller TX and RX ring descriptors
 *
 *
 *  Parameters :
 *  ------------
 *
 *  'pdevice',     IN,    reference for this device  context
 *
 *
 *  Return values :
 *  ---------------
 *
 *  'OK'(=0)
 *
 *
 *
 ************************************************************************/
#ifdef ETH_DEBUG
static
INT32 LAN_8139CPLUS_dump_device( t_LAN_8139CPLUS_device *pdevice ) ;
#endif


/************************************************************************
 *      Static function prototypes, device driver IO functions
 ************************************************************************/


/************************************************************************
 *
 *                          LAN_8139CPLUS_init
 *  Description :
 *  -------------
 *  This service initializes the lan driver and configures
 *  the MAC-address for the 'EN0' LAN interface.
 *  The MAC-address is read during 'init' via the 'syscon' parameter:
 *    -'SYSCON_COM_EN0_MAC_ADDR_ID'.
 *  
 *
 *  Parameters :
 *  ------------
 *
 *  'major',     IN,    major device number
 *  'minor',     IN,    not used
 *  'p_param',   INOUT, not used
 *
 *
 *  Return values :
 *  ---------------
 *
 *  'OK'(=0)
 *
 *
 *
 ************************************************************************/
static
INT32 LAN_8139CPLUS_init(
          UINT32 major,          /* IN: major device number             */
          UINT32 minor,          /* IN: minor device number             */
          void   *p_param ) ;    /* INOUT: device parameter block       */


/************************************************************************
 *
 *                          LAN_8139CPLUS_open
 *  Description :
 *  -------------
 *  This service registers a mac-layer defined receive-handler in the
 *  LAN-drivers ISR-context to allow for interrupt controlled receive
 *  frame processing in the network stack. No external buffer
 *  administration is required as the protocol-layers are responsible for
 *  handling buffer-allocation and data copy-ing to the allocated buffer
 *  payload area. At return from 'receive' handler, the LAN-drivers
 *  local RX-buffer (packet) is released for re-use. After 'open'
 *  has been called, the LAN-driver's 'read' service will call the
 *  registered receive-handler by any frame reception with direct
 *  reference to the LAN-drivers packet space and no read data will be
 *  delivered in the IO-descriptor.
 *
 *
 *
 *  Parameters :
 *  ------------
 *
 *  'major',     IN,    major device number
 *  'minor',     IN,    not used
 *  'p_param',   IN,    LAN variable of type, t_LAN_IO_desc.
 *
 *
 *  Return values :
 *  ---------------
 *
 *  'OK'(=0)
 *
 *
 *
 ************************************************************************/
static
INT32 LAN_8139CPLUS_open(
          UINT32 major,          /* IN: major device number             */
          UINT32 minor,          /* IN: minor device number             */
          t_LAN_OPEN_desc *p_param ) ; /* IN: receive handler reference */


/************************************************************************
 *
 *                          LAN_8139CPLUS_read
 *  Description :
 *  -------------
 *  This service polls the specified LAN interface for any received frame.
 *  If any frame has been received, it will be read into the user allocated
 *  variable, *p_param; if none present, completion = 'ERROR_LAN_NO_FRAME'
 *  will be returned.
 *  
 *
 *  Parameters :
 *  ------------
 *
 *  'major',     IN,    major device number
 *  'minor',     IN,    minor device number for multi device drivers
 *  'p_param',   INOUT, LAN variable of type, t_LAN_IO_desc.
 *
 *
 *  Return values :
 *  ---------------
 *
 *  'OK'(=0)
 *  'ERROR_LAN_NO_FRAME':     no frame present on this LAN interface
 *  'ERROR_LAN_COMM_ERROR':   communication error detected
 *
 *
 ************************************************************************/
static
INT32 LAN_8139CPLUS_read(
          UINT32 major,          /* IN: major device number             */
          UINT32 minor,          /* IN: minor device number             */
          t_LAN_IO_desc *p_param ) ; /* INOUT: LAN frame           */


/************************************************************************
 *
 *                          LAN_8139CPLUS_write
 *  Description :
 *  -------------
 *  This service requests transmission of a frame on the LAN interface.
 *  
 *
 *  Parameters :
 *  ------------
 *
 *  'major',     IN,    major device number
 *  'minor',     IN,    minor device number for multi device drivers
 *  'p_param',   INOUT, LAN variable of type, t_LAN_IO_desc.
 *
 *
 *  Return values :
 *  ---------------
 *
 *  'OK'(=0)
 *  'ERROR_LAN_COMM_ERROR':   communication error detected
 *
 *
 ************************************************************************/
static
INT32 LAN_8139CPLUS_write(
          UINT32 major,          /* IN: major device number             */
          UINT32 minor,          /* IN: minor device number             */
          t_LAN_IO_desc *p_param ) ; /* OUT: frame to transmit     */

/************************************************************************
 *
 *                          LAN_8139CPLUS_ctrl
 *  Description :
 *  -------------
 *  This service requests special service via 'ctrl'
 *  
 *
 *  Parameters :
 *  ------------
 *
 *  'major',     IN,    major device number
 *  'minor',     IN,    minor device number for multi device drivers
 *  'p_param',   INOUT, LAN variable of type, t_LAN_IO_desc.
 *
 *
 *  Return values :
 *  ---------------
 *
 *  'OK'(=0)
 *
 *
 ************************************************************************/
static
INT32 LAN_8139CPLUS_ctrl(
          UINT32 major,          /* IN: major device number             */
          UINT32 minor,          /* IN: minor device number             */
          t_LAN_CTRL_desc *p_param ) ; /* IN-OUT:                       */




/************************************************************************
 *      Implementation : Public functions
 ************************************************************************/


/************************************************************************
 *
 *                          LAN_8139CPLUS_install
 *  Description :
 *  -------------
 *
 *  Installs the RTL8139CPLUS LAN device drivers services in 
 *  the IO system at the reserved device slot, found in the
 *  'sysdev.h' file, which defines all major device numbers.
 *
 *  Note:
 *  This service is the only public declared interface function; all
 *  provided device driver services are static declared, but this
 *  function installs the function pointers in the io-system to
 *  enable the provided public driver services.
 *
 *  Parameters :
 *  ------------
 *
 *  -
 *
 *
 *  Return values :
 *  ---------------
 *
 *  'OK'(=0)
 *  'ERROR_IO_ILLEGAL_MAJOR':  Illegal major device number
 *  'ERROR_IO_NO_SPACE':       Device slot already allocated
 *
 ************************************************************************/
INT32 LAN_8139CPLUS_install( void )
{
    /* pre-initialize local variables and install device services */
    memset( minor_device, sizeof(minor_device), 0) ;
    IO_install(   SYS_MAJOR_LAN_8139CPLUS,         /* major device number */
           (t_io_service) LAN_8139CPLUS_init,      /* 'init'  service     */
           (t_io_service) LAN_8139CPLUS_open,      /* 'open'  service     */
                          NULL,                   /* 'close' service  na */
           (t_io_service) LAN_8139CPLUS_read,      /* 'read'  service     */
           (t_io_service) LAN_8139CPLUS_write,     /* 'write' service     */
           (t_io_service) LAN_8139CPLUS_ctrl ) ;   /* 'ctrl'  service     */
    
    /* initialize RTL8139CPLUS LAN device driver */
    if( IO_init( SYS_MAJOR_LAN_8139CPLUS, 0, NULL ) != OK )
    {   
	/* Should not happen unless board is defect */
	IO_deinstall( SYS_MAJOR_LAN_8139CPLUS );
    }
    return OK;
}



/************************************************************************
 *      Implementation : Static functions
 ************************************************************************/


/************************************************************************
 *      Implementation : Local helper functions
 ************************************************************************/


/************************************************************************
 *
 *                          LAN_8139CPLUS_allocate_memory
 *  Description :
 *  -------------
 *     This routine allocates memory for:
 *
 *     - Init Block
 *     - TX and RX descriptor ring elements
 *     - Tx data buffers
 *     - Rx data buffers 
 *  
 *
 *  Parameters :
 *  ------------
 *
 *  'pdevice',     IN,    reference for this device  context
 *
 *
 *  Return values :
 *  ---------------
 *
 *  'OK'(=0)
 *
 *
 *
 ************************************************************************/
static
INT32 LAN_8139CPLUS_allocate_memory( t_LAN_8139CPLUS_device *pdevice )  
{
    return( OK );
}



/************************************************************************
 *
 *                          LAN_8139CPLUS_init_ring_descriptors
 *  Description :
 *  -------------
 *   This routine initializes the RTL8139CPLUS ring descriptor 
 *   data structures.
 *  
 *
 *  Parameters :
 *  ------------
 *
 *  'pdevice',     IN,    reference for this device  context
 *
 *
 *  Return values :
 *  ---------------
 *
 *  'OK'(=0)
 *
 *
 *
 ************************************************************************/
static
INT32 LAN_8139CPLUS_init_ring_descriptors( t_LAN_8139CPLUS_device *pdevice )              
{
   
    return( OK );
}


/************************************************************************
 *
 *                          LAN_8139CPLUS_stop
 *  Description :
 *  -------------
 *   This routine stops the RTL8139CPLUS LAN controller
 *   by stoping  DMA transfer and resetting the chip.
 *  
 *
 *  Parameters :
 *  ------------
 *
 *  'pdevice',     IN,    reference for this device  context
 *
 *
 *  Return values :
 *  ---------------
 *
 *  'OK'(=0)
 *
 *
 *
 ************************************************************************/
static
INT32 LAN_8139CPLUS_stop( t_LAN_8139CPLUS_device *pdevice )              
{
    UINT32 rvar ;

   

    LAN_8139CPLUS_state = LAN_8139CPLUS_DRIVER_IS_STOPPED ;
    return( rvar ); /* keep compiler happy - return value is not used */
}


/************************************************************************
 *      Implementation : Device driver services
 ************************************************************************/



/************************************************************************
 *
 *                          LAN_8139CPLUS_init
 *  Description :
 *  -------------
 *  This service initializes the lan driver and configures
 *  the MAC-address for the 'EN0' LAN interface.
 *  The MAC-address is read during 'init' via the 'syscon' parameter:
 *    -'SYSCON_COM_EN0_MAC_ADDR_ID'.
 *  
 *
 *  Parameters :
 *  ------------
 *
 *  'major',     IN,    major device number
 *  'minor',     IN,    not used
 *  'p_param',   INOUT, not used
 *
 *
 *  Return values :
 *  ---------------
 *
 *  'OK'(=0)
 *
 *
 *
 ************************************************************************/
static
INT32 LAN_8139CPLUS_init(
          UINT32 major,          /* IN: major device number             */
          UINT32 minor,          /* IN: minor device number             */
          void   *p_param )      /* INOUT: device parameter block       */
{
    int    i;
    t_sys_malloc       mem ;	
    UINT32        ptmp ;
    bool   ic_in_use ;
    UINT32 int_line ;
    INT32  rcode ;
    UINT32 rvar ;
   
#ifdef Board_CPU_VENUS 
    /* check if PHY exists or not */
    /* this check only performs in venus since ethernet phy in venus 
       is external.  In neptune, it is internal in chip.  So it doesn't
       need to check */
    MIIAR = 0x04010000;
    sys_wait_ms(5);
    
   
    if ((MIIAR & 0xfffdb) != 0x17849)
       return (!OK) ;
#endif
  
/*    t_LAN_8139CPLUS_device *pdevice ; cy test*/

    /* get device context for this minor device */
    pdevice = &minor_device[minor] ;

    /* initialize shared services */
    LAN_error_lookup_init() ;
    
    /* get RTL8139CPLUS memory mapped base address */
    IF_ERROR( (rcode), 
              (SYSCON_read(SYSCON_BOARD_RTL8139CPLUS_BASE_ID,
              &(pdevice->p79C973Regs),
              sizeof(pdevice->p79C973Regs)) ) )
    
   
    pdevice->p79C973Regs = (void *)KSEG1(pdevice->p79C973Regs) ;
     /* calculate descriptor size and allocate  */
    mem.size     = TX_DESC_NUM * sizeof(NIC_FD_T) ;
    mem.boundary = 256 ;        
    mem.memory   = (void*) &pTxBDPtr ; 
    rcode = SYSCON_read( SYSCON_BOARD_MALLOC_ID,
                         &mem,
                         sizeof(mem) ) ;
    if (rcode != OK)
    {
        return( rcode ) ;
    }
    pTxBDPtr = (NIC_FD_T *)KSEG1(pTxBDPtr);//cy test
   
    mem.size     = RX_DESC_NUM * sizeof(NIC_FD_T) ;
    mem.boundary = 256 ;        
    mem.memory   = (void*) &pRxBDPtr ; 
    rcode = SYSCON_read( SYSCON_BOARD_MALLOC_ID,
                         &mem,
                         sizeof(mem) ) ;
    if (rcode != OK)
    {
        return( rcode ) ;
    }
    pRxBDPtr = (NIC_FD_T *)KSEG1(pRxBDPtr);//cy test
    	
   

   
	/*  allocate RX Data Buffers */
    mem.size     = RX_DESC_NUM * RX_DESC_BUFFER_SIZE ;
    mem.boundary = CACHE_LINE_SIZE ;
    mem.memory   = (void*)&ptmp ;
    rcode = SYSCON_read( SYSCON_BOARD_MALLOC_ID,
                         &mem,
                         sizeof(t_sys_malloc) ) ;
    if (rcode != OK)
    {
       return( rcode ) ;
    }
    ptmp = PHYS( ptmp ) ;
    
     for(i=0;i<TX_DESC_NUM;i++)
    {
	pTxBDPtr[i].StsLen = 0;
	pTxBDPtr[i].VLan = 0;
	pTxBDPtr[i].DataPtr = 0;
    }
 
    pTxBDPtr[TX_DESC_NUM-1].StsLen |= (EOR_BIT);
    
   
    for(i=0;i<RX_DESC_NUM;i++)
    {	   
      pRxBDPtr[i].StsLen  = (OWN_BIT + RX_DESC_BUFFER_SIZE); //cy test
    
     
      pRxBDPtr[i].VLan    = 0;
     
      pRxBDPtr[i].DataPtr = (ptmp); //cy test
     
      ptmp               += RX_DESC_BUFFER_SIZE ;
     
    }

    pRxBDPtr[RX_DESC_NUM-1].StsLen |= (EOR_BIT);
   
            
    Reset_HW();
    

    if (usr_receive == NULL)
    { 
      
      IF_ERROR( (rcode),
                (SYSCON_read( SYSCON_COM_EN0_INTLINE_ID,
                             &(int_line),
                             sizeof(int_line)) ) )
       
        /* Register interrupt handler */                       /*reserved cyhuang */
      
      EXCEP_register_cpu_isr( int_line,
                           (t_inthandler)LAN_8139CPLUS_receive,
                            pdevice,
		            NULL );
      
    }

    LAN_8139CPLUS_state = LAN_8139CPLUS_DRIVER_IS_STARTED ;

    
    return( OK ) ;
}


/************************************************************************
 *
 *                          LAN_8139CPLUS_open
 *  Description :
 *  -------------
 *  This service registers a mac-layer defined receive-handler in the
 *  LAN-drivers ISR-context to allow for interrupt controlled receive
 *  frame processing in the network stack. No external buffer
 *  administration is required as the protocol-layers are responsible for
 *  handling buffer-allocation and data copy-ing to the allocated buffer
 *  payload area. At return from 'receive' handler, the LAN-drivers
 *  local RX-buffer (packet) is released for re-use. After 'open'
 *  has been called, the LAN-driver's 'read' service will call the
 *  registered receive-handler by any frame reception with direct
 *  reference to the LAN-drivers packet space and no read data will be
 *  delivered in the IO-descriptor.
 *
 *
 *
 *  Parameters :
 *  ------------
 *
 *  'major',     IN,    major device number
 *  'minor',     IN,    not used
 *  'p_param',   IN,    LAN variable of type, t_LAN_IO_desc.
 *
 *
 *  Return values :
 *  ---------------
 *  'OK'(=0)
 *
 *
 *
 ************************************************************************/
static
INT32 LAN_8139CPLUS_open(
          UINT32 major,          /* IN: major device number             */
          UINT32 minor,          /* IN: minor device number             */
          t_LAN_OPEN_desc *p_param )   /* IN: receive handler reference */
{
    UINT32  old_ie ;

    /* register user defined receive handler */
    old_ie = sys_disable_int() ;
    usr_receive = p_param->receive ;
    if(old_ie) sys_enable_int();


    return( OK ) ;
}



/************************************************************************
 *
 *                          LAN_8139CPLUS_read
 *  Description :
 *  -------------
 *  This service polls the specified LAN interface for any received frame.
 *  If any frame has been received, it will be read into the user allocated
 *  variable, *p_param; if none present, completion = 'ERROR_LAN_NO_FRAME'
 *  will be returned.
 *  
 *
 *  Parameters :
 *  ------------
 *
 *  'major',     IN,    major device number
 *  'minor',     IN,    minor device number for multi device drivers
 *  'p_param',   INOUT, LAN variable of type, t_LAN_IO_desc.
 *
 *
 *  Return values :
 *  ---------------
 *
 *  'OK'(=0)
 *  'ERROR_LAN_NO_FRAME':     no frame present on this LAN interface
 *  'ERROR_LAN_COMM_ERROR':   communication error detected
 *
 *
 ************************************************************************/
static
INT32 LAN_8139CPLUS_read(
          UINT32 major,          /* IN: major device number             */
          UINT32 minor,          /* IN: minor device number             */
          t_LAN_IO_desc *p_param )   /* INOUT: LAN frame           */
{
    /* dummy function, as NET_poll, still calls this function */
//    LAN_8139CPLUS_receive(pdevice);
    return( OK ) ;

}

/************************************************************************
 *
 *                          LAN_8139CPLUS_write
 *  Description :
 *  -------------
 *  This service requests transmission of a frame on the LAN interface. It is the caller's
 *  responsibility to fill in all information including the destination and source addresses and
 *  the frame type.  The length parameter gives the number of bytes in the ethernet frame.
 *  The routine will not return until the frame has been transmitted or an error has occured.  If
 *  the frame transmits successfully, OK is returned.  If an error occured, a message is sent to
 *  the serial port and the routine returns non-zero.
 *  
 *
 *  Parameters :
 *  ------------
 *
 *  'major',     IN,    major device number
 *  'minor',     IN,    minor device number for multi device drivers
 *  'p_param',   INOUT, LAN variable of type, t_LAN_IO_desc.
 *
 *
 *  Return values :
 *  ---------------
 *
 *  'OK'(=0)
 *  'ERROR_LAN_COMM_ERROR':   communication error detected
 *
 *
 ************************************************************************/
static
INT32 LAN_8139CPLUS_write(
          UINT32 major,          /* IN: major device number             */
          UINT32 minor,          /* IN: minor device number             */
          t_LAN_IO_desc *p_param )   /* OUT: frame to transmit     */
{
 
    UINT32 status, status2, control ;
    UINT8  *pb   = p_param->data ;
    UINT32 blen  = p_param->length ;
    t_LAN_8139CPLUS_device *pdevice ;
    UINT32 rcode, old_ie;
    int i;


    if ( LAN_8139CPLUS_state == LAN_8139CPLUS_DRIVER_IS_STOPPED )
    {
        return( ERROR_LAN_TXM_ERROR ) ;
    }
   
    /* get device context for this minor device */
    pdevice = &minor_device[minor] ;
    
    old_ie = sys_disable_int() ;
   
    /* flush cache line */  
    /* cyhuang added 2005/07/13 */ 
    for (i= 0 ; i<blen+4 ; i=i+4) 
      sys_flush_cache_line( (void *)(pb+i) );
     
    pTxBDPtr[txBDtail].DataPtr = (PHYS((unsigned int)pb));
    if(blen < 60)
	   blen = 60;
	   
    pTxBDPtr[txBDtail].StsLen &= (~0x00000fff);
    pTxBDPtr[txBDtail].StsLen |= (blen);

    /* trigger to send */
    pTxBDPtr[txBDtail].StsLen |= (OWN_BIT|FS_BIT|LS_BIT|(1<<23));
#if 1    
    __sync();
#else 
    while(((pTxBDPtr[txBDtail].StsLen) &
          (OWN_BIT|FS_BIT|LS_BIT|(1<<23))) !=
          (OWN_BIT|FS_BIT|LS_BIT|(1<<23)));   
#endif                 
    IO_CMD |= (1<<0);
       
	
    i = 100000;
    while((pTxBDPtr[txBDtail].StsLen & (OWN_BIT)) && (i--));
//    while((pTxBDPtr[txBDtail].StsLen & (OWN_BIT)));
          
    /* advance one */
    txBDtail++;
    txBDtail %= TX_DESC_NUM;
    
    if(old_ie) sys_enable_int();
    
    if (i==0)
      return( ERROR_LAN_TXM_ERROR ) ; 
    
    return OK ;
}

/************************************************************************
 *
 *                          LAN_8139CPLUS_ctrl
 *  Description :
 *  -------------
 *  This service requests special service via 'ctrl'
 *  
 *
 *  Parameters :
 *  ------------
 *
 *  'major',     IN,    major device number
 *  'minor',     IN,    minor device number for multi device drivers
 *  'p_param',   INOUT, LAN variable of type, t_LAN_IO_desc.
 *
 *
 *  Return values :
 *  ---------------
 *
 *  'OK'(=0)
 *
 *
 ************************************************************************/
static
INT32 LAN_8139CPLUS_ctrl(
          UINT32 major,          /* IN: major device number             */
          UINT32 minor,          /* IN: minor device number             */
          t_LAN_CTRL_desc *p_param )   /* IN-OUT:                       */
{
    t_LAN_8139CPLUS_device  *pdevice ;
    UINT32                 old_ie ;

    /* get device context for this minor device */
    pdevice = &minor_device[minor] ;

    switch( p_param->command )
    {
      case LAN_CTRL_DISPLAY_STATISTICS:

          LAN_8139CPLUS_dump_status( pdevice ) ;
      break ;

      case LAN_CTRL_STOP_CONTROLLER:
          LAN_8139CPLUS_stop( pdevice ) ;
      break ;

      case LAN_CTRL_START_CONTROLLER:
          /* re-init driver and controller */
          old_ie = sys_disable_int() ;
          LAN_8139CPLUS_init( 0, 0, NULL ) ;
          if(old_ie) sys_enable_int();
      break ;
 #if 0  //cy test     
      case LAN_CTRL_TEST_TX:
          LAN_8139CPLUS_test_tx();
          LAN_8139CPLUS_init( SYS_MAJOR_LAN_8139CPLUS, 0, NULL) ;
      break ;
      
      case LAN_CTRL_TEST_RXTX:
          LAN_8139CPLUS_test_rxtx();
          LAN_8139CPLUS_init( SYS_MAJOR_LAN_8139CPLUS, 0, NULL ) ;
      break ;    
#endif              
  

      default:
      break ;
    }
    return( OK ) ;
}




/************************************************************************
 *
 *                          LAN_8139CPLUS_dump_status
 *  Description :
 *  -------------
 *    Dump all RTL8139CPLUS LAN controller statistics
 * 
 *
 *  Parameters :
 *  ------------
 *
 *  'pdevice',     IN,    reference for this device  context
 *
 *
 *  Return values :
 *  ---------------
 *
 *  'OK'(=0)
 *
 *
 *
 ************************************************************************/
static
void LAN_8139CPLUS_dump_status( t_LAN_8139CPLUS_device *pdevice )
{
   return;
}


/************************************************************************
 *
 *                          LAN_8139CPLUS_dump_descriptors
 *  Description :
 *  -------------
 *    Dump all RTL8139CPLUS LAN controller TX and RX ring descriptors
 * 
 *
 *  Parameters :
 *  ------------
 *
 *  'pdevice',     IN,    reference for this device  context
 *
 *
 *  Return values :
 *  ---------------
 *
 *  'OK'(=0)
 *
 *
 *
 ************************************************************************/
#ifdef ETH_DEBUG
static
INT32 LAN_8139CPLUS_dump_descriptors( t_LAN_8139CPLUS_device *pdevice )
{
 
    return( OK ) ;
}
#endif

/************************************************************************
 *
 *                         LAN_8139CPLUS_receive
 *  Description :
 *  -------------
 *    Receive a packet 
 * 
 *
 *  Parameters :
 *  ------------
 *
 *  'pdevice',     IN,    reference for this device  context
 *
 *
 *  Return values :
 *  ---------------
 *
 *  'OK'(=0)
 *
 *
 *
 ************************************************************************/
static
INT32 LAN_8139CPLUS_receive( t_LAN_8139CPLUS_device *pdevice )
{
    UINT16 status;
    UINT8  *pb  ;
    UINT32 blen ;

  
    /* disable interrupt */
    EIMR = 0 ;
    
   
    status = EISR;
    
    if (!status || (status == 0xFFFF))
	return;
   
    if (status & 0x20)  	   	
    	printf("error:RX desc. unavailable ,status = 0x%x \n",status); 
    	
    if (status & 0x04)  	
    	printf("error:RX runt ,status = 0x%x \n",status); 	

    	    
    if (status & 0x10)
    {
       int i;	
       printf("error:RX FIFO full ,status = 0x%x \n",status); 
       
       
       for(i=0;i<TX_DESC_NUM;i++)
       {
	 pTxBDPtr[i].StsLen = 0;
	 pTxBDPtr[i].VLan = 0;
	 pTxBDPtr[i].DataPtr = 0;
       }
 
       pTxBDPtr[TX_DESC_NUM-1].StsLen |= (EOR_BIT);
       
       for(i=0;i<RX_DESC_NUM;i++)
       {	   
         pRxBDPtr[i].StsLen &= (~0xfff);
         pRxBDPtr[i].StsLen |= (2048);
         pRxBDPtr[i].StsLen |= (OWN_BIT);
       }     
       
       txBDtail = 0;	
       rxBDtail = 0;   
       
       Reset_HW();
    
       return;
      
    }   
       

    EISR = status; 
    
	    
    /* increment number of interrupts */
    pdevice->status.interrupts++;
      
    while((pRxBDPtr[rxBDtail].StsLen & OWN_BIT) ==0)
    {
       pb = (char*)pRxBDPtr[rxBDtail].DataPtr;          /*cyhuang modified 1019 */
       pb = (char*)KSEG1(((UINT32)pb));
   
       blen = (pRxBDPtr[rxBDtail].StsLen);
       blen = ((blen) & (0xfff)) - 4; 
       
       /* call the handler */
       (*usr_receive)( blen, pb ) ;                    /*cyhuang modified 1019 */   
  
       pRxBDPtr[rxBDtail].StsLen &= (~0xfff);
       pRxBDPtr[rxBDtail].StsLen |= (2048);
       pRxBDPtr[rxBDtail].StsLen |= (OWN_BIT);
    
       EthrntRxCPU_Des_Num = rxBDtail;
   
       rxBDtail++;
       rxBDtail %= RX_DESC_NUM;
   
       
       
//       if (rx_work-- ==0)
//           break;
    }
     /* enable interrupt */
   
    EIMR = 0x3f ;
    return OK ;

}


/************************************************************************
 *
 *                          LAN_8139CPLUS_dump_device
 *  Description :
 *  -------------
 *    Dump all RTL8139CPLUS LAN controller TX and RX ring descriptors
 * 
 *
 *  Parameters :
 *  ------------
 *
 *  'pdevice',     IN,    reference for this device  context
 *
 *
 *  Return values :
 *  ---------------
 *
 *  'OK'(=0)
 *
 *
 *
 ************************************************************************/
#ifdef ETH_DEBUG
static
INT32 LAN_8139CPLUS_dump_device( t_LAN_8139CPLUS_device *pdevice )
{
    

    return( OK ) ;
}
#endif
/******************************************************************
*
*
******************************************************************/
static 
void Reset_HW()
{
    t_mac_addr mac_station_adr ;	
    
 
  
    if (SYSCON_read(SYSCON_COM_EN0_MAC_ADDR_ID, 
		            &mac_station_adr, 
			    sizeof(mac_station_adr)) != OK )
    			    
       printf("no mac address \n");	 		    
      		    
                            
    NIC_ID0 = CPU_TO_BE32(*(UINT32 *)mac_station_adr);
    
    NIC_ID1 = CPU_TO_BE16( *(UINT16 *)(mac_station_adr+4))<<16 ;	
   
   

   
    CMDR = 0x01;	 /* Reset */	
   
    
    while ( CMDR & 0x1 );		
   
    CMDR = 0x02;	 /* receive checksum */
  
    
	/* Interrupt Register, ISR, IMR */
    EISR = 0xffff;  	
    EIMR = 0x003f;	
        


    /* setup descriptor */	
    RxFDP = (unsigned int)pRxBDPtr;
    
  
    RxCDO = 0;
   
    TxFDP1 = (unsigned int)pTxBDPtr;
  
   
    TxCDO1 = 0;	
   
    
	/* RCR, don't accept error packet */
    RCR = NoErrAccept;
    
	/* TCR: IFG, Mode */
    TCR = (unsigned long)(TCR_IFG<<TCR_IFG_OFFSET)|(TCR_NORMAL<<TCR_MODE_OFFSET);
   
    Rx_Pse_Des_Thres = RX_FC_THRESHOLD;
  
	/* Rx descriptor Size */	
    EthrntRxCPU_Des_Num = 0; //RX_DESC_NUM-1;
  
    /* 32 descriptors */
    RxRingSize = 0x01;	
  
	/* Flow Control */
    MSR = (TXFCE | RXFCE);
    
    MIIAR = 0x84008000;
    sys_wait_ms(5);
    
#ifdef Board_CPU_NEPTUNE
    if (sys_chiprev == NEPTUNE_REV_A)
    {
    	MIIAR = 0x84000100;    	
    }
    else if (sys_chiprev == NEPTUNE_REV_B)
    {
	    MIIAR = 0x84001000;

		/* Following code is used to strength eth-wire noise filtering. 
		   The delay is necessary and can't be reduce, however, the delay
		   is too long to endure, so we just mark these code */
//    	sys_wait_ms(900);
    	
//    	MIIAR = 0x841f0001;
//    	sys_wait_ms(700);
    	
//    	MIIAR = 0x8412ae54;
//    	sys_wait_ms(5);
    	
//    	MIIAR = 0x841f0000;
//    	sys_wait_ms(5);
    }
    else
    	MIIAR = 0x84001000;
#else    
    MIIAR = 0x84001000;
#endif    
  
    MIIAR = 0x840405e1;
    

    /* Ethernet IO CMD */
    IO_CMD = 0x01110d;  // CMD_CONFIG;
    
    return;
    
}    	


/************************************************************************
 *
 *                          LAN_8139CPLUS_test_tx
 *  Description :
 *  -------------
 *    Test 8139c+ NIC tx is ok or not
 * 
 *
 *  Parameters :
 *  ------------
 *
 *
 *
 *  Return values :
 *  ---------------
 *
 *  'OK'(=0)
 *
 *
 *
 ************************************************************************/
#if 0
int pattern_tx[200]={0xffffffff,0xffffffff,0x23232323,0x45454545,
                     0x01020202,0x02020202,0x02020202,0x02020202,

                     0x10101010,0x10101010,0x10101010,0x10101010}; 
unsigned int desc_tx[512], desc_rx[512];

static          
void LAN_8139CPLUS_test_tx()
{	
	 unsigned int tx_des_num=0;
         unsigned int j;
	 volatile unsigned int* descriptor_tx = (unsigned int*)((((unsigned int)desc_tx + 0xff) & 0x1fffff00) |0xa0000000);
	 volatile unsigned int* descriptor_rx = (unsigned int*)((((unsigned int)desc_rx + 0xff) & 0x1fffff00) |0xa0000000);
  	 
         while(!GETCHAR_CTRLC( DEFAULT_PORT ))
	 {
	    for(j=0;j<63;j++)
	    {   
	      descriptor_tx[(j<<2)]=(0xb0800010);
	      descriptor_tx[(j<<2)+1]=(PHYS((int)pattern_tx));
	      descriptor_tx[(j<<2)+2]=0x0;
	      descriptor_tx[(j<<2)+3]=0x0;
	    }
	  	   
	    descriptor_tx[(63<<2)]=(0xf0800010);
	    descriptor_tx[(63<<2)+1]=(PHYS((int)pattern_tx));
#if 1 //cy test
            CMDR   = 0x01;	 /* Reset */	
                while ( CMDR & 0x1 )
             printf("waiting reset\n");
             	
   
            CMDR   = 0x00;	// /* receive checksum */   
            
            NIC_ID0=0x5dcca000;                                         
            NIC_ID1=0x00004ff5;                    
            MAR1   =0xffffffff;   
            MAR2   =0xffffffff;                    
              
            EIMR   = 0x0000;
            EISR   = 0xffff;
            TCR    = 0x00000c00;
            RCR    = 0x000000ff;
          
            MIIAR  = 0xc4000100;
#endif            
            TxFDP1 =(unsigned int)descriptor_tx;
            TxCDO1 = 0x0;
            RxFDP  =(unsigned int)descriptor_rx;
            RxCDO  =0x0;
            
            IO_CMD =0x000f110d;
            while(descriptor_tx[tx_des_num<<2]& (OWN_BIT))
            {
              printf("descriptor_tx[%d<<2]=0x%x\n",tx_des_num,descriptor_tx[tx_des_num<<2]);//cy test
           
              if (GETCHAR_CTRLC( DEFAULT_PORT )) 
              { printf("ctrl-c-1\n");//cy test
                           IO_CMD=0x0;
                           return;
              }  	
              
            
              printf("send out 1 packet\n");//cy test
            
              tx_des_num++;
              tx_des_num%=64;
            
            }
                                   
         }   
         printf("ctrl-c-2\n");//cy test
         IO_CMD=0x0;                 
         return ;
}

/************************************************************************
 *
 *                          LAN_8139CPLUS_test_rxtx
 *  Description :
 *  -------------
 *    Test 8139c+ NIC rx & tx is ok or not
 * 
 *
 *  Parameters :
 *  ------------
 *
 *
 *
 *  Return values :
 *  ---------------
 *
 *  'OK'(=0)
 *
 *
 *
 ************************************************************************/

unsigned int rx_buffer[64][2048];

          
void LAN_8139CPLUS_test_rxtx()
{	
	
        unsigned int pkt_size;
        unsigned int rx_des_num=0, tx_des_num=0;
        unsigned int j,k;

	volatile unsigned int* descriptor_tx = (unsigned int*)((((unsigned int)desc_tx + 0xff) & 0x1fffff00) |0xa0000000);
	volatile unsigned int* descriptor_rx = (unsigned int*)((((unsigned int)desc_rx + 0xff) & 0x1fffff00) |0xa0000000);
    
             
	for(j=0;j<64;j++)
	{   
	    descriptor_tx[(j<<2)]=0x0;
	    descriptor_tx[(j<<2)+1]=0x0;
	    descriptor_tx[(j<<2)+2]=0x0;
	    descriptor_tx[(j<<2)+3]=0x0;
	}
	 
	for(k=0;k<64;k++) 
	{
	    if(k==63)
	      descriptor_rx[(k<<2)]=(0xc0000800);  //reserved 516 Bytes buffer
	    else
	      descriptor_rx[(k<<2)]=(0x80000800);
	    
	    descriptor_rx[(k<<2)+1]=(PHYS((int)rx_buffer[k]));
	    descriptor_rx[(k<<2)+2]=0x0;
	    descriptor_rx[(k<<2)+3]=0x0;
	}   
     
        CMDR   = 0x01;	 /* Reset */	
        while ( CMDR & 0x1 )
           printf("waiting reset\n");
             	
   
        CMDR   = 0x00;	 /* receive checksum */   
                 
        NIC_ID0=0x5dcca000;
        NIC_ID1=0x00004ff5;
        MAR1=0xffffffff;
        MAR2=0xffffffff;
      
        EISR=0xffff;
        asm("nop");
        EIMR=0x0000;
        asm("nop");
        TCR=0x00000c00;
        RCR=0x000000ff;
           
        MIIAR=0xc4000100;
           
        TxFDP1=(unsigned int)descriptor_tx;
        TxCDO1=0x0;
        RxFDP=(unsigned int)descriptor_rx;
        RxCDO =0x0;
        IO_CMD=0x000f110d;
            
        while(!GETCHAR_CTRLC( DEFAULT_PORT ))
	{
		
		    while((descriptor_rx[(rx_des_num<<2)]&(0x80000000))==0)
		    {
		       printf("got 1 packet\n"); //cy test
		       pkt_size=(descriptor_rx[((rx_des_num)<<2)]&0xff0f0000);
		       pkt_size= (pkt_size);
	               descriptor_tx[(((tx_des_num)&0x3f)<<2)+1]=descriptor_rx[(((rx_des_num)&0x3f)<<2)+1];
	               if(tx_des_num==63)
	               {
	                  descriptor_tx[((tx_des_num)<<2)]=(0xf0000000+pkt_size);   
	                  tx_des_num=0;           
	               }
	               else{
	                  descriptor_tx[((tx_des_num)<<2)]=(0xb0000000+pkt_size);
	                  tx_des_num++;
                       }
                       IO_CMD=0x000f110d;
                       
                       
                       
                       if(rx_des_num==63)
                       {
                          descriptor_rx[((rx_des_num)<<2)]=(0xc0000800); 
	 	          rx_des_num=0;
                       }
	               else
                       {
                          descriptor_rx[(rx_des_num<<2)]=(0x80000800);
	                  rx_des_num++;
	               }       
	               
	               while(descriptor_tx[(tx_des_num<<2)]&(0x80000000))
	               {
                         if (GETCHAR_CTRLC( DEFAULT_PORT )) 
                         {
                           IO_CMD=0x0;
                           return;
                         }  	
                       }  
	               printf("send out 1 packet\n");//cy test
	       
                    }
                                           
	}
	printf("ctrl-c\n");//cy test
	IO_CMD=0x0;
      
        return;
}

#endif
