/************************************************************************
 *
 *      FLASH_PARELLEL.C
 *
 *      The 'FLASH_PARELLEL' module implements the FLASH device driver
 *      interface to be used via 'IO' device driver services:
 *
 *        1) init  device:  configure and initialize FLASH driver
 *        2) open  device:  not used
 *        3) close device:  not used
 *        4) read  device:  not used
 *        5) write device:  write FLASH device
 *        6) ctrl  device:  a) ERASE_SYSTEMFLASH
 *                          b) ERASE_FILEFLASH
 *
 *
 * ######################################################################
 *
 * mips_start_of_legal_notice
 * 
 * Copyright (c) 2003 MIPS Technologies, Inc. All rights reserved.
 *
 *
 * Unpublished rights (if any) reserved under the copyright laws of the
 * United States of America and other countries.
 *
 * This code is proprietary to MIPS Technologies, Inc. ("MIPS
 * Technologies"). Any copying, reproducing, modifying or use of this code
 * (in whole or in part) that is not expressly permitted in writing by MIPS
 * Technologies or an authorized third party is strictly prohibited. At a
 * minimum, this code is protected under unfair competition and copyright
 * laws. Violations thereof may result in criminal penalties and fines.
 *
 * MIPS Technologies reserves the right to change this code to improve
 * function, design or otherwise. MIPS Technologies does not assume any
 * liability arising out of the application or use of this code, or of any
 * error or omission in such code. Any warranties, whether express,
 * statutory, implied or otherwise, including but not limited to the implied
 * warranties of merchantability or fitness for a particular purpose, are
 * excluded. Except as expressly provided in any written license agreement
 * from MIPS Technologies or an authorized third party, the furnishing of
 * this code does not give recipient any license to any intellectual
 * property rights, including any patent rights, that cover this code.
 *
 * This code shall not be exported or transferred for the purpose of
 * reexporting in violation of any U.S. or non-U.S. regulation, treaty,
 * Executive Order, law, statute, amendment or supplement thereto.
 *
 * This code constitutes one or more of the following: commercial computer
 * software, commercial computer software documentation or other commercial
 * items. If the user of this code, or any related documentation of any
 * kind, including related technical data or manuals, is an agency,
 * department, or other entity of the United States government
 * ("Government"), the use, duplication, reproduction, release,
 * modification, disclosure, or transfer of this code, or any related
 * documentation of any kind, is restricted in accordance with Federal
 * Acquisition Regulation 12.212 for civilian agencies and Defense Federal
 * Acquisition Regulation Supplement 227.7202 for military agencies. The use
 * of this code by the Government is further restricted in accordance with
 * the terms of the license agreement(s) and/or applicable contract terms
 * and conditions covering this code from MIPS Technologies or an authorized
 * third party.
 *
 * 
 * mips_end_of_legal_notice
 * 
 *
 ************************************************************************/




/************************************************************************
 *      Include files
 ************************************************************************/

#include <sysdefs.h>
#include <syserror.h>
#include <sysdev.h>
#include <mips.h>
#include <io_api.h>
#include <syscon_api.h>
#include <flash_api.h>

#include <sys_api.h>
#include <stdio.h>
#include <string.h>

#include <project_config.h>

/************************************************************************
 *      Definitions
 ************************************************************************/

#define RTL_FLASHCTRL_REG                  0x1801A014 

/* FLASH-device, relative register address inside a block */
#define FLASH_BLOCKSTATUS_OFS                   0x0008

/* field: LOCK */
#define FLASH_BLOCKSTATUS_LOCK_MSK          0x00010001

/* flash memory ID's */
#define INTEL_MNFCR_CODE			0x0089
#define PARELLEL_DEV1_CODE			0x0014
#define PARELLEL_DEV2_CODE			0x0015

#define FLASH_QUERYQCHAR_OFS                    0x0040
#define FLASH_QUERYRCHAR_OFS                    0x0044
#define FLASH_QUERYYCHAR_OFS                    0x0048

#define FLASH_QUERYQCHAR                    0x00510051
#define FLASH_QUERYRCHAR                    0x00520052
#define FLASH_QUERYYCHAR                    0x00590059

/* status */
#define FLASH_READY				0x0000
#define FLASH_NOT_DETECTED			0x0001
#define FLASH_BUSY				0x0002
#define FLASH_ERROR				0x0003
#define FLASH_TIMEOUT                           0x0004
#define FLASH_INVALID_SECTOR                    0x0005
#define FLASH_SECTOR_LOCKED                     0x0006

/* Commands and masks, etc... */
#define FLASH_STATUS_READY			0x00800080
#define FLASH_STATUS_MASK                       0x00FF00FF
#define FLASH_STATUS_OK                         0x00800080
#define FLASH_STATUS_ERASE                      0x00600060
#define FLASH_STATUS_LOCK                       0x00020002
#define FLASH_STATUS_LOW_VOLTAGE                0x00080008

#define FLASH_READ_COMMAND			0x00FF00FF
#define FLASH_ERASE_COMMAND			0x00200020
#define FLASH_CONFIRM_COMMAND		        0x00D000D0
#define FLASH_CLEAR_STATUS_COMMAND	        0x00500050
#define FLASH_WRITE_WORD_COMMAND	        0x00100010
#define FLASH_WRITE_BUFFER_COMMAND	        0x00E800E8
#define FLASH_STATUS_COMMAND		        0x00700070
#define FLASH_QUERY_COMMAND			0x00980098
#define FLASH_READ_ID_CODES_COMMAND		0x00900090
#define FLASH_CLEAR_LOCK_COMMAND                0x00600060
#define FLASH_LOCK_SECTOR                       0x00010001

/* Timeout values */
#define FLASH_TMOUT_100MS                       100
#define FLASH_TMOUT_250MS                       250
#define FLASH_TMOUT_500MS                       500
#define FLASH_TMOUT_1SEC                        1000
#define FLASH_TMOUT_2SEC                        2000
#define FLASH_TMOUT_2_5SEC                      2500

/* Retry counts */
#define FLASH_MAX_LOOPS				0xFFFFFFFF
#define FLASH_RETRY_5                           5
#define FLASH_RETRY_10                          10
#define FLASH_RETRY_20                          20
#define FLASH_RETRY_50                          50

/* Identification tags for memory devices */
typedef enum flash_device_id
{
	FLASH_SYSTEMFLASH_DEVICE = 0,
	FLASH_MONITORFLASH_DEVICE,
	FLASH_FILEFLASH_DEVICE,
	FLASH_BOOT_DEVICE,
	FLASH_UNKNOWN_DEVICE
} t_flash_device_id ;


/************************************************************************
 *  Macro Definitions
*************************************************************************/

#define WRITE_ENABLE   SYSCON_write( SYSCON_BOARD_SYSTEMFLASH_WRITE_ENABLE_ID, \
                                    NULL, sizeof( UINT32 ) );

#define WRITE_DISABLE  SYSCON_write( SYSCON_BOARD_SYSTEMFLASH_WRITE_DISABLE_ID, \
                                    NULL, sizeof( UINT32 ) );

/************************************************************************
 *      Public variables
 ************************************************************************/
extern UINT32 rtl_monitorflash_size;


/************************************************************************
 *      Static variables
 ************************************************************************/

static char* flash_error_string[] = 
{
    /* ERROR_FLASH_PROGRAM_ERROR      */ "Flash device failure",
    /* ERROR_FLASH_INVALID_ADDRESS    */ "Invalid address",
    /* ERROR_FLASH_INVALID_COMMAND    */ "Internal ERROR: Invalid control command",
    /* ERROR_FLASH_TIME_OUT           */ "Internal ERROR: Flash device timed out during operation",
    /* ERROR_FLASH_VERIFY_ERROR       */ "Data verify error",
    /* ERROR_FLASH_LOCKED             */ "Some sectors are locked",
    /* ERROR_FLASH_ERASE_ERROR        */ "Sector has erase error",
    /* ERROR_FLASH_LOW_VOLTAGE        */ "Low programming voltage detected",
    /* ERROR_FLASH_WRITE_PROTECTED    */ "Flash is write protected",
    /* ERROR_FLASH_FILE_FLASH_PROT    */ "Environment FLASH is write protected",
    /* ERROR_FLASH_FILE_FLASH_LOCK    */ "Environment FLASH is lock-bit protected",
    /* ERROR_FLASH_MONITOR_FLASH_LOCK */ "Some MONITOR FLASH sector(s) locked",
    /* ERROR_FLASH_QRY_NOT_FOUND      */ "CFI Query-ID string of FLASH not found",
    /* ERROR_FLASH_BOOT_WRITE_PROTECTED */ "Write access to this area not allowed"
} ;

static char* flash_error_hint_string[] = 
{
    /* ERROR_FLASH_PROGRAM_ERROR      */ NULL,
    /* ERROR_FLASH_INVALID_ADDRESS    */ NULL,
    /* ERROR_FLASH_INVALID_COMMAND    */ NULL,
    /* ERROR_FLASH_TIME_OUT           */ NULL,
    /* ERROR_FLASH_VERIFY_ERROR       */ "Check flash has been erased before programming",
    /* ERROR_FLASH_LOCKED             */ "Unlock sector(s) before programming",
    /* ERROR_FLASH_ERASE_ERROR        */ NULL,
    /* ERROR_FLASH_LOW_VOLTAGE        */ NULL,
    /* ERROR_FLASH_WRITE_PROTECTED    */ "Disable write protection: Switch S1-3",
    /* ERROR_FLASH_FILE_FLASH_PROT    */ "Check programming addresses",
    /* ERROR_FLASH_FILE_FLASH_LOCK    */ "Disable 'clear lock-bit' protection: (MFWR-jumper) must be fitted",
    /* ERROR_FLASH_MONITOR_FLASH_LOCK */ NULL,
    /* ERROR_FLASH_QRY_NOT_FOUND      */ NULL,
    /* ERROR_FLASH_BOOT_WRITE_PROTECTED */ NULL
} ;

static UINT32 flash_last_error ;
static char   flash_diag_msg[160] ;


/* these variables are initialized at 'init' with the physical
   address boundaries of:

     a) system FLASH
     b) monitor FLASH
     c) file FLASH.

   Following rule is to apply:
     'start' <= physical-device-address-space < 'end'
*/

static UINT32  systemflash_phys_start ;
static UINT32  systemflash_phys_end ;
static UINT32  systemflash_block_size ;
static UINT32  systemflash_bank_count ;
static UINT32  systemflash_block_count ;

static UINT32  monitorflash_phys_start ;
static UINT32  monitorflash_phys_end ;
static UINT32  monitorflash_block_size ;

static UINT32  fileflash_phys_start ;
static UINT32  fileflash_phys_end ;
static UINT32  fileflash_block_size ;



#define FLASH_ADDR_BASE		0xbfcfffff

 
#define MXIC        0xc2
#define SPANSION    0x01
#define ATMEL       0x1f
#define AMIC        0x37
#define EON         0x7f
#define FUJITSU     0x04


#define MX29LV160C_BOTTOM     0x000049
#define MX29LV160C_TOP        0x0000c4
#define MX29LV320C_BOTTOM     0x0000a8
#define MX29LV320C_TOP        0x0000a7
#define MX29LV640B_BOTTOM     0x0000cb
#define MX29LV640B_TOP        0x0000c9
#define MX29LV640M_BOTTOM     0x00107e
#define MX29LV640M_TOP        0x01107e
#define MX29LV128M_BOTTOM     0x00117e
#define MX29LV128M_TOP        0x01117e
#define MX29LV128D_BOTTOM     0x00007a
#define MX29LV128D_TOP        0x00007e
#define MX29LV320M_BOTTOM     0x001a7e
#define MX29LV320M_TOP        0x011a7e


#define S29AL016D_BOTTOM      0x000049
#define S29AL016D_TOP         0x0000c4
#define S29GL32M_R4_BOTTOM    0x001a7e
#define S29GL32M_R3_TOP       0x011a7e
#define S29GL64M_R4_BOTTOM    0x00107e
#define S29GL64M_R3_TOP       0x01107e
#define S29GL128N             0x01217e
	
#define AT49BV163AT           0xc2	
#define AT49BV163A            0xc0 

#define A29L160A_BOTTOM       0x000049
#define A29L160A_TOP          0x0000c4

#define EN29LV640B_TOP        0x0000c9
#define EN29LV640B_BOTTOM     0x0000cb

#define MBM29DL640E_TOP       0x00007e

typedef struct 
{
    unsigned char              mid ;
    unsigned char         did_cycle;
    unsigned int              did ;
    unsigned int    *flash_sectors ;
    unsigned int      total_sector ;
    unsigned int     size;
    
}
p_device_type;




p_device_type devicetype;;


unsigned int at16_top_flash_sectors[ 39 ] = 
{
	0x000000,  0x010000, 
	0x020000,  0x030000,  
	0x040000,  0x050000, 
	0x060000,  0x070000,  
	0x080000,  0x090000,  
	0x0a0000,  0x0b0000, 
	0x0c0000,  0x0d0000,  
	0x0e0000,  0x0f0000, 
	0x100000,  0x110000,  
	0x120000,  0x130000,  
	0x140000,  0x150000, 
	0x160000,  0x170000,  
	0x180000,  0x190000,  
	0x1a0000,  0x1b0000, 
	0x1c0000,  0x1d0000,  
	0x1e0000,  0x1f0000, 	  
	0x1f2000,  0x1f4000,
	0x1f6000,  0x1f8000,
	0x1fa000,  0x1fc000,
	0x1fe000
};

unsigned int at16_bottom_flash_sectors[ 39 ] = 
{
	0x000000,  0x002000, 
	0x004000,  0x006000,  
	0x008000,  0x00a000, 
	0x00c000,  0x00e000,  
	0x010000,  0x020000,  
	0x030000,  0x040000,
	0x050000,  0x060000,  
	0x070000,  0x080000,
	0x090000,  0x0a0000,  
	0x0b0000,  0x0c0000,
	0x0d0000,  0x0e0000,  
	0x0f0000,  0x100000,
	 
        0x110000,  
	0x120000,  0x130000,  
	0x140000,  0x150000, 
	0x160000,  0x170000,  
	0x180000,  0x190000,  
	0x1a0000,  0x1b0000, 
	0x1c0000,  0x1d0000,  
	0x1e0000,  0x1f0000 	  
	
};


unsigned int mx16_top_flash_sectors[ 35 ] = 
{
	0x000000,  0x010000, 
	0x020000,  0x030000,  
	0x040000,  0x050000, 
	0x060000,  0x070000,  
	0x080000,  0x090000,  
	0x0a0000,  0x0b0000, 
	0x0c0000,  0x0d0000,  
	0x0e0000,  0x0f0000, 
	0x100000,  0x110000,  
	0x120000,  0x130000,  
	0x140000,  0x150000, 
	0x160000,  0x170000,  
	0x180000,  0x190000,  
	0x1a0000,  0x1b0000, 
	0x1c0000,  0x1d0000,  
	0x1e0000,  0x1f0000, 	  
	0x1f8000, 
	0x1fa000,  0x1fc000
};

unsigned int mx16_bottom_flash_sectors[ 35 ] = 
{
	0x000000, 
	0x004000,  0x006000,  
	0x008000, 
	0x010000,  0x020000,  
	0x030000,  0x040000, 
	0x050000,  0x060000,  
	0x070000,  0x080000, 
	0x090000,  0x0a0000,  
	0x0b0000,  0x0c0000,  
	0x0d0000,  0x0e0000, 
	0x0f0000,  0x100000,  
	0x110000,  0x120000,  
	0x130000,  0x140000, 
	0x150000,  0x160000,  
	0x170000,  0x180000, 
	0x190000,  0x1a0000, 
	0x1b0000,  0x1c0000,  
	0x1d0000,  0x1e0000, 
	0x1f0000 
	
};

unsigned int sp32_top_flash_sectors[ 71 ] = 
{
	0x000000,  0x010000, 
	0x020000,  0x030000,  
	0x040000,  0x050000, 
	0x060000,  0x070000,  
	0x080000,  0x090000,  
	0x0a0000,  0x0b0000, 
	0x0c0000,  0x0d0000,  
	0x0e0000,  0x0f0000, 
	0x100000,  0x110000,  
	0x120000,  0x130000,  
	0x140000,  0x150000, 
	0x160000,  0x170000,  
	0x180000,  0x190000,  
	0x1a0000,  0x1b0000, 
	0x1c0000,  0x1d0000,  
	0x1e0000,  0x1f0000, 
	0x200000,  0x210000, 
	0x220000,  0x230000,  
	0x240000,  0x250000, 
	0x260000,  0x270000,  
	0x280000,  0x290000,  
	0x2a0000,  0x2b0000, 
	0x2c0000,  0x2d0000,  
	0x2e0000,  0x2f0000, 
	0x300000,  0x310000,  
	0x320000,  0x330000,  
	0x340000,  0x350000, 
	0x360000,  0x370000,  
	0x380000,  0x390000,  
	0x3a0000,  0x3b0000, 
	0x3c0000,  0x3d0000,  
	0x3e0000,  0x3f0000,
	0x3f2000,  0x3f4000,  
	0x3f6000,  0x3f8000, 
	0x3fa000,  0x3fc000,  
	0x3fe000
};

unsigned int sp32_bottom_flash_sectors[ 71 ] = 
{
	0x000000,  0x002000, 
	0x004000,  0x006000,  
	0x008000,  0x00a000, 
	0x00c000,  0x00e000,  
	0x010000,  0x020000,  
	0x030000,  0x040000, 
	0x050000,  0x060000,  
	0x070000,  0x080000, 
	0x090000,  0x0a0000,  
	0x0b0000,  0x0c0000,  
	0x0d0000,  0x0e0000, 
	0x0f0000,  0x100000,  
	0x110000,  0x120000,  
	0x130000,  0x140000, 
	0x150000,  0x160000,  
	0x170000,  0x180000, 
	0x190000,  0x1a0000, 
	0x1b0000,  0x1c0000,  
	0x1d0000,  0x1e0000, 
	0x1f0000,  0x200000,  
	0x210000,  0x220000,  
	0x230000,  0x240000, 
	0x250000,  0x260000,  
	0x270000,  0x280000, 
	0x290000,  0x2a0000,  
	0x2b0000,  0x2c0000,  
	0x2d0000,  0x2e0000, 
	0x2f0000,  0x300000,  
	0x310000,  0x320000,  
	0x330000,  0x340000, 
	0x350000,  0x360000,  
	0x370000,  0x380000,
	0x390000,  0x3a0000,  
	0x3b0000,  0x3c0000, 
	0x3d0000,  0x3e0000,  
	0x3f0000
};

unsigned int mx64_top_flash_sectors[ 135 ] = 
{
	0x000000,  0x010000, 0x020000,  0x030000,	  
	0x040000,  0x050000, 0x060000,  0x070000, 	 
	0x080000,  0x090000, 0x0a0000,  0x0b0000,	 
	0x0c0000,  0x0d0000, 0x0e0000,  0x0f0000, 	 
	0x100000,  0x110000, 0x120000,  0x130000, 	  
	0x140000,  0x150000, 0x160000,  0x170000,	  
	0x180000,  0x190000, 0x1a0000,  0x1b0000, 	 
	0x1c0000,  0x1d0000, 0x1e0000,  0x1f0000, 	 
	0x200000,  0x210000, 0x220000,  0x230000,	  
	0x240000,  0x250000, 0x260000,  0x270000,	  
	0x280000,  0x290000, 0x2a0000,  0x2b0000, 	 
	0x2c0000,  0x2d0000, 0x2e0000,  0x2f0000, 	 
	0x300000,  0x310000, 0x320000,  0x330000, 	  
	0x340000,  0x350000, 0x360000,  0x370000,	  
	0x380000,  0x390000, 0x3a0000,  0x3b0000, 	 
	0x3c0000,  0x3d0000, 0x3e0000,  0x3f0000,
	
	0x400000,  0x410000, 0x420000,  0x430000,	  
	0x440000,  0x450000, 0x460000,  0x470000, 	 
	0x480000,  0x490000, 0x4a0000,  0x4b0000,	 
	0x4c0000,  0x4d0000, 0x4e0000,  0x4f0000, 	 
	0x500000,  0x510000, 0x520000,  0x530000, 	  
	0x540000,  0x550000, 0x560000,  0x570000,	  
	0x580000,  0x590000, 0x5a0000,  0x5b0000, 	 
	0x5c0000,  0x5d0000, 0x5e0000,  0x5f0000, 	 
	0x600000,  0x610000, 0x620000,  0x630000,	  
	0x640000,  0x650000, 0x660000,  0x670000,	  
	0x680000,  0x690000, 0x6a0000,  0x6b0000, 	 
	0x6c0000,  0x6d0000, 0x6e0000,  0x6f0000, 	 
	0x700000,  0x710000, 0x720000,  0x730000, 	  
	0x740000,  0x750000, 0x760000,  0x770000,	  
	0x780000,  0x790000, 0x7a0000,  0x7b0000, 	 
	0x7c0000,  0x7d0000, 0x7e0000,  0x7f0000,  
	
	0x7f2000,  0x7f4000, 0x7f6000,  0x7f8000, 	 
	0x7fa000,  0x7fc000, 0x7fe000 
	
};

unsigned int mx64_bottom_flash_sectors[ 135 ] = 
{
	0x000000,  0x002000, 0x004000,  0x006000,
	0x008000,  0x00a000, 0x00c000,  0x00e000, 
	  
	0x010000,  0x020000, 0x030000,	  
	0x040000,  0x050000, 0x060000,  0x070000, 	 
	0x080000,  0x090000, 0x0a0000,  0x0b0000,	 
	0x0c0000,  0x0d0000, 0x0e0000,  0x0f0000, 	 
	0x100000,  0x110000, 0x120000,  0x130000, 	  
	0x140000,  0x150000, 0x160000,  0x170000,	  
	0x180000,  0x190000, 0x1a0000,  0x1b0000, 	 
	0x1c0000,  0x1d0000, 0x1e0000,  0x1f0000, 	 
	0x200000,  0x210000, 0x220000,  0x230000,	  
	0x240000,  0x250000, 0x260000,  0x270000,	  
	0x280000,  0x290000, 0x2a0000,  0x2b0000, 	 
	0x2c0000,  0x2d0000, 0x2e0000,  0x2f0000, 	 
	0x300000,  0x310000, 0x320000,  0x330000, 	  
	0x340000,  0x350000, 0x360000,  0x370000,	  
	0x380000,  0x390000, 0x3a0000,  0x3b0000, 	 
	0x3c0000,  0x3d0000, 0x3e0000,  0x3f0000,
	
	0x400000,  0x410000, 0x420000,  0x430000,	  
	0x440000,  0x450000, 0x460000,  0x470000, 	 
	0x480000,  0x490000, 0x4a0000,  0x4b0000,	 
	0x4c0000,  0x4d0000, 0x4e0000,  0x4f0000, 	 
	0x500000,  0x510000, 0x520000,  0x530000, 	  
	0x540000,  0x550000, 0x560000,  0x570000,	  
	0x580000,  0x590000, 0x5a0000,  0x5b0000, 	 
	0x5c0000,  0x5d0000, 0x5e0000,  0x5f0000, 	 
	0x600000,  0x610000, 0x620000,  0x630000,	  
	0x640000,  0x650000, 0x660000,  0x670000,	  
	0x680000,  0x690000, 0x6a0000,  0x6b0000, 	 
	0x6c0000,  0x6d0000, 0x6e0000,  0x6f0000, 	 
	0x700000,  0x710000, 0x720000,  0x730000, 	  
	0x740000,  0x750000, 0x760000,  0x770000,	  
	0x780000,  0x790000, 0x7a0000,  0x7b0000, 	 
	0x7c0000,  0x7d0000, 0x7e0000,  0x7f0000,  
	
	
};

static unsigned int fujitsu64_top_flash_sectors[ 142 ] = 
{
	0x000000,  0x002000, 0x004000,  0x006000,
	0x008000,  0x00a000, 0x00c000,  0x00e000,

	0x010000,  0x020000, 0x030000,
	0x040000,  0x050000, 0x060000,  0x070000,
	0x080000,  0x090000, 0x0a0000,  0x0b0000,
	0x0c0000,  0x0d0000, 0x0e0000,  0x0f0000,
	0x100000,  0x110000, 0x120000,  0x130000,
	0x140000,  0x150000, 0x160000,  0x170000,
	0x180000,  0x190000, 0x1a0000,  0x1b0000,
	0x1c0000,  0x1d0000, 0x1e0000,  0x1f0000,
	0x200000,  0x210000, 0x220000,  0x230000,
	0x240000,  0x250000, 0x260000,  0x270000,
	0x280000,  0x290000, 0x2a0000,  0x2b0000,
	0x2c0000,  0x2d0000, 0x2e0000,  0x2f0000,
	0x300000,  0x310000, 0x320000,  0x330000,
	0x340000,  0x350000, 0x360000,  0x370000,
	0x380000,  0x390000, 0x3a0000,  0x3b0000,
	0x3c0000,  0x3d0000, 0x3e0000,  0x3f0000,
	0x400000,  0x410000, 0x420000,  0x430000,
	0x440000,  0x450000, 0x460000,  0x470000,
	0x480000,  0x490000, 0x4a0000,  0x4b0000,
	0x4c0000,  0x4d0000, 0x4e0000,  0x4f0000,
	0x500000,  0x510000, 0x520000,  0x530000,
	0x540000,  0x550000, 0x560000,  0x570000,
	0x580000,  0x590000, 0x5a0000,  0x5b0000,
	0x5c0000,  0x5d0000, 0x5e0000,  0x5f0000,
	0x600000,  0x610000, 0x620000,  0x630000,
	0x640000,  0x650000, 0x660000,  0x670000,
	0x680000,  0x690000, 0x6a0000,  0x6b0000,
	0x6c0000,  0x6d0000, 0x6e0000,  0x6f0000,
	0x700000,  0x710000, 0x720000,  0x730000,
	0x740000,  0x750000, 0x760000,  0x770000,
	0x780000,  0x790000, 0x7a0000,  0x7b0000,
	0x7c0000,  0x7d0000, 0x7e0000,  0x7f0000,

	0x7f2000,  0x7f4000, 0x7f6000,  0x7f8000,
	0x7fa000,  0x7fc000, 0x7fe000,
};
unsigned int mx128_top_flash_sectors[ 263 ] = 
{
	0x000000,  0x010000, 0x020000,  0x030000,	  
	0x040000,  0x050000, 0x060000,  0x070000, 	 
	0x080000,  0x090000, 0x0a0000,  0x0b0000,	 
	0x0c0000,  0x0d0000, 0x0e0000,  0x0f0000, 	 
	0x100000,  0x110000, 0x120000,  0x130000, 	  
	0x140000,  0x150000, 0x160000,  0x170000,	  
	0x180000,  0x190000, 0x1a0000,  0x1b0000, 	 
	0x1c0000,  0x1d0000, 0x1e0000,  0x1f0000, 	 
	0x200000,  0x210000, 0x220000,  0x230000,	  
	0x240000,  0x250000, 0x260000,  0x270000,	  
	0x280000,  0x290000, 0x2a0000,  0x2b0000, 	 
	0x2c0000,  0x2d0000, 0x2e0000,  0x2f0000, 	 
	0x300000,  0x310000, 0x320000,  0x330000, 	  
	0x340000,  0x350000, 0x360000,  0x370000,	  
	0x380000,  0x390000, 0x3a0000,  0x3b0000, 	 
	0x3c0000,  0x3d0000, 0x3e0000,  0x3f0000,
	
	0x400000,  0x410000, 0x420000,  0x430000,	  
	0x440000,  0x450000, 0x460000,  0x470000, 	 
	0x480000,  0x490000, 0x4a0000,  0x4b0000,	 
	0x4c0000,  0x4d0000, 0x4e0000,  0x4f0000, 	 
	0x500000,  0x510000, 0x520000,  0x530000, 	  
	0x540000,  0x550000, 0x560000,  0x570000,	  
	0x580000,  0x590000, 0x5a0000,  0x5b0000, 	 
	0x5c0000,  0x5d0000, 0x5e0000,  0x5f0000, 	 
	0x600000,  0x610000, 0x620000,  0x630000,	  
	0x640000,  0x650000, 0x660000,  0x670000,	  
	0x680000,  0x690000, 0x6a0000,  0x6b0000, 	 
	0x6c0000,  0x6d0000, 0x6e0000,  0x6f0000, 	 
	0x700000,  0x710000, 0x720000,  0x730000, 	  
	0x740000,  0x750000, 0x760000,  0x770000,	  
	0x780000,  0x790000, 0x7a0000,  0x7b0000, 	 
	0x7c0000,  0x7d0000, 0x7e0000,  0x7f0000, 
	
	0x800000,  0x810000, 0x820000,  0x830000,	  
	0x840000,  0x850000, 0x860000,  0x870000, 	 
	0x880000,  0x890000, 0x8a0000,  0x8b0000,	 
	0x8c0000,  0x8d0000, 0x8e0000,  0x8f0000, 	 
	0x900000,  0x910000, 0x920000,  0x930000, 	  
	0x940000,  0x950000, 0x960000,  0x970000,	  
	0x980000,  0x990000, 0x9a0000,  0x9b0000, 	 
	0x9c0000,  0x9d0000, 0x9e0000,  0x9f0000, 	 
	0xa00000,  0xa10000, 0xa20000,  0xa30000,	  
	0xa40000,  0xa50000, 0xa60000,  0xa70000,	  
	0xa80000,  0xa90000, 0xaa0000,  0xab0000, 	 
	0xac0000,  0xad0000, 0xae0000,  0xaf0000, 	 
	0xb00000,  0xb10000, 0xb20000,  0xb30000, 	  
	0xb40000,  0xb50000, 0xb60000,  0xb70000,	  
	0xb80000,  0xb90000, 0xba0000,  0xbb0000, 	 
	0xbc0000,  0xbd0000, 0xbe0000,  0xbf0000,
	
	0xc00000,  0xc10000, 0xc20000,  0xc30000,	  
	0xc40000,  0xc50000, 0xc60000,  0xc70000, 	 
	0xc80000,  0xc90000, 0xca0000,  0xcb0000,	 
	0xcc0000,  0xcd0000, 0xce0000,  0xcf0000, 	 
	0xd00000,  0xd10000, 0xd20000,  0xd30000, 	  
	0xd40000,  0xd50000, 0xd60000,  0xd70000,	  
	0xd80000,  0xd90000, 0xda0000,  0xdb0000, 	 
	0xdc0000,  0xdd0000, 0xde0000,  0xdf0000, 	 
	0xe00000,  0xe10000, 0xe20000,  0xe30000,	  
	0xe40000,  0xe50000, 0xe60000,  0xe70000,	  
	0xe80000,  0xe90000, 0xea0000,  0xeb0000, 	 
	0xec0000,  0xed0000, 0xee0000,  0xef0000, 	 
	0xf00000,  0xf10000, 0xf20000,  0xf30000, 	  
	0xf40000,  0xf50000, 0xf60000,  0xf70000,	  
	0xf80000,  0xf90000, 0xfa0000,  0xfb0000, 	 
	0xfc0000,  0xfd0000, 0xfe0000,  0xff0000,   
	
	0xff2000,  0xff4000, 0xff6000,  0xff8000, 	 
	0xffa000,  0xffc000, 0xffe000 
	
};

unsigned int mx128_bottom_flash_sectors[ 263 ] = 
{
	0x000000,  0x002000, 0x004000,  0x006000,  
	0x008000,  0x00a000, 0x00c000,  0x00e000,
	0x010000,  0x020000, 0x030000,	  
	0x040000,  0x050000, 0x060000,  0x070000, 	 
	0x080000,  0x090000, 0x0a0000,  0x0b0000,	 
	0x0c0000,  0x0d0000, 0x0e0000,  0x0f0000, 	 
	0x100000,  0x110000, 0x120000,  0x130000, 	  
	0x140000,  0x150000, 0x160000,  0x170000,	  
	0x180000,  0x190000, 0x1a0000,  0x1b0000, 	 
	0x1c0000,  0x1d0000, 0x1e0000,  0x1f0000, 	 
	0x200000,  0x210000, 0x220000,  0x230000,	  
	0x240000,  0x250000, 0x260000,  0x270000,	  
	0x280000,  0x290000, 0x2a0000,  0x2b0000, 	 
	0x2c0000,  0x2d0000, 0x2e0000,  0x2f0000, 	 
	0x300000,  0x310000, 0x320000,  0x330000, 	  
	0x340000,  0x350000, 0x360000,  0x370000,	  
	0x380000,  0x390000, 0x3a0000,  0x3b0000, 	 
	0x3c0000,  0x3d0000, 0x3e0000,  0x3f0000,
	
	0x400000,  0x410000, 0x420000,  0x430000,	  
	0x440000,  0x450000, 0x460000,  0x470000, 	 
	0x480000,  0x490000, 0x4a0000,  0x4b0000,	 
	0x4c0000,  0x4d0000, 0x4e0000,  0x4f0000, 	 
	0x500000,  0x510000, 0x520000,  0x530000, 	  
	0x540000,  0x550000, 0x560000,  0x570000,	  
	0x580000,  0x590000, 0x5a0000,  0x5b0000, 	 
	0x5c0000,  0x5d0000, 0x5e0000,  0x5f0000, 	 
	0x600000,  0x610000, 0x620000,  0x630000,	  
	0x640000,  0x650000, 0x660000,  0x670000,	  
	0x680000,  0x690000, 0x6a0000,  0x6b0000, 	 
	0x6c0000,  0x6d0000, 0x6e0000,  0x6f0000, 	 
	0x700000,  0x710000, 0x720000,  0x730000, 	  
	0x740000,  0x750000, 0x760000,  0x770000,	  
	0x780000,  0x790000, 0x7a0000,  0x7b0000, 	 
	0x7c0000,  0x7d0000, 0x7e0000,  0x7f0000, 
	
	0x800000,  0x810000, 0x820000,  0x830000,	  
	0x840000,  0x850000, 0x860000,  0x870000, 	 
	0x880000,  0x890000, 0x8a0000,  0x8b0000,	 
	0x8c0000,  0x8d0000, 0x8e0000,  0x8f0000, 	 
	0x900000,  0x910000, 0x920000,  0x930000, 	  
	0x940000,  0x950000, 0x960000,  0x970000,	  
	0x980000,  0x990000, 0x9a0000,  0x9b0000, 	 
	0x9c0000,  0x9d0000, 0x9e0000,  0x9f0000, 	 
	0xa00000,  0xa10000, 0xa20000,  0xa30000,	  
	0xa40000,  0xa50000, 0xa60000,  0xa70000,	  
	0xa80000,  0xa90000, 0xaa0000,  0xab0000, 	 
	0xac0000,  0xad0000, 0xae0000,  0xaf0000, 	 
	0xb00000,  0xb10000, 0xb20000,  0xb30000, 	  
	0xb40000,  0xb50000, 0xb60000,  0xb70000,	  
	0xb80000,  0xb90000, 0xba0000,  0xbb0000, 	 
	0xbc0000,  0xbd0000, 0xbe0000,  0xbf0000,
	
	0xc00000,  0xc10000, 0xc20000,  0xc30000,	  
	0xc40000,  0xc50000, 0xc60000,  0xc70000, 	 
	0xc80000,  0xc90000, 0xca0000,  0xcb0000,	 
	0xcc0000,  0xcd0000, 0xce0000,  0xcf0000, 	 
	0xd00000,  0xd10000, 0xd20000,  0xd30000, 	  
	0xd40000,  0xd50000, 0xd60000,  0xd70000,	  
	0xd80000,  0xd90000, 0xda0000,  0xdb0000, 	 
	0xdc0000,  0xdd0000, 0xde0000,  0xdf0000, 	 
	0xe00000,  0xe10000, 0xe20000,  0xe30000,	  
	0xe40000,  0xe50000, 0xe60000,  0xe70000,	  
	0xe80000,  0xe90000, 0xea0000,  0xeb0000, 	 
	0xec0000,  0xed0000, 0xee0000,  0xef0000, 	 
	0xf00000,  0xf10000, 0xf20000,  0xf30000, 	  
	0xf40000,  0xf50000, 0xf60000,  0xf70000,	  
	0xf80000,  0xf90000, 0xfa0000,  0xfb0000, 	 
	0xfc0000,  0xfd0000, 0xfe0000,  0xff0000,   
	
	
	
	
};

unsigned int sp128_flash_sectors[ 128 ] =           /* 128 */
{
	0x000000,  0x020000,  0x040000, 0x060000, 
	0x080000,  0x0a0000,  0x0c0000, 0x0e0000,
	0x100000,  0x120000,  0x140000, 0x160000, 
	0x180000,  0x1a0000,  0x1c0000, 0x1e0000,
	0x200000,  0x220000,  0x240000, 0x260000, 
	0x280000,  0x2a0000,  0x2c0000, 0x2e0000,
	0x300000,  0x320000,  0x340000, 0x360000, 
	0x380000,  0x3a0000,  0x3c0000, 0x3e0000,
	0x400000,  0x420000,  0x440000, 0x460000, 
	0x480000,  0x4a0000,  0x4c0000, 0x4e0000,
	0x500000,  0x520000,  0x540000, 0x560000, 
	0x580000,  0x5a0000,  0x5c0000, 0x5e0000,
	0x600000,  0x620000,  0x640000, 0x660000, 
	0x680000,  0x6a0000,  0x6c0000, 0x6e0000,
	0x700000,  0x720000,  0x740000, 0x760000, 
	0x780000,  0x7a0000,  0x7c0000, 0x7e0000,

	0x800000,  0x820000,  0x840000, 0x860000, 
	0x880000,  0x8a0000,  0x8c0000, 0x8e0000,
	0x900000,  0x920000,  0x940000, 0x960000, 
	0x980000,  0x9a0000,  0x9c0000, 0x9e0000,
	0xa00000,  0xa20000,  0xa40000, 0xa60000, 
	0xa80000,  0xaa0000,  0xac0000, 0xae0000,
	0xb00000,  0xb20000,  0xb40000, 0xb60000, 
	0xb80000,  0xba0000,  0xbc0000, 0xbe0000,
	0xc00000,  0xc20000,  0xc40000, 0xc60000, 
	0xc80000,  0xca0000,  0xcc0000, 0xce0000,
	0xd00000,  0xd20000,  0xd40000, 0xd60000, 
	0xd80000,  0xda0000,  0xdc0000, 0xde0000,
	0xe00000,  0xe20000,  0xe40000, 0xe60000, 
	0xe80000,  0xea0000,  0xec0000, 0xee0000,
	0xf00000,  0xf20000,  0xf40000, 0xf60000, 
	0xf80000,  0xfa0000,  0xfc0000, 0xfe0000
	
};








static p_device_type device[] = 
{
 {MXIC    ,  3,  MX29LV160C_BOTTOM  ,  mx16_bottom_flash_sectors , 35  , 0x200000 },    
 {MXIC    ,  3,  MX29LV160C_TOP     ,  mx16_top_flash_sectors    , 35  , 0x200000 },    
 {MXIC    ,  1,  MX29LV320C_BOTTOM  ,  sp32_bottom_flash_sectors , 71  , 0x400000 },
 {MXIC    ,  1,  MX29LV320C_TOP     ,  sp32_top_flash_sectors    , 71  , 0x400000 }, 
 {MXIC    ,  1,  MX29LV640B_BOTTOM  ,  mx64_bottom_flash_sectors , 135 , 0x800000 },
 {MXIC    ,  1,  MX29LV640B_TOP     ,  mx64_top_flash_sectors    , 135 , 0x800000 },
 {MXIC    ,  3,  MX29LV320M_BOTTOM  ,  sp32_bottom_flash_sectors , 71  , 0x400000 },
 {MXIC    ,  3,  MX29LV320M_TOP     ,  sp32_top_flash_sectors    , 71  , 0x400000 }, 
 {MXIC    ,  3,  MX29LV640M_BOTTOM  ,  mx64_bottom_flash_sectors , 135 , 0x800000 },
 {MXIC    ,  3,  MX29LV640M_TOP     ,  mx64_top_flash_sectors    , 135 , 0x800000 },
 {MXIC    ,  3,  MX29LV128M_BOTTOM  ,  mx128_bottom_flash_sectors, 263 , 0x1000000},
 {MXIC    ,  3,  MX29LV128M_TOP     ,  mx128_top_flash_sectors   , 263 , 0x1000000},
 {MXIC    ,  1,  MX29LV128D_BOTTOM  ,  mx128_bottom_flash_sectors , 263    ,0x1000000} ,
 {MXIC    ,  1,  MX29LV128D_TOP     ,  mx128_top_flash_sectors    , 263    ,0x1000000} ,
 {SPANSION,  3,  S29AL016D_BOTTOM   ,  mx16_bottom_flash_sectors , 35  , 0x200000 },
 {SPANSION,  3,  S29AL016D_TOP      ,  mx16_top_flash_sectors    , 35  , 0x200000 },
 {SPANSION,  3,  S29GL64M_R4_BOTTOM ,  mx64_bottom_flash_sectors , 135 , 0x800000 },
 {SPANSION,  3,  S29GL64M_R3_TOP    ,  mx64_top_flash_sectors    , 135 , 0x800000 },
 {SPANSION,  3,  S29GL32M_R4_BOTTOM ,  sp32_bottom_flash_sectors , 71  , 0x400000 },
 {SPANSION,  3,  S29GL32M_R3_TOP    ,  sp32_top_flash_sectors    , 71  , 0x400000 },
 {SPANSION,  3,  S29GL128N          ,  sp128_flash_sectors       , 128 , 0x1000000},
 {ATMEL   ,  1,  AT49BV163AT        ,  at16_top_flash_sectors    , 39  , 0x200000 },
 {ATMEL   ,  1,  AT49BV163A         ,  at16_bottom_flash_sectors , 39  , 0x200000 }, 
 {AMIC    ,  1,  A29L160A_BOTTOM    ,  mx16_bottom_flash_sectors , 35  , 0x200000 },
 {AMIC    ,  1,  A29L160A_TOP       ,  mx16_top_flash_sectors    , 35  , 0x200000 },
 {EON     ,  1,  EN29LV640B_BOTTOM  ,  mx64_bottom_flash_sectors , 135 , 0x800000 },
 {EON     ,  1,  EN29LV640B_TOP     ,  mx64_top_flash_sectors    , 135 , 0x800000 },
 {FUJITSU ,  1,  MBM29DL640E_TOP    ,  fujitsu64_top_flash_sectors, 142, 0x800000 },
} ; 

#define DEV_SIZE	(sizeof(device)/sizeof(p_device_type))



/************************************************************************
 *      Static function prototypes
 ************************************************************************/
 
static unsigned int get_sector_index(p_device_type dev_type,  UINT32 p_addr );

/************************************************************************
 *
 *                          FLASH_PARELLEL_init
 *  Description :
 *  -------------
 * This service initializes the FLASH driver to handle all
 * FLASH devices on this board environment.
 *  
 *
 *  Parameters :
 *  ------------
 *
 *  'major',     IN,    major device number
 *  'minor',     IN,    not used
 *  'p_param',   INOUT, not used
 *
 *
 *  Return values :
 *  ---------------
 *
 *  'OK'(=0)
 *
 *
 *
 ************************************************************************/
static
INT32 FLASH_PARELLEL_init(
          UINT32 major,          /* IN: major device number             */
          UINT32 minor,          /* IN: minor device number             */
          void   *p_param ) ;    /* INOUT: device parameter block       */



/************************************************************************
 *
 *                          FLASH_PARELLEL_write
 *  Description :
 *  -------------
 *  This service writes data into a specified address location, which
 *  can be in either SYSTEM-FLASH or FILE-FLASH space. Default is RAM.
 *  
 *
 *  Parameters :
 *  ------------
 *
 *  'major',     IN,    major device number
 *  'minor',     IN,    minor device number for multi device drivers
 *  'p_param',   IN,    variable of type, t_FLASH_write_descriptor.
 *
 *
 *  Return values :
 *  ---------------
 *
 * 'OK' = 0x00:                         data has been stored
 *  ERROR_FLASH_PROGRAM_ERROR           Flash device failure
 *  ERROR_FLASH_INVALID_ADDRESS,        Physical address not impl.    
 *
 *
 ************************************************************************/
static
INT32 FLASH_PARELLEL_write(
          UINT32 major,          /* IN: major device number             */
          UINT32 minor,          /* IN: minor device number             */
          t_FLASH_write_descriptor *p_param ) ; /* IN: write data       */


/************************************************************************
 *
 *                          FLASH_PARELLEL_ctrl
 *  Description :
 *  -------------
 *  This service comprise following specific FLASH services:
 *    1) 'ERASE_SYSTEMFLASH'
 *    2) 'ERASE_FILEFLASH'
 *    3) 'FLASH_CTRL_ERASE_FLASH_AREA'
 *    4) 'FLASH_CTRL_INQUIRE_FLASH_DEVICEID'
 *    5) 'FLASH_CTRL_TEST_SYSTEMFLASH'
 *  
 *
 *  Parameters :
 *  ------------
 *
 *  'major',     IN,    major device number
 *  'minor',     IN,    minor device number for multi device drivers
 *  'p_param',   IN,    variable of type, t_FLASH_ctrl_descriptor.
 *
 *
 *  Return values :
 *  ---------------
 *
 * 'OK' = 0x00:                         FLASH service completed successfully
 *  ERROR_FLASH_PROGRAM_ERROR           Flash device failure
 *
 *
 ************************************************************************/
static
INT32 FLASH_PARELLEL_ctrl(
          UINT32 major,          /* IN: major device number             */
          UINT32 minor,          /* IN: minor device number             */
          t_FLASH_ctrl_descriptor *p_param ) ;  /* IN: write data       */





/************************************************************************
 *
 *                          FLASH_PARELLEL_is_file_flash_write_protected
 *  Description :
 *  -------------
 *  Check if file flash id write protected
 *  
 *
 *  Parameters :
 *  ------------
 *
 *
 *
 *  Return values :
 *  ---------------
 *
 *  FALSE: not write protected
 *  TRUE:  write protected
 *
 *
 ************************************************************************/
static
bool FLASH_PARELLEL_is_file_flash_write_protected( void ) ;

/************************************************************************
 *
 *                          FLASH_PARELLEL_is_monitor_flash_write_protected
 *  Description :
 *  -------------
 *  Check if file flash id write protected
 *
 *
 *  Parameters :
 *  ------------
 *
 *
 *
 *  Return values :
 *  ---------------
 *
 *  FALSE: not write protected
 *  TRUE:  write protected
 *
 *
 ************************************************************************/
static
bool FLASH_PARELLEL_is_monitor_flash_write_protected( void ) ;


/************************************************************************
 *
 *                          FLASH_PARELLEL_devicetype
 *  Description :
 *  -------------
 *  Derive the memory device type from 'physical address'
 *  
 *
 *  Parameters :
 *  ------------
 *
 *  'physadr': physical address inside a 512 MByte space
 *
 *
 *  Return values :
 *  ---------------
 *
 *  t_flash_device_id:    memory device type detected.
 *
 *
 ************************************************************************/
static
t_flash_device_id FLASH_PARELLEL_devicetype( UINT32 physadr ) ;


/************************************************************************
 *
 *                          FLASH_PARELLEL_program_systemram
 *  Description :
 *  -------------
 *  Programs (store data) in the system RAM device.
 *  
 *
 *  Parameters :
 *  ------------
 *
 *  'p_param',   IN,    variable of type, t_FLASH_write_descriptor.
 *
 *
 *  Return values :
 *  ---------------
 *
 * 'OK' = 0x00:             Data stored in system RAM
 *
 *
 ************************************************************************/
static
INT32 FLASH_PARELLEL_program_systemram( t_FLASH_write_descriptor *p_param ) ;


/************************************************************************
 *
 *                          FLASH_PARELLEL_program_flash
 *  Description :
 *  -------------
 *  Programs a Intel 28Fxxx-compliant flash device.
 *  
 *
 *  Parameters :
 *  ------------
 *
 *  'p_param',   IN,    variable of type, t_FLASH_write_descriptor.
 *
 *
 *  Return values :
 *  ---------------
 *
 * 'OK' = 0x00:                         FLASH programmed succesfully
 *  ERROR_FLASH_PROGRAM_ERROR           Flash device failure
 *
 *
 ************************************************************************/
static
INT32 FLASH_PARELLEL_program_flash( t_FLASH_write_descriptor *p_param ) ;






/************************************************************************
 *
 *                          FLASH_PARELLEL_erase_fileflash
 *  Description :
 *  -------------
 *  Erase file flash, which is the last block (in each FLASH device)
 *  of the monitor flash.
 *  
 *
 *  Parameters :
 *  ------------
 *
 *  -
 *
 *  Return values :
 *  ---------------
 *
 * 'OK' = 0x00:                         File FLASH erased succesfully
 *  ERROR_FLASH_PROGRAM_ERROR           Flash device failure
 *
 *
 ************************************************************************/
static
INT32 FLASH_PARELLEL_erase_fileflash( void ) ;


/************************************************************************
 *
 *                          FLASH_PARELLEL_erase_flasharea
 *  Description :
 *  -------------
 *  Erase flash area; i.e. the driver erases the flash blocks inside 
 *  the specified memory area.
 *  
 *
 *  Parameters :
 *  ------------
 * 
 *  'p_param',   IN,    variable of type, t_FLASH_ctrl_descriptor.
 *
 *
 *  Return values :
 *  ---------------
 *
 * 'OK' = 0x00:                         FLASH erased succesfully
 *  ERROR_FLASH_PROGRAM_ERROR           Flash device failure
 *  ERROR_FLASH_INVALID_ADDRESS         Address area not inside FLASH
 *                                      devices
 *
 *
 ************************************************************************/
static
INT32 FLASH_PARELLEL_erase_flasharea( t_FLASH_ctrl_descriptor *p_param ) ;

 
/************************************************************************
 *
 *                          FLASH_PARELLEL_inquire_deviceid
 *  Description :
 *  -------------
 *  Inquire flash device id.
 *  
 *
 *  Parameters :
 *  ------------
 * 
 *  
 *
 *
 *  Return values :
 *  ---------------
 *
 * 'OK' = 0x00:                         
 *  ERROR_FLASH_DEVICE_ID        
 *
 *
 ************************************************************************/
static
INT32 FLASH_PARELLEL_inquire_deviceid( void) ;






/************************************************************************
 *
 *                          FLASH_PARELLEL_wait_ready
 *  Description :
 *  -------------
 *  
 *  Await FLASH operation completes.
 *  
 *
 *  Parameters :
 *  ------------
 *
 *  -
 *
 *  Return values :
 *  ---------------
 *
 * 'OK' = 0x00:                         File FLASH erased succesfully
 *  ERROR_FLASH_PROGRAM_ERROR           Flash device failure
 *
 *
 ************************************************************************/
static
INT32 FLASH_PARELLEL_wait_ready(UINT32 pw, UINT32 retry ) ;



/************************************************************************
 *
 *                          FLASH_error_lookup
 *  Description :
 *  -------------
 *  Lookup error code to error string(s)
 *  
 *
 *  Parameters :
 *  ------------
 * 
 *  'p_param',   INOUT,    variable of type, t_sys_error_string.
 *
 *
 *  Return values :
 *  ---------------
 *
 * 'OK' = 0x00:  
 *
 *
 ************************************************************************/
static
INT32 FLASH_error_lookup( t_sys_error_string *p_param ) ;


/************************************************************************
 *      Implementation : Public functions
 ************************************************************************/


/************************************************************************
 *
 *                          FLASH_PARELLEL_install
 *  Description :
 *  -------------
 *
 *  Installs the PARELLEL FLASH device drivers services in 
 *  the IO system at the reserved device slot, found in the
 *  'sysdev.h' file, which defines all major device numbers.
 *
 *  Note:
 *  This service is the only public declared interface function; all
 *  provided device driver services are static declared, but this
 *  function installs the function pointers in the io-system to
 *  enable the provided public driver services.
 *
 *  Parameters :
 *  ------------
 *
 *  -
 *
 *
 *  Return values :
 *  ---------------
 *
 *  'OK'(=0)
 *  'ERROR_IO_ILLEGAL_MAJOR':  Illegal major device number
 *  'ERROR_IO_NO_SPACE':       Device slot already allocated
 *
 ************************************************************************/
INT32 FLASH_PARELLEL_install( void )
{
    t_sys_error_lookup_registration registration ;

    /* register lookup syserror */
    registration.prefix = SYSERROR_DOMAIN( ERROR_FLASH ) ;
    registration.lookup = FLASH_error_lookup ;
    SYSCON_write( SYSCON_ERROR_REGISTER_LOOKUP_ID,
                  &registration, 
                  sizeof( registration ) );
   
    /* pre-initialize local variables and install device services */
    IO_install( SYS_MAJOR_FLASH,         /* major device number */
           (t_io_service) FLASH_PARELLEL_init,    /* 'init'  service     */
                          NULL,                 /* 'open'  service  na */
                          NULL,                 /* 'close' service  na */
                          NULL,                 /* 'read'  service     */
           (t_io_service) FLASH_PARELLEL_write,   /* 'write' service     */
           (t_io_service) FLASH_PARELLEL_ctrl ) ; /* 'ctrl'  service     */

    /* call our own 'init' service */
    return IO_init( SYS_MAJOR_FLASH, 0, NULL);
  
     
 
}


/************************************************************************
 *      Implementation : Public functions
 ************************************************************************/


/************************************************************************
 *
 *                          FLASH_install
 *  Description :
 *  -------------
 *
 *  Install FLASH_SPI_install function.  When Board_FLASH_SUPPORT is 
 *  defined as 'PARALLEL' type, the function FLASH_install would active,
 *  otherwise, it is disable at compiling time
 *
 *  Note:
 *
 *  Parameters :
 *
 *  Return values :
 *
 ************************************************************************/
#if defined(Board_FLASH_Support_PARALLEL)
INT32 FLASH_install( void )
{
	return FLASH_PARELLEL_install();
}
#endif


/************************************************************************
 *      Implementation : Static functions
 ************************************************************************/

/************************************************************************
 *
 *                          FLASH_PARELLEL_init
 *  Description :
 *  -------------
 *  This service initializes the FLASH_PARELLEL driver.
 *  
 *
 *  Parameters :
 *  ------------
 *
 *  'major',     IN,    major device number
 *  'minor',     IN,    not used
 *  'p_param',   INOUT, not used
 *
 *
 *  Return values :
 *  ---------------
 *
 *  'OK'(=0)
 *
 *
 *
 ************************************************************************/
static
INT32 FLASH_PARELLEL_init(
          UINT32 major,          /* IN: major device number             */
          UINT32 minor,          /* IN: minor device number             */
          void   *p_param )      /* INOUT: device parameter block       */
{
    int rcode, i ;
    unsigned char mid, did0, did1, did2;

   
    /* 0xb801a014=0xffff */
    REG32(KSEG1(RTL_FLASHCTRL_REG)) = 0xffff;
   
   
    

    

   
    
    /* initialize system profile: FLASH, address boundaries */
   
   

   

    

    
    REG8(KSEG1(0xbfc00000) + 0xf556) = 0xaa ;
    REG8(KSEG1(0xbfc00000) + 0xfaa9) = 0x55 ;
    REG8(KSEG1(0xbfc00000) + 0xf556) = 0x90 ;
    mid  = REG8(0xbfc00000 + 0xfffc);
    did0 = REG8(0xbfc00000 + 0xfffe);
    did1 = REG8(0xbfc00000 + 0xffe0);
    did2 = REG8(0xbfc00000 + 0xffe2);
    
    for (i = 0; i< DEV_SIZE; i++)
           if ( device[i].mid == mid ) 
           {
              	
              switch (device[i].did_cycle) 
              {
              	case 1 : 
              	         if ( (device[i].did&0xff) == did0 )       
              	         {         	       
                              goto device_found;     
                         }     
                         break;
                case 2 :
                         if (( (device[i].did&0xff) == did0 )&&( ((device[i].did>>8) &0xff) == did1 ))
                         {
                             	
                              goto device_found; 
                         }     
                         break; 
                case 3 :
                                      
                         if (( (device[i].did&0xff) == did0 )&&( ((device[i].did>>8) &0xff) == did1 )
                            &&( ((device[i].did>>16) &0xff) == did2 ))
                         {
                              	   
                              goto device_found; 
                         }     
                         break;                          
                default : 
                         break;
                                                     	
              }	        
           }  
           
    if (i== DEV_SIZE)
       return ( !OK ) ; 
              
device_found :

    
    devicetype = device[i];  
    rtl_monitorflash_size = device[i].size;
   
   
    /* send Reset command */
    
    REG8(KSEG1((0xbfc00000)))        = 0xf0; 
    
    
    
    /* MONITOR FLASH: */
    /* get base for monitor flash  */
#if 0    
    rcode = SYSCON_read( SYSCON_BOARD_MONITORFLASH_BASE_ID,
                         &monitorflash_phys_start,
                         sizeof(monitorflash_phys_start) ) ;
    if (rcode != OK)
    {
        monitorflash_phys_start = 0 ;
    }    

    monitorflash_phys_start = PHYS(monitorflash_phys_start) ;
    
    

    /* get size for monitor flash  */
    rcode = SYSCON_read( SYSCON_BOARD_MONITORFLASH_SIZE_ID,
                         &monitorflash_phys_end,
                         sizeof(monitorflash_phys_end) ) ;
    if (rcode != OK)
    {
        monitorflash_phys_end = 0 ;
    }
#endif


    monitorflash_phys_start = 0x1fd00000 - rtl_monitorflash_size ; 
    
    monitorflash_phys_end   = 0x1fd00000 ;

    /* FILE FLASH: */
    /* get base for file flash  */
    rcode = SYSCON_read( SYSCON_BOARD_FILEFLASH_BASE_ID,
                         &fileflash_phys_start,
                         sizeof(fileflash_phys_start) ) ;
    if (rcode != OK)
    {
        fileflash_phys_start = 0 ;
    }
    fileflash_phys_start = PHYS(fileflash_phys_start) ;

    /* get size for file flash  */
    rcode = SYSCON_read( SYSCON_BOARD_FILEFLASH_SIZE_ID,
                         &fileflash_phys_end,
                         sizeof(fileflash_phys_end) ) ;
    if (rcode != OK)
    {
        fileflash_phys_end = 0 ;
    }
    fileflash_phys_end = fileflash_phys_start + fileflash_phys_end ;


#if 0
    if (p_param != NULL) 
    {
        /*  Hack that may be used e.g. for test of monitor flash.
	 *  Not used by YAMON.
	 */
        fileflash_phys_start = monitorflash_phys_start;
        monitorflash_phys_start = 0xffffffff;
    }

  

    /* get monitor flash sector size */
    rcode = SYSCON_read( SYSCON_BOARD_MONITORFLASH_SECTORSIZE_ID,
                         &monitorflash_block_size,
                         sizeof(monitorflash_block_size) ) ;
    monitorflash_block_size = 2 * monitorflash_block_size ;
    if (rcode != OK)
    {
        monitorflash_block_size = 0 ;
    }

    /* get file flash sector size */
    rcode = SYSCON_read( SYSCON_BOARD_FILEFLASH_SECTORSIZE_ID,
                         &fileflash_block_size,
                         sizeof(fileflash_block_size) ) ;
    fileflash_block_size = 2 * fileflash_block_size ;
    if (rcode != OK)
    {
        fileflash_block_size = 0 ;
    }
#endif    
    
    return( OK ) ;
}


/************************************************************************
 *
 *                          FLASH_PARELLEL_write
 *  Description :
 *  -------------
 *  This service writes data into a specified address location, which
 *  can be in either SYSTEM-FLASH or FILE-FLASH space. Default is RAM.
 *  
 *
 *  Parameters :
 *  ------------
 *
 *  'major',     IN,    major device number
 *  'minor',     IN,    minor device number for multi device drivers
 *  'p_param',   IN,    variable of type, t_FLASH_write_descriptor.
 *
 *
 *  Return values :
 *  ---------------
 *
 * 'OK' = 0x00:                         data has been stored
 *  ERROR_FLASH_PROGRAM_ERROR           Flash device failure
 *  ERROR_FLASH_INVALID_ADDRESS,        Physical address not impl.    
 *
 *
 ************************************************************************/
static
INT32 FLASH_PARELLEL_write(
          UINT32 major,          /* IN: major device number             */
          UINT32 minor,          /* IN: minor device number             */
          t_FLASH_write_descriptor *p_param )   /* IN: write data       */
{
    int rcode = OK ;

    /* set to default */
    flash_diag_msg[0] = 0 ;

    switch (FLASH_PARELLEL_devicetype(PHYS(p_param->adr)))
    {
       

        case FLASH_MONITORFLASH_DEVICE:
           
            
            /* program monitor FLASH */
            rcode = FLASH_PARELLEL_program_flash( p_param ) ;

           
          break ;

        default:
            
            /* program any RAM */
            rcode = FLASH_PARELLEL_program_systemram( p_param ) ;
          break ;
    }
    flash_last_error = rcode ;
    return( rcode ) ;
}


/************************************************************************
 *
 *                          FLASH_PARELLEL_ctrl
 *  Description :
 *  -------------
 *  This service comprise following specific FLASH services:
 *    1) 'ERASE_SYSTEMFLASH'
 *    2) 'ERASE_FILEFLASH'
 *    3) 'FLASH_CTRL_ERASE_FLASH_AREA'
 *    4) 'FLASH_CTRL_INQUIRE_FLASH_DEVICEID'
 *    5) 'FLASH_CTRL_TEST_SYSTEMFLASH'
 *  
 *
 *  Parameters :
 *  ------------
 *
 *  'major',     IN,    major device number
 *  'minor',     IN,    minor device number for multi device drivers
 *  'p_param',   IN,    variable of type, t_FLASH_ctrl_descriptor.
 *
 *
 *  Return values :
 *  ---------------
 *
 * 'OK' = 0x00:                         FLASH service completed successfully
 *  ERROR_FLASH_PROGRAM_ERROR           Flash device failure
 *
 *
 ************************************************************************/
static
INT32 FLASH_PARELLEL_ctrl(
          UINT32 major,          /* IN: major device number             */
          UINT32 minor,          /* IN: minor device number             */
          t_FLASH_ctrl_descriptor *p_param )    /* IN: write data       */
{
    int    rcode = OK ;

    /* set to default */
    flash_diag_msg[0] = 0 ;
    


    switch(p_param->command)
    {
       
 
        case FLASH_CTRL_ERASE_FILEFLASH:
            if (devicetype.did == S29GL128N)
            {	
               t_FLASH_write_descriptor p_param2 ;
               UINT32 addr;
               	

               for (addr = 0 ; addr < 0x10000 ; addr++)
                 *(char *)(0xa0a00000 +addr) = REG8(KSEG1(fileflash_phys_start)-
                                               0x10000+ addr); 
        
               FLASH_PARELLEL_erase_fileflash() ;
        
               p_param2.adr    = KSEG1(fileflash_phys_start)- 0x10000 ;
               p_param2.length = 0x10000 ;
               p_param2.buffer = (UINT8 *)0xa0a00000;
        
               FLASH_PARELLEL_program_flash( &p_param2 ) ;
            } 
            else
               rcode = FLASH_PARELLEL_erase_fileflash() ;
          break ;

        case FLASH_CTRL_ERASE_FLASH_AREA: 
            rcode = FLASH_PARELLEL_erase_flasharea(p_param) ;
          break ;

        case FLASH_CTRL_INQUIRE_FLASH_DEVICEID:
            rcode = FLASH_PARELLEL_inquire_deviceid() ;
          break ;

      

        case FLASH_CTRL_WRITE_FILEFLASH:
            /* program file FLASH */
           
            rcode = FLASH_PARELLEL_program_flash( p_param->wr_param ) ;
            if (rcode ==  ERROR_FLASH_LOCKED)
            {
                if ( FLASH_PARELLEL_is_file_flash_write_protected() )
                {
                    rcode = ERROR_FLASH_FILE_FLASH_LOCK ;
                }
            }


          break ;

        

        default:
            rcode = ERROR_FLASH_INVALID_COMMAND ;
          break ;
    }
    flash_last_error = rcode ;
    return( rcode ) ;
}


/************************************************************************
 *      Local helper functions
 ************************************************************************/

/************************************************************************
 *
 *                          FLASH_PARELLEL_devicetype
 *  Description :
 *  -------------
 *  Derive the memory device type from 'physical address'
 *  
 *
 *  Parameters :
 *  ------------
 *
 *  'physadr': physical address inside a 512 MByte space
 *
 *
 *  Return values :
 *  ---------------
 *
 *  t_flash_device_id:    memory device type detected.
 *
 *
 ************************************************************************/
static
t_flash_device_id FLASH_PARELLEL_devicetype( UINT32 physadr )
{
  
    /* check for monitor FLASH */
    if ((monitorflash_phys_start <= physadr) && (physadr < monitorflash_phys_end))
    {
    	
        return(FLASH_MONITORFLASH_DEVICE) ;
    }

    

    /* device not known */
    return(FLASH_UNKNOWN_DEVICE) ;
}


/************************************************************************
 *
 *                          FLASH_PARELLEL_program_systemram
 *  Description :
 *  -------------
 *  Programs (store data) in the system RAM device.
 *  
 *
 *  Parameters :
 *  ------------
 *
 *  'p_param',   IN,    variable of type, t_FLASH_write_descriptor.
 *
 *
 *  Return values :
 *  ---------------
 *
 * 'OK' = 0x00:             Data stored in system RAM
 *
 *
 ************************************************************************/
static
INT32 FLASH_PARELLEL_program_systemram( t_FLASH_write_descriptor *p_param )
{
    int len ;
    UINT8 *psrc ;
    UINT8 *pdst ;

    /* set addresses */
    psrc = (UINT8*) p_param->buffer  ;
    pdst = (UINT8*) p_param->adr ;
    len  = p_param->length  ;

    /* call our speedy memcpy */
    memcpy( pdst, psrc, len ) ;
    return(OK) ;
}

/************************************************************************
 *
 *                          FLASH_PARELLEL_program_flash
 *  Description :
 *  -------------
 *  Programs a EN29LV160 flash device.
 *  
 *
 *  Parameters :
 *  ------------
 *
 *  'p_param',   IN,    variable of type, t_FLASH_write_descriptor.
 *
 *
 *  Return values :
 *  ---------------
 *
 * 'OK' = 0x00:                         FLASH programmed succesfully
 *  ERROR_FLASH_PROGRAM_ERROR           Flash device failure
 *
 *
 ************************************************************************/
static
INT32 FLASH_PARELLEL_program_flash( t_FLASH_write_descriptor *p_param )
{
    int             rcode = OK ;
    volatile UINT8  dw ;    
    int             i;
    UINT8          *psrc ;
    UINT32          pdst ;
    UINT32          size = 0;
    UINT32          base_addr = 0;
    UINT32          pdst_base_addr;
    UINT32          old_ie, len;
    
  

    /* convert addresses to kseg1 */
    psrc = p_param->buffer ;
    pdst = p_param->adr ;
    len  = p_param->length ;

    
    /* check any special case */
    if (len <= 0)
    {
        return(OK) ;
    }
    
    /* Disable interrupts */
    old_ie = sys_disable_int();
    
    while(len --)
    {  
       dw = REG8(psrc);
     
    /* issue 1st write cycle */
       REG8(KSEG1((monitorflash_phys_start&0xffff0000)+0xf556))  = 0xaa;
    
    /* issue 2nd write cycle */
       REG8(KSEG1((monitorflash_phys_start&0xffff0000)+0xfaa9))  = 0x55;
    
    /* issue 3rd write cycle */
       REG8(KSEG1((monitorflash_phys_start&0xffff0000)+0xf556))  = 0xa0;
    
    /* issue 4th write cycle */
       REG8(KSEG1(pdst)) = dw ;
       
       
       
       i = 100000;          
       while ( REG8(KSEG1(pdst)) != dw ) 
        if ( --i == 0)
            break;   
    /* Verify programmed byte */
       
  
       if ( REG8(KSEG1(pdst)) != dw )
       {    
           
            printf("Data check read: (0x%08x)=0x%x, Data written: 0x%x", KSEG1(pdst), REG8(KSEG1(pdst)), dw) ; 
            return(ERROR_FLASH_VERIFY_ERROR) ;
       }
            
       if ( (len%1024) == 1023)
       {

            printf(".");//cy test     
       }      
       pdst ++ ;
       psrc ++ ;
       
       
     
    }
   
    /* Restore interrupt enable status */
    if(old_ie)
        sys_enable_int();

    return( rcode ) ;
}



/************************************************************************
 *
 *                          FLASH_PARELLEL_erase_fileflash
 *  Description :
 *  -------------
 *  Erase file flash, which is the last block (in each FLASH device)
 *  of the monitor flash.
 *  
 *
 *  Parameters :
 *  ------------
 *
 *  -
 *
 *  Return values :
 *  ---------------
 *
 * 'OK' = 0x00:                         File FLASH erased succesfully
 *  ERROR_FLASH_PROGRAM_ERROR           Flash device failure
 *
 *
 ************************************************************************/
static
INT32 FLASH_PARELLEL_erase_fileflash( void )
{
    int             rcode = OK ,i;
  
    
     
  
    /* issue 1st erase cycle */
    REG8(KSEG1((fileflash_phys_start&0xffff0000)+0xf556))  = 0xaa;
    
    /* issue 2nd erase cycle */
    REG8(KSEG1((fileflash_phys_start&0xffff0000)+0xfaa9))  = 0x55;
    
    /* issue 3rd erase cycle */
    REG8(KSEG1((fileflash_phys_start&0xffff0000)+0xf556))  = 0x80;
    
    /* issue 4th erase cycle */
    REG8(KSEG1((fileflash_phys_start&0xffff0000)+0xf556))  = 0xaa;
    
    /* issue 5th erase cycle */
    REG8(KSEG1((fileflash_phys_start&0xffff0000)+0xfaa9))  = 0x55;
    
    /* issue 6th erase cycle */
    REG8(KSEG1(fileflash_phys_start))                     = 0x30;
    
    for (i=0 ; i<20 ; i++);   //cy test
          
         
    while(REG8(KSEG1(fileflash_phys_start))!=0xFF);
         

    return( rcode ) ;
}

/************************************************************************
 *
 *                          FLASH_PARELLEL_erase_flasharea
 *  Description :
 *  -------------
 *  Erase flash area; i.e. the driver erases the flash blocks inside 
 *  the specified memory area.
 *  
 *
 *  Parameters :
 *  ------------
 * 
 *  'p_param',   IN,    variable of type, t_FLASH_ctrl_descriptor.
 *
 *
 *  Return values :
 *  ---------------
 *
 * 'OK' = 0x00:                         FLASH erased succesfully
 *  ERROR_FLASH_PROGRAM_ERROR           Flash device failure
 *  ERROR_FLASH_INVALID_ADDRESS         Address area not inside FLASH
 *                                      devices
 *
 *
 ************************************************************************/
static
INT32 FLASH_PARELLEL_erase_flasharea( t_FLASH_ctrl_descriptor *p_param )
{
    int rcode = OK ;
    int index, i, j;
    UINT32 sector_address_begin, sector_address_end;
    bool   monitor_flash = FALSE ;
   
   
    /* erase FLASH area */
    index = get_sector_index(devicetype, KSEG1(p_param->user_physadr));
  
    sector_address_begin = devicetype.flash_sectors[index];
                         
    sector_address_end = devicetype.flash_sectors[get_sector_index
                           (devicetype, KSEG1(p_param->user_physadr +
                                  p_param->user_length - 1))] ;
                      
    for (i = sector_address_begin; i>=sector_address_end && index >=0; i=devicetype.flash_sectors[--index])
    { 
      
     
      /* issue 1st erase cycle */
      REG8(KSEG1((monitorflash_phys_start&0xffff0000)+0xf556))  = 0xaa;
    
      /* issue 2nd erase cycle */
      REG8(KSEG1((monitorflash_phys_start&0xffff0000)+0xfaa9))  = 0x55;
    
      /* issue 3rd erase cycle */
      REG8(KSEG1((monitorflash_phys_start&0xffff0000)+0xf556))  = 0x80;
    
      /* issue 4th erase cycle */
      REG8(KSEG1((monitorflash_phys_start&0xffff0000)+0xf556))  = 0xaa;
    
      /* issue 5th erase cycle */
      REG8(KSEG1((monitorflash_phys_start&0xffff0000)+0xfaa9))  = 0x55;
    
      /* issue 6th erase cycle */
      REG8(KSEG1(FLASH_ADDR_BASE-i))                 = 0x30;

     
      for (j=0 ; j<20 ; j++);   
                  
      while(REG8(KSEG1(FLASH_ADDR_BASE-i))!=0xFF);
      
      printf("to 0x%x erased\n",FLASH_ADDR_BASE-i);//cy test
      
    }  
         

    return( rcode ) ;
}

/************************************************************************
 *
 *                          FLASH_PARELLEL_erase_chip
 *  Description :
 *  -------------
 *  Erase whole chip; 
 *  
 *
 *  Parameters :
 *  ------------
 * 
 *  'p_param',   IN,    variable of type, t_FLASH_ctrl_descriptor.
 *
 *
 *  Return values :
 *  ---------------
 *
 * 'OK' = 0x00:                         FLASH erased succesfully
 *  ERROR_FLASH_PROGRAM_ERROR           Flash device failure
 *  ERROR_FLASH_INVALID_ADDRESS         Address area not inside FLASH
 *                                      devices
 *
 *
 ************************************************************************/
static
INT32 FLASH_PARELLEL_erase_chip( t_FLASH_ctrl_descriptor *p_param )
{
    int rcode = OK ;
    int  i;
    
    /* issue 1st erase cycle */
    REG8(KSEG1((monitorflash_phys_start&0xffff0000)+0xf556))  = 0xaa;
    
    /* issue 2nd erase cycle */
    REG8(KSEG1((monitorflash_phys_start&0xffff0000)+0xfaa9))  = 0x55;
    
    /* issue 3rd erase cycle */
    REG8(KSEG1((monitorflash_phys_start&0xffff0000)+0xf556))  = 0x80;
    
    /* issue 4th erase cycle */
    REG8(KSEG1((monitorflash_phys_start&0xffff0000)+0xf556))  = 0xaa;
    
    /* issue 5th erase cycle */
    REG8(KSEG1((monitorflash_phys_start&0xffff0000)+0xfaa9))  = 0x55;
    
    /* issue 6th erase cycle */
    REG8(KSEG1((monitorflash_phys_start&0xffff0000)+0xf556))  = 0x10;
             
    while(REG8(KSEG1(0xbfc00000))!=0xFF);
     
    return( rcode ) ;
}

/************************************************************************
 *
 *                          FLASH_PARELLEL_inquire_deviceid
 *  Description :
 *  -------------
 *  Inquire flash device id (Top Boot or Bottom Boot);
 *  
 *
 *  Parameters :
 *  ------------
 * 
 
 *
 *
 *  Return values :
 *  ---------------
 *
 * 'OK' = 0x00:                         
 *  ERROR_FLASH_DEVICE_ID               
 *
 *
 ************************************************************************/
static
INT32 FLASH_PARELLEL_inquire_deviceid(void)
{
    int rcode = OK ;
    
     
    return( rcode ) ;

   
}







/************************************************************************
 *
 *                          FLASH_error_lookup
 *  Description :
 *  -------------
 *  Lookup error code to error string(s)
 *  
 *
 *  Parameters :
 *  ------------
 * 
 *  'p_param',   INOUT,    variable of type, t_sys_error_string.
 *
 *
 *  Return values :
 *  ---------------
 *
 * 'OK' = 0x00:  
 *
 *
 ************************************************************************/
static
INT32 FLASH_error_lookup( t_sys_error_string *p_param )
{
    UINT32 t, i ;

    i = 0 ;
    p_param->count = 0 ;
    t = SYSERROR_ID( p_param->syserror ) ;
    if ( flash_last_error == p_param->syserror )
    {
        /* check for recognized error code */
        if (t < sizeof(flash_error_string)/sizeof(char*) )
        {
            /* fill in mandatory error message string */
            p_param->strings[SYSCON_ERRORMSG_IDX] = flash_error_string[t] ;
            i++ ;

            /* check for diagnose message */
            if ( flash_diag_msg[0] != 0 )
            {
                /* fill in optional diagnose message string */
                p_param->strings[SYSCON_DIAGMSG_IDX] = flash_diag_msg ;
                i++ ;
            }

            /* check for hint message */
            if ( flash_error_hint_string[t] != NULL)
            {
                /* fill in optional hint message string */
                p_param->strings[SYSCON_HINTMSG_IDX] = flash_error_hint_string[t] ;
                i++ ;
            }
        }
        p_param->count      = i ;
    }
 
    /* delete context */
    flash_last_error  = OK ;
    return(OK) ;
}



/************************************************************************
 *
 *                          FLASH_PARELLEL_is_file_flash_write_protected
 *  Description :
 *  -------------
 *  Check if file flash id write protected
 *  
 *
 *  Parameters :
 *  ------------
 *
 *
 *
 *  Return values :
 *  ---------------
 *
 *  FALSE: not write protected
 *  TRUE:  write protected
 *
 *
 ************************************************************************/
static
bool FLASH_PARELLEL_is_file_flash_write_protected( void )
{
    UINT32 wr_protect ;

    /* check in SYSCON */
    SYSCON_read( SYSCON_BOARD_FILEFLASH_WRITE_PROTECTED_ID,
                 &wr_protect,
                 sizeof(wr_protect) ) ;
    if (wr_protect)
    {
        return( TRUE ) ;
    }
    else
    {
        return( FALSE ) ;
    }
}


/************************************************************************
 *
 *                          FLASH_PARELLEL_is_monitor_flash_write_protected
 *  Description :
 *  -------------
 *  Check if file flash id write protected
 *  
 *
 *  Parameters :
 *  ------------
 *
 *
 *
 *  Return values :
 *  ---------------
 *
 *  FALSE: not write protected
 *  TRUE:  write protected
 *
 *
 ************************************************************************/
static
bool FLASH_PARELLEL_is_monitor_flash_write_protected( void )
{
    UINT32 wr_protect ;

    /* check in SYSCON */
    SYSCON_read( SYSCON_BOARD_MONITORFLASH_WRITE_PROTECTED_ID,
                 &wr_protect,
                 sizeof(wr_protect) ) ;
    if (wr_protect)
    {
        return( TRUE ) ;
    }
    else
    {
        return( FALSE ) ;
    }
}
/************************************************************************
 *
 *                          get_sectorindex
 *  Description :
 *  -------------
 *  get the sector index of flash 
 *  
 *
 *  Parameters :
 *  ------------
 *
 *
 *
 *  Return values :
 *  ---------------
 *
 *  
 *
 *
 ************************************************************************/

static unsigned int get_sector_index(p_device_type dev_type, UINT32 p_addr)
{
	int i;
	int addr = p_addr;
	

	addr = FLASH_ADDR_BASE-addr;
        
	if ( addr > 0xffffff ) 
		return (unsigned int)-1;
		
	for( i=dev_type.total_sector-1; i>=0; i--)
	{      
		if(addr >= dev_type.flash_sectors[i])
			break;
	}
	
	return (unsigned int) i;
}

