
/************************************************************************
 *
 *      ide.c
 *
 *      The 'IDE' module implements the IDE driver
 *      interface to be used via 'IO' device driver services:
 *
 *        1) init  device:  configure and initialize IDE driver
 *        2) open  device:  not used
 *        3) close device:  not used
 *        4) read  device:  not used
 *        5) write device:  not used
 *        6) ctrl  device:  a) READ  sector
 *                          b) WRITE sector
 *			    c) READ info
 *
 * ######################################################################
 *
 * mips_start_of_legal_notice
 * 
 * Copyright (c) 2003 MIPS Technologies, Inc. All rights reserved.
 *
 *
 * Unpublished rights (if any) reserved under the copyright laws of the
 * United States of America and other countries.
 *
 * This code is proprietary to MIPS Technologies, Inc. ("MIPS
 * Technologies"). Any copying, reproducing, modifying or use of this code
 * (in whole or in part) that is not expressly permitted in writing by MIPS
 * Technologies or an authorized third party is strictly prohibited. At a
 * minimum, this code is protected under unfair competition and copyright
 * laws. Violations thereof may result in criminal penalties and fines.
 *
 * MIPS Technologies reserves the right to change this code to improve
 * function, design or otherwise. MIPS Technologies does not assume any
 * liability arising out of the application or use of this code, or of any
 * error or omission in such code. Any warranties, whether express,
 * statutory, implied or otherwise, including but not limited to the implied
 * warranties of merchantability or fitness for a particular purpose, are
 * excluded. Except as expressly provided in any written license agreement
 * from MIPS Technologies or an authorized third party, the furnishing of
 * this code does not give recipient any license to any intellectual
 * property rights, including any patent rights, that cover this code.
 *
 * This code shall not be exported or transferred for the purpose of
 * reexporting in violation of any U.S. or non-U.S. regulation, treaty,
 * Executive Order, law, statute, amendment or supplement thereto.
 *
 * This code constitutes one or more of the following: commercial computer
 * software, commercial computer software documentation or other commercial
 * items. If the user of this code, or any related documentation of any
 * kind, including related technical data or manuals, is an agency,
 * department, or other entity of the United States government
 * ("Government"), the use, duplication, reproduction, release,
 * modification, disclosure, or transfer of this code, or any related
 * documentation of any kind, is restricted in accordance with Federal
 * Acquisition Regulation 12.212 for civilian agencies and Defense Federal
 * Acquisition Regulation Supplement 227.7202 for military agencies. The use
 * of this code by the Government is further restricted in accordance with
 * the terms of the license agreement(s) and/or applicable contract terms
 * and conditions covering this code from MIPS Technologies or an authorized
 * third party.
 *
 * 
 * mips_end_of_legal_notice
 * 
 *
 ************************************************************************/


/************************************************************************
 *      Include files
 ************************************************************************/

#include <sysdefs.h>
#include <syserror.h>
#include <sysdev.h>

#include <sys_api.h>
#include <io_api.h>
#include <ide_api.h>
#include <stdio.h>
#include <string.h>


#define TIMER      0xfffff
/************************************************************************
 *      Definitions
 ************************************************************************/
 
#define MIS_GP1DATO_REG           0x1801b10c
#define IDE_PRIMARY_IO_CMD_BASE   0x18012000
#define IDE_SECONDARY_IO_CMD_BASE 0x18012100

#define IDE_DATA_OFS		0
#define IDE_ERROR_OFS		1*4
#define IDE_SECTOR_COUNT_OFS	2*4
#define IDE_SECTOR_NUMBER_OFS	3*4
#define IDE_CYL_LOW_OFS		4*4
#define IDE_CYL_HI_OFS		5*4
#define IDE_DRIVE_HEAD_OFS	6*4
#define IDE_COMMAND_OFS		7*4
#define IDE_STATUS_OFS		7*4
#define IDE_DEV_CTRL_OFS        8*4
#define IDE_ALT_STATUS_OFS      8*4

#define ATA_DEV0_OFS            9*4
#define ATA_DEV1_OFS           10*4
#define ATA_PIO0_OFS           11*4
#define ATA_PIO1_OFS           12*4
#define ATA_RST_OFS            23*4  

/* IDE ERROR Register field */
#define IDE_ERROR_BBK_SHF	7
#define IDE_ERROR_BBK_MSK	(MSK(1) << IDE_ERROR_BBK_SHF)
#define IDE_ERROR_BBK_BIT	IDE_ERROR_BBK_MSK

#define IDE_ERROR_UNC_SHF	6
#define IDE_ERROR_UNC_MSK	(MSK(1) << IDE_ERROR_UNC_SHF)
#define IDE_ERROR_UNC_BIT	IDE_ERROR_UNC_MSK

#define IDE_ERROR_MC_SHF	5
#define IDE_ERROR_MC_MSK	(MSK(1) << IDE_ERROR_MC_SHF)
#define IDE_ERROR_MC_BIT	IDE_ERROR_MC_MSK

#define IDE_ERROR_IDNF_SHF	4
#define IDE_ERROR_IDNF_MSK	(MSK(1) << IDE_ERROR_IDNF_SHF)
#define IDE_ERROR_IDNF_BIT	IDE_ERROR_IDNF_MSK

#define IDE_ERROR_MCR_SHF	3
#define IDE_ERROR_MCR_MSK	(MSK(1) << IDE_ERROR_MCR_SHF)
#define IDE_ERROR_MCR_BIT	IDE_ERROR_MCR_MSK

#define IDE_ERROR_ABRT_SHF	2
#define IDE_ERROR_ABRT_MSK	(MSK(1) << IDE_ERROR_ABRT_SHF)
#define IDE_ERROR_ABRT_BIT	IDE_ERROR_ABRT_MSK

#define IDE_ERROR_TK0NF_SHF	1
#define IDE_ERROR_TK0NF_MSK	(MSK(1) << IDE_ERROR_TK0NF_SHF)
#define IDE_ERROR_TK0NF_BIT	IDE_ERROR_TK0NF_MSK

#define IDE_ERROR_AMNF_SHF	0
#define IDE_ERROR_AMNF_MSK	(MSK(1) << IDE_ERROR_AMNF_SHF)
#define IDE_ERROR_AMNF_BIT	IDE_ERROR_AMNF_MSK

/* IDE DEVICE/HEAD Register field */
#define IDE_DRIVE_HEAD_DEV_SHF	4 
#define IDE_DRIVE_HEAD_DEV_MSK  (MSK(1) << IDE_DRIVE_HEAD_DEV_SHF)
#define IDE_DRIVE_HEAD_DEV_BIT  IDE_DRIVE_HEAD_DEV_MSK

/* IDE STATUS Register field */
#define IDE_STATUS_ERR_SHF	0
#define IDE_STATUS_ERR_MSK	(MSK(1) << IDE_STATUS_ERR_SHF)
#define IDE_STATUS_ERR_BIT	IDE_STATUS_ERR_MSK

#define IDE_STATUS_DRQ_SHF	3
#define IDE_STATUS_DRQ_MSK	(MSK(1) << IDE_STATUS_DRQ_SHF)
#define IDE_STATUS_DRQ_BIT	IDE_STATUS_DRQ_MSK

#define IDE_STATUS_DSC_SHF	4
#define IDE_STATUS_DSC_MSK	(MSK(1) << IDE_STATUS_DSC_SHF)
#define IDE_STATUS_DSC_BIT	IDE_STATUS_DSC_MSK

#define IDE_STATUS_DF_SHF	5
#define IDE_STATUS_DF_MSK	(MSK(1) << IDE_STATUS_DF_SHF)
#define IDE_STATUS_DF_BIT	IDE_STATUS_DF_MSK

#define IDE_STATUS_DRDY_SHF	6
#define IDE_STATUS_DRDY_MSK	(MSK(1) << IDE_STATUS_DRDY_SHF)
#define IDE_STATUS_DRDY_BIT	IDE_STATUS_DRDY_MSK

#define IDE_STATUS_BSY_SHF	7
#define IDE_STATUS_BSY_MSK	(MSK(1) << IDE_STATUS_BSY_SHF)
#define IDE_STATUS_BSY_BIT	IDE_STATUS_BSY_MSK

#define IDE_DEV_CTRL_SRST_SHF   2
#define IDE_DEV_CTRL_SRST_MSK	(MSK(1) << IDE_DEV_CTRL_SRST_SHF)
#define IDE_DEV_CTRL_SRST_BIT	IDE_DEV_CTRL_SRST_MSK

/*IDE SECTOR COUNT Register field */
#define IDE_SECTOR_COUNT_REL_SHF 2 
#define IDE_SECTOR_COUNT_REL_MSK (MSK(1) << IDE_SECTOR_COUNT_REL_SHF)
#define IDE_SECTOR_COUNT_REL_BIT IDE_SECTOR_COUNT_REL_MSK

#define IDE_SECTOR_COUNT_IO_SHF  2 
#define IDE_SECTOR_COUNT_IO_MSK  (MSK(1) << IDE_SECTOR_COUNT_IO_SHF)
#define IDE_SECTOR_COUNT_IO_BIT  IDE_SECTOR_COUNT_IO_MSK

#define IDE_SECTOR_COUNT_CD_SHF  2 
#define IDE_SECTOR_COUNT_CD_MSK  (MSK(1) << IDE_SECTOR_COUNT_CD_SHF)
#define IDE_SECTOR_COUNT_CD_BIT  IDE_SECTOR_COUNT_CD_MSK




#define IO_ADDR(ofs)		(ide_io_base + (ofs))
#define IO32(ofs)		REG32(KSEG1(IO_ADDR(ofs)))
#define IO16(ofs)		REG16(KSEG1(IO_ADDR(ofs)))
#define IO8(ofs)		REG8(KSEG1(IO_ADDR(ofs)))


#define IDE_DEV_MASTER			0
#define IDE_DEV_SLAVE			1

#define ATA_PKT_CMD             0xA0 
#define ATA_IDENTITY_PKT_CMD    0xA1     

#define ATAPI_CMD_PKT_LEN       12

#define GPCMD_READ_10           0x28
#define GPCMD_TEST_UNIT_READY	0x00
#define GPCMD_STARTSTOP_UNIT    0x1b




// T1R
#define VENUS_ATA_PIO_0_T1R                     (70)
#define VENUS_ATA_PIO_1_T1R                     (50)
#define VENUS_ATA_PIO_2_T1R                     (30)
#define VENUS_ATA_PIO_3_T1R                     (30)
#define VENUS_ATA_PIO_4_T1R                     (25)

// T2R
#define VENUS_ATA_PIO_0_T2R                     (290)
#define VENUS_ATA_PIO_1_T2R                     (290)
#define VENUS_ATA_PIO_2_T2R                     (290)
#define VENUS_ATA_PIO_3_T2R                     (80)
#define VENUS_ATA_PIO_4_T2R                     (70)

// T3R
#define VENUS_ATA_PIO_0_T3R                     (600- VENUS_ATA_PIO_0_T2R)
#define VENUS_ATA_PIO_1_T3R                     (383- VENUS_ATA_PIO_1_T2R)
#define VENUS_ATA_PIO_2_T3R                     (330- VENUS_ATA_PIO_2_T2R)
#define VENUS_ATA_PIO_3_T3R                     (180- VENUS_ATA_PIO_3_T2R)
#define VENUS_ATA_PIO_4_T3R                     (120- VENUS_ATA_PIO_4_T2R)

// T1D
#define VENUS_ATA_PIO_0_T1D                     (70)
#define VENUS_ATA_PIO_1_T1D                     (50)
#define VENUS_ATA_PIO_2_T1D                     (30)
#define VENUS_ATA_PIO_3_T1D                     (30)
#define VENUS_ATA_PIO_4_T1D                     (25)

// T2D
#define VENUS_ATA_PIO_0_T2D                     (165)
#define VENUS_ATA_PIO_1_T2D                     (125)
#define VENUS_ATA_PIO_2_T2D                     (100)
#define VENUS_ATA_PIO_3_T2D                     (80)
#define VENUS_ATA_PIO_4_T2D                     (70)

// T3D
#define VENUS_ATA_PIO_0_T3D                     ((unsigned int)(600- VENUS_ATA_PIO_0_T2D))
#define VENUS_ATA_PIO_1_T3D                     ((unsigned int)(383- VENUS_ATA_PIO_1_T2D))
#define VENUS_ATA_PIO_2_T3D                     ((unsigned int)(240- VENUS_ATA_PIO_2_T2D))
#define VENUS_ATA_PIO_3_T3D                     ((unsigned int)(180- VENUS_ATA_PIO_3_T2D))
#define VENUS_ATA_PIO_4_T3D                     ((unsigned int)(120- VENUS_ATA_PIO_4_T2D))



#define VENUS_CYCLE_TIME                        5       // 5 ns

// PIO mode
#define VENUS_REG_PIOISTER_TIME_SHIFT_T1R           (0)
#define VENUS_REG_PIOISTER_TIME_SHIFT_T2R           (5)
#define VENUS_REG_PIOISTER_TIME_SHIFT_T3R           (11)
#define VENUS_REG_PIOISTER_TIME_SHIFT_T1D           (16)
#define VENUS_REG_PIOISTER_TIME_SHIFT_T2D           (21)
#define VENUS_REG_PIOISTER_TIME_SHIFT_T3D           (27)

#define REG_PIO(mode, time) (((VENUS_ATA_PIO_ ## mode ## _ ## time)/ (VENUS_CYCLE_TIME)+ 1)<< (VENUS_REG_PIOISTER_TIME_SHIFT_ ## time))

unsigned int venus_pio_timing[]={
    (REG_PIO(0, T3D)| REG_PIO(0, T2D)| REG_PIO(0, T1D)| REG_PIO(0, T3R)| REG_PIO(0, T2R)| REG_PIO(0 , T1R)), // XFER_PIO_0
    (REG_PIO(1, T3D)| REG_PIO(1, T2D)| REG_PIO(1, T1D)| REG_PIO(1, T3R)| REG_PIO(1, T2R)| REG_PIO(1 , T1R)), // XFER_PIO_1
    (REG_PIO(2, T3D)| REG_PIO(2, T2D)| REG_PIO(2, T1D)| REG_PIO(2, T3R)| REG_PIO(2, T2R)| REG_PIO(2 , T1R)), // XFER_PIO_2
    (REG_PIO(3, T3D)| REG_PIO(3, T2D)| REG_PIO(3, T1D)| REG_PIO(3, T3R)| REG_PIO(3, T2R)| REG_PIO(3 , T1R)), // XFER_PIO_3
    (REG_PIO(4, T3D)| REG_PIO(4, T2D)| REG_PIO(4, T1D)| REG_PIO(4, T3R)| REG_PIO(4, T2R)| REG_PIO(4 , T1R))  // XFER_PIO_4
};
/************************************************************************
 *  Macro Definitions
*************************************************************************/

/************************************************************************
 *      Public variables
 ************************************************************************/
UINT16 HDD_CYL_COUNT;
UINT16 HDD_HEAD_COUNT;
UINT16 HDD_SEC_PER_TRACK;
UINT32 ide_minor_hdd =0xff;
UINT32 ide_minor_cdrom;

/************************************************************************
 *      Static variables
 ************************************************************************/

static char* ide_error_string[] = 
{
    /* ERROR_IDE_INVALID_COMMAND */  
    "Internal ERROR: Invalid control command",

    /* ERROR_IDE_UNKNOWN_DEVICE */  
    "Internal ERROR: Illegal minor number",

    /* ERROR_IDE_NULL_BUFFER */	
    "Internal ERROR: NULL buffer",

    /* ERROR_IDE_UNAVAILABLE_DEVICE */
    "Device unavailable",

    /* ERROR_IDE_BAD_BLOCK */
    "Bad block detected",

    /* ERROR_IDE_UNCORRECTABLE_DATA */
    "Uncorrectable data error",

    /* ERROR_IDE_MEDIA_CHANGE */
    "Media change",

    /* ERROR_IDE_ID_NOT_FOUND */
    "ID not found (damaged or non-existent sector)",

    /* ERROR_IDE_MEDIA_CHANGE_REQUESTED */
    "Media change requested",

    /* ERROR_IDE_ABORTED */
    "Aborted command (illegal command or disk drive error)",

    /* ERROR_IDE_TRACK_0_NOT_FOUND */
    "Track 0 not found",

    /* ERROR_IDE_ADDRESS_MARK_NOT_FOUND	*/
    "Address mark not found",

    /* ERROR_IDE_UNKNOWN */
    "Unknown IDE error"
};


static UINT32 sector;	      /* Last sector read (or failed)		*/
static bool   sector_valid;   /* 'sector' valid for error handling	*/
static char   diag_msg[40];   /* Diagnostics error message		*/
UINT32 ide_io_base;


/************************************************************************
 *      Static function prototypes
 ************************************************************************/
static INT32 
ide_ctrl(
    UINT32                major,     /* IN: major device number		*/
    UINT32		  minor,     /* IN: minor device number		*/
    t_ide_ctrl_descriptor *p_param ) /* IN: write data			*/
;

static UINT32
get_error( void );

/************************************************************************
 *      Implementation : Static functions
 ************************************************************************/


/************************************************************************
 *
 *                          ide_init
 *  Description :
 *  -------------
 *
 *  This service initializes the IDE driver.
 *
 *  Parameters :
 *  ------------
 *
 *  'major',     IN,    major device number
 *  'minor',     IN,    not used
 *  'p_param',   INOUT, not used
 *
 *
 *  Return values :
 *  ---------------
 *
 *  'OK'(=0)
 *
 ************************************************************************/
static INT32 
ide_init(
    UINT32 major,          /* IN: major device number             */
    UINT32 minor,          /* IN: minor device number             */
    bool  ide_init_first   
     )
{   
    unsigned int id0, id1, id2, id3, id4;
    int i; 
  
    UINT8 dev;
    t_ide_ctrl_descriptor  ide_ctl;
    UINT32 rc;   
    UINT32 timer;
    
    

    
    switch( minor )
    {
      case IDE_MINOR_PRIMARY_MASTER :
        ide_io_base = IDE_PRIMARY_IO_CMD_BASE;
        dev	    = IDE_DEV_MASTER;

	break;
      case IDE_MINOR_PRIMARY_SLAVE :
        ide_io_base = IDE_PRIMARY_IO_CMD_BASE;
        dev	    = IDE_DEV_SLAVE;

	break;
      case IDE_MINOR_SECONDARY_MASTER :
        ide_io_base = IDE_SECONDARY_IO_CMD_BASE;
        dev	    = IDE_DEV_MASTER;
	break;
      case IDE_MINOR_SECONDARY_SLAVE :
        ide_io_base = IDE_SECONDARY_IO_CMD_BASE;
        dev	    = IDE_DEV_SLAVE;
	break;
      default :
        return ERROR_IDE_UNKNOWN_DEVICE;
    }
#if 1
    
    if (ide_init_first == 1)  
    if (minor == IDE_MINOR_PRIMARY_MASTER || minor == IDE_MINOR_SECONDARY_MASTER)
    {
      
     //IDE device reset	
      IO32(ATA_RST_OFS)  =  0x0 ;	
      sys_sync();
      sys_wait_ms(1);
      // disable IDE device reset
      IO32(ATA_RST_OFS)  =  0x1 ;
      // set ATA0/DEV0 timing
      // set ATA0/DEV1 timing
      IO32(ATA_DEV0_OFS) = (1<<16) | (3<<19) | (3<<23) | (6<<26) ;
      IO32(ATA_DEV1_OFS) = (1<<16) | (3<<19) | (3<<23) | (6<<26) ;
    // set ATA0/PIO0 timing
      IO32(ATA_PIO0_OFS) = venus_pio_timing[0];//(2<<27)|(6<<21)|(3<<16)|(2<<11)|(10<<5)|3 ;
      IO32(ATA_PIO1_OFS) = venus_pio_timing[0];//(2<<27)|(6<<21)|(3<<16)|(2<<11)|(10<<5)|3 ;
    
    }  
#endif    
    
   
    IO32(IDE_DRIVE_HEAD_OFS)=(dev << IDE_DRIVE_HEAD_DEV_SHF)|0xE0 ;
   
    
    timer = 0; 
    
   
    while( IO32(IDE_STATUS_OFS) & IDE_STATUS_BSY_MSK )
        if (timer++ >TIMER/4) //
        {  
           return (ERROR_IDE_UNKNOWN);
        }
     
   
    IO32(IDE_DEV_CTRL_OFS) = IDE_DEV_CTRL_SRST_BIT;
   
    
    /* delay for a while */ 
    timer=0x800000;   //for sata timing                      
    while(timer--);
    
    
    
    IO32(IDE_DEV_CTRL_OFS) = 0 ;
    
   
    /* delay for a while */ 
     timer=TIMER/256;                     
    while(timer--);
    
   
   
    timer = 0; 
    while( IO32(IDE_STATUS_OFS) & (IDE_STATUS_BSY_MSK | IDE_STATUS_DRQ_MSK))
    {   
        if (timer++ >5*TIMER)
        {  
           return (ERROR_IDE_UNKNOWN);		
	}
    }		
    

    /* After software reset, the signature is stored in
       the Command Block registers. Refer to p.210 in T13/1153D
     */ 
  
    
    IO32(IDE_DRIVE_HEAD_OFS)=(dev << IDE_DRIVE_HEAD_DEV_SHF)|0xE0 ;
    
    id0=IO32(IDE_CYL_LOW_OFS);
    id1=IO32(IDE_CYL_HI_OFS);
    id2=IO32(IDE_SECTOR_COUNT_OFS);
    id3=IO32(IDE_SECTOR_NUMBER_OFS);
	id4=IO32(IDE_DRIVE_HEAD_OFS);
		
    
 
    if ((IO32(IDE_SECTOR_COUNT_OFS)!= 0x1) || (IO32(IDE_SECTOR_NUMBER_OFS)!= 0x1))
    {  
    	
    	printf("ide%d signature error :\n", minor);
    	printf("0x%x 0x%x 0x%x 0x%x 0x%x 0x%x 0x%x\n",IO32(IDE_SECTOR_COUNT_OFS)
                                      ,IO32(IDE_SECTOR_NUMBER_OFS)  
                                      , id0, id1, id2, id3, id4);	
        return OK;
    }    
		
    
		
    if ((IO32(IDE_CYL_LOW_OFS)==0x14)&&(IO32(IDE_CYL_HI_OFS)==0xEB)) 
    {   
    	
    	ide_ctl.command         = IDE_TEST_UNIT_READY;	  
    	rc = ide_ctrl( SYS_MAJOR_IDE, minor, &ide_ctl );
    	 	
        if(rc != OK)	  
        {
           
           return rc;
        }   
        ide_minor_cdrom = minor ;
           
    	printf("\nide %x with packet command feature set\n", minor);
    	
    	
    }	
    else if ((IO32(IDE_CYL_LOW_OFS)==0x00)&&(IO32(IDE_CYL_HI_OFS)==0x00)) 
    {   
    	
    	
        ide_ctl.command         = IDE_CMD_IDENTIFY;	
        
        rc = ide_ctrl( SYS_MAJOR_IDE, minor, &ide_ctl ); 
     
        if(rc != OK)
        {   
           return rc;
        }
       
        HDD_CYL_COUNT     = ide_ctl.u.identify.cyl_count;
        HDD_HEAD_COUNT    = ide_ctl.u.identify.head_count;
        HDD_SEC_PER_TRACK = ide_ctl.u.identify.sectors_per_track;

        ide_minor_hdd = minor ;
        printf("ide %x w/o packet command feature set\n", minor);
    }    

    
    IO32(IDE_DEV_CTRL_OFS)= 2;      /*pending */
    /*intEn=nIEN: disable interrupt */
   
   
  
   
    return OK;
}


/************************************************************************
 *
 *                          ide_ctrl
 *  Description :
 *  -------------
 *  This service comprises following specific IDE services:
 *
 *    1) IDE_CMD_READ
 *    2) IDE_CMD_WRITE
 *    3) IDE_CMD_IDENTIFY
 *
 *  Parameters :
 *  ------------
 *
 *  'major',     IN,    major device number
 *  'minor',     IN,    minor device number for multi device drivers
 *  'p_param',   IN,    variable of type, t_ide_ctrl_descriptor.
 *
 *  Return values :
 *  ---------------
 *
 *  OK (0x00):                   IDE service completed successfully
 *  Other :			 See error codes in ide_api.h
 *
 ************************************************************************/
static INT32 
ide_ctrl(
    UINT32                major,     /* IN: major device number		*/
    UINT32		  minor,     /* IN: minor device number		*/
    t_ide_ctrl_descriptor *p_param ) /* IN: write data			*/
{
    UINT32          i,t,len;
    UINT32	    sector_count;
    UINT8	    dev;
    UINT16	    buf_id[IDE_BYTES_PER_SECTOR/2]; /* 16 bit aligned  */

    t_ide_identify  *identify_struct;
    UINT8	    *buffer;
    UINT8	    status;
    UINT32	    data32;
    UINT8           cmd[12];
    UINT32          timer=0;

    /* Sector number is not valid for error handling any more */
    
    
    sector_valid = FALSE;

    switch( minor )
    {
      case IDE_MINOR_PRIMARY_MASTER :
        ide_io_base = IDE_PRIMARY_IO_CMD_BASE;
        dev	    = IDE_DEV_MASTER;
	break;
      case IDE_MINOR_PRIMARY_SLAVE :
        ide_io_base = IDE_PRIMARY_IO_CMD_BASE;
        dev	    = IDE_DEV_SLAVE;
	break;
      case IDE_MINOR_SECONDARY_MASTER :
        ide_io_base = IDE_SECONDARY_IO_CMD_BASE;
        dev	    = IDE_DEV_MASTER;
	break;
      case IDE_MINOR_SECONDARY_SLAVE :
        ide_io_base = IDE_SECONDARY_IO_CMD_BASE;
        dev	    = IDE_DEV_SLAVE;
	break;
      default :
        return ERROR_IDE_UNKNOWN_DEVICE;
    }

    switch( p_param->command )
    {
      case IDE_PACKET_CMD_READ  : 
      case IDE_CMD_READ  : 
      case IDE_CMD_WRITE :       
       
        sector          = p_param->u.sector.sector;
	sector_count    = p_param->u.sector.count;
	buffer          = p_param->u.sector.buffer;
	if( buffer == NULL ) 
	    return ERROR_IDE_NULL_BUFFER;
	break;
      case IDE_CMD_IDENTIFY : 
	sector_count    = 1;
        buffer          = (UINT8 *)buf_id;
	break;	
	
      case IDE_CMD_INIT_DEV_PARAMETERS :
        break;
      	
      case IDE_TEST_UNIT_READY :
        break;
      	
      default : 
	return ERROR_IDE_INVALID_COMMAND;
    }

    /**** Determine if device is available ****/
  
    /* Select device */
    IO32(IDE_DRIVE_HEAD_OFS) = (dev << IDE_DRIVE_HEAD_DEV_SHF)|0xE0;

   
  

    /* Now perform the requested operation */
    switch (p_param->command)
    {
      case IDE_PACKET_CMD_READ :	
    
      {   
    	 /* Wait until BSY is cleared (may take up to 30 seconds, depending
          * on configuration).
         */
        timer = 0; 
        while( IO32(IDE_ALT_STATUS_OFS) & IDE_STATUS_BSY_MSK )
           if (timer++ >TIMER)
           {  
               return (ERROR_IDE_UNKNOWN);
     	   } 
        
	/* Issue command */
        IO32(IDE_COMMAND_OFS) = ATA_PKT_CMD;
        memset(cmd, 0, sizeof(cmd));
        cmd[0] = GPCMD_READ_10;
        /*
         * fill in lba
         */
        cmd[2] = (sector >> 24) & 0xff;
        cmd[3] = (sector >> 16) & 0xff;
        cmd[4] = (sector >>  8) & 0xff;
        cmd[5] = sector & 0xff;
 
        /*
         * and transfer length
         */
        cmd[7] = (sector_count >> 8) & 0xff;
        cmd[8] = sector_count & 0xff;
         
        timer = 0; 
        while(IO32(IDE_ALT_STATUS_OFS) & (IDE_STATUS_BSY_MSK |
                                         IDE_STATUS_DRQ_MSK) !=
              IDE_STATUS_DRQ_BIT)
              if (timer++ >100000)
              {
               return (ERROR_IDE_UNKNOWN);
              } 
        timer = 0;     
        while(IO32(IDE_SECTOR_COUNT_OFS) & (IDE_SECTOR_COUNT_REL_MSK |
                                           IDE_SECTOR_COUNT_IO_MSK  |
                                           IDE_SECTOR_COUNT_CD_MSK  ) !=
              IDE_SECTOR_COUNT_CD_BIT)
              if (timer++ >100000)
              {
               return (ERROR_IDE_UNKNOWN);        
              }             
        for (i = 0; i< ATAPI_CMD_PKT_LEN; i+=2)   
        {    
             IO32(IDE_DATA_OFS) =  *(UINT16*)(cmd+i);                
        }     
        
        timer = 0;
        while(IO32(IDE_ALT_STATUS_OFS) & (IDE_STATUS_BSY_MSK))
              if (timer++ >100000)
              {
               return (ERROR_IDE_UNKNOWN);
              }
         /* Check if read caused error */
        if( IO32(IDE_ALT_STATUS_OFS) & IDE_STATUS_ERR_BIT )  /*reserved cy test */
        {  
	    return get_error();      
        }
        if (IO32(IDE_ALT_STATUS_OFS) & (IDE_STATUS_DRQ_MSK) !=
              IDE_STATUS_DRQ_BIT)
        {     
            return (ERROR_IDE_UNKNOWN);   /*reserved cy test */
        }
        if (IO32(IDE_SECTOR_COUNT_OFS) & (IDE_SECTOR_COUNT_IO_MSK  |
                                         IDE_SECTOR_COUNT_CD_MSK  )!=
              IDE_SECTOR_COUNT_IO_BIT)
        {          
            return (ERROR_IDE_UNKNOWN);   /*reserved cy test */
        }        
        
         
	/* get byte count(Cylinder High & Cylinder Low) used for PIO only
           len is word count */
           
	len=((IO32(IDE_CYL_HI_OFS)&0xff)<<8)  +
	    (IO32(IDE_CYL_LOW_OFS)&0xff);	
	
	

	
	for (i=0;i<len;i++) 
	{   
	    data32=IO32(IDE_DATA_OFS);
	  
	   /* Byte oriented data */
	    buffer[0] = (data32 >> 0)  & 0xff;
	    buffer[1] = (data32 >> 8)  & 0xff; 
	    
	    buffer+= 2;
	}        
	    
       
       
	
      }
      break ;
    
    case IDE_TEST_UNIT_READY:
    
    
      {
    	
    	
    	 /* Wait until BSY is cleared (may take up to 30 seconds, depending
          * on configuration).
          */
        timer = 0;  
        while( IO32(IDE_ALT_STATUS_OFS) & IDE_STATUS_BSY_MSK )
            if (timer++ >100000)
            { 
               return (ERROR_IDE_UNKNOWN);
            }   
        
    	/* Issue command */
        IO32(IDE_COMMAND_OFS) = ATA_PKT_CMD;
        memset(cmd, 0, sizeof(cmd));
        cmd[0] = GPCMD_TEST_UNIT_READY;
       
        timer =0 ;
        while(IO32(IDE_ALT_STATUS_OFS) & (IDE_STATUS_BSY_MSK |
                                         IDE_STATUS_DRQ_MSK) !=
              IDE_STATUS_DRQ_BIT)
              if (timer++ >100000)
              {
               return (ERROR_IDE_UNKNOWN);   
              }    
        timer =0 ;                           
        while(IO32(IDE_SECTOR_COUNT_OFS) & (IDE_SECTOR_COUNT_REL_MSK |
                                           IDE_SECTOR_COUNT_IO_MSK  |
                                           IDE_SECTOR_COUNT_CD_MSK  ) !=
              IDE_SECTOR_COUNT_CD_BIT)
              if (timer++ >100000)
              {
               return (ERROR_IDE_UNKNOWN);        
              }           
        for (i = 0; i< ATAPI_CMD_PKT_LEN; i+=2)   
        {   
             IO32(IDE_DATA_OFS) =  *(UINT16*)(cmd+i);                
        }     
       
        timer = 0;
        
        while(IO32(IDE_ALT_STATUS_OFS) & (IDE_STATUS_BSY_MSK |
                                         IDE_STATUS_DRQ_MSK) !=
              IDE_STATUS_DRQ_BIT)
              if (timer++ >100000)
              {
               return (ERROR_IDE_UNKNOWN);   
              }    
      
        if (IO32(IDE_SECTOR_COUNT_OFS) & (IDE_SECTOR_COUNT_IO_MSK  |
                                         IDE_SECTOR_COUNT_CD_MSK  )!=
              (IDE_SECTOR_COUNT_IO_BIT|IDE_SECTOR_COUNT_CD_BIT ))
        {      
            return (ERROR_IDE_UNKNOWN);   /*reserved cy test */
        }    
       

      
    
      }
      break;
      
    case   IDE_CMD_READ:
    case   IDE_CMD_WRITE:
    case   IDE_CMD_IDENTIFY:
    
   
     {
     /* Wait until BSY is cleared (may take up to 30 seconds, depending
      * on configuration).
      */
   
      timer = 0;
     
      while( IO32(IDE_STATUS_OFS) & IDE_STATUS_BSY_MSK )
          if (timer++ >TIMER)
          {   
               return (ERROR_IDE_UNKNOWN);
          } 
          
#if 0      
     /* Devices present and functional should now have the DRDY bit set */
      if( ( IO32(IDE_STATUS_OFS) & (IDE_STATUS_DRDY_MSK | 
				 IDE_STATUS_DF_MSK) ) !=
	      IDE_STATUS_DRDY_BIT )
      {     
        return ERROR_IDE_UNAVAILABLE_DEVICE;
      }
#endif  
     /* Now perform the requested operation */
      while( sector_count-- )
      {
          if( (p_param->command != IDE_CMD_IDENTIFY) && 
	      (sector & 0xF0000000) )
	  {   
	    /* Bits 31:28 are not used for IDE */
            
            /* Sector number is now valid for error handling */
	      sector_valid = TRUE;
	      return ERROR_IDE_ID_NOT_FOUND;
	  }
        
          /* Wait until device is ready */
          timer = 0;
          while( !(IO32(IDE_STATUS_OFS) & IDE_STATUS_DRDY_BIT) )
              if (timer++ >5*TIMER)
              { 
                return (ERROR_IDE_UNKNOWN);
              } 
          if( p_param->command != IDE_CMD_IDENTIFY )
          {
            	
	    /* Setup sector (LBA mode) */
	    IO32(IDE_SECTOR_COUNT_OFS)  = 1;

	       			    
	    IO32(IDE_SECTOR_NUMBER_OFS) = (sector >> 0)  & 0xFF;
	    IO32(IDE_CYL_LOW_OFS)       = (sector >> 8)  & 0xFF;
	    IO32(IDE_CYL_HI_OFS)        = (sector >> 16) & 0xFF;
	    IO32(IDE_DRIVE_HEAD_OFS)    = ((sector >> 24) & 0x0F) | 
	       			     (dev << IDE_DRIVE_HEAD_DEV_SHF) | 
		  			 0xE0;
          }		  			 
        

          /* Issue command */
          IO32(IDE_COMMAND_OFS) = p_param->command;

          timer=0;
          while( IO32(IDE_STATUS_OFS) & IDE_STATUS_BSY_MSK )
          /* need more time to wait */
          if (timer++ > 20*TIMER)
          {   
               return (ERROR_IDE_UNKNOWN);
          } 
          
          /*  Wait until device is ready to exchange data or
	   *  an error is detected 
	   */
	
	  timer = 0;
	  do
	  {
	    status = IO32(IDE_STATUS_OFS);
	
	    if (timer++ >4*TIMER)
	    { 
               return (ERROR_IDE_UNKNOWN);
            }   
	    
	  }
          while( ((status & IDE_STATUS_DRQ_BIT) == 0) &&
	         ((status & IDE_STATUS_ERR_BIT) == 0) )
	      ;
         
          
   

          if( p_param->command != IDE_CMD_WRITE )
          {
            
	    /* Read data from device sector buffer */
            for( i=0; i < IDE_BYTES_PER_SECTOR / 4 *2; i++ )
	    {
	        data32 = IO32(IDE_DATA_OFS);

	        /* Byte oriented data */
	        buffer[0] = (data32 >> 0)  & 0xff;
	        buffer[1] = (data32 >> 8)  & 0xff;
	        
                buffer+= 2;
                
       
               
	    }
    
	    status = IO32(IDE_STATUS_OFS);
	    
	    /* Check if read caused error */
    
            if( status & IDE_STATUS_ERR_BIT )
            {   
	        return get_error();
	    }
	
          }
          else
          {
            /* Write data to device sector buffer */
            for( i=0; i < IDE_BYTES_PER_SECTOR / 4*2; i++ )
	    {
	        data32 = (buffer[0] << 0)  |
		         (buffer[1] << 8) ; 

	        IO32(IDE_DATA_OFS) = data32;

		buffer+= 2;
	    }

	    /*  Wait until device is done.
	     *  This will be the case when DRQ and BSY are both cleared.
	     */
	    timer = 0;
	    do
	    {
	      status = IO32(IDE_STATUS_OFS);
	      if (timer++ >TIMER) {
	      
               return (ERROR_IDE_UNKNOWN);
              } 

	    }
            while( ((status & IDE_STATUS_DRQ_BIT) ) ||
	         ((status & IDE_STATUS_BSY_BIT) ) )
	      ; 
          
             if( status & IDE_STATUS_ERR_BIT ) 
             {  
                return get_error();
             }
            
          }

        sector++;
      }	 
      
      if( p_param->command == IDE_CMD_IDENTIFY )
      {
	identify_struct = &p_param->u.identify;

        /* Decode buffer */
        i = 0;
  
       
        identify_struct->config    =		          buf_id[i++];
	identify_struct->cyl_count =			  buf_id[i++];

        identify_struct->reserved0 =			  buf_id[i++];
        identify_struct->head_count=			  buf_id[i++];

	for( t=0; t<2; t++ )
	{
            identify_struct->obsolete0[t] =		  buf_id[i++];
	}        

	identify_struct->sectors_per_track =		  buf_id[i++];

	for( t=0; t<3; t++ )    
	{
	    identify_struct->vendor_specific0[t] =	  buf_id[i++];
	}

	for( t=0; t<10; t++ )
	{
	  identify_struct->serial_number[t*2]   =	  (UINT8)(buf_id[i] >> 8);
	  identify_struct->serial_number[t*2+1] =	  (UINT8)(buf_id[i++] & 0xFF);
	}

	for( t=0; t<2; t++ )
	{
	    identify_struct->obsolete1[t] =		  buf_id[i++];
	}

        identify_struct->ecc_count =			  buf_id[i++];

	for( t=0; t<4; t++ )
	{
	    identify_struct->firmware_rev[t*2] =	  (UINT8)(buf_id[i] >> 8);
	    identify_struct->firmware_rev[t*2+1] =	  (UINT8)(buf_id[i++] & 0xFF);
	}

	for( t=0; t<20; t++ )
	{
	    identify_struct->model_name[t*2] =		  (UINT8)(buf_id[i] >> 8);
	    identify_struct->model_name[t*2+1] =	  (UINT8)(buf_id[i++] & 0xFF);
	}

        identify_struct->sectors_per_interrupt =          buf_id[i++];
        identify_struct->reserved1 =			  buf_id[i++];
        identify_struct->capabilities =			  buf_id[i++];
        identify_struct->reserved2 =			  buf_id[i++];
        identify_struct->timing_mode_pio =		  buf_id[i++];
        identify_struct->timing_mode_dma =		  buf_id[i++];
        identify_struct->application =			  buf_id[i++];
        identify_struct->cyl_count_apparent =		  buf_id[i++];
   
        identify_struct->head_count_apparent =		  buf_id[i++];
        identify_struct->sectors_per_track_apparent =	  buf_id[i++];

        identify_struct->capacity_apparent =              (UINT32)buf_id[i] |
						          ((UINT32)buf_id[i+1] << 16);
        i+=2;
							  
        identify_struct->sectors_per_interrupt_apparent = buf_id[i++];

        identify_struct->lba_sector_count =		  (UINT32)buf_id[i] |
							  ((UINT32)buf_id[i+1] << 16);
        i+=2;
							  
        identify_struct->modes_single_dma =		  buf_id[i++];
        identify_struct->modes_multiple_dma =		  buf_id[i++];
        identify_struct->modes_adv_pio =		  buf_id[i++];
        identify_struct->min_cycle_time_dma =		  buf_id[i++];
        identify_struct->recommended_cycle_time_dma =	  buf_id[i++];
        identify_struct->minimum_cycle_time_pio =	  buf_id[i++];
        identify_struct->minimom_cycle_time_pio_iordy =	  buf_id[i++];

	for( t=0; t<59; t++ )
	{
            identify_struct->reserved3[59] =		  buf_id[i++];
	}

	for( t=0; t<32; t++ )
	{
            identify_struct->vendor_specific1[32] =	  buf_id[i++];
	}

	for( t=0; t<96; t++ )
	{
	    identify_struct->reserved4[96] =		  buf_id[i++];
	}
      }  
     }
     break;

    } 
    
    return OK;
}


/************************************************************************
 *                          get_error
 ************************************************************************/
static UINT32
get_error( void )
{
    UINT32 error = IO32(IDE_ERROR_OFS);
    
    
    /* Sector number is now valid for error handling */
  

    if(      error & IDE_ERROR_BBK_BIT )
        return ERROR_IDE_BAD_BLOCK;
    else if( error & IDE_ERROR_UNC_BIT )
        return ERROR_IDE_UNCORRECTABLE_DATA;
    else if( error & IDE_ERROR_MC_BIT )
        return ERROR_IDE_MEDIA_CHANGE;
    else if( error & IDE_ERROR_IDNF_BIT )
        return ERROR_IDE_ID_NOT_FOUND;
    else if( error & IDE_ERROR_MCR_BIT )
        return ERROR_IDE_MEDIA_CHANGE_REQUESTED;
    else if( error & IDE_ERROR_ABRT_BIT )
        return ERROR_IDE_ABORTED;
    else if( error & IDE_ERROR_TK0NF_BIT )
        return ERROR_IDE_TRACK_0_NOT_FOUND;
    else if( error & IDE_ERROR_AMNF_BIT )
        return ERROR_IDE_ADDRESS_MARK_NOT_FOUND;
    else
        return ERROR_IDE_UNKNOWN;
        
        
}





/************************************************************************
 *      Implementation : Public functions
 ************************************************************************/


/************************************************************************
 *
 *                          ide_install
 *  Description :
 *  -------------
 *
 *  Installs the IDE device driver services in the IO system at the 
 *  reserved device slot, found in the 'sysdev.h' file, which defines 
 *  all major device numbers.
 *
 *  Note:
 *  This service is the only public declared interface function; all
 *  provided device driver services are static declared, but this
 *  function installs the function pointers in the io-system to
 *  enable the provided public driver services.
 *
 *  Parameters :
 *  ------------
 *
 *  -
 *
 *  Return values :
 *  ---------------
 *
 *  'OK'(=0)
 *  'ERROR_IO_ILLEGAL_MAJOR':  Illegal major device number
 *  'ERROR_IO_NO_SPACE':       Device slot already allocated
 *
 ************************************************************************/
INT32 
ide_install( void )
{   
    
    UINT32       *buffer;
    
   

    /* Install device services */
    IO_install( SYS_MAJOR_IDE,                  /* major device number */
                    (t_io_service)ide_init,	/* 'init'  service     */
                    NULL,			/* 'open'  service  na */
                    NULL,			/* 'close' service  na */
                    NULL,			/* 'read'  service  na */
		    NULL,			/* 'write' service  na */
                    (t_io_service)ide_ctrl ) ;	/* 'ctrl'  service     */

  

    /* call our own 'init' service */
   IO_init( SYS_MAJOR_IDE, IDE_MINOR_PRIMARY_MASTER, 1);
   
   
  
   IO_init( SYS_MAJOR_IDE, IDE_MINOR_PRIMARY_SLAVE, 1); 
   
   
   IO_init( SYS_MAJOR_IDE, IDE_MINOR_SECONDARY_MASTER, 1);
   
   
   IO_init( SYS_MAJOR_IDE, IDE_MINOR_SECONDARY_SLAVE, 1);
   
    
   
   
     
   return OK;
}
