/* 
LzmaTest.c
Test application for LZMA Decoder

This file written and distributed to public domain by Igor Pavlov.
This file is part of LZMA SDK 4.26 (2005-08-05)
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <syscon_api.h>

#include "LzmaDecode.h"

const char *kCantReadMessage = "Can not read input file";
const char *kCantWriteMessage = "Can not write output file";
const char *kCantAllocateMessage = "Can not allocate memory";



int MyReadFileAndCheck(char *in_addr, char *data, size_t size)
{ 
   int i ;	
   for (i = 0; i<size; i++)
     *(char *)((int)data+i) = *(char *)((int)in_addr+i );
   return 0;  	  
} 



int MyWriteFileAndCheck(char *in_addr, const char *data, size_t size)
 { 
   int i ;	
   for (i = 0; i<size; i++)
     *(char *)((int)in_addr+i) = *(char *)((int)data+i );
   return 0;  	  
} 

#ifdef _LZMA_IN_CB
#define kInBufferSize (1 << 15)
typedef struct _CBuffer
{
  ILzmaInCallback InCallback;
  FILE *File;
  unsigned char Buffer[kInBufferSize];
} CBuffer;

int LzmaReadCompressed(void *object, const unsigned char **buffer, SizeT *size)
{
  CBuffer *b = (CBuffer *)object;
  *buffer = b->Buffer;
  *size = (SizeT)MyReadFile(b->File, b->Buffer, kInBufferSize);
  return LZMA_RESULT_OK;
}
CBuffer g_InBuffer;

#endif

#ifdef _LZMA_OUT_READ
#define kOutBufferSize (1 << 15)
unsigned char g_OutBuffer[kOutBufferSize];
#endif

int PrintError(char *buffer, const char *message)
{
  sprintf(buffer + strlen(buffer), "\nError: ");
  sprintf(buffer + strlen(buffer), message);
  return 1;
}

int Lzma_main(char *inAddr, char *outAddr, long length, char *rs)
{
  /* We use two 32-bit integers to construct 64-bit integer for file size.
     You can remove outSizeHigh, if you don't need >= 4GB supporting,
     or you can use UInt64 outSize, if your compiler supports 64-bit integers*/
  UInt32 outSize = 0;
  UInt32 outSizeHigh = 0;
  
  SizeT outSizeFull;
  unsigned char *outStream;
  
  
  int waitEOS = 1; 
  /* waitEOS = 1, if there is no uncompressed size in headers, 
   so decoder will wait EOS (End of Stream Marker) in compressed stream */

  
  SizeT compressedSize;
  unsigned char *inStream;
  

  CLzmaDecoderState state;  /* it's about 24-80 bytes structure, if int is 32-bit */
  unsigned char properties[LZMA_PROPERTIES_SIZE];

  int res;
  t_sys_malloc       mem ;

 
  compressedSize = (SizeT)(length - (LZMA_PROPERTIES_SIZE + 8));
 
  
  /* Read LZMA properties for compressed stream */

  MyReadFileAndCheck(inAddr, properties, sizeof(properties));
  
 
  inAddr += sizeof(properties);
  /* Read uncompressed size */

  {
    int i;
    for (i = 0; i < 8; i++)
    {
      unsigned char b;
      MyReadFileAndCheck(inAddr, &b, 1);
      inAddr += 1;
     
      if (b != 0xFF)
        waitEOS = 0;
      if (i < 4)
        outSize += (UInt32)(b) << (i * 8);
      else
        outSizeHigh += (UInt32)(b) << ((i - 4) * 8);
    }
    
  
    if (waitEOS)
      return PrintError(rs, "Stream with EOS marker is not supported");
    outSizeFull = (SizeT)outSize;
    if (sizeof(SizeT) >= 8)
      outSizeFull |= (((SizeT)outSizeHigh << 16) << 16);
    else if (outSizeHigh != 0 || (UInt32)(SizeT)outSize != outSize)
      return PrintError(rs, "Too big uncompressed stream");
  
    
  }
 
  /* Decode LZMA properties and allocate memory */

  if (LzmaDecodeProperties(&state.Properties, properties, LZMA_PROPERTIES_SIZE) != LZMA_RESULT_OK)
    return PrintError(rs, "Incorrect stream properties");
    
 
  mem.size     = LzmaGetNumProbs(&state.Properties) * sizeof(CProb) ;
  mem.boundary =  sizeof(CProb) ;        /* sizeof(char) ;  */
  mem.memory   = (void*) &(state.Probs) ; 
  SYSCON_read( SYSCON_FREEMEM_ALLOC_ID,
                 &mem,
                 sizeof(mem) ) ;
 
//  state.Probs = (CProb *)malloc(LzmaGetNumProbs(&state.Properties) * sizeof(CProb));

 
  if (outSizeFull == 0)
    outStream = 0;
  else
  { 
    mem.size     = outSizeFull ;
    mem.boundary = sizeof(char) ;        
    mem.memory   = (void*) &outStream ; 
    SYSCON_read( SYSCON_FREEMEM_ALLOC_ID,
                 &mem,
                 sizeof(mem) ) ;	
  	
  } 	
  
//    outStream = (unsigned char *)malloc(outSizeFull);
  

  
  if (compressedSize == 0)
    inStream = 0;
  else
  { 
    mem.size     = compressedSize ;
    mem.boundary = sizeof(char) ;        
    mem.memory   = (void*) &inStream ; 
    SYSCON_read( SYSCON_FREEMEM_ALLOC_ID,
                 &mem,
                 sizeof(mem) ) ;
    	
  }  	
//    inStream = (unsigned char *)malloc(compressedSize);
 
  
  if (state.Probs == 0 
    
    || (outStream == 0 && outSizeFull != 0)
    
   
    || (inStream == 0 && compressedSize != 0)
    
    )
  {
  	
//    free(state.Probs);
//    #ifdef _LZMA_OUT_READ
//    free(state.Dictionary);
//    #else
//    free(outStream);
//    #endif
//    #ifndef _LZMA_IN_CB
//    free(inStream);
//    #endif
    return PrintError(rs, kCantAllocateMessage);
  }

  /* Decompress */


  MyReadFileAndCheck(inAddr, inStream, compressedSize);
  

  
 
  {
    
    SizeT inProcessed;
    
    SizeT outProcessed;
    res = LzmaDecode(&state,
      
      inStream, compressedSize, &inProcessed,
      
      outStream, outSizeFull, &outProcessed);
   
    if (res != 0)
    {
      printf("\nDecoding error = %d\n", res);
      res = 1;
     
    }
    else if (outAddr != 0)
    {
     
      MyWriteFileAndCheck(outAddr, outStream, (size_t)outProcessed);
             
    }
  }
 
  printf("lzma finished \n"); //cy test 
//  free(state.Probs);
//  #ifdef _LZMA_OUT_READ
//  free(state.Dictionary);
//  #else
//  free(outStream);
//  #endif
//  #ifndef _LZMA_IN_CB
//  free(inStream);
//  #endif
  return res;
}




