#include <project_config.h>
#include <extern_param.h>
#include "flashdev_p.h"
#include "flashdev_s.h"


#define FLASH_TYPE_SERIAL   	0xde
#define FLASH_TYPE_PARALLEL		0xbe

//#define PARAMETER_TMP_ADDR		0xa0300000
#define PARAMETER_TMP_ADDR		0xa0500000
#define LINUX_KERNEL_TMP_ADDR		0xa0600000


extern unsigned char array[];
extern unsigned char array_end;
extern unsigned char linux[];
extern unsigned char linux_end;
extern unsigned char logo[];
extern unsigned char logo_end;
extern unsigned char logo2[];
extern unsigned char logo2_end;
extern unsigned char logo3[];
extern unsigned char logo3_end;

/************************************************************************
 *
 *                          dvrmain
 *  Description :
 *  -------------
 *  main function of flash writer 
 *  
 *
 *  Parameters :
 *  ------------
 *
 *
 *
 *  Return values :
 *  ---------------
 *
 *  
 *
 *
 ************************************************************************/
int dvrmain(void)
{	
    UINT8 *array_addr = (UINT8 *)0xbfc00000;		//assign bootcode start address
    UINT8 *logo_addr;
    UINT8 *logo2_addr;
    UINT8 *logo3_addr;
    UINT8 *linux_sec1_addr;
    UINT8 *linux_sec2_addr;
    UINT8 *env_param_addr = (UINT8 *)0xbfbf0000;	//assign bootcode parameter address
    
    UINT8 *temp_ptr = (UINT8 *)0xbfc00000;
    UINT8 *src_addr;
    UINT8 *dst_addr;

    UINT32  array_size = (unsigned int )(&array_end - array);
    UINT32  logo_size =  (unsigned int )(&logo_end  - logo);
    UINT32  logo2_size = (unsigned int )(&logo2_end - logo2);
    UINT32  logo3_size = (unsigned int )(&logo3_end - logo3);
#if defined(Rescue_Source_USB) && defined(Board_USB_Driver_Enabled)
    UINT32  linux_size = (unsigned int ) 0;
#else /* defined(Rescue_Source_USB) && defined(Board_USB_Driver_Enabled) */
    UINT32  linux_size = (unsigned int )(&linux_end - linux);
#endif /* defined(Rescue_Source_USB) && defined(Board_USB_Driver_Enabled) */
    UINT32  linux_sec1_size;
    UINT32  linux_sec2_size;
    
    int (*do_erase)(void  *, UINT8 * , UINT32);
    int (*do_write)(void *, UINT8 *, UINT8 *, UINT32);
    int (*do_identify)(void **);
    void (*do_init)();
	void (*do_exit)();
    
    UINT8 flash_type;
    UINT32 idx;
    UINT32 temp;
    t_extern_param param;
	void *device;
    
    //define minimal erase block size and total block from 0xbfc0_0000~0xbfcf_ffff
    int  block_size = 0x1000;
    int  total_block_cnt = 0x100000 / block_size;
    
    /***************************************************
     * determine flash type on DUT and correspond tools
     ***************************************************/
    /* if 0: parallel flash. 1: serial flash */
#ifdef Board_CPU_VENUS
    if(REG32(0xb801a800) >> 31)
#else /* Board_CPU_VENUS */
    if(REG32(0xb801a200) >> 16)
#endif /* Board_CPU_VENUS */
    {
    	dump_str("flash type: serial", 18);
    	flash_type = FLASH_TYPE_SERIAL;
		do_erase = do_erase_s;
    	do_write = do_write_s;
    	do_identify = do_identify_s;
    	do_init = do_init_s;
    	do_exit = do_exit_s;
    }
    else
    {
    	dump_str("flash type: parallel", 20);
    	flash_type = FLASH_TYPE_PARALLEL;
   		do_erase = do_erase_p;
   		do_write = do_write_p;
   		do_identify = do_identify_p;
   		do_init = do_init_p;
   		do_exit = do_exit_p;    	
    }

    /***************************************************
     * calculate images start address
     ***************************************************/
    /* evaluate bootcode occupied space */
    temp = (array_size / block_size ) + 1;
    total_block_cnt -= temp;
    temp_ptr = (UINT8 *)(array_addr + (temp * block_size));
    
    /* evaluate logo occupied space */
    if (logo_size > 0)
    {
    	logo_addr = temp_ptr;
		
		//calculate next data section start address
		temp = (logo_size / block_size ) + 1;
        total_block_cnt -= temp;
    	temp_ptr = (unsigned char *)(logo_addr + (temp * block_size));
    }
    else
    {
    	logo_addr = 0x0;
    }
    
    /* evaluate logo2 occupied space */
    if (logo2_size > 0)
    {
    	logo2_addr = temp_ptr;
		
		//calculate next data section start address
		temp = (logo2_size / block_size ) + 1;
        total_block_cnt -= temp;
    	temp_ptr = (unsigned char *)(logo2_addr + (temp * block_size));
    }
    else
    {
    	logo2_addr = 0x0;
    }

    /* evaluate logo3 occupied space */
    if (logo3_size > 0)
    {
    	logo3_addr = temp_ptr;
		
		//calculate next data section start address
		temp = (logo3_size / block_size ) + 1;
        total_block_cnt -= temp;
    	temp_ptr = (unsigned char *)(logo3_addr + (temp * block_size));
    }
    else
    {
    	logo3_addr = 0x0;
    }
    
    /* evaluate rescue occupied space */
    if (linux_size > 0)
    {
        /* assign linux section1 address */
        linux_sec1_addr = temp_ptr;
        
        /* calculate linux section2 address: */
        temp = (linux_size / block_size ) + 1;
        if (total_block_cnt >= temp)
        {
            linux_sec1_size = linux_size;
            linux_sec2_addr = 0;
            linux_sec2_size = 0;
        }
        else
        {
        	UINT32 boundary = 0xbfbf0000;
            
            linux_sec1_size = total_block_cnt * block_size;
            linux_sec2_size = linux_size - linux_sec1_size;
            	
            temp = temp - total_block_cnt;
                
            linux_sec2_addr = (UINT8 *)( boundary - (temp * block_size));
            linux_sec2_size = linux_size - linux_sec1_size;
        }
    }
    else
    {
        linux_sec1_addr = 0;
        linux_sec1_size = 0;
        linux_sec2_addr = 0;
        linux_sec2_size = 0;
    }
    
#if defined(Rescue_Source_USB) && defined(Board_USB_Driver_Enabled)
    /* Keep starting offset of the next free space. */
    linux_sec1_addr = temp_ptr;
    linux_sec1_size = total_block_cnt * block_size;
#endif /* defined(Rescue_Source_USB) && defined(Board_USB_Driver_Enabled) */

    /***************************************************
     * assign value to param object
     ***************************************************/
    param.flash_type = flash_type;
    
    param.region     = 0;
    param.mac_hi     = Param_MAC_hi;
    param.mac_lo     = Param_MAC_lo;
    
    param.logo_img_saddr = logo_addr;
    param.logo_img_len   = logo_size;
#if defined(Logo_Type_PAL)
    param.logo_type   = 1;
#elif defined(Logo_Type_NTSC)
    param.logo_type   = 0;
#endif
    param.logo_offset   = Logo_Offset;
    param.logo_reg_5370 = Logo_Reg5370;
    param.logo_reg_5374 = Logo_Reg5374;
    param.logo_reg_5378 = Logo_Reg5378;
    param.logo_reg_537c = Logo_Reg537c;

	//clear logo value if logo2, logo3 was not used
	param.logo2_img_saddr = param.logo3_img_saddr = 0;
	param.logo2_img_len   = param.logo3_img_len   = 0;
    param.logo2_type      = param.logo3_type      = 0;
	param.logo2_offset    = param.logo3_offset    = 0;
	param.logo2_reg_5370  = param.logo3_reg_5370  = 0;
	param.logo2_reg_5374  = param.logo3_reg_5374  = 0;
	param.logo2_reg_5378  = param.logo3_reg_5378  = 0;
	param.logo2_reg_537c  = param.logo3_reg_537c  = 0;

#if defined(Logo2_Source_FLASH)
    param.logo2_img_saddr = logo2_addr;
    param.logo2_img_len   = logo2_size;
  #if defined(Logo2_Type_PAL)
    param.logo2_type   = 1;
  #elif defined(Logo2_Type_NTSC)
    param.logo2_type   = 0;
  #endif
    param.logo2_offset   = Logo2_Offset;
    param.logo2_reg_5370 = Logo2_Reg5370;
    param.logo2_reg_5374 = Logo2_Reg5374;
    param.logo2_reg_5378 = Logo2_Reg5378;
    param.logo2_reg_537c = Logo2_Reg537c;  
#endif

#if defined(Logo3_Source_FLASH)
    param.logo3_img_saddr = logo3_addr;
    param.logo3_img_len   = logo3_size  ;
  #if defined(Logo3_Type_PAL)
   	param.logo3_type   = 1  ;
  #elif defined(Logo3_Type_NTSC)
   	param.logo3_type   = 0  ;
  #endif
    param.logo3_offset   = Logo3_Offset;
    param.logo3_reg_5370 = Logo3_Reg5370;
    param.logo3_reg_5374 = Logo3_Reg5374;
    param.logo3_reg_5378 = Logo3_Reg5378;
    param.logo3_reg_537c = Logo3_Reg537c;
#endif
    
    param.rescue_img_size        = linux_size;
    param.rescue_img_part0_saddr = linux_sec1_addr;
    param.rescue_img_part0_len   = linux_sec1_size;
    param.rescue_img_part1_saddr = linux_sec2_addr;
    param.rescue_img_part1_len   = linux_sec2_size;
    
    param.env_param_saddr = env_param_addr;

    /***************************************************
     * write param obj to array content
     ***************************************************/    
    src_addr = (UINT8 *)&param;
    dst_addr = array + 0x1010;
    for (idx = 0; idx < sizeof(t_extern_param); idx++ )
        dst_addr[idx] = src_addr[idx];
    
    /***************************************************
     * copy parameters in flash to DDR
     ***************************************************/
    for (idx = 0; idx < 0x10000; idx +=4)
    	REG32(PARAMETER_TMP_ADDR + idx) = REG32( 0xbfbf0000 + idx);
       
#if defined(Rescue_Source_USB) && defined(Board_USB_Driver_Enabled)
    /***************************************************
     * copy Linux kernel from flash to DDR
     ***************************************************/
    for (idx = 0; idx < linux_sec1_size; idx +=4)
    	REG32(LINUX_KERNEL_TMP_ADDR + idx) = REG32(linux_sec1_addr + idx);
#endif /* defined(Rescue_Source_USB) && defined(Board_USB_Driver_Enabled) */

    /*******************************************************
     * identyfy flash type and program code to flash device
     *******************************************************/
    //dump 'Begin' string to RS232 
    if ((*do_identify)(&device) < 0)
    {
    	dump_str("error identify flash type!!", 27);
    	return -1;
    }

    dump_str("Begin", 6);
    
    (*do_init)();
    
    //start erase and write image to flash
    if ((*do_erase)(device, array_addr, 0x100000) !=0 )
		return -2;

    if ((*do_write)(device, array, array_addr, array_size)!= 0 )
        return -3;
        
    if (logo_size > 0)
    {
	    if ((*do_write)(device, logo, logo_addr, logo_size)!= 0)
    	    return -4;
    }
    
    if (logo2_size > 0)
    {
	    if ((*do_write)(device, logo2, logo2_addr, logo2_size)!= 0)
    	    return -5;
    }

    if (logo3_size > 0)
    {
	    if ((*do_write)(device, logo3, logo3_addr, logo3_size)!= 0)
    	    return -5;
    }
    
    if (linux_size > 0)
    {
        //write linux part 1
        // rescue part1 must in 0xbfc0_0000 ~ 0xbfcf_ffff, so do not need to erase
        if ((*do_write)( device, linux, linux_sec1_addr, linux_sec1_size)!=0)
            return -6;
        
        //if linux part 2 exists, write it to flash
        if (linux_sec2_addr != 0)
        {
            /* rescue part2 located in 0xbfb0_0000 ~ 0xbfbf_ffff, so need to erase flash
               and parameter region */
            if ((*do_erase)( device, linux_sec2_addr, linux_sec2_size + 0x10000)!=0)
            	return -7;
            if ((*do_write)( device, (UINT8 *)(linux + linux_sec1_size),
                          linux_sec2_addr, linux_sec2_size)!=0)
                return -8;
            if ((*do_write)( device, (UINT8 *)(PARAMETER_TMP_ADDR),
                          (UINT8 *)0xbfbf0000, 0x10000)!=0)
                return -9;
        }
    }
#if defined(Rescue_Source_USB) && defined(Board_USB_Driver_Enabled)
    else {
        if (linux_sec1_size) {
            if ((*do_write)( device, (UINT8 *)(LINUX_KERNEL_TMP_ADDR),
                          (UINT8 *)linux_sec1_addr, linux_sec1_size)!=0) {
                return -0xa;
            }
        }
    }
#endif /* defined(Rescue_Source_USB) && defined(Board_USB_Driver_Enabled) */
    
    /* Reset Command */
    (*do_exit)();

    //dump 'Finish' string to RS232 
    dump_str("Finish", 6);
    
    return 0;
}
