#include <linux/kernel.h>
#include <linux/config.h>
#include <linux/kernel.h>
#include <linux/ioport.h>
#include <linux/module.h>
#include <linux/delay.h>
#include <linux/slab.h>
#include <linux/init.h>
#include <linux/interrupt.h>
#include <linux/wait.h>
#include <linux/fs.h>
#include <linux/cdev.h>
#include <linux/devfs_fs_kernel.h>
#include "cec.h"
#include "cec_mars.h"


/*------------------------------------------------------------------
 * Func : cec_dev_ioctl
 *
 * Desc : ioctl function of cec dev
 *
 * Parm : inode : inode of dev
 *        file  : context of file
 *        cmd   : control command
 *        arg   : arguments
 *         
 * Retn : 0 : success, others fail  
 *------------------------------------------------------------------*/
static 
int cec_dev_ioctl(
    struct inode*           inode, 
    struct file*            file,
    unsigned int            cmd, 
    unsigned long           arg
    )
{            
    mars_cec*       p_cec = (mars_cec*) file->private_data;
    cec_msg         msg;    
    unsigned char   buff[64];
    unsigned char   init;
    unsigned char   rcv_len;
            
    cec_dbg("%s\n",__FUNCTION__);			
    
    switch(cmd)
    {
    case CEC_ENABLE:        
        return mars_cec_enable(p_cec, (arg) ? 1 : 0);
        
    case CEC_SET_LOGICAL_ADDRESS: 
              
        return mars_cec_set_logical_addr(p_cec, (unsigned char) arg);
            
    case CEC_SEND_MESSAGE:         
        
        if (copy_from_user(&msg, (cec_msg __user *)arg, sizeof(cec_msg)))              
			return -EFAULT;                
        
        if (msg.len > 64)
            return -ENOMEM;
            
        if (copy_from_user(buff, (unsigned char __user *)msg.buf, msg.len))
            return -EFAULT;			                			
            
        return mars_cec_send_message(p_cec, buff, msg.len);
    
    case CEC_RCV_MESSAGE:
        
        if (copy_from_user(&msg, (cec_msg __user *)arg, sizeof(cec_msg)))              
			return -EFAULT;     
			
        if (mars_cec_rcv_message(p_cec, buff, 64, &rcv_len)<0)			
            return -EFAULT;
    
        if (rcv_len > msg.len) 
        {
            cec_warning("cec : read message failed, msg size (%d) more than msg buffer size(%d)\n", rcv_len, msg.len);                            
            return -ENOMEM;
        }

        if (copy_to_user((unsigned char __user *) msg.buf, buff, rcv_len)<0)
        {
            cec_warning("cec : read message failed, copy msg to user failed\n");
            return -EFAULT;
        }

        return rcv_len;
                
    default:    
        cec_warning("cec : unknown ioctl cmd %08x\n", cmd);        
        return -EFAULT;          
    }
}



/*------------------------------------------------------------------
 * Func : cec_dev_open
 *
 * Desc : open function of cec dev
 *
 * Parm : inode : inode of dev
 *        file  : context of file
 *         
 * Retn : 0 : success, others fail  
 *------------------------------------------------------------------*/
static int cec_dev_open(struct inode *inode, struct file *file)
{
    mars_cec* p_cec;
    
    cec_dbg("%s\n",__FUNCTION__);	
    
    p_cec =  mars_cec_open();
    
    if (!p_cec)
    {
        cec_warning("cec : unable to alloc cec handle\n");
        return -ENODEV;
    }
    
    mars_cec_init(p_cec);    
	
	file->private_data = p_cec;
	
	return 0;    	
}


/*------------------------------------------------------------------
 * Func : cec_dev_release
 *
 * Desc : release function of cec dev
 *
 * Parm : inode : inode of dev
 *        file  : context of file
 *         
 * Retn : 0 : success, others fail  
 *------------------------------------------------------------------*/
static int cec_dev_release(
    struct inode*           inode, 
    struct file*            file
    )
{    
    mars_cec* p_cec = (mars_cec*) file->private_data;
    
    cec_dbg("%s\n",__FUNCTION__);	   
        
    mars_cec_enable(p_cec, 0);
    
    mars_cec_close(p_cec);
    
	return 0;
}



static struct file_operations cec_dev_fops = 
{
	.owner		= THIS_MODULE,	
	.ioctl		= cec_dev_ioctl,
	.open		= cec_dev_open,
	.release	= cec_dev_release,
};



static dev_t    dev;
struct cdev     my_cdev;


/*------------------------------------------------------------------
 * Func : cec_dev_init
 *
 * Desc : cec dev init function
 *
 * Parm : N/A
 *         
 * Retn : 0 : success, others fail  
 *------------------------------------------------------------------*/
static int __init cec_dev_init(void)
{         
    if (alloc_chrdev_region(&dev, 0, 1, "cec")!=0)    
        goto err_alloc_chr_dev_failed;    
    
    cdev_init(&my_cdev, &cec_dev_fops);
    
	if (cdev_add(&my_cdev, dev, 1)<0)
	{		    
	    cec_dbg("cec : register character dev failed\n");    
	    goto err_register_chr_dev_failed;		    
    }
    
    // create devicd node    
	devfs_mk_dir("cec");	
    devfs_mk_cdev(dev, S_IFCHR|S_IRUSR|S_IWUSR, "cec/0");
	        
    return 0;

//** Error handler **
err_register_chr_dev_failed:    
    unregister_chrdev_region(MAJOR(dev), 1);
    
err_alloc_chr_dev_failed:

    return -EFAULT;
}


/*------------------------------------------------------------------
 * Func : cec_dev_exit
 *
 * Desc : cec dev exit function
 *
 * Parm : N/A
 *         
 * Retn : 0 : success, others fail  
 *------------------------------------------------------------------*/
static void __exit cec_dev_exit(void)
{   
    cdev_del(&my_cdev);    
    unregister_chrdev_region(dev, 1);
    devfs_remove("cec/0");
    devfs_remove("cec");    	
}


module_init(cec_dev_init);
module_exit(cec_dev_exit);
