/* ------------------------------------------------------------------------- 
   cec_mars.c  cec driver for Realtek Neptune/Mars           
   ------------------------------------------------------------------------- 
    Copyright (C) 2008 Kevin Wang <kevin_wang@realtek.com.tw>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.    
----------------------------------------------------------------------------
Update List :
----------------------------------------------------------------------------
    1.0     |   20080911    |   Kevin Wang  | 1) create phase
----------------------------------------------------------------------------
    1.0a    |   20080917    |   Kevin Wang  | 1) porting to kernel land
----------------------------------------------------------------------------
    1.0b    |   20080918    |   Kevin Wang 
----------------------------------------------------------------------------    
    1) change the prototype of mar_cec_rcv_msg / mars_cec_send_message    
    2) fix TX control logic of CEC
    3) change TX to interrupt mode
    4) terminate all blocked tx/rx thread when chip is disabled
-----------------------------------------------------------------------------
    1.0c    |   20080925    |   Kevin Wang  | 1) Fix CEC timing
----------------------------------------------------------------------------*/
#include <linux/kernel.h>
#include <linux/config.h>
#include <linux/kernel.h>
#include <linux/ioport.h>
#include <linux/module.h>
#include <linux/delay.h>
#include <linux/slab.h>
#include <linux/init.h>
#include <linux/jiffies.h>
#include <linux/interrupt.h>
#include <linux/wait.h>
#include <asm/io.h>
#include "cec.h"
#include "cec_mars.h"
#include "cec_mars_reg.h"

static mars_cec         m_cec;
static unsigned char    cec_count = 0;

#if 1
#define write_reg32(addr, val)      (writel((u32)(val), (volatile void __iomem*) (addr)))
#define read_reg32(addr)            ((u32) readl((volatile void __iomem*) (addr)))
#else

void write_reg32(unsigned long addr, unsigned long value)
{    
    writel((u32)(value), (volatile void __iomem*) (addr));
    cec_dbg("w : addr=%p, val = %p\n", addr, value);   
}

unsigned long read_reg32(unsigned long addr)
{    
    u32 val = ((u32) readl((volatile void __iomem*) (addr)));
    cec_dbg("r : addr=%p, val = %p\n", addr, val);       
    return val;
}
#endif



/*------------------------------------------------------------------
 * Func : mars_cec_isr
 *
 * Desc : isr of venus i2c
 *
 * Parm : p_this : handle of venus i2c 
 *        dev_id : handle of the mars_cec
 *        regs   :
 *         
 * Retn : IRQ_NONE, IRQ_HANDLED
 *------------------------------------------------------------------*/
static 
irqreturn_t mars_cec_isr(
    int                     this_irq, 
    void*                   dev_id, 
    struct pt_regs*         regs
    )
{        
    mars_cec* p_this  = (mars_cec*) dev_id;                        
    unsigned long event = read_reg32(MIS_ISR);
    
    if ((event & (CEC_TX_INT | CEC_RX_INT))==0)    
        return IRQ_NONE;    
    
    if (event & CEC_TX_INT)    
        mars_cec_do_xmit(p_this);            

    if (event & CEC_RX_INT)    
        mars_cec_do_rcv(p_this);    

    write_reg32(MIS_ISR, event);
        
    return IRQ_HANDLED;
}



/*------------------------------------------------------------------
 * Func : mars_cec_open
 *
 * Desc : open a mars cec module
 *
 * Parm : N/A
 *         
 * Retn : handle of cec module 
 *------------------------------------------------------------------*/
mars_cec* mars_cec_open()
{        
    if (cec_count)
        return NULL;

    mars_cec_init(&m_cec);
    
    if (request_irq(MISC_IRQ, mars_cec_isr, SA_INTERRUPT | SA_SHIRQ, "MARS CEC", &m_cec) < 0) 
    {
        cec_warning("cec : open mars cec failed, unable to request irq#%d\n", MISC_IRQ);	    		
        return NULL;
    }
  
    return &m_cec;            
}



/*------------------------------------------------------------------
 * Func : mars_cec_close
 *
 * Desc : close a mars cec module
 *
 * Parm : p_this : handle of mars cec 
 *         
 * Retn : N/A  
 *------------------------------------------------------------------*/
void mars_cec_close(mars_cec* p_this)
{
    if (p_this == &m_cec) 
    {        
        mars_cec_enable(p_this, 0);                
        free_irq(MISC_IRQ ,p_this);
        cec_count=0;
    }
}



/*------------------------------------------------------------------
 * Func : mars_cec_do_xmit
 *
 * Desc : cec xmit function for mars ces
 *
 * Parm : p_this : handle of mars cec 
 *         
 * Retn : N/A  
 *------------------------------------------------------------------*/
void mars_cec_do_xmit(mars_cec* p_this)
{
    mars_cec_xmit*  p_xmit = &p_this->xmit;
    unsigned long   val;
    unsigned char   len;  
    unsigned int    i;      
        
    switch (p_xmit->state)
    {
    case WAIT_XMIT:                                    
                
        val = (p_xmit->dest & 0x0F);
        p_xmit->state = XMIT;
        p_xmit->xmit_data_len = 0;                         
        p_this->xmit.timeout = jiffies + (HZ>>2);         
        
        // Fill Data to Buffer (max 16B)                        
        for (i=0; i<16 && p_xmit->xmit_data_len < p_xmit->data_len; len++, p_xmit->xmit_data_len++)
            write_reg32(MIS_CEC_TX_FIFO, p_xmit->p_data[p_xmit->xmit_data_len]);
                                
        val |= TX_EN | TX_INT_EN;
        
        if (p_xmit->xmit_data_len != p_xmit->data_len)
            val |= TX_CONTINUOUS;
                    
        write_reg32(MIS_CEC_TX0,  val);
        
        init_completion(&p_xmit->complete);
        wait_for_completion(&p_xmit->complete);
                                
        break;
        
    case XMIT:
        
        if ((read_reg32(MIS_CEC_TX0) & TX_EN)==0)
            goto end_xmit;
            
        if (time_after(jiffies, p_xmit->timeout))
            goto end_xmit;
          
        if ((read_reg32(MIS_CEC_TX1) & TX_INT)==0)
            break;

        if (p_xmit->xmit_data_len < p_xmit->data_len)
        {
            for (i=0; i<8 && p_xmit->xmit_data_len < p_xmit->data_len; len++, p_xmit->xmit_data_len++)
                write_reg32(MIS_CEC_TX_FIFO, p_xmit->p_data[p_xmit->xmit_data_len]);

            if (p_xmit->xmit_data_len==p_xmit->data_len)
                write_reg32(MIS_CEC_TX0, read_reg32(MIS_CEC_TX0) & ~TX_CONTINUOUS);            
        }
                                    
        write_reg32(MIS_CEC_TX1, TX_INT);                                                           
    }       
    
    return;

end_xmit:

    write_reg32(MIS_CEC_TX0, 0);                    
    
    if (read_reg32(MIS_CEC_TX1) & TX_EOM)
    {
        p_xmit->status = XMIT_OK;
    }
    else if (time_after(jiffies, p_xmit->timeout))        
    {
        p_xmit->status = XMIT_TIMEOUT;    
    }        
    else
    {            
        p_xmit->status = XMIT_FAIL;
    }        
        
    p_xmit->state  = XMIT_COMPLETE;    
        
    complete(&p_xmit->complete);        // wakeup complete    
    
}



/*------------------------------------------------------------------
 * Func : mars_cec_do_rcv
 *
 * Desc : cec rcv function for mars ces
 *
 * Parm : p_this : handle of mars cec 
 *         
 * Retn : N/A  
 *------------------------------------------------------------------*/
void mars_cec_do_rcv(
    mars_cec*               p_this
    )
{    
    mars_cec_rcv*   p_rcv = &p_this->rcv;
    unsigned long   event;
    unsigned int    len;
    unsigned int    i;    
    unsigned char*  ptr;        
        
    switch (p_rcv->state)
    {
    case WAIT_RCV:                                    
            
        // Reset RX                                                       
        write_reg32(MIS_CEC_RX0, RX_RST);
        msleep(100); 
        write_reg32(MIS_CEC_RX0, 0);
                   
        // Rset RX state                                                  
        p_rcv->rcv_data_len = 0;
        p_rcv->status       = RCV_OK;
        p_rcv->state        = RCV;
        
        // start RCV                            
        write_reg32(MIS_CEC_RX0, RX_EN);
        
        break;
        
    case RCV:
                                        
        event = read_reg32(MIS_CEC_RX1);
                  
        if (event & RX_INT)
        {            
            len = event & 0x0F;            
                        
            if (len)
            {                                                                 
                if (p_rcv->rcv_data_len + len > p_rcv->buff_len)
                    goto end_rcv;

                if (p_rcv->rcv_data_len==0) 
                {                    
                    p_rcv->p_buff[0] = (read_reg32(MIS_CEC_RX0) & 0xF) <<4;         // init : 
                    p_rcv->p_buff[0]|= (read_reg32(MIS_CEC_CR0) & 0xF);             // dest : this device
                    p_rcv->rcv_data_len++;   
                }
                                    
                ptr = &p_rcv->p_buff[p_rcv->rcv_data_len];
                for (i=0; i<len; i++)
                    *(ptr++) = (unsigned char) (read_reg32(MIS_CEC_RX_FIFO)& 0xFF);                                
   
                p_rcv->rcv_data_len += len;                                    
            }                              
                              
            write_reg32(MIS_CEC_RX1, RX_INT); 
        }       
        
        if ((read_reg32(MIS_CEC_RX0)& RX_EN)==0)        
            goto end_rcv;            
    }
    
    return;

end_rcv :
    write_reg32(MIS_CEC_RX0, 0);
    p_rcv->state  = RCV_COMPLETE;
    p_rcv->status = (event & RX_EOM) ? RCV_OK : RCV_FAIL;       
} 



/*------------------------------------------------------------------
 * Func : mars_cec_start_xmit
 *
 * Desc : start xmit message
 *
 * Parm : p_this   : handle of mars cec 
 *         
 * Retn : 0 : for success, others : fail
 *------------------------------------------------------------------*/
int mars_cec_start_xmit(
    mars_cec*               p_this    
    )
{   
    if (p_this->xmit.state != WAIT_XMIT)
        return -1;            
    
    write_reg32(MIS_CEC_TX0, TX_RST);
    msleep(1); 
    write_reg32(MIS_CEC_TX0, 0);
    
    mars_cec_do_xmit(p_this);                   // kick off 
    
    while(p_this->xmit.state != XMIT_COMPLETE)            
        msleep(10);           
        
    p_this->xmit.state = IDEL;
    
    return (p_this->xmit.status==XMIT_OK) ? 0 : -1;
}




/*------------------------------------------------------------------
 * Func : mars_cec_stop_xmit
 *
 * Desc : stop xmit message
 *
 * Parm : p_this   : handle of mars cec 
 *         
 * Retn : 0 : for success, others : fail
 *------------------------------------------------------------------*/
int mars_cec_stop_xmit(
    mars_cec*               p_this    
    )
{   
    if (p_this->xmit.state == XMIT)
    {
        write_reg32(MIS_CEC_TX0, 0);
        mars_cec_do_xmit(p_this);
    }
    return 0;        
}

/*------------------------------------------------------------------
 * Func : mars_cec_start_rcv
 *
 * Desc : start rcv message
 *
 * Parm : p_this   : handle of mars cec 
 *         
 * Retn : 0 : for success, others : fail
 *
 * see also : mars_cec_stop_rcv
 * Notations : 
 *      
 *  Anyone who invoke the function will be blocked until the cec module receives message.   
 *------------------------------------------------------------------*/
int mars_cec_start_rcv(
    mars_cec*               p_this    
    )
{
    if (p_this->rcv.state != WAIT_RCV)
        return -1;
    
    while(p_this->rcv.state != RCV_COMPLETE)
    {        
        //cec_dbg("p_this->rcv.state=%d\n",p_this->rcv.state);
        mars_cec_do_rcv(p_this);
        msleep(100);                
    }
        
    p_this->rcv.state = IDEL;           // Reset to IDEL State
    
    return (p_this->rcv.status==RCV_OK) ? 0 : -1;
}



/*------------------------------------------------------------------
 * Func : mars_cec_stop_rcv
 *
 * Desc : stop rcv message.
 *
 * Parm : p_this   : handle of mars cec 
 *         
 * Retn : 0 : for success, others : fail
 *------------------------------------------------------------------*/
int mars_cec_stop_rcv(
    mars_cec*               p_this    
    )
{    
    if (p_this->rcv.state == RCV)
    {
        write_reg32(MIS_CEC_RX0, 0);
        mars_cec_do_rcv(p_this);        
    }
    
    return 0;
}

/*------------------------------------------------------------------
 * Func : mars_cec_init
 *
 * Desc : init mars ces module
 *
 * Parm : p_this   : handle of mars cec 
 *         
 * Retn : 0 : success, others : fail
 *------------------------------------------------------------------*/
int mars_cec_init(
    mars_cec*               p_this    
    )
{    
    write_reg32(MIS_CEC_CR0, 0x0f);        // Disable CEC
    write_reg32(MIS_CEC_RX0, 0);           // RX Disable 
    write_reg32(MIS_CEC_TX0, 0);           // TX Disable
    write_reg32(MIS_CEC_RT1, 5);           // Retry times = 5
    write_reg32(MIS_CEC_CR2, 0x22);           // Retry times = 5        
    write_reg32(MIS_CEC_TX_DATA2, 0x26);           // Retry times = 5    
   
    memset(p_this, 0, sizeof(mars_cec));    // Reset Xmit function
    return 0;
}


/*------------------------------------------------------------------
 * Func : mars_cec_enable
 *
 * Desc : enable ces module
 *
 * Parm : p_this   : handle of mars cec 
 *        on_off   : 0 : disable, others enable
 *         
 * Retn : 0 for success, others failed
 *------------------------------------------------------------------*/
int mars_cec_enable(
    mars_cec*               p_this,
    unsigned char           on_off
    )
{
    unsigned long val = read_reg32(MIS_CEC_CR0) & 0x3F;        
        
    if (on_off)                
    {   
        cec_info("cec : enabled\n"); 
        write_reg32(MIS_CEC_CR0, val | CEC_MODE_NORMAL);
        p_this->status.enable = 1;                
    }
    else
    {
        cec_info("cec : disabled\n");
        write_reg32(MIS_CEC_CR0, val);
        
        mars_cec_stop_xmit(p_this);
        mars_cec_stop_rcv(p_this);        
        p_this->status.enable = 0;
    }
    return 0;
}





/*------------------------------------------------------------------
 * Func : mars_cec_send_message
 *
 * Desc : load message to mars ces
 *
 * Parm : p_this   : handle of mars cec  
 *        p_data   : message data buffer
 *        data_len : length of message in byte
 *         
 * Retn : 0 : success, others fail  
 *------------------------------------------------------------------*/
int mars_cec_send_message(
    mars_cec*               p_this,     
    unsigned char*          p_data,
    unsigned char           data_len
    )
{
    int i;                    
    
    if (!p_this->status.enable)
        return -1;        
        
    if (p_this->xmit.state != IDEL)
        return -1;        
           
    p_this->xmit.dest           = (p_data[0] & 0xf);
    p_this->xmit.p_data         = &p_data[1];
    p_this->xmit.data_len       = data_len-1;
    p_this->xmit.xmit_data_len  = 0;
    p_this->xmit.state          = WAIT_XMIT;    
    p_this->xmit.status         = XMIT_OK;    
    
#if 1
    cec_dbg("SEND Msg to <%d> : ",p_this->xmit.dest);
    for (i=1; i < data_len; i++)
        cec_dbg("%02x ", p_data[i]);
    cec_dbg("\n");
#endif
    
#if 0               
    return mars_cec_start_xmit(p_this);         
#else    
    
    if (mars_cec_start_xmit(p_this))
    {
        if (p_this->xmit.status == XMIT_TIMEOUT)        
            cec_dbg("xmit timeout\n");        
        else
            cec_dbg("xmit abort\n");
                            
        return -1;
    }
    else
    {   
        cec_dbg("success\n");
        return 0;
    }
#endif            
}    





/*------------------------------------------------------------------
 * Func : mars_cec_rcv_message
 *
 * Desc : receive message from mars ces
 *
 * Parm : p_this    : handle of mars cec  
 *        p_buff    : message data buffer
 *        buff_len  : length of message data buffer in byte
 *        p_rcv_len : received data length
 *         
 * Retn : 0 : success, others fail    
 *------------------------------------------------------------------*/
int mars_cec_rcv_message(
    mars_cec*               p_this,     
    unsigned char*          p_buff,
    unsigned char           buff_len,
    unsigned char*          p_rcv_len
    )
{
    if (!p_this->status.enable)
        return -1;
        
    if (p_this->rcv.state != IDEL || p_buff==NULL)
        return -1;
                
    p_this->rcv.p_buff       = p_buff;
    p_this->rcv.buff_len     = buff_len;
    p_this->rcv.rcv_data_len = 0;
    p_this->rcv.state        = WAIT_RCV;    
    p_this->rcv.status       = RCV_OK;        
    
    if (mars_cec_start_rcv(p_this)<0)
    {
        *p_rcv_len = 0;
        return -1;    
    }        
        
    *p_rcv_len = p_this->rcv.rcv_data_len;
    return 0;    
}    



/*------------------------------------------------------------------
 * Func : mars_cec_set_logical_addr
 *
 * Desc : set logical address of mars ces
 *
 * Parm : p_this   : handle of mars cec 
 *        log_addr : logical address  
 *         
 * Retn : 0 : success, others fail  
 *------------------------------------------------------------------*/
int mars_cec_set_logical_addr(
    mars_cec*               p_this, 
    unsigned char           log_addr    
    )
{
    unsigned long val = read_reg32(MIS_CEC_CR0);
    
    if (log_addr>0x0F) 
    {
        cec_warning("cec : illegal logical address %02x\n", log_addr);
        return -EFAULT;
    }    
        
    val = val & ~0xF;
    write_reg32(MIS_CEC_CR0, val | log_addr);
    
    cec_dbg("cec : logical address = %02x\n", log_addr);    
    
    return 0;            
}    
