

#include <linux/config.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/pci.h>
#include <linux/init.h>
#include <linux/blkdev.h>
#include <linux/delay.h>
#include <linux/interrupt.h>
#include "scsi.h"
#include <scsi/scsi_host.h>
#include <linux/libata.h>
#include <asm/r4kcache.h>
#include <asm/r4kcache.h>

#include "sata_mars.h"
#include "sata_mars_cp.h"
#include "debug.h"

#define DRV_NAME			"SATA_DRV"
#define DRV_VERSION			"0.1"
#define	MARS_SATA_IRQ		2

#define	MARS_SATA_REG_RANGE	0x100

static int mars_sata_match(struct device *dev, struct device_driver *drv);

static u8 mars_ata_check_status(struct ata_port *ap);
void mars_ata_dma_setup(struct ata_queued_cmd *qc);
void mars_ata_dma_start(struct ata_queued_cmd *qc);
u8 mars_ata_dma_status(struct ata_port *ap);
void mars_ata_dma_stop(struct ata_queued_cmd *qc);
void mars_ata_dma_irq_clear(struct ata_port *ap);
static int mars_init_sata (struct device *dev);
//static void mars_remove(struct device *dev);
static int mars_remove(struct device *dev);
void mars_ata_dma_reset(struct ata_port *ap);

static u32 mars_sata_scr_read (struct ata_port *ap, unsigned int sc_reg);
static void mars_sata_scr_write (struct ata_port *ap, unsigned int sc_reg, u32 val);
static void mdio_setting(u8, u16,u8);
static void init_mdio (u8);

struct bus_type sata_bus_type = {
	.name		= "SATA_BUS",
	.match		= mars_sata_match,
};

void sata_sb1_release(struct device * dev)
{
	printk(KERN_INFO "sata_mars: sata_sb1_release() called\n");
}

static struct device sata_sb1 = {
	.bus_id		= "SATA_DEV",
	.release	= sata_sb1_release,
	.bus		= &sata_bus_type,
};

static struct device_driver sata_driver = {
	.name			= DRV_NAME,
	.bus			= &sata_bus_type,
	.probe			= mars_init_sata,
	.remove			= mars_remove,
};

static Scsi_Host_Template mars_sht = {
	.module			        = THIS_MODULE,
	.name			        = DRV_NAME,
	.ioctl			        = ata_scsi_ioctl,
	.queuecommand		    = ata_scsi_queuecmd,
	.eh_strategy_handler    = ata_scsi_error,
	.can_queue		        = ATA_DEF_QUEUE,
	.this_id		        = ATA_SHT_THIS_ID,
	.sg_tablesize		    = ATA_MAX_PRD-1, /*mars limite is 255*/
	.max_sectors		    = ATA_MAX_SECTORS,
	.cmd_per_lun		    = ATA_SHT_CMD_PER_LUN,
	.emulated		        = ATA_SHT_EMULATED,
	.use_clustering		    = ATA_SHT_USE_CLUSTERING,
	.proc_name		        = DRV_NAME,
	.dma_boundary		    = ATA_DMA_BOUNDARY,
	.slave_configure	    = ata_scsi_slave_config,
	.bios_param		        = ata_std_bios_param,
	.ordered_flush		    = 1,
};

static struct ata_port_operations mars_ops = {
	.port_disable   = ata_port_disable,
	.tf_load		= ata_tf_load,
	.tf_read		= ata_tf_read,
	.exec_command   = ata_exec_command,
	.dev_select		= ata_std_dev_select,
	.phy_reset		= sata_phy_reset,
	.check_status   = mars_ata_check_status,
	.bmdma_setup    = mars_ata_dma_setup,
	.bmdma_start    = mars_ata_dma_start,
	.bmdma_stop		= mars_ata_dma_stop,
	.bmdma_status   = mars_ata_dma_status,
	.irq_clear		= mars_ata_dma_irq_clear,
	.qc_prep		= ata_qc_prep,
	.qc_issue		= ata_qc_issue_prot,
	.eng_timeout    = ata_eng_timeout,
	.irq_handler    = ata_interrupt,
	.scr_read		= mars_sata_scr_read,
	.scr_write		= mars_sata_scr_write,
	.port_start		= ata_port_start,
	.port_stop		= ata_port_stop,
	.host_stop		= ata_host_stop,
};

static struct ata_port_info mars_port_info = {
	.sht		= &mars_sht,
	.host_flags	= ATA_FLAG_SATA | ATA_FLAG_SATA_RESET | ATA_FLAG_NO_LEGACY |
			      ATA_FLAG_MMIO | ATA_FLAG_PIO_DMA,
	.pio_mask	= 0x1f,
	.mwdma_mask	= 0x7,
	.udma_mask	= 0x7f,
	.port_ops	= &mars_ops,
};


MODULE_AUTHOR("abevau");
MODULE_DESCRIPTION("host controller driver for Realtek mars-SATA controller");
MODULE_LICENSE("GPL");
MODULE_VERSION(DRV_VERSION);


/**
Tell if a marssata device structure has a matching mars sata device id structure
**/

static int mars_sata_match(struct device *dev, struct device_driver *drv)
{
	return 1;
}


static u32 mars_sata_scr_read (struct ata_port *ap, unsigned int sc_reg)
{
	marsinfo("mars_sata_scr_read\n");
    void __iomem *mmio = (void __iomem *) (ap->ioaddr.scr_addr + (sc_reg * 4));
    return readl(mmio);
}

static void mars_sata_scr_write (struct ata_port *ap, unsigned int sc_reg, u32 val)
{
	marsinfo("mars_sata_scr_write\n");
    void __iomem *mmio = (void __iomem *) (ap->ioaddr.scr_addr + (sc_reg * 4));
    writel(val, mmio);
}

static u8 mars_ata_check_status(struct ata_port *ap)
{
	marsinfo("mars_ata_check_status\n");
    void __iomem *mmio = (void __iomem *) ap->ioaddr.status_addr;
    marslolo("status:%x\n",(u8)readl(mmio)&0xff);
    return (u8)readl(mmio)&0xff;
}

extern int MARS_CP_EN;
extern u32 CP_KEY[];
void mars_ata_dma_setup(struct ata_queued_cmd *qc)
{

	struct ata_port *ap = qc->ap;

	marsinfo("mars_ata_dma_setup\n");


#ifdef SHOW_PRD
	if(0){
		u16 i;
		u32 addr;
		struct scatterlist *sg = qc->sg;
		printk(KERN_WARNING"prd alloc, virt %p, dma %llx\n", ap->prd, (unsigned long long) ap->prd_dma);
		addr = (u32) sg_dma_address(sg);
		addr = cpu_to_le32(addr);
		printk(KERN_WARNING"n_elem=%d\n",qc->n_elem);

		for(i=0;i<qc->n_elem;i++){
			struct ata_prd *prdtmp;
			prdtmp=ap->prd+i;
			printk(KERN_WARNING"prdlist[%d]=addr:0x%08x len:0x%08x\n",i,prdtmp->addr,prdtmp->flags_len);
		}
	}

#endif


	if(MARS_CP_EN)
	{
		u32 reg_data;
		int port_gap=0;;

		marslolo("MARS_CP_EN A\n");
		marslolo("CP_KEY[0]=%x\n",CP_KEY[0]);
		marslolo("CP_KEY[1]=%x\n",CP_KEY[1]);

		writel(CP_KEY[0], (void __iomem *) (ACP_CP0_KEY0+0x10*ap->port_no));
		writel(CP_KEY[1], (void __iomem *) (ACP_CP0_KEY1+0x10*ap->port_no));

		reg_data=readl((void __iomem *) ACP_CTRL);
		if(ap->port_no==1)
		{
			port_gap=10;
		}
		reg_data=reg_data & ~(1<<(5+port_gap)) |(1<<port_gap)|(0x150<<port_gap);
		marslolo("cp reg-a=%x\n",reg_data);

		writel(reg_data, (void __iomem *) ACP_CTRL);
		marslolo("cp reg-b=%x\n",readl((void __iomem *) ACP_CTRL));

	}

	writel((u32)ap->prd_dma, (void __iomem *) (SATA0_PRDPTR+(ap->port_no * MARS_BANK_GAP )));
	ap->ops->exec_command(ap, &qc->tf);
}

void mars_ata_dma_start(struct ata_queued_cmd *qc)
{
	struct ata_port *ap = qc->ap;
	unsigned int rw = (qc->tf.flags & ATA_TFLAG_WRITE);
	u32 dmactl;

	marsinfo("mars_ata_dma_start\n");

	if(rw){
		writel((u32)0x01, (void __iomem *) (SATA0_DMACR+(ap->port_no * MARS_BANK_GAP )));	//alex add
	}else{
		writel((u32)0x02, (void __iomem *) (SATA0_DMACR+(ap->port_no * MARS_BANK_GAP )));	//alex add
	}

	//writel((u32)0x03, (void __iomem *) (SATA0_DMACR+(ap->port_no * MARS_BANK_GAP )));

#ifdef FORCE_SYNC
	mb();
	dma_cache_wback_inv((u32) ap->prd,ATA_PRD_TBL_SZ);
#endif

#ifdef CHG_BUF_LEN
	//printk(KERN_WARNING "sata%u: CHG_BUF_LEN\n",ap->id);
	void __iomem *mmio2 = (void __iomem *) (SATA0_DMAMISC+(ap->port_no * MARS_BANK_GAP));
	writel(readl(mmio2 ) & ~(0x0f<<12)|(0x0a<<12), mmio2);
#endif

	void __iomem *mmio = (void __iomem *) (SATA0_DMACTRL+(ap->port_no * MARS_BANK_GAP));

#ifdef MARS_LOLO_DEBUG
	if(0){
		dmactl=readl(mmio);
		marslolo("chk DMACTRL:0x%x\n",dmactl);
		dmactl=0;
	}
#endif

	dmactl = (1<<21) | (0<<16) | ((qc->n_elem&0xff)<<8) | ((!rw<<3) | 0x01);

	if(MARS_CP_EN)
	{
		marslolo("MARS_CP_EN B\n");
		dmactl|=(1<<6);
	}
	marslolo("dmactl:0x%x\n",dmactl);

#ifdef FORCE_SYNC
	mb();
	dma_cache_wback_inv((u32) ap->prd,ATA_PRD_TBL_SZ);
#endif
	writel(dmactl, mmio);
}

u8 mars_ata_dma_status(struct ata_port *ap)
{
	marsinfo("mars_ata_dma_status\n");
	void __iomem *mmio = (void __iomem *) (SATA0_INTPR+(ap->port_no * MARS_BANK_GAP) );

	if(readl(mmio)&0x80000000)
		return ATA_DMA_INTR;

	return 0;

}

void mars_ata_dma_reset(struct ata_port *ap)
{
	unsigned int reginfo;
	int bank=ap->port_no;
	unsigned int polltime;

	//if(bank==0)	return;

	marsinfo("mars_ata_dma_reset\n");

	reginfo=readl((void __iomem *)CRT_SOFT_RESET1);
	marslolo("0x%08x_a=0x%08x\n",CRT_SOFT_RESET1,reginfo);

	reginfo=readl((void __iomem *)CRT_CLOCK_ENABLE1);
	marslolo("0x%08x_a=0x%08x\n",CRT_CLOCK_ENABLE1,reginfo);

	reginfo=readl((void __iomem *)SB1_MASK);
	marslolo("0x%08x_a=0x%08x\n",SB1_MASK,reginfo);

	/*Mask SB1 sata*/
	reginfo=readl((void __iomem *)SB1_MASK);

	polltime=0;
	writel(reginfo|(0x08<<bank),(void __iomem *)SB1_MASK);
	while(((readl((void __iomem *)SB1_MASK)>>16)&(0x08<<bank))!=(0x08<<bank)){
		udelay(1);
		if(polltime>5000){	/*max waiting 5ms*/
			printk(KERN_WARNING "sata%u: SB1 check fail!\n",ap->id);
			break;
		}
		polltime++;
	}
	marslolo("polltime-c: %u\n",polltime);

	/* reset phy*/
	reginfo=readl((void __iomem *)CRT_SOFT_RESET1);
	writel(reginfo&(~(0x10<<bank)),(void __iomem *)CRT_SOFT_RESET1);

	/*reset SATA MAC clock*/
	reginfo=readl((void __iomem *)CRT_CLOCK_ENABLE1);
	writel(reginfo&(~(0x04<<bank)),(void __iomem *)CRT_CLOCK_ENABLE1);

	/*reset SATA MAC */
	reginfo=readl((void __iomem *)CRT_SOFT_RESET1);
	writel(reginfo&(~(0x04<<bank)),(void __iomem *)CRT_SOFT_RESET1);

	/*release SB1 Mask*/
	reginfo=readl((void __iomem *)SB1_MASK);
	writel(reginfo&(~(0x08<<bank)),(void __iomem *)SB1_MASK);

	/*release SATA MAC*/
	reginfo=readl((void __iomem *)CRT_SOFT_RESET1);
	writel(reginfo|(0x14<<bank),(void __iomem *)CRT_SOFT_RESET1);

	/*release SATA MAC clock*/
	reginfo=readl((void __iomem *)CRT_CLOCK_ENABLE1);
	writel(reginfo|(0x04<<bank),(void __iomem *)CRT_CLOCK_ENABLE1);

	reginfo=readl((void __iomem *)CRT_SOFT_RESET1);
	marslolo("0x%08x_b=0x%08x\n",CRT_SOFT_RESET1,reginfo);

	reginfo=readl((void __iomem *)CRT_CLOCK_ENABLE1);
	marslolo("0x%08x_b=0x%08x\n",CRT_CLOCK_ENABLE1,reginfo);

	reginfo=readl((void __iomem *)SB1_MASK);
	marslolo("0x%08x_b=0x%08x\n",SB1_MASK,reginfo);

	init_mdio(bank);

	reginfo=readl((void __iomem *)(SATA0_ENABLE+bank*MARS_BANK_GAP));
	marslolo("0x%08x-a=0x%08x\n",(SATA0_ENABLE+bank*MARS_BANK_GAP),reginfo);

	writel(0x01, (void __iomem *)(SATA0_ENABLE+bank*MARS_BANK_GAP));

	reginfo=readl((void __iomem *)(SATA0_ENABLE+bank*MARS_BANK_GAP));
	marslolo("0x%08x-b=0x%08x\n",(SATA0_ENABLE+bank*MARS_BANK_GAP),reginfo);


	reginfo=readl((void __iomem *)(SATA0_SCR2+bank*MARS_BANK_GAP));
	marslolo("0x%08x-a=0x%08x\n",(SATA0_SCR2+bank*MARS_BANK_GAP),reginfo);

	writel(0x300,(void __iomem *) (SATA0_SCR2+bank*MARS_BANK_GAP));

	reginfo=readl((void __iomem *)(SATA0_SCR2+bank*MARS_BANK_GAP));
	marslolo("0x%08x_b=0x%08x\n",(SATA0_SCR2+bank*MARS_BANK_GAP),reginfo);

	/*check ide status*/
	reginfo=readl((void __iomem *) (SATA0_CDR7+bank*MARS_BANK_GAP))&0xff;
	marslolo("sata%u: E status-a:0x%02x\n",ap->id,reginfo);
	polltime=0;
	while(reginfo & (ATA_BUSY|ATA_ERR)){
		mdelay(1);
		reginfo=readl((void __iomem *) (SATA0_CDR7+bank*MARS_BANK_GAP))&0xff;

		if(polltime>500){	/*max waiting 0.5 second */
			marslolo("sata%u: E status-b:0x%02x\n",ap->id,reginfo);
			printk(KERN_WARNING "sata%u: phy reset timeout(DMA)!\n",ap->id);
			break;
		}
		polltime++;
	}
	marslolo("polltime-a: %u\n",polltime);
	marslolo("sata%u: E status-c:0x%02x\n",ap->id,reginfo);
}


void mars_ata_dma_stop(struct ata_queued_cmd *qc)
{

	struct ata_port *ap = qc->ap;
	void __iomem *mmio;
	unsigned int polltime;

	marsinfo("sata%u: mars_ata_dma_stop\n",ap->id);


	/*clear DMACTRL go bit */
	mmio = (void __iomem *)(SATA0_DMACTRL+(ap->port_no * MARS_BANK_GAP));

	if((readl(mmio)&0x08)==0){	//read memory(write data to device)
		marsflow("E=>1\n");

#ifdef POLLING_IPF
		if(1){
			void __iomem *mmio2;
			mmio2 = (void __iomem *)(SATA0_INTPR+(ap->port_no * MARS_BANK_GAP));
			polltime=0;
			//printk(KERN_WARNING "sata%u: INTPR-a:0x%08x\n",ap->id,readl(mmio2));
			while((readl(mmio2)|0x80000000)==0){
				mdelay(1);
				if(polltime>10){
                	marsflow("E=>1-1\n");
                	printk(KERN_WARNING "sata%u: DMA write to HD error!\n",ap->id);
                	printk(KERN_WARNING "sata%u: INTPR:0x%08x\n",ap->id,readl(mmio2));
				    break;
				}
				polltime++;
			}
		}
#endif
		writel(0x00, mmio);
	}else{						//write memory(read data from device)
		marsflow("E=>2\n");
		polltime=0;
		while((readl(mmio)&0x01)){
			mdelay(1);
			if(polltime>500){
                marsflow("E=>2-1\n");
                writel(0x00, mmio);
                printk(KERN_WARNING "sata%u: DMA read from HD error !\n",ap->id);

#ifdef SHOW_IPF
		void __iomem *mmio3;
		mmio3 = (void __iomem *)(SATA0_INTPR+(ap->port_no * MARS_BANK_GAP));
		printk(KERN_WARNING "sata%u: INTPR-b:0x%08x\n",ap->id,readl(mmio3));
#endif
                //mars_ata_dma_reset(ap);
			    break;
			}
			polltime++;
		}
		marslolo("polltime-b: %u\n",polltime);
		marsflow("E=>3\n");
	}
	writel(0x00, (void __iomem *) (SATA0_DMACR+(ap->port_no * MARS_BANK_GAP )));

	ata_altstatus(ap);
}

void mars_ata_dma_irq_clear(struct ata_port *ap)
{
	void __iomem *mmio;

	marsinfo("mars_ata_dma_irq_clear\n");

	/*clear interrupt */
	mmio = (void __iomem *)(SATA0_INTPR+(ap->port_no * MARS_BANK_GAP));
	readl(mmio);
	writel(readl(mmio), mmio);


}

static void mars_ata_std_ports(struct ata_ioports *ioaddr)
{
	marsinfo("mars_ata_std_ports\n");

	ioaddr->data_addr       = ioaddr->cmd_addr + (ATA_REG_DATA<<2);
	ioaddr->error_addr      = ioaddr->cmd_addr + (ATA_REG_ERR<<2);
	ioaddr->feature_addr    = ioaddr->cmd_addr + (ATA_REG_FEATURE<<2);
	ioaddr->nsect_addr      = ioaddr->cmd_addr + (ATA_REG_NSECT<<2);
	ioaddr->lbal_addr       = ioaddr->cmd_addr + (ATA_REG_LBAL<<2);
	ioaddr->lbam_addr       = ioaddr->cmd_addr + (ATA_REG_LBAM<<2);
	ioaddr->lbah_addr       = ioaddr->cmd_addr + (ATA_REG_LBAH<<2);
	ioaddr->device_addr     = ioaddr->cmd_addr + (ATA_REG_DEVICE<<2);
	ioaddr->status_addr     = ioaddr->cmd_addr + (ATA_REG_STATUS<<2);
	ioaddr->command_addr    = ioaddr->cmd_addr + (ATA_REG_CMD<<2);
}



static void mdio_setting(u8 address, u16 data,u8 bank)
{
	marsinfo("mdio_setting\n");
	void __iomem *mmio = (void __iomem *) SATA0_MDIOCTRL+(bank*MARS_BANK_GAP);
	writel((data<<16) | (address<<8) | 0x01, mmio);
	//while(!(readl((void __iomem *) SATA0_MDIOCTRL)&0x00000010));
	udelay(50);


}

/*
static u16 mdio_getting(u16 address,u8 bank)
{
	marsinfo("mdio_getting\n");
	void __iomem *mmio = (void __iomem *) (SATA0_MDIOCTRL+(bank*MARS_BANK_GAP));
	writel(address<<8, mmio);
	//while(!(readl((void __iomem *) SATA0_MDIOCTRL)&0x00000010));
	udelay(50);
	return (u16)(readl(mmio)>>16);

}
*/
static void init_mdio (u8 bank)
{
#ifdef HEAT_DOWN12p5
	mdio_setting(SATA_PHY_BPCR, 0x6900,bank);
#endif
#ifdef HEAT_DOWN25
	mdio_setting(SATA_PHY_BPCR, 0x6d00,bank);
#endif

	mdio_setting(SATA_PHY_TCR0, 0x623C,bank);
	mdio_setting(SATA_PHY_RCR0, 0x7C65,bank);

	mdio_setting(SATA_PHY_PCR,  0x3020,bank);
	mdio_setting(SATA_PHY_RCR1, 0x12DE,bank);
	mdio_setting(SATA_PHY_RCR2, 0xC2B8,bank);
	mdio_setting(SATA_PHY_TCR1, 0x2900,bank);

}

static int mars_init_sata (struct device *dev)
{
	unsigned int bank_num;
	unsigned int bank=2.;
	struct ata_probe_ent *probe_ent;

	marsinfo("mars_init_sata\n");

	probe_ent = (struct ata_probe_ent *)ata_probe_ent_alloc(dev, &mars_port_info);

	if (!probe_ent)
	{
		return -ENOMEM;
	}

	probe_ent->n_ports      = bank;	//2 bank
	probe_ent->irq          = 2 + MARS_SATA_IRQ;
	probe_ent->irq_flags    = SA_SHIRQ;

#ifdef PRIORITY_SATA0
	if(1){ /*pull up priority 0f SATA0 */
		printk(KERN_WARNING "PRIORITY_SATA0\n");
		writel(0x100, (void __iomem *)SB1_PRIORITY2);
		writel(0x18f, (void __iomem *)SB1_PRIORITY5);
	}
#endif

#ifdef LOW_PRIORITY_SATA0
	if(1){ /*pull up priority 0f SATA0 */
		printk(KERN_WARNING "LOW_PRIORITY_SATA0\n");
		writel(0x1FF, (void __iomem *)SB1_PRIORITY2);
		writel(0x18f, (void __iomem *)SB1_PRIORITY4);
	}
#endif

#ifdef CHK_SATA_CLK
	if(1){
		unsigned int reginfo;
		reginfo=readl((void __iomem *)CRT_CLOCK_ENABLE1);
		if(reginfo&0x08==0)
			bank=1;
	}
#endif

	for(bank_num=0; bank_num < probe_ent->n_ports;bank_num++)
	{
		if (!request_mem_region(MARS_SATA0_BASE+(bank_num*MARS_BANK_GAP), MARS_SATA_REG_RANGE, DRV_NAME))
			goto err_out;

		probe_ent->port[bank_num].cmd_addr         = (unsigned long) SATA0_CDR0+(bank_num*MARS_BANK_GAP);
		probe_ent->port[bank_num].altstatus_addr   = (unsigned long) SATA0_CLR0+(bank_num*MARS_BANK_GAP);
		probe_ent->port[bank_num].ctl_addr         = (unsigned long) SATA0_CLR0+(bank_num*MARS_BANK_GAP);
		probe_ent->port[bank_num].scr_addr         = (unsigned long) SATA0_SCR0+(bank_num*MARS_BANK_GAP);

		mars_ata_std_ports(&probe_ent->port[bank_num]);

		writel(0x0, (void __iomem *) (SATA0_TIMEOUT+(bank_num*MARS_BANK_GAP)));

#ifdef PINGOUT_FORCE_PHY_G1
		writel(0x0000020f, (void __iomem *) (SATA0_PINOUT+(bank_num*MARS_BANK_GAP)));
#endif

#ifdef PINGOUT_FORCE_PHY_G2
		writel(0x0000030f, (void __iomem *) (SATA0_PINOUT+(bank_num*MARS_BANK_GAP)));
#endif
		init_mdio(bank_num);

		/*force perform interface communication initialzation*/
		printk(KERN_WARNING "sata%u: FORCE_COMRESET\n",bank_num+1);

#ifdef PHY_AUTO
		//writel(0x00000301, SATA0_SCR2+(bank_num*MARS_BANK_GAP));
		//writel(0x00000300, SATA0_SCR2+(bank_num*MARS_BANK_GAP));
#endif

#ifdef PHY1G5
		//writel(0x00000311, SATA0_SCR2+(bank_num*MARS_BANK_GAP));
		//writel(0x00000310, SATA0_SCR2+(bank_num*MARS_BANK_GAP));
#endif

#ifdef PHY3G0
		//writel(0x00000321, SATA0_SCR2+(bank_num*MARS_BANK_GAP));
		//writel(0x00000320, SATA0_SCR2+(bank_num*MARS_BANK_GAP));
#endif

		//printk(KERN_WARNING "sata%u: SATA0_SCR2=0x%08x\n",bank_num+1,readl(SATA0_SCR2+(bank_num*MARS_BANK_GAP)));
		//printk(KERN_WARNING "sata%u: SATA0_SCR0=0x%08x\n",bank_num+1,readl(SATA0_SCR0+(bank_num*MARS_BANK_GAP)));

	    /*enable SATA int*/
		writel(0x01, (void __iomem *) (SATA0_ENABLE+(bank_num*MARS_BANK_GAP)));

	}
	ata_device_add(probe_ent);
	kfree(probe_ent);

	return 0;

err_out:
	//release_region(MARS_SATA0_BASE, MARS_SATA_GAP*14);
	release_region(MARS_SATA0_BASE, MARS_SATA_REG_RANGE);
	return -ENOMEM;

}

//static void mars_remove(struct device *dev)
static int mars_remove(struct device *dev)

{
	struct ata_host_set *host_set = dev_get_drvdata(dev);
	struct ata_port *ap;
	u32 i;

	marsinfo("mars_remove\n");

	for(i=0; i<host_set->n_ports; i++){
		ap = host_set->ports[i];
		scsi_remove_host(ap->host);
	}

	free_irq(host_set->irq, host_set);

	for(i=0; i<host_set->n_ports; i++){
		ap = host_set->ports[i];

		ata_scsi_release(ap->host);

		scsi_host_put(ap->host);
	}

	if(host_set->ops->host_stop)
		host_set->ops->host_stop(host_set);

	kfree(host_set);

	release_region(MARS_SATA0_BASE, MARS_SATA_REG_RANGE);
	//release_region(MARS_SATA0_BASE+MARS_BANK_GAP*host_set->n_ports, MARS_SATA_REG_RANGE);

	dev_set_drvdata(dev, NULL);
	return 0;
}

static int __init mars_init(void)
{
	int error;

    marsinfo("mars_init\n");
	error = bus_register(&sata_bus_type);
	if (error) return error;
	else {
		device_register(&sata_sb1);
		error = driver_register(&sata_driver);
		return error;
	}
}

static void __exit mars_exit(void)
{
	driver_unregister(&sata_driver);
	device_unregister(&sata_sb1);
	bus_unregister(&sata_bus_type);

}

module_init(mars_init);
module_exit(mars_exit);

