#!/usr/local/bin/perl

use strict;

sub get_latest_change_num; # (void) -> $change_num
sub get_branch_spec; # (void) -> $branch_spec
sub get_checked_out_count; # (void) -> $count


my $branch_spec = get_branch_spec();
my $checked_out_count = get_checked_out_count();
my $checked_out_string;
if ($checked_out_count > 0)
  {
    $checked_out_string = "[+";
    if ($checked_out_count == 1)
      {
        $checked_out_string .= "1 locally modified file";
      }
    else
      {
        $checked_out_string .= "$checked_out_count locally modified files";
      }
    $checked_out_string .= "]";
  }
else
  {
    $checked_out_string = "";
  }

my $max_change_num = get_latest_change_num();
my $min_change_num = 1;

while (1)
  {
    if ($max_change_num < $min_change_num)
      {
        print STDOUT "<<no change set matches>>$checked_out_string";
        exit 0;
      }

    my $try_num = int(($min_change_num + $max_change_num + 1) / 2);
    my $try_text = `p4 sync -n \@$try_num`;
    my @mismatching_files = split(/\n/, $try_text);
    while (1)
      {
        if ($#mismatching_files < 0)
          {
            print STDOUT "$branch_spec\@$try_num$checked_out_string";
            exit 0;
          }

        my $this_mismatch = shift(@mismatching_files);
        while (($#mismatching_files >= 0) &&
               ($this_mismatch =~
                m/^(\w|\/|\_|\.|\-)+\#(\d+) \- added as (.*)$/))
          {
            $this_mismatch = shift(@mismatching_files);
          }

        if ($this_mismatch =~ m/^\.\.\. \/\/depot(.*) \- must resolve \#.* before submitting$/)
          {
            next;
          }
        if (!($this_mismatch =~ m/^(\w|\/|\_|\.|\-)+\#(\d+) \- (.*)$/))
          {
            die "Can't understand information from Perforce about a change " .
                "set mismatch (`$this_mismatch')";
          }
        my $other_num = $2;
        my $tail = $3;

        if ($tail =~ m/^is opened /)
          {
            next;
          }

        my $file;
        if ($tail =~ m/^updating (.*)$/)
          {
            $file = $1;
          }
        elsif ($tail =~ m/^deleted as (.*)$/)
          {
            $other_num = 0;
            $file = $1;
          }
        elsif ($tail =~ m/^added as (.*)$/)
          {
            $file = $1;
          }
        else
          {
            die "Can't understand information from Perforce about a change " .
                "set mismatch (`$tail')";
          }
        my $have_num;
        if ($tail =~ m/^added as (.*)$/)
          {
            $have_num = 0;
          }
        else
          {
            my $have_text = `p4 have $file` ||
                    die "Unable to get current revision number of file " .
                    "`$file' from Perforce";
            if (!($have_text =~ m/^(\w|\/|\_|\.|\-)+\#(\d+) \- (.*)$/))
              {
                die "Can't understand information from Perforce about the " .
                    "current revision number of file `$file'";
              }
            $have_num = $2;
          }
        if ($have_num > $other_num)
          {
            $min_change_num = $try_num + 1;
          }
        else
          {
            $max_change_num = $try_num - 1;
          }

        my $log_text = `p4 filelog $file`;
        my @log_lines = split(/\n/, $log_text);
        while ($#log_lines >= 0)
          {
            my $log_entry = shift(@log_lines);
            if (!($log_entry =~ m/^\.\.\. \#(\d+) change (\d+) /))
              {
                next;
              }
            my $file_num = $1;
            my $change_num = $2;
            if ($file_num <= $have_num)
              {
                if ($min_change_num < $change_num)
                  {
                    $min_change_num = $change_num;
                  }
              }
            else
              {
                if ($max_change_num >= $change_num)
                  {
                    $max_change_num = ($change_num - 1);
                  }
              }
          }

        last;
      }
  }

sub get_latest_change_num # (void) -> $change_num
  {
    my $change_text = `p4 changes -m 1` ||
            die "Unable to get the latest change number from Perforce";
    if (!($change_text =~ m/^Change (\d+) on /))
      {
        die "Can't understand information from Perforce about the latest " .
            "change number";
      }
    return $1;
  }

sub get_branch_spec # (void) -> $branch_spec
  {
    my $spec_text = `p4 client -o` ||
            die "Unable to get the client specification from Perforce";
    my @spec_lines = split(/\n/, $spec_text);
    if ($#spec_lines < 0)
      {
        die "Can't understand information from Perforce about the client " .
            "specification";
      }
    my $line_num = $#spec_lines;
    while (($line_num > 0) &&
           ($spec_lines[$line_num] =~ m/^(\s)*-\/\/depot\/((\w|\/|\_|\.|\-)+)\/\.\.\.\s/))
      {
        --$line_num;
      }
    if (!($spec_lines[$line_num] =~ m/^(\s)*\/\/depot\/((\w|\/|\_|\.|\-)+)\/\.\.\.\s/))
      {
        die "Can't understand information from Perforce about the client " .
            "specification";
      }
    return "//depot/" . $2 . "/...";
  }

sub get_checked_out_count # (void) -> $count
  {
    my $text = `p4 opened`;
    my @lines = split(/\n/, $text);
    return $#lines + 1;
  }
