/*
 * This is the interface to the webhooks shared library, which allows shared
 * libraries to add hooks to the BroadNAS web server.
 *
 * Copyright 2003-2004, Broadcom Corporation
 * All Rights Reserved.
 *
 * This is UNPUBLISHED PROPRIETARY SOURCE CODE of Broadcom Corporation;
 * the contents of this file may not be disclosed to third parties, copied
 * or duplicated in any form, in whole or in part, without the prior
 * written permission of Broadcom Corporation.
 */

#ifndef WEBHOOKS_H
#define WEBHOOKS_H

#include "typedefs.h"
#include <stdio.h>
#include <stdarg.h>
#include <stddef.h>


/*
 *  There are three types of web hooks: web hook functions, web accessable
 *  variables, and post file handlers.
 */

/*
 *      Web Hook Functions
 *      ------------------
 *
 *  A web hook function allows a web page to call a function in the server.
 *  The following is the interface for adding such hook functions.
 *
 *  The webs_t type provides a handle for the hook function to access the web
 *  server state.  Using such a handle it can get dynamic parameters that were
 *  passed to the server at run time and it can write to the web page that will
 *  be sent to the browser.  Note that because of limitations in C's ability to
 *  encapsulate interfaces, by looking at this file you can easily see that
 *  webs_t is actually a FILE * and you could use it interchangably with that
 *  type, but you should never do that.  You should only use it as the
 *  interface described here allows because in the future it might change to a
 *  different type.
 *
 *  The hook_type type defines the type that each hook function must have. 
 *  When the hook is activated, it is passed the eid (not currently used), the
 *  webs_t (the server state handle), and an argc and argv to give the static
 *  parameters of the hook call.
 */
typedef FILE *webs_t;
typedef int (*hook_type)(int eid, webs_t wp, int argc, char **argv);


/*
 *  The register_web_hook() function is used to register a new web hook by
 *  name.  This function should be used only at initialization time, when a
 *  library is loaded, to register all the hooks.  Once the server is
 *  initialized, this function would normally not be used again.
 *
 *  The lookup_web_hook() function is used by the server when a web hook
 *  reference is found in a web page, to find the function it needs to call.
 */
extern void register_web_hook(const char *hook_name, hook_type hook_function);
extern hook_type lookup_web_hook(const char *hook_name);

/*
 *  The following three functions are for internal use and should never be
 *  called directly from outside this module.  They are exposed here only so
 *  that they can be used in the macros below.
 */
extern void init_cgi(char *query, size_t init_count);
extern char * get_cgi(char *name);
extern void set_cgi(char *name, char *value);

/*
 *  The following four macros provide the four ways a hook can communicate with
 *  the web server state.  Note that each requires a wp parameter of type
 *  webs_t.
 *
 *  The websWrite() macro allows the hook to send output to the web page.  This
 *  output is typically dynamically-generated Javascript data or code to be
 *  used by the web page.
 *
 *  The vwebsWrite() macro is similar but uses a va_arg parameter.  It is to
 *  websWrite() as vfprintf() is to fprintf().
 *
 *  The websGetVar() macro allows a hook to read a dynamic parameter passed to
 *  the web server.  It takes a default value argument in addition to the
 *  variable name.
 *
 *  Finally, the websError() macro allows the hook to notify the server that a
 *  serious error has occurred.  This is typically only used when the hook is
 *  called incorrectly, is out of memory, or has some similar serious internal
 *  error.  Errors that can be caused by end users normally would call
 *  Javascript error handling functions to gracefully notify the user rather
 *  than using this macro.
 */
#define websWrite(wp, fmt, args...) ({ int TMPVAR = fprintf(wp, fmt, ## args); fflush(wp); TMPVAR; })
#define vwebsWrite(wp, fmt, arg) ({ int TMPVAR = vfprintf(wp, fmt, arg); fflush(wp); TMPVAR; })
#define websGetVar(wp, var, default) (get_cgi(var) ? : default)
#define websError(wp, code, msg, args...) fprintf(wp, msg, ## args)
extern int ejArgs(int argc, char **argv, char *fmt, ...);


/*
 *      Web Accessable Variables
 *      ------------------------
 *
 *  The web server provides an interface to the web pages to read and write
 *  certain variables.  In nearly all cases, these variables map to NVRAM
 *  variables, but there is no requirement that they do.
 *
 *  Each web accessable variable is described by a structure of type
 *  var_hook_info.  Different libraries can provide access to different
 *  variables.  For each variable, the library that provides access fills in
 *  the var_hook_info structure for that variable and registers the information
 *  with the webhooks library.
 *
 *  The var_hook_info structure has six fields.  The ``name'' field is the name
 *  used internally for the variable.  It is this name that is used by the web
 *  pages to pass the variable information to the server.  The ``longname''
 *  field is the human-readable version of the name.  It is this string that is
 *  used when referring to the variable when communicating with the user, as in
 *  error messages about the variable.  The third field is the ``validate''
 *  field.  This is a pointer to the function that is called when the variable
 *  is being set to a new value.  It is somewhat mis-named -- rather than just
 *  validating that the new value is OK for that variable, it also must take
 *  whatever action is necessary to actually set the variable to the new value
 *  if it is valid.  For example, for a var_hook_info structure corresponding
 *  to an integer NVRAM variable, the validation function would have to check
 *  that the new value was an integer, issue an error if it was not, and set
 *  the NVRAM variable if it was valid.  The ``argv'' field provides additional
 *  parameters for the ``validate'' function.  This is to allow a single
 *  validate function to be used for several different variables with similar
 *  but different validation criteria.  For example, there is a
 *  validate_choice() function that can be used as the validation function for
 *  any NVRAM variable that is allowed to be set to only a small, finite
 *  pre-devined list of choices by setting argv to the list of legal values.
 *  The fourth field is the nullok field.  This field is a boolean that defines
 *  whether or not an empty value is legal for that variable.  Finally, the
 *  restart_needed_bits is used to specify what running programs need to be
 *  restarted if the specified variable is changed.  For example, if a variable
 *  that specifies whether or not NFS is enabled might require restarting the
 *  NFS daemon but not the httpd, samba, or other services.  By waiting until
 *  all variables are updated before restarting anything, the server can avoid
 *  multiple restarts of the same service when more than one variable relating
 *  to that service is changed at the same time and it can also avoid
 *  restarting the web server itself while it is in the middle of processing
 *  requests from the browser, all of which would be impossible if the restarts
 *  were done directly by the validation functions instead of using these bits.
 *  The restart_needed_bits field is, as the name implies, a bit mask with each
 *  bit corresponding to a different service.  The bits are defined by
 *  enumeration constants later in this file.
 */

typedef struct var_hook_info var_hook_info;

struct var_hook_info
{
	char *name;
	char *longname;
	void (*validate)(webs_t wp, char *value, var_hook_info *v);
	char **argv;
	bool nullok;
	u32 restart_needed_bits;
};

/*
 *  The following enumeration constants and #define of identifiers with the
 *  RESTART_ prefix define the meaning of various bits in the
 *  restart_needed_bits of the var_hook_info structure.
 */
enum
{
	RESTART_SAMBA = 0x1,
	RESTART_NFS = 0x2,
	RESTART_HTTPD = 0x4,
	RESTART_DNS = 0x8,
	RESTART_DHCPD = 0x10,
	RESTART_UPNP = 0x20,
	RESTART_NAS = 0x40,
	RESTART_SYSLOG = 0x80,
	RESTART_NETWORKING = 0x100,
	RESTART_UPNP_AV = 0x200
};

#define RESTART_ALL \
		(RESTART_SAMBA | RESTART_NFS | RESTART_HTTPD | RESTART_DNS | \
		 RESTART_DHCPD | RESTART_UPNP | RESTART_NAS | RESTART_SYSLOG | \
		 RESTART_NETWORKING | RESTART_UPNP_AV)

/*
 *  The ARGV() macro is a helper utility for packaging up arguments for the
 *  argv field of the var_hook_info structure.  For example the argv field
 *  could be statically initialized with the code fragment ARGV("a", "b", "c")
 *  to produce an argv field of three parameters, "a", "b", and "c".
 *
 *  The XSTR() macro is a helper macro for making an integer value from a macro
 *  into a string, to be used in an ARGV() call.  For example, if MY_MIN was
 *  defined in some header file to be the integer 17 and MY_MAX was defined in
 *  that header file to be the integer 37, then
 *  ARGV(XSTR(MY_MIN - 10), XSTR(MY_MAX - 10)) would give an argv of two
 *  strings, "7" and "27".
 *
 *  The STR() macro is a helper macro used by the XSTR() macro and isn't
 *  intended to be used by itself.
 *
 *  The char_t macro is for legacy code that uses this instead of char.
 */
#define ARGV(args...) ((char *[]) { args, NULL })
#define XSTR(s) STR(s)
#define STR(s) #s
typedef char char_t;

/*
 *  The register_web_accessable_variable() function is used at library init
 *  time to register the variables whose interface is provided by that library.
 *  It should never be called after initialization is finished and the server
 *  is active.
 *
 *  The web_accessable_variable_count() and get_web_accessable_variable_info()
 *  functions allow the table of all registered variable information to be
 *  accessed.  The former function gives the number of entries in the table and
 *  the later provides a pointer to any particular entry.
 */
extern void register_web_accessable_variable(const var_hook_info *var_info);
extern size_t web_accessable_variable_count(void);
extern var_hook_info *get_web_accessable_variable_info(size_t var_num);

/*
 *  The hook_set_var() function is used by variable validation functions to
 *  actually set a new value for a variable once it has been determined that
 *  the new value is valid.  Note that this can only be used for variables that
 *  map to NVRAM variables.  It doesn't do a commit of the NVRAM changes
 *  itself; instead, it sets the commit_needed flag so that the server can do
 *  the commits later, so only one commit is needed if multiple variables are
 *  being set at the same time.  It checks to see if the new value is equal to
 *  the old one and skips the setting of the variable and commit flag if so to
 *  avoid unnecessary commits when there is no real change.
 *
 *  The invalid_variable() function is used by variable validation functions to
 *  report attempts to set variables to invalid values.  The message format is
 *  treated as a printf() format and the optional arguments correspond to that
 *  to allow arbitrary error messages.  The variable name and the illegal value
 *  it was being set to should not be included in the message because they are
 *  automatically added to what is presented to the user by the
 *  invalid_variable() function.
 */
extern void hook_set_var(const var_hook_info *to_set, const char *new_value);
extern void invalid_variable(const var_hook_info *the_variable,
		char *invalid_value, webs_t wp, const char *message_format, ...);

/*
 *  The following are some useful utility functions for variable validation.
 *
 *  The validate_list() function allows validation of a variable whose value in
 *  NVRAM is required to be a space-separated list of values, where each
 *  individual value is validated by the ``valid'' function parameter.  The
 *  list is represented in the interface to the web browser as a series of
 *  individual variables named <name>0, <name>1, etc. where <name> is the
 *  ``name'' field of the variable.  The ``valid'' function is given the webs_t
 *  pointer as its first argument, a pointer to the individual item value as
 *  its second parameter, and a pointer to the overall variable (for use in
 *  error reporting) as its third parameter.  It should return FALSE if and
 *  only if the value is invalid.  Invalid values are omitted and the valid
 *  ones are concatedated, with a space between each, to form the new value for
 *  the NVRAM variable.  Note that the validate_list() function doesn't have
 *  the right type to be put directly into the ``validate'' field of a
 *  var_hook_info structure because the ``valid'' parameter can't be specified,
 *  but a simple wrapper function can be created that simply calls
 *  validate_list() and passes it the appropriate ``valid'' function.  There
 *  would be one such wrapper function for each ``valid'' function.
 *
 *  The valid_choice() function checks to see whether the new value is one of
 *  those listed in the argv field of the variable info, and if not reports the
 *  error and returns FALSE.
 *
 *  The validate_choice() function uses the valid_choice() function to check
 *  validity, and if the value is valid, sets the variable to the new value.
 *
 *  The valid_range() function checks to see that the new value is a decimal
 *  integer in the range specified by the argv field of the var_hook_info
 *  structure.  The argv field should contain two parameters, first the decimal
 *  string for the low end of the range and second the decimal string for the
 *  high end of the range.  For example ARGV("5", "7") for argv would specify
 *  that the values 5, 6, and 7 were legal for that variable but no other
 *  values were.
 *
 *  The validate_range() function checks that the new value is only digits and
 *  and an optional + or - prefix.  If so, it uses the valid_range() function
 *  to check that the value was in the proper range.  If everything checks out
 *  OK, it sets the variable to the new value.
 *
 *  The valid_name() function checks to see that the new value is a name with a
 *  number of characters specified by minimum and maximum values in argv.  For
 *  example, ARGV("12", "15") would require the name to be between 12 and 15
 *  characters long.
 *
 *  The validate_name() function uses the valid_name() function to check
 *  validity, and if the value is good, sets the variable to the new value.
 *
 *  The valid_ipaddr() function checks to see that the new value is a valid IP
 *  address.
 *
 *  The validate_ipaddr() function uses the valid_ipaddr() function to check
 *  validity, and if the new value is valid, sets the variable to this new
 *  value.
 *
 *  The validate_ipaddrs() function uses the validate_list() function with the
 *  valid_ipaddr() function to validate a list of IP addresses.
 */
extern void validate_list(webs_t wp, char *value, var_hook_info *v,
						  bool (*valid)(webs_t, char *, var_hook_info *));
extern bool valid_choice(webs_t wp, char *value, var_hook_info *v);
extern void validate_choice(webs_t wp, char *value, var_hook_info *v);
extern bool valid_range(webs_t wp, char *value, var_hook_info *v);
extern void validate_range(webs_t wp, char *value, var_hook_info *v);
extern int valid_name(webs_t wp, char *value, var_hook_info *v);
extern void validate_name(webs_t wp, char *value, var_hook_info *v);
extern bool valid_ipaddr(webs_t wp, char *value, var_hook_info *v);
extern void validate_ipaddr(webs_t wp, char *value, var_hook_info *v);
extern void validate_ipaddrs(webs_t wp, char *value, var_hook_info *v);


/*
 *  The websJavascriptOut() function is a utility function for writing
 *  Javascript code out in an HTML context.  It automatically puts the
 *  specified code in a <script>...</script> wrapper.
 *
 *  The websJavascriptCall() function is a utility function that is even more
 *  specific, for when a call is to be made to a Javascript function with a
 *  single string argument.
 */
extern int websJavascriptOut(webs_t wp, char *format, ...);
extern int websJavascriptCall(webs_t wp, char *function, char *format, ...);


/*
 *      Post File Handlers
 *      ------------------
 *
 *  A post file handler is a function that handles a file being sent to the
 *  server by the browser through a POST operation.  The server uses the
 *  ``name'' field in the post to determine which handler to use.  Note that
 *  this is not the ``filename'' field which gives the name of the file as the
 *  user sees it -- rather, the ``name'' field in the post operation specifies
 *  the name of the HTML input item associated with the file, which is
 *  something that is not normally directly visible to the user.  The
 *  interfaces in this file allow post file handlers to be registered by name
 *  and looked up by name.
 *
 *  The post_file_handler_type type defines the type that each post file
 *  handler function must have.  When it is activated, it is passed the URL in
 *  the url parameter, a pointer to the stream from which the data for the file
 *  may be read in the ``stream'' parameter, and a number of bytes in the file
 *  in the ``len'' parameter.
 *
 *  The register_post_file_handler() function is called to register a post file
 *  handler.  All such calls should be made during initialization time, not
 *  after the web server is servicing requests.
 *
 *  The lookup_post_file_handler() is for use by the web server, to look up the
 *  handler for a particular file post operation.
 */
typedef void (*post_file_handler_type)(char *url, FILE *stream, int len);

extern void register_post_file_handler(const char *handler_name,
		post_file_handler_type handler_function);
extern post_file_handler_type lookup_post_file_handler(
		const char *handler_name);


/*
 *      General Hook Support
 *      --------------------
 *
 *  The following six variables can be used by hooks to communicate with the
 *  core server code.  They can be used by hook functions, validation functions
 *  for web accessable variables, and post file handlers.
 *
 *  The restart_needed_bits are a bitfield specifying what services need to be
 *  restarted.  They are normally set automatically from the
 *  restart_needed_bits field of the var_hook_info structure by the
 *  hook_set_var(), but they can be set manually to force restarts.  Note that
 *  hooks should only or-in bits to this field, never clear any of the bits
 *  because that might wipe out information for other restarts that are needed
 *  but not yet complete.
 *
 *  The reboot_needed variable is similar, but with just one bit that tells the
 *  server whether a reboot of the entire system should be done once the
 *  current web requests have been serviced.
 *
 *  The complete_restart_needed global is similar to the reboot_needed variable
 *  but instead of indicating a need to reboot the system, it just indicates a
 *  need to shutdown and restart all user-level services.
 *
 *  The commit_needed flag is used to indicate that at least one NVRAM variable
 *  has been changed so once the current request has finished being serviced,
 *  an NVRAM commit operation should take place.
 *
 *  The non_nvram_changes_made flag is used to indicate that even if the
 *  commit_needed flag has not been set, some state has been updated.  This is
 *  used by the server to communicate results of variable setting back to the
 *  browser -- the server indicates whether or not any state has been changed.
 *
 *  The end_of_page_hook pointer is used to tell the server that a special
 *  function should be called when the serving of the current page is complete.
 *  This is used for special functionality, such as a soft reboot, that is
 *  sometimes required as a result of executing a hook but which shouldn't be
 *  done in the middle of serving the page because it may cause the web server
 *  to go down before it has completed serving the page.
 */
extern u32 restart_needed_bits;
extern bool reboot_needed;
extern bool complete_restart_needed;
extern bool commit_needed;
extern bool non_nvram_changes_made;
extern void (*end_of_page_hook)(void);


#endif /* WEBHOOKS_H */
