
/*
   raidlib.c : Multiple Devices tools for Linux
               Copyright (C) 1994-96 Marc ZYNGIER
	       <zyngier@ufr-info-p7.ibp.fr> or
	       <maz@gloups.fdn.fr>
	       Copyright (C) 1998 Erik Troan
	       <ewt@redhat.com>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.
   
   You should have received a copy of the GNU General Public License
   (for example /usr/src/linux/COPYING); if not, write to the Free
   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  
*/

#include "common.h"
#include "parser.h"
#include "raidlib.h"
#include "version.h"

#include <string.h>
#include <popt.h>
#include <asm/page.h>
#include <sys/sysmacros.h>

#ifndef PAGE_SIZE
#define PAGE_SIZE (sizeof(long)*1024)
#endif

mdu_version_t md_ver;

#define OLD_MDTOOLS ((md_ver.major == 0) && (md_ver.minor < 0.50))

static int save_errno = 0;

md_cfg_entry_t *cfg_head = NULL, *cfg = NULL;
int do_quiet_flag = 0;

static int fake_stat_from_table(const char *file_name, struct stat *buf);

int open_or_die (char *file)
{
  int fd;

  fd = open(file, O_RDWR);
  if (fd == -1) {
    if (errno == EACCES)
        fprintf(stderr, "need to be run as root.\n");
    else if (errno == ENOSYS)
        fprintf(stderr, "kernel has no RAID support.\n");
    else if (errno == EEXIST)
        fprintf(stderr, "device already running!\n");
    else if (errno == ENOENT)
        fprintf(stderr, "no MD device file in /dev.\n");
    else
    	perror (file);
    exit (EXIT_FAILURE);
  }

  return fd;
}

int getpageshift(void)
{
   int i,pagesize,pageshift=0;

   pagesize=getpagesize();
   for (i=1; i <= 8*sizeof(int); i++) {
       if( pagesize & (1UL << i)) {
           pageshift += i;
       }
   }
   return pageshift;
}

/*
 * converts a size in bytes to 'factor' metrics, which is a
 * kernel-internal way of dealing with chunk sizes and stuff.
 * It's the number of pages within the given number.
 */
static int s2f (int bytes)
{
    int factor, c, i;
    int kbytes, pageshift;

    if (bytes % 1024) {
	fprintf (stderr, "chunk_size must be an integral number of k\n");
	return 0;
    }
    kbytes = bytes >> 10;
   
    pageshift=getpageshift();
    factor = kbytes >> (pageshift - 10);
 
    for (i=1, c=0; i<kbytes; i*=2)
	if (factor & i) c++;
    
    if ((kbytes*MD_BLK_SIZ) % PAGE_SIZE || c!=1) {
	fprintf (stderr, "Cannot handle %dk chunks. Defaulting to %dk\n",
		 kbytes, 1 << (pageshift-10));
	return (0);
    }

    for (i=0; !(factor & (1 << i)); i++);
    return (i);
}

__u32 get_random(void)
{
        __u32 num;
        int fd;

        fd = open("/dev/urandom", O_RDONLY);
        if (fd != -1 && read(fd, &num, 4) == 4) {
#if DEBUG
                printf("raid set magic: %x\n", num);
#endif
                close(fd);
        } else {
                num = rand();
#if DEBUG
                printf("raid set magic (pseudo-random): %x\n", num);
#endif
        }
        return num;
}

static int do_mdrun (int fd, char *dev, mdu_param_t *param) {
    int rc;
    int pers;

	/* Old versions of md (< 0.50) used this instead. Thank Ingo
	   for fixing it. */
#define MD_PERSONALITY_SHIFT    16

/* These are the ioctls for md versions < 0.50 */
#define REGISTER_MD_DEV         _IO (MD_MAJOR, 1)
#define START_MD                _IO (MD_MAJOR, 2)
#define STOP_MD                 _IO (MD_MAJOR, 3)

	if (OLD_MDTOOLS) {
	    pers = param->personality << MD_PERSONALITY_SHIFT;
	    if ((param->personality == LINEAR || param->personality == RAID0)
		&& param->chunk_size)
		    pers |= s2f(param->chunk_size);
	    rc = ioctl (fd, START_MD, (unsigned long)pers);
	    return 0;
	}

    if ((rc = ioctl (fd, RUN_ARRAY, (unsigned long)param))) {
	    save_errno=errno;
	    switch (save_errno) {
	      case EBUSY:
		fprintf(stderr,"%s: already running\n",dev);
		break;
		/* fall through */
	      default:
		perror (dev);
	    }
	    errno=save_errno;
	    return 1;
    }
    return 0;
}
static int do_mdstart (int fd, char *dev, dev_t rdev) {
    int rc;

    if ((rc = ioctl (fd, START_ARRAY, (unsigned long)rdev))) {
	    save_errno=errno;
	    switch (save_errno) {
	      case EEXIST:
	      case EBUSY:
		fprintf(stderr,"%s: already running\n",dev);
		break;
		/* fall through */
	      default:
		perror (dev);
	    }
	    errno=save_errno;
	    return 1;
    }
    return 0;
}

int do_raidstart_rw (int fd, char *dev)
{
    int func = RESTART_ARRAY_RW;
    int rc;
    struct stat s;

    fstat (fd, &s);
    
    if (major (s.st_rdev) != MD_MAJOR) {
	fprintf(stderr,"%s: not an MD device!\n",dev);
        exit(EXIT_FAILURE);
    }

    if (OLD_MDTOOLS)
        exit(EXIT_FAILURE);

    if ((rc = ioctl (fd, func, 0UL))) {
	    save_errno = errno;
	    switch (save_errno) {
	      case EBUSY:
		fprintf(stderr,"%s: device already read-write!\n",dev);
		break;
	      case ENXIO:
		fprintf(stderr,"%s: not running!\n",dev);
		break;
	      case EINVAL:
		fprintf(stderr,"%s: old kernel?\n",dev);
		/* fall through */
	      default:
		perror (dev);
	    }
	    errno = save_errno;
	    return 1;
    }
    return 0;
}

int do_raidstop (int fd, char *dev, int ro)
{
    int func = ro ? STOP_ARRAY_RO : STOP_ARRAY;
    int rc;
    struct stat s;


    fstat (fd, &s);
    
    if (major (s.st_rdev) != MD_MAJOR) {
	fprintf(stderr,"%s: not an MD device!\n",dev);
        exit(EXIT_FAILURE);
    }

    if (OLD_MDTOOLS)
	if (func == STOP_ARRAY) {
		rc = ioctl(fd, STOP_MD, 0UL);
    		return 0;
	}

    if ((rc = ioctl (fd, func, 0UL))) {
	    save_errno = errno;
	    switch (save_errno) {
	      case ENODEV:
	      case ENXIO:
		fprintf(stderr,"%s: device already stopped\n",dev);
		break;
	      case EINVAL:
		if (func == STOP_ARRAY_RO) {
			fprintf(stderr,"%s: old kernel!\n",dev);
			break;
		}
		/* fall through */
	      default:
		perror (dev);
	    }
	    errno = save_errno;
	    return 1;
    }
    return 0;
}

int do_raidhotadd (int md_fd, char * disk_name, char * md_name)
{
    int rc;
    struct stat s;

    if (OLD_MDTOOLS) {
	fprintf(stderr,"kernel does not support hot-add!\n");
        exit(EXIT_FAILURE);
    }
	
    rc = fstat (md_fd, &s);
    if (rc != 0) {
	fprintf(stderr,"%s: no such device.\n",md_name);
        exit(EXIT_FAILURE);
    }
    if (major (s.st_rdev) != MD_MAJOR) {
	fprintf(stderr,"%s: not an MD device!\n",md_name);
        exit(EXIT_FAILURE);
    }

    rc = stat (disk_name, &s);
    if (rc != 0)
	rc = fake_stat_from_table(disk_name, &s);
    if (rc != 0) {
	fprintf(stderr,"%s: no such device.\n",disk_name);
        exit(EXIT_FAILURE);
    }

    rc = ioctl (md_fd, HOT_ADD_DISK, (unsigned long)s.st_rdev);
    if (rc) {
	    save_errno = errno;
  	    fprintf(stderr,"%s: can not hot-add disk: ", md_name);
	    switch (save_errno) {
	      case ENXIO:
		fprintf(stderr,"disk does not exist!\n");
		break;
	      case EBUSY:
		fprintf(stderr,"disk busy!\n");
		break;
	      case ENOSPC:
		fprintf(stderr,"too small disk!\n");
		break;
	      case ENODEV:
		fprintf(stderr,"array not running!\n");
		break;
	      case EINVAL:
		fprintf(stderr,"invalid argument.\n");
		break;
	      default:
		perror (md_name);
	    }
	    errno = save_errno;
	    return 1;
    }
    return 0;
}

int do_raidhotgenerateerror (int md_fd, char * disk_name, char * md_name)
{
    int rc;
    struct stat s;

    if (OLD_MDTOOLS) {
	fprintf(stderr,"kernel does not support hot-generate-error!\n");
        exit(EXIT_FAILURE);
    }
	
    rc = fstat (md_fd, &s);
    if (rc != 0) {
	fprintf(stderr,"%s: no such device.\n",md_name);
        exit(EXIT_FAILURE);
    }
    if (major (s.st_rdev) != MD_MAJOR) {
	fprintf(stderr,"%s: not an MD device!\n",md_name);
        exit(EXIT_FAILURE);
    }

    rc = stat (disk_name, &s);
    if (rc != 0)
	rc = fake_stat_from_table(disk_name, &s);
    if (rc != 0) {
	fprintf(stderr,"%s: no such device.\n",disk_name);
        exit(EXIT_FAILURE);
    }

    rc = ioctl (md_fd, HOT_GENERATE_ERROR, (unsigned long)s.st_rdev);
    if (rc) {
	    save_errno = errno;
  	    fprintf(stderr,"%s: can not hot-generate-error disk: ", md_name);
	    switch (save_errno) {
	      case ENXIO:
		fprintf(stderr,"disk not in array!\n");
		break;
	      case EBUSY:
		fprintf(stderr,"disk busy!\n");
		break;
	      case ENOSPC:
		fprintf(stderr,"too small disk!\n");
		break;
	      case ENODEV:
		fprintf(stderr,"array not running!\n");
		break;
	      case EINVAL:
		fprintf(stderr,"invalid argument.\n");
		break;
	      default:
		perror (md_name);
	    }
	    errno = save_errno;
	    return 1;
    }
    return 0;
}

int do_raidhotremove (int md_fd, char * disk_name, char * md_name)
{
    int rc;
    struct stat s;

    if (OLD_MDTOOLS) {
	fprintf(stderr,"kernel does not support hot-remove!\n");
        exit(EXIT_FAILURE);
    }
	
    rc = fstat (md_fd, &s);
    if (rc != 0) {
	fprintf(stderr,"%s: no such device.\n",md_name);
        exit(EXIT_FAILURE);
    }
    if (major (s.st_rdev) != MD_MAJOR) {
	fprintf(stderr,"%s: not an MD device!\n",md_name);
        exit(EXIT_FAILURE);
    }

    rc = stat (disk_name, &s);
    if (rc != 0)
	rc = fake_stat_from_table(disk_name, &s);
    if (rc != 0) {
	fprintf(stderr,"%s: no such device.\n",disk_name);
        exit(EXIT_FAILURE);
    }

    rc = ioctl (md_fd, HOT_REMOVE_DISK, (unsigned long)s.st_rdev);
    if (rc) {
	    save_errno = errno;
  	    fprintf(stderr,"%s: can not hot-remove disk: ", md_name);
	    switch (save_errno) {
	      case ENXIO:
		fprintf(stderr,"disk not in array!\n");
		break;
	      case EBUSY:
		fprintf(stderr,"disk busy!\n");
		break;
	      case ENOSPC:
		fprintf(stderr,"too small disk!\n");
		break;
	      case ENODEV:
		fprintf(stderr,"array not running!\n");
		break;
	      case EINVAL:
		fprintf(stderr,"invalid argument.\n");
		break;
	      default:
		perror (md_name);
	    }
	    errno = save_errno;
	    return 1;
    }
    return 0;
}

int do_raidsetfaulty (int md_fd, char * disk_name, char * md_name)
{
	int rc;
	struct stat s;

	if (OLD_MDTOOLS) {
		fprintf(stderr,"kernel does not support hot-add!\n");
		exit(EXIT_FAILURE);
	}

	fstat (md_fd, &s);
	if (major (s.st_rdev) != MD_MAJOR) {
		fprintf(stderr,"%s: not an MD device!\n",md_name);
		exit(EXIT_FAILURE);
	}

	stat (disk_name, &s);

	rc = ioctl (md_fd, SET_DISK_FAULTY, (unsigned long)s.st_rdev);
	if (rc) {
		save_errno = errno;
		fprintf(stderr,"%s: can not set disk faulty: ", md_name);
		switch (save_errno) {
			case ENXIO:
				fprintf(stderr,"disk does not exist!\n");
				break;
			case EBUSY:
				fprintf(stderr,"disk busy!\n");
				break;
			case ENOSPC:
				fprintf(stderr,"too small disk!\n");
				break;
			case ENODEV:
				fprintf(stderr,"array not running!\n");
				break;
			case EINVAL:
				fprintf(stderr,"invalid argument.\n");
				break;
			default:
				perror (md_name);
		}
		errno = save_errno;
		return 1;
	}
	return 0;
}

int handleOneConfig(enum raidFunc func, md_cfg_entry_t * cfg) {
    int rc = 0;
    int fd = -1;
    mdu_param_t param;

    switch (func) {
      case raidstart:
      {
	struct stat s;

	stat (cfg->device_name[0], &s);

	fd = open_or_die(cfg->md_name);
	if (do_mdstart (fd, cfg->md_name, s.st_rdev)) rc++;
	break;
      }

      case raidrun:
	memset(&param,0,sizeof(param));
	switch (cfg->array.param.level) {
	  case -4: param.personality = MULTIPATH; break;
	  case -3: param.personality = HSM; break;
	  case -2: param.personality = TRANSLUCENT; break;
	  case -1: param.personality = LINEAR; break;
	  case 0:  param.personality = RAID0; break;
	  case 1:  param.personality = RAID1; break;
	  case 4:
	  case 5:  param.personality = RAID5; break;
	  default: exit (EXIT_FAILURE);
	}

	param.chunk_size = cfg->array.param.chunk_size;

	fd = open_or_die(cfg->md_name);
	if (do_mdrun (fd, cfg->md_name, &param)) rc++;
	break;

      case raidstop:
	return 1;

      case raidstop_ro:
	return 1;

      case raidhotremove:
	return 1;

      case raidhotadd:
	return 1;

      case raidhotgenerateerror:
	return 1;

      case raidsetfaulty:
        return 1;
    }
    close (fd);

    return rc;
}

int prepare_raidlib (void)
{
    if (getMdVersion(&md_ver))
        return EXIT_FAILURE;
    return 0;
}

static int fake_stat_from_table(const char *file_name, struct stat *buf)
{
    /*
     *  This function is pretty ugly.  The reason it's here is to handle disks
     *  or partitions that go away while still in use.  In particular, if a
     *  disk is removed unexpectedly while it or one of its partitions is still
     *  in use by an MD device, if the kernel detects that, it immediately
     *  invalidates the device for that disk and its partition.  Then,
     *  user-level code comes along to clean up and wants to run raidhotremove
     *  in some cases to remove those devices that have gone away.  The trouble
     *  is that raidhotremove takes the name of the device as its argument, and
     *  tries to stat() that device.  But in this case, the device has gone
     *  away, so the stat fails.
     *
     *  Since the MD driver uses major and minor numbers internally to track
     *  the component devices, and even the ioctl() to hot remove a disk uses
     *  the major and minor numbers instead of a path, a cleaner solution would
     *  be to pass the major and minor numbers to the raidhotremove program on
     *  its command line instead of the path.  But we'd also need to have
     *  /proc/mdstat show the numbers instead of the device names, since it's
     *  in there that we look for the components (or else we'd have to have
     *  this very same mapping table somewhere else).  Changing the interfaces
     *  to /proc/mdstat and the command-line interface to raidhotremove are
     *  pretty drastic, so for now I'm going with this table.  It whould work
     *  just as well, since on our system I know exactly which devices are
     *  possible as components of an MD device, even if it is hard to maintain
     *  if new devices are added or device names or major/minor numbers change
     *  (we've already changed the mapping of minor numbers for SCSI disks).
     *
     *          --Chris Wilson, August 12, 2004, for Broadcom NASoC
     */

    static struct { int major; int minor; const char *name; } path_table[] =
    {
	{   8,   0, "/dev/scsi/host1/bus0/target0/lun0/disc"},
	{   8,   1, "/dev/scsi/host1/bus0/target0/lun0/part1"},
	{   8,   2, "/dev/scsi/host1/bus0/target0/lun0/part2"},
	{   8,   3, "/dev/scsi/host1/bus0/target0/lun0/part3"},
	{   8,   4, "/dev/scsi/host1/bus0/target0/lun0/part4"},
	{   8,   5, "/dev/scsi/host1/bus0/target0/lun0/part5"},
	{   8,   6, "/dev/scsi/host1/bus0/target0/lun0/part6"},
	{   8,   7, "/dev/scsi/host1/bus0/target0/lun0/part7"},
	{   8,   8, "/dev/scsi/host1/bus0/target0/lun0/part8"},
	{   8,   9, "/dev/scsi/host1/bus0/target0/lun0/part9"},
	{   8,  10, "/dev/scsi/host1/bus0/target0/lun0/part10"},
	{   8,  11, "/dev/scsi/host1/bus0/target0/lun0/part11"},
	{   8,  12, "/dev/scsi/host1/bus0/target0/lun0/part12"},
	{   8,  13, "/dev/scsi/host1/bus0/target0/lun0/part13"},
	{   8,  14, "/dev/scsi/host1/bus0/target0/lun0/part14"},
	{   8,  15, "/dev/scsi/host1/bus0/target0/lun0/part15"},
	{   8,  16, "/dev/scsi/host1/bus0/target0/lun0/part16"},
	{   8,  17, "/dev/scsi/host1/bus0/target0/lun0/part17"},
	{   8,  18, "/dev/scsi/host1/bus0/target0/lun0/part18"},
	{   8,  19, "/dev/scsi/host1/bus0/target0/lun0/part19"},
	{   8,  20, "/dev/scsi/host1/bus0/target0/lun0/part20"},
	{   8,  21, "/dev/scsi/host1/bus0/target0/lun0/part21"},
	{   8,  22, "/dev/scsi/host1/bus0/target0/lun0/part22"},
	{   8,  23, "/dev/scsi/host1/bus0/target0/lun0/part23"},
	{   8,  24, "/dev/scsi/host1/bus0/target0/lun0/part24"},
	{   8,  25, "/dev/scsi/host1/bus0/target0/lun0/part25"},
	{   8,  26, "/dev/scsi/host1/bus0/target0/lun0/part26"},
	{   8,  27, "/dev/scsi/host1/bus0/target0/lun0/part27"},
	{   8,  28, "/dev/scsi/host1/bus0/target0/lun0/part28"},
	{   8,  29, "/dev/scsi/host1/bus0/target0/lun0/part29"},
	{   8,  30, "/dev/scsi/host1/bus0/target0/lun0/part30"},
	{   8,  31, "/dev/scsi/host1/bus0/target0/lun0/part31"},
	{   8,  32, "/dev/scsi/host1/bus0/target0/lun0/part32"},
	{   8,  33, "/dev/scsi/host1/bus0/target0/lun0/part33"},
	{   8,  34, "/dev/scsi/host1/bus0/target0/lun0/part34"},
	{   8,  35, "/dev/scsi/host1/bus0/target0/lun0/part35"},
	{   8,  36, "/dev/scsi/host1/bus0/target0/lun0/part36"},
	{   8,  37, "/dev/scsi/host1/bus0/target0/lun0/part37"},
	{   8,  38, "/dev/scsi/host1/bus0/target0/lun0/part38"},
	{   8,  39, "/dev/scsi/host1/bus0/target0/lun0/part39"},
	{   8,  40, "/dev/scsi/host1/bus0/target0/lun0/part40"},
	{   8,  41, "/dev/scsi/host1/bus0/target0/lun0/part41"},
	{   8,  42, "/dev/scsi/host1/bus0/target0/lun0/part42"},
	{   8,  43, "/dev/scsi/host1/bus0/target0/lun0/part43"},
	{   8,  44, "/dev/scsi/host1/bus0/target0/lun0/part44"},
	{   8,  45, "/dev/scsi/host1/bus0/target0/lun0/part45"},
	{   8,  46, "/dev/scsi/host1/bus0/target0/lun0/part46"},
	{   8,  47, "/dev/scsi/host1/bus0/target0/lun0/part47"},
	{   8,  48, "/dev/scsi/host1/bus0/target0/lun0/part48"},
	{   8,  49, "/dev/scsi/host1/bus0/target0/lun0/part49"},
	{   8,  50, "/dev/scsi/host1/bus0/target0/lun0/part50"},
	{   8,  51, "/dev/scsi/host1/bus0/target0/lun0/part51"},
	{   8,  52, "/dev/scsi/host1/bus0/target0/lun0/part52"},
	{   8,  53, "/dev/scsi/host1/bus0/target0/lun0/part53"},
	{   8,  54, "/dev/scsi/host1/bus0/target0/lun0/part54"},
	{   8,  55, "/dev/scsi/host1/bus0/target0/lun0/part55"},
	{   8,  56, "/dev/scsi/host1/bus0/target0/lun0/part56"},
	{   8,  57, "/dev/scsi/host1/bus0/target0/lun0/part57"},
	{   8,  58, "/dev/scsi/host1/bus0/target0/lun0/part58"},
	{   8,  59, "/dev/scsi/host1/bus0/target0/lun0/part59"},
	{   8,  60, "/dev/scsi/host1/bus0/target0/lun0/part60"},
	{   8,  61, "/dev/scsi/host1/bus0/target0/lun0/part61"},
	{   8,  62, "/dev/scsi/host1/bus0/target0/lun0/part62"},
	{   8,  63, "/dev/scsi/host1/bus0/target0/lun0/part63"},
	{   8,  64, "/dev/scsi/host1/bus0/target1/lun0/disc"},
	{   8,  65, "/dev/scsi/host1/bus0/target1/lun0/part1"},
	{   8,  66, "/dev/scsi/host1/bus0/target1/lun0/part2"},
	{   8,  67, "/dev/scsi/host1/bus0/target1/lun0/part3"},
	{   8,  68, "/dev/scsi/host1/bus0/target1/lun0/part4"},
	{   8,  69, "/dev/scsi/host1/bus0/target1/lun0/part5"},
	{   8,  70, "/dev/scsi/host1/bus0/target1/lun0/part6"},
	{   8,  71, "/dev/scsi/host1/bus0/target1/lun0/part7"},
	{   8,  72, "/dev/scsi/host1/bus0/target1/lun0/part8"},
	{   8,  73, "/dev/scsi/host1/bus0/target1/lun0/part9"},
	{   8,  74, "/dev/scsi/host1/bus0/target1/lun0/part10"},
	{   8,  75, "/dev/scsi/host1/bus0/target1/lun0/part11"},
	{   8,  76, "/dev/scsi/host1/bus0/target1/lun0/part12"},
	{   8,  77, "/dev/scsi/host1/bus0/target1/lun0/part13"},
	{   8,  78, "/dev/scsi/host1/bus0/target1/lun0/part14"},
	{   8,  79, "/dev/scsi/host1/bus0/target1/lun0/part15"},
	{   8,  80, "/dev/scsi/host1/bus0/target1/lun0/part16"},
	{   8,  81, "/dev/scsi/host1/bus0/target1/lun0/part17"},
	{   8,  82, "/dev/scsi/host1/bus0/target1/lun0/part18"},
	{   8,  83, "/dev/scsi/host1/bus0/target1/lun0/part19"},
	{   8,  84, "/dev/scsi/host1/bus0/target1/lun0/part20"},
	{   8,  85, "/dev/scsi/host1/bus0/target1/lun0/part21"},
	{   8,  86, "/dev/scsi/host1/bus0/target1/lun0/part22"},
	{   8,  87, "/dev/scsi/host1/bus0/target1/lun0/part23"},
	{   8,  88, "/dev/scsi/host1/bus0/target1/lun0/part24"},
	{   8,  89, "/dev/scsi/host1/bus0/target1/lun0/part25"},
	{   8,  90, "/dev/scsi/host1/bus0/target1/lun0/part26"},
	{   8,  91, "/dev/scsi/host1/bus0/target1/lun0/part27"},
	{   8,  92, "/dev/scsi/host1/bus0/target1/lun0/part28"},
	{   8,  93, "/dev/scsi/host1/bus0/target1/lun0/part29"},
	{   8,  94, "/dev/scsi/host1/bus0/target1/lun0/part30"},
	{   8,  95, "/dev/scsi/host1/bus0/target1/lun0/part31"},
	{   8,  96, "/dev/scsi/host1/bus0/target1/lun0/part32"},
	{   8,  97, "/dev/scsi/host1/bus0/target1/lun0/part33"},
	{   8,  98, "/dev/scsi/host1/bus0/target1/lun0/part34"},
	{   8,  99, "/dev/scsi/host1/bus0/target1/lun0/part35"},
	{   8, 100, "/dev/scsi/host1/bus0/target1/lun0/part36"},
	{   8, 101, "/dev/scsi/host1/bus0/target1/lun0/part37"},
	{   8, 102, "/dev/scsi/host1/bus0/target1/lun0/part38"},
	{   8, 103, "/dev/scsi/host1/bus0/target1/lun0/part39"},
	{   8, 104, "/dev/scsi/host1/bus0/target1/lun0/part40"},
	{   8, 105, "/dev/scsi/host1/bus0/target1/lun0/part41"},
	{   8, 106, "/dev/scsi/host1/bus0/target1/lun0/part42"},
	{   8, 107, "/dev/scsi/host1/bus0/target1/lun0/part43"},
	{   8, 108, "/dev/scsi/host1/bus0/target1/lun0/part44"},
	{   8, 109, "/dev/scsi/host1/bus0/target1/lun0/part45"},
	{   8, 110, "/dev/scsi/host1/bus0/target1/lun0/part46"},
	{   8, 111, "/dev/scsi/host1/bus0/target1/lun0/part47"},
	{   8, 112, "/dev/scsi/host1/bus0/target1/lun0/part48"},
	{   8, 113, "/dev/scsi/host1/bus0/target1/lun0/part49"},
	{   8, 114, "/dev/scsi/host1/bus0/target1/lun0/part50"},
	{   8, 115, "/dev/scsi/host1/bus0/target1/lun0/part51"},
	{   8, 116, "/dev/scsi/host1/bus0/target1/lun0/part52"},
	{   8, 117, "/dev/scsi/host1/bus0/target1/lun0/part53"},
	{   8, 118, "/dev/scsi/host1/bus0/target1/lun0/part54"},
	{   8, 119, "/dev/scsi/host1/bus0/target1/lun0/part55"},
	{   8, 120, "/dev/scsi/host1/bus0/target1/lun0/part56"},
	{   8, 121, "/dev/scsi/host1/bus0/target1/lun0/part57"},
	{   8, 122, "/dev/scsi/host1/bus0/target1/lun0/part58"},
	{   8, 123, "/dev/scsi/host1/bus0/target1/lun0/part59"},
	{   8, 124, "/dev/scsi/host1/bus0/target1/lun0/part60"},
	{   8, 125, "/dev/scsi/host1/bus0/target1/lun0/part61"},
	{   8, 126, "/dev/scsi/host1/bus0/target1/lun0/part62"},
	{   8, 127, "/dev/scsi/host1/bus0/target1/lun0/part63"},
	{   8, 128, "/dev/scsi/host1/bus0/target2/lun0/disc"},
	{   8, 129, "/dev/scsi/host1/bus0/target2/lun0/part1"},
	{   8, 130, "/dev/scsi/host1/bus0/target2/lun0/part2"},
	{   8, 131, "/dev/scsi/host1/bus0/target2/lun0/part3"},
	{   8, 132, "/dev/scsi/host1/bus0/target2/lun0/part4"},
	{   8, 133, "/dev/scsi/host1/bus0/target2/lun0/part5"},
	{   8, 134, "/dev/scsi/host1/bus0/target2/lun0/part6"},
	{   8, 135, "/dev/scsi/host1/bus0/target2/lun0/part7"},
	{   8, 136, "/dev/scsi/host1/bus0/target2/lun0/part8"},
	{   8, 137, "/dev/scsi/host1/bus0/target2/lun0/part9"},
	{   8, 138, "/dev/scsi/host1/bus0/target2/lun0/part10"},
	{   8, 139, "/dev/scsi/host1/bus0/target2/lun0/part11"},
	{   8, 140, "/dev/scsi/host1/bus0/target2/lun0/part12"},
	{   8, 141, "/dev/scsi/host1/bus0/target2/lun0/part13"},
	{   8, 142, "/dev/scsi/host1/bus0/target2/lun0/part14"},
	{   8, 143, "/dev/scsi/host1/bus0/target2/lun0/part15"},
	{   8, 144, "/dev/scsi/host1/bus0/target2/lun0/part16"},
	{   8, 145, "/dev/scsi/host1/bus0/target2/lun0/part17"},
	{   8, 146, "/dev/scsi/host1/bus0/target2/lun0/part18"},
	{   8, 147, "/dev/scsi/host1/bus0/target2/lun0/part19"},
	{   8, 148, "/dev/scsi/host1/bus0/target2/lun0/part20"},
	{   8, 149, "/dev/scsi/host1/bus0/target2/lun0/part21"},
	{   8, 150, "/dev/scsi/host1/bus0/target2/lun0/part22"},
	{   8, 151, "/dev/scsi/host1/bus0/target2/lun0/part23"},
	{   8, 152, "/dev/scsi/host1/bus0/target2/lun0/part24"},
	{   8, 153, "/dev/scsi/host1/bus0/target2/lun0/part25"},
	{   8, 154, "/dev/scsi/host1/bus0/target2/lun0/part26"},
	{   8, 155, "/dev/scsi/host1/bus0/target2/lun0/part27"},
	{   8, 156, "/dev/scsi/host1/bus0/target2/lun0/part28"},
	{   8, 157, "/dev/scsi/host1/bus0/target2/lun0/part29"},
	{   8, 158, "/dev/scsi/host1/bus0/target2/lun0/part30"},
	{   8, 159, "/dev/scsi/host1/bus0/target2/lun0/part31"},
	{   8, 160, "/dev/scsi/host1/bus0/target2/lun0/part32"},
	{   8, 161, "/dev/scsi/host1/bus0/target2/lun0/part33"},
	{   8, 162, "/dev/scsi/host1/bus0/target2/lun0/part34"},
	{   8, 163, "/dev/scsi/host1/bus0/target2/lun0/part35"},
	{   8, 164, "/dev/scsi/host1/bus0/target2/lun0/part36"},
	{   8, 165, "/dev/scsi/host1/bus0/target2/lun0/part37"},
	{   8, 166, "/dev/scsi/host1/bus0/target2/lun0/part38"},
	{   8, 167, "/dev/scsi/host1/bus0/target2/lun0/part39"},
	{   8, 168, "/dev/scsi/host1/bus0/target2/lun0/part40"},
	{   8, 169, "/dev/scsi/host1/bus0/target2/lun0/part41"},
	{   8, 170, "/dev/scsi/host1/bus0/target2/lun0/part42"},
	{   8, 171, "/dev/scsi/host1/bus0/target2/lun0/part43"},
	{   8, 172, "/dev/scsi/host1/bus0/target2/lun0/part44"},
	{   8, 173, "/dev/scsi/host1/bus0/target2/lun0/part45"},
	{   8, 174, "/dev/scsi/host1/bus0/target2/lun0/part46"},
	{   8, 175, "/dev/scsi/host1/bus0/target2/lun0/part47"},
	{   8, 176, "/dev/scsi/host1/bus0/target2/lun0/part48"},
	{   8, 177, "/dev/scsi/host1/bus0/target2/lun0/part49"},
	{   8, 178, "/dev/scsi/host1/bus0/target2/lun0/part50"},
	{   8, 179, "/dev/scsi/host1/bus0/target2/lun0/part51"},
	{   8, 180, "/dev/scsi/host1/bus0/target2/lun0/part52"},
	{   8, 181, "/dev/scsi/host1/bus0/target2/lun0/part53"},
	{   8, 182, "/dev/scsi/host1/bus0/target2/lun0/part54"},
	{   8, 183, "/dev/scsi/host1/bus0/target2/lun0/part55"},
	{   8, 184, "/dev/scsi/host1/bus0/target2/lun0/part56"},
	{   8, 185, "/dev/scsi/host1/bus0/target2/lun0/part57"},
	{   8, 186, "/dev/scsi/host1/bus0/target2/lun0/part58"},
	{   8, 187, "/dev/scsi/host1/bus0/target2/lun0/part59"},
	{   8, 188, "/dev/scsi/host1/bus0/target2/lun0/part60"},
	{   8, 189, "/dev/scsi/host1/bus0/target2/lun0/part61"},
	{   8, 190, "/dev/scsi/host1/bus0/target2/lun0/part62"},
	{   8, 191, "/dev/scsi/host1/bus0/target2/lun0/part63"},
	{   8, 192, "/dev/scsi/host1/bus0/target3/lun0/disc"},
	{   8, 193, "/dev/scsi/host1/bus0/target3/lun0/part1"},
	{   8, 194, "/dev/scsi/host1/bus0/target3/lun0/part2"},
	{   8, 195, "/dev/scsi/host1/bus0/target3/lun0/part3"},
	{   8, 196, "/dev/scsi/host1/bus0/target3/lun0/part4"},
	{   8, 197, "/dev/scsi/host1/bus0/target3/lun0/part5"},
	{   8, 198, "/dev/scsi/host1/bus0/target3/lun0/part6"},
	{   8, 199, "/dev/scsi/host1/bus0/target3/lun0/part7"},
	{   8, 200, "/dev/scsi/host1/bus0/target3/lun0/part8"},
	{   8, 201, "/dev/scsi/host1/bus0/target3/lun0/part9"},
	{   8, 202, "/dev/scsi/host1/bus0/target3/lun0/part10"},
	{   8, 203, "/dev/scsi/host1/bus0/target3/lun0/part11"},
	{   8, 204, "/dev/scsi/host1/bus0/target3/lun0/part12"},
	{   8, 205, "/dev/scsi/host1/bus0/target3/lun0/part13"},
	{   8, 206, "/dev/scsi/host1/bus0/target3/lun0/part14"},
	{   8, 207, "/dev/scsi/host1/bus0/target3/lun0/part15"},
	{   8, 208, "/dev/scsi/host1/bus0/target3/lun0/part16"},
	{   8, 209, "/dev/scsi/host1/bus0/target3/lun0/part17"},
	{   8, 210, "/dev/scsi/host1/bus0/target3/lun0/part18"},
	{   8, 211, "/dev/scsi/host1/bus0/target3/lun0/part19"},
	{   8, 212, "/dev/scsi/host1/bus0/target3/lun0/part20"},
	{   8, 213, "/dev/scsi/host1/bus0/target3/lun0/part21"},
	{   8, 214, "/dev/scsi/host1/bus0/target3/lun0/part22"},
	{   8, 215, "/dev/scsi/host1/bus0/target3/lun0/part23"},
	{   8, 216, "/dev/scsi/host1/bus0/target3/lun0/part24"},
	{   8, 217, "/dev/scsi/host1/bus0/target3/lun0/part25"},
	{   8, 218, "/dev/scsi/host1/bus0/target3/lun0/part26"},
	{   8, 219, "/dev/scsi/host1/bus0/target3/lun0/part27"},
	{   8, 220, "/dev/scsi/host1/bus0/target3/lun0/part28"},
	{   8, 221, "/dev/scsi/host1/bus0/target3/lun0/part29"},
	{   8, 222, "/dev/scsi/host1/bus0/target3/lun0/part30"},
	{   8, 223, "/dev/scsi/host1/bus0/target3/lun0/part31"},
	{   8, 224, "/dev/scsi/host1/bus0/target3/lun0/part32"},
	{   8, 225, "/dev/scsi/host1/bus0/target3/lun0/part33"},
	{   8, 226, "/dev/scsi/host1/bus0/target3/lun0/part34"},
	{   8, 227, "/dev/scsi/host1/bus0/target3/lun0/part35"},
	{   8, 228, "/dev/scsi/host1/bus0/target3/lun0/part36"},
	{   8, 229, "/dev/scsi/host1/bus0/target3/lun0/part37"},
	{   8, 230, "/dev/scsi/host1/bus0/target3/lun0/part38"},
	{   8, 231, "/dev/scsi/host1/bus0/target3/lun0/part39"},
	{   8, 232, "/dev/scsi/host1/bus0/target3/lun0/part40"},
	{   8, 233, "/dev/scsi/host1/bus0/target3/lun0/part41"},
	{   8, 234, "/dev/scsi/host1/bus0/target3/lun0/part42"},
	{   8, 235, "/dev/scsi/host1/bus0/target3/lun0/part43"},
	{   8, 236, "/dev/scsi/host1/bus0/target3/lun0/part44"},
	{   8, 237, "/dev/scsi/host1/bus0/target3/lun0/part45"},
	{   8, 238, "/dev/scsi/host1/bus0/target3/lun0/part46"},
	{   8, 239, "/dev/scsi/host1/bus0/target3/lun0/part47"},
	{   8, 240, "/dev/scsi/host1/bus0/target3/lun0/part48"},
	{   8, 241, "/dev/scsi/host1/bus0/target3/lun0/part49"},
	{   8, 242, "/dev/scsi/host1/bus0/target3/lun0/part50"},
	{   8, 243, "/dev/scsi/host1/bus0/target3/lun0/part51"},
	{   8, 244, "/dev/scsi/host1/bus0/target3/lun0/part52"},
	{   8, 245, "/dev/scsi/host1/bus0/target3/lun0/part53"},
	{   8, 246, "/dev/scsi/host1/bus0/target3/lun0/part54"},
	{   8, 247, "/dev/scsi/host1/bus0/target3/lun0/part55"},
	{   8, 248, "/dev/scsi/host1/bus0/target3/lun0/part56"},
	{   8, 249, "/dev/scsi/host1/bus0/target3/lun0/part57"},
	{   8, 250, "/dev/scsi/host1/bus0/target3/lun0/part58"},
	{   8, 251, "/dev/scsi/host1/bus0/target3/lun0/part59"},
	{   8, 252, "/dev/scsi/host1/bus0/target3/lun0/part60"},
	{   8, 253, "/dev/scsi/host1/bus0/target3/lun0/part61"},
	{   8, 254, "/dev/scsi/host1/bus0/target3/lun0/part62"},
	{   8, 255, "/dev/scsi/host1/bus0/target3/lun0/part63"},

	{  65,   0, "/dev/scsi/host2/bus0/target0/lun0/disc"},
	{  65,   1, "/dev/scsi/host2/bus0/target0/lun0/part1"},
	{  65,   2, "/dev/scsi/host2/bus0/target0/lun0/part2"},
	{  65,   3, "/dev/scsi/host2/bus0/target0/lun0/part3"},
	{  65,   4, "/dev/scsi/host2/bus0/target0/lun0/part4"},
	{  65,   5, "/dev/scsi/host2/bus0/target0/lun0/part5"},
	{  65,   6, "/dev/scsi/host2/bus0/target0/lun0/part6"},
	{  65,   7, "/dev/scsi/host2/bus0/target0/lun0/part7"},
	{  65,   8, "/dev/scsi/host2/bus0/target0/lun0/part8"},
	{  65,   9, "/dev/scsi/host2/bus0/target0/lun0/part9"},
	{  65,  10, "/dev/scsi/host2/bus0/target0/lun0/part10"},
	{  65,  11, "/dev/scsi/host2/bus0/target0/lun0/part11"},
	{  65,  12, "/dev/scsi/host2/bus0/target0/lun0/part12"},
	{  65,  13, "/dev/scsi/host2/bus0/target0/lun0/part13"},
	{  65,  14, "/dev/scsi/host2/bus0/target0/lun0/part14"},
	{  65,  15, "/dev/scsi/host2/bus0/target0/lun0/part15"},
	{  65,  16, "/dev/scsi/host2/bus0/target0/lun0/part16"},
	{  65,  17, "/dev/scsi/host2/bus0/target0/lun0/part17"},
	{  65,  18, "/dev/scsi/host2/bus0/target0/lun0/part18"},
	{  65,  19, "/dev/scsi/host2/bus0/target0/lun0/part19"},
	{  65,  20, "/dev/scsi/host2/bus0/target0/lun0/part20"},
	{  65,  21, "/dev/scsi/host2/bus0/target0/lun0/part21"},
	{  65,  22, "/dev/scsi/host2/bus0/target0/lun0/part22"},
	{  65,  23, "/dev/scsi/host2/bus0/target0/lun0/part23"},
	{  65,  24, "/dev/scsi/host2/bus0/target0/lun0/part24"},
	{  65,  25, "/dev/scsi/host2/bus0/target0/lun0/part25"},
	{  65,  26, "/dev/scsi/host2/bus0/target0/lun0/part26"},
	{  65,  27, "/dev/scsi/host2/bus0/target0/lun0/part27"},
	{  65,  28, "/dev/scsi/host2/bus0/target0/lun0/part28"},
	{  65,  29, "/dev/scsi/host2/bus0/target0/lun0/part29"},
	{  65,  30, "/dev/scsi/host2/bus0/target0/lun0/part30"},
	{  65,  31, "/dev/scsi/host2/bus0/target0/lun0/part31"},
	{  65,  32, "/dev/scsi/host2/bus0/target0/lun0/part32"},
	{  65,  33, "/dev/scsi/host2/bus0/target0/lun0/part33"},
	{  65,  34, "/dev/scsi/host2/bus0/target0/lun0/part34"},
	{  65,  35, "/dev/scsi/host2/bus0/target0/lun0/part35"},
	{  65,  36, "/dev/scsi/host2/bus0/target0/lun0/part36"},
	{  65,  37, "/dev/scsi/host2/bus0/target0/lun0/part37"},
	{  65,  38, "/dev/scsi/host2/bus0/target0/lun0/part38"},
	{  65,  39, "/dev/scsi/host2/bus0/target0/lun0/part39"},
	{  65,  40, "/dev/scsi/host2/bus0/target0/lun0/part40"},
	{  65,  41, "/dev/scsi/host2/bus0/target0/lun0/part41"},
	{  65,  42, "/dev/scsi/host2/bus0/target0/lun0/part42"},
	{  65,  43, "/dev/scsi/host2/bus0/target0/lun0/part43"},
	{  65,  44, "/dev/scsi/host2/bus0/target0/lun0/part44"},
	{  65,  45, "/dev/scsi/host2/bus0/target0/lun0/part45"},
	{  65,  46, "/dev/scsi/host2/bus0/target0/lun0/part46"},
	{  65,  47, "/dev/scsi/host2/bus0/target0/lun0/part47"},
	{  65,  48, "/dev/scsi/host2/bus0/target0/lun0/part48"},
	{  65,  49, "/dev/scsi/host2/bus0/target0/lun0/part49"},
	{  65,  50, "/dev/scsi/host2/bus0/target0/lun0/part50"},
	{  65,  51, "/dev/scsi/host2/bus0/target0/lun0/part51"},
	{  65,  52, "/dev/scsi/host2/bus0/target0/lun0/part52"},
	{  65,  53, "/dev/scsi/host2/bus0/target0/lun0/part53"},
	{  65,  54, "/dev/scsi/host2/bus0/target0/lun0/part54"},
	{  65,  55, "/dev/scsi/host2/bus0/target0/lun0/part55"},
	{  65,  56, "/dev/scsi/host2/bus0/target0/lun0/part56"},
	{  65,  57, "/dev/scsi/host2/bus0/target0/lun0/part57"},
	{  65,  58, "/dev/scsi/host2/bus0/target0/lun0/part58"},
	{  65,  59, "/dev/scsi/host2/bus0/target0/lun0/part59"},
	{  65,  60, "/dev/scsi/host2/bus0/target0/lun0/part60"},
	{  65,  61, "/dev/scsi/host2/bus0/target0/lun0/part61"},
	{  65,  62, "/dev/scsi/host2/bus0/target0/lun0/part62"},
	{  65,  63, "/dev/scsi/host2/bus0/target0/lun0/part63"},
	{  65,  64, "/dev/scsi/host2/bus0/target1/lun0/disc"},
	{  65,  65, "/dev/scsi/host2/bus0/target1/lun0/part1"},
	{  65,  66, "/dev/scsi/host2/bus0/target1/lun0/part2"},
	{  65,  67, "/dev/scsi/host2/bus0/target1/lun0/part3"},
	{  65,  68, "/dev/scsi/host2/bus0/target1/lun0/part4"},
	{  65,  69, "/dev/scsi/host2/bus0/target1/lun0/part5"},
	{  65,  70, "/dev/scsi/host2/bus0/target1/lun0/part6"},
	{  65,  71, "/dev/scsi/host2/bus0/target1/lun0/part7"},
	{  65,  72, "/dev/scsi/host2/bus0/target1/lun0/part8"},
	{  65,  73, "/dev/scsi/host2/bus0/target1/lun0/part9"},
	{  65,  74, "/dev/scsi/host2/bus0/target1/lun0/part10"},
	{  65,  75, "/dev/scsi/host2/bus0/target1/lun0/part11"},
	{  65,  76, "/dev/scsi/host2/bus0/target1/lun0/part12"},
	{  65,  77, "/dev/scsi/host2/bus0/target1/lun0/part13"},
	{  65,  78, "/dev/scsi/host2/bus0/target1/lun0/part14"},
	{  65,  79, "/dev/scsi/host2/bus0/target1/lun0/part15"},
	{  65,  80, "/dev/scsi/host2/bus0/target1/lun0/part16"},
	{  65,  81, "/dev/scsi/host2/bus0/target1/lun0/part17"},
	{  65,  82, "/dev/scsi/host2/bus0/target1/lun0/part18"},
	{  65,  83, "/dev/scsi/host2/bus0/target1/lun0/part19"},
	{  65,  84, "/dev/scsi/host2/bus0/target1/lun0/part20"},
	{  65,  85, "/dev/scsi/host2/bus0/target1/lun0/part21"},
	{  65,  86, "/dev/scsi/host2/bus0/target1/lun0/part22"},
	{  65,  87, "/dev/scsi/host2/bus0/target1/lun0/part23"},
	{  65,  88, "/dev/scsi/host2/bus0/target1/lun0/part24"},
	{  65,  89, "/dev/scsi/host2/bus0/target1/lun0/part25"},
	{  65,  90, "/dev/scsi/host2/bus0/target1/lun0/part26"},
	{  65,  91, "/dev/scsi/host2/bus0/target1/lun0/part27"},
	{  65,  92, "/dev/scsi/host2/bus0/target1/lun0/part28"},
	{  65,  93, "/dev/scsi/host2/bus0/target1/lun0/part29"},
	{  65,  94, "/dev/scsi/host2/bus0/target1/lun0/part30"},
	{  65,  95, "/dev/scsi/host2/bus0/target1/lun0/part31"},
	{  65,  96, "/dev/scsi/host2/bus0/target1/lun0/part32"},
	{  65,  97, "/dev/scsi/host2/bus0/target1/lun0/part33"},
	{  65,  98, "/dev/scsi/host2/bus0/target1/lun0/part34"},
	{  65,  99, "/dev/scsi/host2/bus0/target1/lun0/part35"},
	{  65, 100, "/dev/scsi/host2/bus0/target1/lun0/part36"},
	{  65, 101, "/dev/scsi/host2/bus0/target1/lun0/part37"},
	{  65, 102, "/dev/scsi/host2/bus0/target1/lun0/part38"},
	{  65, 103, "/dev/scsi/host2/bus0/target1/lun0/part39"},
	{  65, 104, "/dev/scsi/host2/bus0/target1/lun0/part40"},
	{  65, 105, "/dev/scsi/host2/bus0/target1/lun0/part41"},
	{  65, 106, "/dev/scsi/host2/bus0/target1/lun0/part42"},
	{  65, 107, "/dev/scsi/host2/bus0/target1/lun0/part43"},
	{  65, 108, "/dev/scsi/host2/bus0/target1/lun0/part44"},
	{  65, 109, "/dev/scsi/host2/bus0/target1/lun0/part45"},
	{  65, 110, "/dev/scsi/host2/bus0/target1/lun0/part46"},
	{  65, 111, "/dev/scsi/host2/bus0/target1/lun0/part47"},
	{  65, 112, "/dev/scsi/host2/bus0/target1/lun0/part48"},
	{  65, 113, "/dev/scsi/host2/bus0/target1/lun0/part49"},
	{  65, 114, "/dev/scsi/host2/bus0/target1/lun0/part50"},
	{  65, 115, "/dev/scsi/host2/bus0/target1/lun0/part51"},
	{  65, 116, "/dev/scsi/host2/bus0/target1/lun0/part52"},
	{  65, 117, "/dev/scsi/host2/bus0/target1/lun0/part53"},
	{  65, 118, "/dev/scsi/host2/bus0/target1/lun0/part54"},
	{  65, 119, "/dev/scsi/host2/bus0/target1/lun0/part55"},
	{  65, 120, "/dev/scsi/host2/bus0/target1/lun0/part56"},
	{  65, 121, "/dev/scsi/host2/bus0/target1/lun0/part57"},
	{  65, 122, "/dev/scsi/host2/bus0/target1/lun0/part58"},
	{  65, 123, "/dev/scsi/host2/bus0/target1/lun0/part59"},
	{  65, 124, "/dev/scsi/host2/bus0/target1/lun0/part60"},
	{  65, 125, "/dev/scsi/host2/bus0/target1/lun0/part61"},
	{  65, 126, "/dev/scsi/host2/bus0/target1/lun0/part62"},
	{  65, 127, "/dev/scsi/host2/bus0/target1/lun0/part63"},
	{  65, 128, "/dev/scsi/host2/bus0/target2/lun0/disc"},
	{  65, 129, "/dev/scsi/host2/bus0/target2/lun0/part1"},
	{  65, 130, "/dev/scsi/host2/bus0/target2/lun0/part2"},
	{  65, 131, "/dev/scsi/host2/bus0/target2/lun0/part3"},
	{  65, 132, "/dev/scsi/host2/bus0/target2/lun0/part4"},
	{  65, 133, "/dev/scsi/host2/bus0/target2/lun0/part5"},
	{  65, 134, "/dev/scsi/host2/bus0/target2/lun0/part6"},
	{  65, 135, "/dev/scsi/host2/bus0/target2/lun0/part7"},
	{  65, 136, "/dev/scsi/host2/bus0/target2/lun0/part8"},
	{  65, 137, "/dev/scsi/host2/bus0/target2/lun0/part9"},
	{  65, 138, "/dev/scsi/host2/bus0/target2/lun0/part10"},
	{  65, 139, "/dev/scsi/host2/bus0/target2/lun0/part11"},
	{  65, 140, "/dev/scsi/host2/bus0/target2/lun0/part12"},
	{  65, 141, "/dev/scsi/host2/bus0/target2/lun0/part13"},
	{  65, 142, "/dev/scsi/host2/bus0/target2/lun0/part14"},
	{  65, 143, "/dev/scsi/host2/bus0/target2/lun0/part15"},
	{  65, 144, "/dev/scsi/host2/bus0/target2/lun0/part16"},
	{  65, 145, "/dev/scsi/host2/bus0/target2/lun0/part17"},
	{  65, 146, "/dev/scsi/host2/bus0/target2/lun0/part18"},
	{  65, 147, "/dev/scsi/host2/bus0/target2/lun0/part19"},
	{  65, 148, "/dev/scsi/host2/bus0/target2/lun0/part20"},
	{  65, 149, "/dev/scsi/host2/bus0/target2/lun0/part21"},
	{  65, 150, "/dev/scsi/host2/bus0/target2/lun0/part22"},
	{  65, 151, "/dev/scsi/host2/bus0/target2/lun0/part23"},
	{  65, 152, "/dev/scsi/host2/bus0/target2/lun0/part24"},
	{  65, 153, "/dev/scsi/host2/bus0/target2/lun0/part25"},
	{  65, 154, "/dev/scsi/host2/bus0/target2/lun0/part26"},
	{  65, 155, "/dev/scsi/host2/bus0/target2/lun0/part27"},
	{  65, 156, "/dev/scsi/host2/bus0/target2/lun0/part28"},
	{  65, 157, "/dev/scsi/host2/bus0/target2/lun0/part29"},
	{  65, 158, "/dev/scsi/host2/bus0/target2/lun0/part30"},
	{  65, 159, "/dev/scsi/host2/bus0/target2/lun0/part31"},
	{  65, 160, "/dev/scsi/host2/bus0/target2/lun0/part32"},
	{  65, 161, "/dev/scsi/host2/bus0/target2/lun0/part33"},
	{  65, 162, "/dev/scsi/host2/bus0/target2/lun0/part34"},
	{  65, 163, "/dev/scsi/host2/bus0/target2/lun0/part35"},
	{  65, 164, "/dev/scsi/host2/bus0/target2/lun0/part36"},
	{  65, 165, "/dev/scsi/host2/bus0/target2/lun0/part37"},
	{  65, 166, "/dev/scsi/host2/bus0/target2/lun0/part38"},
	{  65, 167, "/dev/scsi/host2/bus0/target2/lun0/part39"},
	{  65, 168, "/dev/scsi/host2/bus0/target2/lun0/part40"},
	{  65, 169, "/dev/scsi/host2/bus0/target2/lun0/part41"},
	{  65, 170, "/dev/scsi/host2/bus0/target2/lun0/part42"},
	{  65, 171, "/dev/scsi/host2/bus0/target2/lun0/part43"},
	{  65, 172, "/dev/scsi/host2/bus0/target2/lun0/part44"},
	{  65, 173, "/dev/scsi/host2/bus0/target2/lun0/part45"},
	{  65, 174, "/dev/scsi/host2/bus0/target2/lun0/part46"},
	{  65, 175, "/dev/scsi/host2/bus0/target2/lun0/part47"},
	{  65, 176, "/dev/scsi/host2/bus0/target2/lun0/part48"},
	{  65, 177, "/dev/scsi/host2/bus0/target2/lun0/part49"},
	{  65, 178, "/dev/scsi/host2/bus0/target2/lun0/part50"},
	{  65, 179, "/dev/scsi/host2/bus0/target2/lun0/part51"},
	{  65, 180, "/dev/scsi/host2/bus0/target2/lun0/part52"},
	{  65, 181, "/dev/scsi/host2/bus0/target2/lun0/part53"},
	{  65, 182, "/dev/scsi/host2/bus0/target2/lun0/part54"},
	{  65, 183, "/dev/scsi/host2/bus0/target2/lun0/part55"},
	{  65, 184, "/dev/scsi/host2/bus0/target2/lun0/part56"},
	{  65, 185, "/dev/scsi/host2/bus0/target2/lun0/part57"},
	{  65, 186, "/dev/scsi/host2/bus0/target2/lun0/part58"},
	{  65, 187, "/dev/scsi/host2/bus0/target2/lun0/part59"},
	{  65, 188, "/dev/scsi/host2/bus0/target2/lun0/part60"},
	{  65, 189, "/dev/scsi/host2/bus0/target2/lun0/part61"},
	{  65, 190, "/dev/scsi/host2/bus0/target2/lun0/part62"},
	{  65, 191, "/dev/scsi/host2/bus0/target2/lun0/part63"},
	{  65, 192, "/dev/scsi/host2/bus0/target3/lun0/disc"},
	{  65, 193, "/dev/scsi/host2/bus0/target3/lun0/part1"},
	{  65, 194, "/dev/scsi/host2/bus0/target3/lun0/part2"},
	{  65, 195, "/dev/scsi/host2/bus0/target3/lun0/part3"},
	{  65, 196, "/dev/scsi/host2/bus0/target3/lun0/part4"},
	{  65, 197, "/dev/scsi/host2/bus0/target3/lun0/part5"},
	{  65, 198, "/dev/scsi/host2/bus0/target3/lun0/part6"},
	{  65, 199, "/dev/scsi/host2/bus0/target3/lun0/part7"},
	{  65, 200, "/dev/scsi/host2/bus0/target3/lun0/part8"},
	{  65, 201, "/dev/scsi/host2/bus0/target3/lun0/part9"},
	{  65, 202, "/dev/scsi/host2/bus0/target3/lun0/part10"},
	{  65, 203, "/dev/scsi/host2/bus0/target3/lun0/part11"},
	{  65, 204, "/dev/scsi/host2/bus0/target3/lun0/part12"},
	{  65, 205, "/dev/scsi/host2/bus0/target3/lun0/part13"},
	{  65, 206, "/dev/scsi/host2/bus0/target3/lun0/part14"},
	{  65, 207, "/dev/scsi/host2/bus0/target3/lun0/part15"},
	{  65, 208, "/dev/scsi/host2/bus0/target3/lun0/part16"},
	{  65, 209, "/dev/scsi/host2/bus0/target3/lun0/part17"},
	{  65, 210, "/dev/scsi/host2/bus0/target3/lun0/part18"},
	{  65, 211, "/dev/scsi/host2/bus0/target3/lun0/part19"},
	{  65, 212, "/dev/scsi/host2/bus0/target3/lun0/part20"},
	{  65, 213, "/dev/scsi/host2/bus0/target3/lun0/part21"},
	{  65, 214, "/dev/scsi/host2/bus0/target3/lun0/part22"},
	{  65, 215, "/dev/scsi/host2/bus0/target3/lun0/part23"},
	{  65, 216, "/dev/scsi/host2/bus0/target3/lun0/part24"},
	{  65, 217, "/dev/scsi/host2/bus0/target3/lun0/part25"},
	{  65, 218, "/dev/scsi/host2/bus0/target3/lun0/part26"},
	{  65, 219, "/dev/scsi/host2/bus0/target3/lun0/part27"},
	{  65, 220, "/dev/scsi/host2/bus0/target3/lun0/part28"},
	{  65, 221, "/dev/scsi/host2/bus0/target3/lun0/part29"},
	{  65, 222, "/dev/scsi/host2/bus0/target3/lun0/part30"},
	{  65, 223, "/dev/scsi/host2/bus0/target3/lun0/part31"},
	{  65, 224, "/dev/scsi/host2/bus0/target3/lun0/part32"},
	{  65, 225, "/dev/scsi/host2/bus0/target3/lun0/part33"},
	{  65, 226, "/dev/scsi/host2/bus0/target3/lun0/part34"},
	{  65, 227, "/dev/scsi/host2/bus0/target3/lun0/part35"},
	{  65, 228, "/dev/scsi/host2/bus0/target3/lun0/part36"},
	{  65, 229, "/dev/scsi/host2/bus0/target3/lun0/part37"},
	{  65, 230, "/dev/scsi/host2/bus0/target3/lun0/part38"},
	{  65, 231, "/dev/scsi/host2/bus0/target3/lun0/part39"},
	{  65, 232, "/dev/scsi/host2/bus0/target3/lun0/part40"},
	{  65, 233, "/dev/scsi/host2/bus0/target3/lun0/part41"},
	{  65, 234, "/dev/scsi/host2/bus0/target3/lun0/part42"},
	{  65, 235, "/dev/scsi/host2/bus0/target3/lun0/part43"},
	{  65, 236, "/dev/scsi/host2/bus0/target3/lun0/part44"},
	{  65, 237, "/dev/scsi/host2/bus0/target3/lun0/part45"},
	{  65, 238, "/dev/scsi/host2/bus0/target3/lun0/part46"},
	{  65, 239, "/dev/scsi/host2/bus0/target3/lun0/part47"},
	{  65, 240, "/dev/scsi/host2/bus0/target3/lun0/part48"},
	{  65, 241, "/dev/scsi/host2/bus0/target3/lun0/part49"},
	{  65, 242, "/dev/scsi/host2/bus0/target3/lun0/part50"},
	{  65, 243, "/dev/scsi/host2/bus0/target3/lun0/part51"},
	{  65, 244, "/dev/scsi/host2/bus0/target3/lun0/part52"},
	{  65, 245, "/dev/scsi/host2/bus0/target3/lun0/part53"},
	{  65, 246, "/dev/scsi/host2/bus0/target3/lun0/part54"},
	{  65, 247, "/dev/scsi/host2/bus0/target3/lun0/part55"},
	{  65, 248, "/dev/scsi/host2/bus0/target3/lun0/part56"},
	{  65, 249, "/dev/scsi/host2/bus0/target3/lun0/part57"},
	{  65, 250, "/dev/scsi/host2/bus0/target3/lun0/part58"},
	{  65, 251, "/dev/scsi/host2/bus0/target3/lun0/part59"},
	{  65, 252, "/dev/scsi/host2/bus0/target3/lun0/part60"},
	{  65, 253, "/dev/scsi/host2/bus0/target3/lun0/part61"},
	{  65, 254, "/dev/scsi/host2/bus0/target3/lun0/part62"},
	{  65, 255, "/dev/scsi/host2/bus0/target3/lun0/part63"},

	{  66,   0, "/dev/scsi/host3/bus0/target0/lun0/disc"},
	{  66,   1, "/dev/scsi/host3/bus0/target0/lun0/part1"},
	{  66,   2, "/dev/scsi/host3/bus0/target0/lun0/part2"},
	{  66,   3, "/dev/scsi/host3/bus0/target0/lun0/part3"},
	{  66,   4, "/dev/scsi/host3/bus0/target0/lun0/part4"},
	{  66,   5, "/dev/scsi/host3/bus0/target0/lun0/part5"},
	{  66,   6, "/dev/scsi/host3/bus0/target0/lun0/part6"},
	{  66,   7, "/dev/scsi/host3/bus0/target0/lun0/part7"},
	{  66,   8, "/dev/scsi/host3/bus0/target0/lun0/part8"},
	{  66,   9, "/dev/scsi/host3/bus0/target0/lun0/part9"},
	{  66,  10, "/dev/scsi/host3/bus0/target0/lun0/part10"},
	{  66,  11, "/dev/scsi/host3/bus0/target0/lun0/part11"},
	{  66,  12, "/dev/scsi/host3/bus0/target0/lun0/part12"},
	{  66,  13, "/dev/scsi/host3/bus0/target0/lun0/part13"},
	{  66,  14, "/dev/scsi/host3/bus0/target0/lun0/part14"},
	{  66,  15, "/dev/scsi/host3/bus0/target0/lun0/part15"},
	{  66,  16, "/dev/scsi/host3/bus0/target0/lun0/part16"},
	{  66,  17, "/dev/scsi/host3/bus0/target0/lun0/part17"},
	{  66,  18, "/dev/scsi/host3/bus0/target0/lun0/part18"},
	{  66,  19, "/dev/scsi/host3/bus0/target0/lun0/part19"},
	{  66,  20, "/dev/scsi/host3/bus0/target0/lun0/part20"},
	{  66,  21, "/dev/scsi/host3/bus0/target0/lun0/part21"},
	{  66,  22, "/dev/scsi/host3/bus0/target0/lun0/part22"},
	{  66,  23, "/dev/scsi/host3/bus0/target0/lun0/part23"},
	{  66,  24, "/dev/scsi/host3/bus0/target0/lun0/part24"},
	{  66,  25, "/dev/scsi/host3/bus0/target0/lun0/part25"},
	{  66,  26, "/dev/scsi/host3/bus0/target0/lun0/part26"},
	{  66,  27, "/dev/scsi/host3/bus0/target0/lun0/part27"},
	{  66,  28, "/dev/scsi/host3/bus0/target0/lun0/part28"},
	{  66,  29, "/dev/scsi/host3/bus0/target0/lun0/part29"},
	{  66,  30, "/dev/scsi/host3/bus0/target0/lun0/part30"},
	{  66,  31, "/dev/scsi/host3/bus0/target0/lun0/part31"},
	{  66,  32, "/dev/scsi/host3/bus0/target0/lun0/part32"},
	{  66,  33, "/dev/scsi/host3/bus0/target0/lun0/part33"},
	{  66,  34, "/dev/scsi/host3/bus0/target0/lun0/part34"},
	{  66,  35, "/dev/scsi/host3/bus0/target0/lun0/part35"},
	{  66,  36, "/dev/scsi/host3/bus0/target0/lun0/part36"},
	{  66,  37, "/dev/scsi/host3/bus0/target0/lun0/part37"},
	{  66,  38, "/dev/scsi/host3/bus0/target0/lun0/part38"},
	{  66,  39, "/dev/scsi/host3/bus0/target0/lun0/part39"},
	{  66,  40, "/dev/scsi/host3/bus0/target0/lun0/part40"},
	{  66,  41, "/dev/scsi/host3/bus0/target0/lun0/part41"},
	{  66,  42, "/dev/scsi/host3/bus0/target0/lun0/part42"},
	{  66,  43, "/dev/scsi/host3/bus0/target0/lun0/part43"},
	{  66,  44, "/dev/scsi/host3/bus0/target0/lun0/part44"},
	{  66,  45, "/dev/scsi/host3/bus0/target0/lun0/part45"},
	{  66,  46, "/dev/scsi/host3/bus0/target0/lun0/part46"},
	{  66,  47, "/dev/scsi/host3/bus0/target0/lun0/part47"},
	{  66,  48, "/dev/scsi/host3/bus0/target0/lun0/part48"},
	{  66,  49, "/dev/scsi/host3/bus0/target0/lun0/part49"},
	{  66,  50, "/dev/scsi/host3/bus0/target0/lun0/part50"},
	{  66,  51, "/dev/scsi/host3/bus0/target0/lun0/part51"},
	{  66,  52, "/dev/scsi/host3/bus0/target0/lun0/part52"},
	{  66,  53, "/dev/scsi/host3/bus0/target0/lun0/part53"},
	{  66,  54, "/dev/scsi/host3/bus0/target0/lun0/part54"},
	{  66,  55, "/dev/scsi/host3/bus0/target0/lun0/part55"},
	{  66,  56, "/dev/scsi/host3/bus0/target0/lun0/part56"},
	{  66,  57, "/dev/scsi/host3/bus0/target0/lun0/part57"},
	{  66,  58, "/dev/scsi/host3/bus0/target0/lun0/part58"},
	{  66,  59, "/dev/scsi/host3/bus0/target0/lun0/part59"},
	{  66,  60, "/dev/scsi/host3/bus0/target0/lun0/part60"},
	{  66,  61, "/dev/scsi/host3/bus0/target0/lun0/part61"},
	{  66,  62, "/dev/scsi/host3/bus0/target0/lun0/part62"},
	{  66,  63, "/dev/scsi/host3/bus0/target0/lun0/part63"},
	{  66,  64, "/dev/scsi/host3/bus0/target1/lun0/disc"},
	{  66,  65, "/dev/scsi/host3/bus0/target1/lun0/part1"},
	{  66,  66, "/dev/scsi/host3/bus0/target1/lun0/part2"},
	{  66,  67, "/dev/scsi/host3/bus0/target1/lun0/part3"},
	{  66,  68, "/dev/scsi/host3/bus0/target1/lun0/part4"},
	{  66,  69, "/dev/scsi/host3/bus0/target1/lun0/part5"},
	{  66,  70, "/dev/scsi/host3/bus0/target1/lun0/part6"},
	{  66,  71, "/dev/scsi/host3/bus0/target1/lun0/part7"},
	{  66,  72, "/dev/scsi/host3/bus0/target1/lun0/part8"},
	{  66,  73, "/dev/scsi/host3/bus0/target1/lun0/part9"},
	{  66,  74, "/dev/scsi/host3/bus0/target1/lun0/part10"},
	{  66,  75, "/dev/scsi/host3/bus0/target1/lun0/part11"},
	{  66,  76, "/dev/scsi/host3/bus0/target1/lun0/part12"},
	{  66,  77, "/dev/scsi/host3/bus0/target1/lun0/part13"},
	{  66,  78, "/dev/scsi/host3/bus0/target1/lun0/part14"},
	{  66,  79, "/dev/scsi/host3/bus0/target1/lun0/part15"},
	{  66,  80, "/dev/scsi/host3/bus0/target1/lun0/part16"},
	{  66,  81, "/dev/scsi/host3/bus0/target1/lun0/part17"},
	{  66,  82, "/dev/scsi/host3/bus0/target1/lun0/part18"},
	{  66,  83, "/dev/scsi/host3/bus0/target1/lun0/part19"},
	{  66,  84, "/dev/scsi/host3/bus0/target1/lun0/part20"},
	{  66,  85, "/dev/scsi/host3/bus0/target1/lun0/part21"},
	{  66,  86, "/dev/scsi/host3/bus0/target1/lun0/part22"},
	{  66,  87, "/dev/scsi/host3/bus0/target1/lun0/part23"},
	{  66,  88, "/dev/scsi/host3/bus0/target1/lun0/part24"},
	{  66,  89, "/dev/scsi/host3/bus0/target1/lun0/part25"},
	{  66,  90, "/dev/scsi/host3/bus0/target1/lun0/part26"},
	{  66,  91, "/dev/scsi/host3/bus0/target1/lun0/part27"},
	{  66,  92, "/dev/scsi/host3/bus0/target1/lun0/part28"},
	{  66,  93, "/dev/scsi/host3/bus0/target1/lun0/part29"},
	{  66,  94, "/dev/scsi/host3/bus0/target1/lun0/part30"},
	{  66,  95, "/dev/scsi/host3/bus0/target1/lun0/part31"},
	{  66,  96, "/dev/scsi/host3/bus0/target1/lun0/part32"},
	{  66,  97, "/dev/scsi/host3/bus0/target1/lun0/part33"},
	{  66,  98, "/dev/scsi/host3/bus0/target1/lun0/part34"},
	{  66,  99, "/dev/scsi/host3/bus0/target1/lun0/part35"},
	{  66, 100, "/dev/scsi/host3/bus0/target1/lun0/part36"},
	{  66, 101, "/dev/scsi/host3/bus0/target1/lun0/part37"},
	{  66, 102, "/dev/scsi/host3/bus0/target1/lun0/part38"},
	{  66, 103, "/dev/scsi/host3/bus0/target1/lun0/part39"},
	{  66, 104, "/dev/scsi/host3/bus0/target1/lun0/part40"},
	{  66, 105, "/dev/scsi/host3/bus0/target1/lun0/part41"},
	{  66, 106, "/dev/scsi/host3/bus0/target1/lun0/part42"},
	{  66, 107, "/dev/scsi/host3/bus0/target1/lun0/part43"},
	{  66, 108, "/dev/scsi/host3/bus0/target1/lun0/part44"},
	{  66, 109, "/dev/scsi/host3/bus0/target1/lun0/part45"},
	{  66, 110, "/dev/scsi/host3/bus0/target1/lun0/part46"},
	{  66, 111, "/dev/scsi/host3/bus0/target1/lun0/part47"},
	{  66, 112, "/dev/scsi/host3/bus0/target1/lun0/part48"},
	{  66, 113, "/dev/scsi/host3/bus0/target1/lun0/part49"},
	{  66, 114, "/dev/scsi/host3/bus0/target1/lun0/part50"},
	{  66, 115, "/dev/scsi/host3/bus0/target1/lun0/part51"},
	{  66, 116, "/dev/scsi/host3/bus0/target1/lun0/part52"},
	{  66, 117, "/dev/scsi/host3/bus0/target1/lun0/part53"},
	{  66, 118, "/dev/scsi/host3/bus0/target1/lun0/part54"},
	{  66, 119, "/dev/scsi/host3/bus0/target1/lun0/part55"},
	{  66, 120, "/dev/scsi/host3/bus0/target1/lun0/part56"},
	{  66, 121, "/dev/scsi/host3/bus0/target1/lun0/part57"},
	{  66, 122, "/dev/scsi/host3/bus0/target1/lun0/part58"},
	{  66, 123, "/dev/scsi/host3/bus0/target1/lun0/part59"},
	{  66, 124, "/dev/scsi/host3/bus0/target1/lun0/part60"},
	{  66, 125, "/dev/scsi/host3/bus0/target1/lun0/part61"},
	{  66, 126, "/dev/scsi/host3/bus0/target1/lun0/part62"},
	{  66, 127, "/dev/scsi/host3/bus0/target1/lun0/part63"},
	{  66, 128, "/dev/scsi/host3/bus0/target2/lun0/disc"},
	{  66, 129, "/dev/scsi/host3/bus0/target2/lun0/part1"},
	{  66, 130, "/dev/scsi/host3/bus0/target2/lun0/part2"},
	{  66, 131, "/dev/scsi/host3/bus0/target2/lun0/part3"},
	{  66, 132, "/dev/scsi/host3/bus0/target2/lun0/part4"},
	{  66, 133, "/dev/scsi/host3/bus0/target2/lun0/part5"},
	{  66, 134, "/dev/scsi/host3/bus0/target2/lun0/part6"},
	{  66, 135, "/dev/scsi/host3/bus0/target2/lun0/part7"},
	{  66, 136, "/dev/scsi/host3/bus0/target2/lun0/part8"},
	{  66, 137, "/dev/scsi/host3/bus0/target2/lun0/part9"},
	{  66, 138, "/dev/scsi/host3/bus0/target2/lun0/part10"},
	{  66, 139, "/dev/scsi/host3/bus0/target2/lun0/part11"},
	{  66, 140, "/dev/scsi/host3/bus0/target2/lun0/part12"},
	{  66, 141, "/dev/scsi/host3/bus0/target2/lun0/part13"},
	{  66, 142, "/dev/scsi/host3/bus0/target2/lun0/part14"},
	{  66, 143, "/dev/scsi/host3/bus0/target2/lun0/part15"},
	{  66, 144, "/dev/scsi/host3/bus0/target2/lun0/part16"},
	{  66, 145, "/dev/scsi/host3/bus0/target2/lun0/part17"},
	{  66, 146, "/dev/scsi/host3/bus0/target2/lun0/part18"},
	{  66, 147, "/dev/scsi/host3/bus0/target2/lun0/part19"},
	{  66, 148, "/dev/scsi/host3/bus0/target2/lun0/part20"},
	{  66, 149, "/dev/scsi/host3/bus0/target2/lun0/part21"},
	{  66, 150, "/dev/scsi/host3/bus0/target2/lun0/part22"},
	{  66, 151, "/dev/scsi/host3/bus0/target2/lun0/part23"},
	{  66, 152, "/dev/scsi/host3/bus0/target2/lun0/part24"},
	{  66, 153, "/dev/scsi/host3/bus0/target2/lun0/part25"},
	{  66, 154, "/dev/scsi/host3/bus0/target2/lun0/part26"},
	{  66, 155, "/dev/scsi/host3/bus0/target2/lun0/part27"},
	{  66, 156, "/dev/scsi/host3/bus0/target2/lun0/part28"},
	{  66, 157, "/dev/scsi/host3/bus0/target2/lun0/part29"},
	{  66, 158, "/dev/scsi/host3/bus0/target2/lun0/part30"},
	{  66, 159, "/dev/scsi/host3/bus0/target2/lun0/part31"},
	{  66, 160, "/dev/scsi/host3/bus0/target2/lun0/part32"},
	{  66, 161, "/dev/scsi/host3/bus0/target2/lun0/part33"},
	{  66, 162, "/dev/scsi/host3/bus0/target2/lun0/part34"},
	{  66, 163, "/dev/scsi/host3/bus0/target2/lun0/part35"},
	{  66, 164, "/dev/scsi/host3/bus0/target2/lun0/part36"},
	{  66, 165, "/dev/scsi/host3/bus0/target2/lun0/part37"},
	{  66, 166, "/dev/scsi/host3/bus0/target2/lun0/part38"},
	{  66, 167, "/dev/scsi/host3/bus0/target2/lun0/part39"},
	{  66, 168, "/dev/scsi/host3/bus0/target2/lun0/part40"},
	{  66, 169, "/dev/scsi/host3/bus0/target2/lun0/part41"},
	{  66, 170, "/dev/scsi/host3/bus0/target2/lun0/part42"},
	{  66, 171, "/dev/scsi/host3/bus0/target2/lun0/part43"},
	{  66, 172, "/dev/scsi/host3/bus0/target2/lun0/part44"},
	{  66, 173, "/dev/scsi/host3/bus0/target2/lun0/part45"},
	{  66, 174, "/dev/scsi/host3/bus0/target2/lun0/part46"},
	{  66, 175, "/dev/scsi/host3/bus0/target2/lun0/part47"},
	{  66, 176, "/dev/scsi/host3/bus0/target2/lun0/part48"},
	{  66, 177, "/dev/scsi/host3/bus0/target2/lun0/part49"},
	{  66, 178, "/dev/scsi/host3/bus0/target2/lun0/part50"},
	{  66, 179, "/dev/scsi/host3/bus0/target2/lun0/part51"},
	{  66, 180, "/dev/scsi/host3/bus0/target2/lun0/part52"},
	{  66, 181, "/dev/scsi/host3/bus0/target2/lun0/part53"},
	{  66, 182, "/dev/scsi/host3/bus0/target2/lun0/part54"},
	{  66, 183, "/dev/scsi/host3/bus0/target2/lun0/part55"},
	{  66, 184, "/dev/scsi/host3/bus0/target2/lun0/part56"},
	{  66, 185, "/dev/scsi/host3/bus0/target2/lun0/part57"},
	{  66, 186, "/dev/scsi/host3/bus0/target2/lun0/part58"},
	{  66, 187, "/dev/scsi/host3/bus0/target2/lun0/part59"},
	{  66, 188, "/dev/scsi/host3/bus0/target2/lun0/part60"},
	{  66, 189, "/dev/scsi/host3/bus0/target2/lun0/part61"},
	{  66, 190, "/dev/scsi/host3/bus0/target2/lun0/part62"},
	{  66, 191, "/dev/scsi/host3/bus0/target2/lun0/part63"},
	{  66, 192, "/dev/scsi/host3/bus0/target3/lun0/disc"},
	{  66, 193, "/dev/scsi/host3/bus0/target3/lun0/part1"},
	{  66, 194, "/dev/scsi/host3/bus0/target3/lun0/part2"},
	{  66, 195, "/dev/scsi/host3/bus0/target3/lun0/part3"},
	{  66, 196, "/dev/scsi/host3/bus0/target3/lun0/part4"},
	{  66, 197, "/dev/scsi/host3/bus0/target3/lun0/part5"},
	{  66, 198, "/dev/scsi/host3/bus0/target3/lun0/part6"},
	{  66, 199, "/dev/scsi/host3/bus0/target3/lun0/part7"},
	{  66, 200, "/dev/scsi/host3/bus0/target3/lun0/part8"},
	{  66, 201, "/dev/scsi/host3/bus0/target3/lun0/part9"},
	{  66, 202, "/dev/scsi/host3/bus0/target3/lun0/part10"},
	{  66, 203, "/dev/scsi/host3/bus0/target3/lun0/part11"},
	{  66, 204, "/dev/scsi/host3/bus0/target3/lun0/part12"},
	{  66, 205, "/dev/scsi/host3/bus0/target3/lun0/part13"},
	{  66, 206, "/dev/scsi/host3/bus0/target3/lun0/part14"},
	{  66, 207, "/dev/scsi/host3/bus0/target3/lun0/part15"},
	{  66, 208, "/dev/scsi/host3/bus0/target3/lun0/part16"},
	{  66, 209, "/dev/scsi/host3/bus0/target3/lun0/part17"},
	{  66, 210, "/dev/scsi/host3/bus0/target3/lun0/part18"},
	{  66, 211, "/dev/scsi/host3/bus0/target3/lun0/part19"},
	{  66, 212, "/dev/scsi/host3/bus0/target3/lun0/part20"},
	{  66, 213, "/dev/scsi/host3/bus0/target3/lun0/part21"},
	{  66, 214, "/dev/scsi/host3/bus0/target3/lun0/part22"},
	{  66, 215, "/dev/scsi/host3/bus0/target3/lun0/part23"},
	{  66, 216, "/dev/scsi/host3/bus0/target3/lun0/part24"},
	{  66, 217, "/dev/scsi/host3/bus0/target3/lun0/part25"},
	{  66, 218, "/dev/scsi/host3/bus0/target3/lun0/part26"},
	{  66, 219, "/dev/scsi/host3/bus0/target3/lun0/part27"},
	{  66, 220, "/dev/scsi/host3/bus0/target3/lun0/part28"},
	{  66, 221, "/dev/scsi/host3/bus0/target3/lun0/part29"},
	{  66, 222, "/dev/scsi/host3/bus0/target3/lun0/part30"},
	{  66, 223, "/dev/scsi/host3/bus0/target3/lun0/part31"},
	{  66, 224, "/dev/scsi/host3/bus0/target3/lun0/part32"},
	{  66, 225, "/dev/scsi/host3/bus0/target3/lun0/part33"},
	{  66, 226, "/dev/scsi/host3/bus0/target3/lun0/part34"},
	{  66, 227, "/dev/scsi/host3/bus0/target3/lun0/part35"},
	{  66, 228, "/dev/scsi/host3/bus0/target3/lun0/part36"},
	{  66, 229, "/dev/scsi/host3/bus0/target3/lun0/part37"},
	{  66, 230, "/dev/scsi/host3/bus0/target3/lun0/part38"},
	{  66, 231, "/dev/scsi/host3/bus0/target3/lun0/part39"},
	{  66, 232, "/dev/scsi/host3/bus0/target3/lun0/part40"},
	{  66, 233, "/dev/scsi/host3/bus0/target3/lun0/part41"},
	{  66, 234, "/dev/scsi/host3/bus0/target3/lun0/part42"},
	{  66, 235, "/dev/scsi/host3/bus0/target3/lun0/part43"},
	{  66, 236, "/dev/scsi/host3/bus0/target3/lun0/part44"},
	{  66, 237, "/dev/scsi/host3/bus0/target3/lun0/part45"},
	{  66, 238, "/dev/scsi/host3/bus0/target3/lun0/part46"},
	{  66, 239, "/dev/scsi/host3/bus0/target3/lun0/part47"},
	{  66, 240, "/dev/scsi/host3/bus0/target3/lun0/part48"},
	{  66, 241, "/dev/scsi/host3/bus0/target3/lun0/part49"},
	{  66, 242, "/dev/scsi/host3/bus0/target3/lun0/part50"},
	{  66, 243, "/dev/scsi/host3/bus0/target3/lun0/part51"},
	{  66, 244, "/dev/scsi/host3/bus0/target3/lun0/part52"},
	{  66, 245, "/dev/scsi/host3/bus0/target3/lun0/part53"},
	{  66, 246, "/dev/scsi/host3/bus0/target3/lun0/part54"},
	{  66, 247, "/dev/scsi/host3/bus0/target3/lun0/part55"},
	{  66, 248, "/dev/scsi/host3/bus0/target3/lun0/part56"},
	{  66, 249, "/dev/scsi/host3/bus0/target3/lun0/part57"},
	{  66, 250, "/dev/scsi/host3/bus0/target3/lun0/part58"},
	{  66, 251, "/dev/scsi/host3/bus0/target3/lun0/part59"},
	{  66, 252, "/dev/scsi/host3/bus0/target3/lun0/part60"},
	{  66, 253, "/dev/scsi/host3/bus0/target3/lun0/part61"},
	{  66, 254, "/dev/scsi/host3/bus0/target3/lun0/part62"},
	{  66, 255, "/dev/scsi/host3/bus0/target3/lun0/part63"},

	{  67,   0, "/dev/scsi/host4/bus0/target0/lun0/disc"},
	{  67,   1, "/dev/scsi/host4/bus0/target0/lun0/part1"},
	{  67,   2, "/dev/scsi/host4/bus0/target0/lun0/part2"},
	{  67,   3, "/dev/scsi/host4/bus0/target0/lun0/part3"},
	{  67,   4, "/dev/scsi/host4/bus0/target0/lun0/part4"},
	{  67,   5, "/dev/scsi/host4/bus0/target0/lun0/part5"},
	{  67,   6, "/dev/scsi/host4/bus0/target0/lun0/part6"},
	{  67,   7, "/dev/scsi/host4/bus0/target0/lun0/part7"},
	{  67,   8, "/dev/scsi/host4/bus0/target0/lun0/part8"},
	{  67,   9, "/dev/scsi/host4/bus0/target0/lun0/part9"},
	{  67,  10, "/dev/scsi/host4/bus0/target0/lun0/part10"},
	{  67,  11, "/dev/scsi/host4/bus0/target0/lun0/part11"},
	{  67,  12, "/dev/scsi/host4/bus0/target0/lun0/part12"},
	{  67,  13, "/dev/scsi/host4/bus0/target0/lun0/part13"},
	{  67,  14, "/dev/scsi/host4/bus0/target0/lun0/part14"},
	{  67,  15, "/dev/scsi/host4/bus0/target0/lun0/part15"},
	{  67,  16, "/dev/scsi/host4/bus0/target0/lun0/part16"},
	{  67,  17, "/dev/scsi/host4/bus0/target0/lun0/part17"},
	{  67,  18, "/dev/scsi/host4/bus0/target0/lun0/part18"},
	{  67,  19, "/dev/scsi/host4/bus0/target0/lun0/part19"},
	{  67,  20, "/dev/scsi/host4/bus0/target0/lun0/part20"},
	{  67,  21, "/dev/scsi/host4/bus0/target0/lun0/part21"},
	{  67,  22, "/dev/scsi/host4/bus0/target0/lun0/part22"},
	{  67,  23, "/dev/scsi/host4/bus0/target0/lun0/part23"},
	{  67,  24, "/dev/scsi/host4/bus0/target0/lun0/part24"},
	{  67,  25, "/dev/scsi/host4/bus0/target0/lun0/part25"},
	{  67,  26, "/dev/scsi/host4/bus0/target0/lun0/part26"},
	{  67,  27, "/dev/scsi/host4/bus0/target0/lun0/part27"},
	{  67,  28, "/dev/scsi/host4/bus0/target0/lun0/part28"},
	{  67,  29, "/dev/scsi/host4/bus0/target0/lun0/part29"},
	{  67,  30, "/dev/scsi/host4/bus0/target0/lun0/part30"},
	{  67,  31, "/dev/scsi/host4/bus0/target0/lun0/part31"},
	{  67,  32, "/dev/scsi/host4/bus0/target0/lun0/part32"},
	{  67,  33, "/dev/scsi/host4/bus0/target0/lun0/part33"},
	{  67,  34, "/dev/scsi/host4/bus0/target0/lun0/part34"},
	{  67,  35, "/dev/scsi/host4/bus0/target0/lun0/part35"},
	{  67,  36, "/dev/scsi/host4/bus0/target0/lun0/part36"},
	{  67,  37, "/dev/scsi/host4/bus0/target0/lun0/part37"},
	{  67,  38, "/dev/scsi/host4/bus0/target0/lun0/part38"},
	{  67,  39, "/dev/scsi/host4/bus0/target0/lun0/part39"},
	{  67,  40, "/dev/scsi/host4/bus0/target0/lun0/part40"},
	{  67,  41, "/dev/scsi/host4/bus0/target0/lun0/part41"},
	{  67,  42, "/dev/scsi/host4/bus0/target0/lun0/part42"},
	{  67,  43, "/dev/scsi/host4/bus0/target0/lun0/part43"},
	{  67,  44, "/dev/scsi/host4/bus0/target0/lun0/part44"},
	{  67,  45, "/dev/scsi/host4/bus0/target0/lun0/part45"},
	{  67,  46, "/dev/scsi/host4/bus0/target0/lun0/part46"},
	{  67,  47, "/dev/scsi/host4/bus0/target0/lun0/part47"},
	{  67,  48, "/dev/scsi/host4/bus0/target0/lun0/part48"},
	{  67,  49, "/dev/scsi/host4/bus0/target0/lun0/part49"},
	{  67,  50, "/dev/scsi/host4/bus0/target0/lun0/part50"},
	{  67,  51, "/dev/scsi/host4/bus0/target0/lun0/part51"},
	{  67,  52, "/dev/scsi/host4/bus0/target0/lun0/part52"},
	{  67,  53, "/dev/scsi/host4/bus0/target0/lun0/part53"},
	{  67,  54, "/dev/scsi/host4/bus0/target0/lun0/part54"},
	{  67,  55, "/dev/scsi/host4/bus0/target0/lun0/part55"},
	{  67,  56, "/dev/scsi/host4/bus0/target0/lun0/part56"},
	{  67,  57, "/dev/scsi/host4/bus0/target0/lun0/part57"},
	{  67,  58, "/dev/scsi/host4/bus0/target0/lun0/part58"},
	{  67,  59, "/dev/scsi/host4/bus0/target0/lun0/part59"},
	{  67,  60, "/dev/scsi/host4/bus0/target0/lun0/part60"},
	{  67,  61, "/dev/scsi/host4/bus0/target0/lun0/part61"},
	{  67,  62, "/dev/scsi/host4/bus0/target0/lun0/part62"},
	{  67,  63, "/dev/scsi/host4/bus0/target0/lun0/part63"},
	{  67,  64, "/dev/scsi/host4/bus0/target1/lun0/disc"},
	{  67,  65, "/dev/scsi/host4/bus0/target1/lun0/part1"},
	{  67,  66, "/dev/scsi/host4/bus0/target1/lun0/part2"},
	{  67,  67, "/dev/scsi/host4/bus0/target1/lun0/part3"},
	{  67,  68, "/dev/scsi/host4/bus0/target1/lun0/part4"},
	{  67,  69, "/dev/scsi/host4/bus0/target1/lun0/part5"},
	{  67,  70, "/dev/scsi/host4/bus0/target1/lun0/part6"},
	{  67,  71, "/dev/scsi/host4/bus0/target1/lun0/part7"},
	{  67,  72, "/dev/scsi/host4/bus0/target1/lun0/part8"},
	{  67,  73, "/dev/scsi/host4/bus0/target1/lun0/part9"},
	{  67,  74, "/dev/scsi/host4/bus0/target1/lun0/part10"},
	{  67,  75, "/dev/scsi/host4/bus0/target1/lun0/part11"},
	{  67,  76, "/dev/scsi/host4/bus0/target1/lun0/part12"},
	{  67,  77, "/dev/scsi/host4/bus0/target1/lun0/part13"},
	{  67,  78, "/dev/scsi/host4/bus0/target1/lun0/part14"},
	{  67,  79, "/dev/scsi/host4/bus0/target1/lun0/part15"},
	{  67,  80, "/dev/scsi/host4/bus0/target1/lun0/part16"},
	{  67,  81, "/dev/scsi/host4/bus0/target1/lun0/part17"},
	{  67,  82, "/dev/scsi/host4/bus0/target1/lun0/part18"},
	{  67,  83, "/dev/scsi/host4/bus0/target1/lun0/part19"},
	{  67,  84, "/dev/scsi/host4/bus0/target1/lun0/part20"},
	{  67,  85, "/dev/scsi/host4/bus0/target1/lun0/part21"},
	{  67,  86, "/dev/scsi/host4/bus0/target1/lun0/part22"},
	{  67,  87, "/dev/scsi/host4/bus0/target1/lun0/part23"},
	{  67,  88, "/dev/scsi/host4/bus0/target1/lun0/part24"},
	{  67,  89, "/dev/scsi/host4/bus0/target1/lun0/part25"},
	{  67,  90, "/dev/scsi/host4/bus0/target1/lun0/part26"},
	{  67,  91, "/dev/scsi/host4/bus0/target1/lun0/part27"},
	{  67,  92, "/dev/scsi/host4/bus0/target1/lun0/part28"},
	{  67,  93, "/dev/scsi/host4/bus0/target1/lun0/part29"},
	{  67,  94, "/dev/scsi/host4/bus0/target1/lun0/part30"},
	{  67,  95, "/dev/scsi/host4/bus0/target1/lun0/part31"},
	{  67,  96, "/dev/scsi/host4/bus0/target1/lun0/part32"},
	{  67,  97, "/dev/scsi/host4/bus0/target1/lun0/part33"},
	{  67,  98, "/dev/scsi/host4/bus0/target1/lun0/part34"},
	{  67,  99, "/dev/scsi/host4/bus0/target1/lun0/part35"},
	{  67, 100, "/dev/scsi/host4/bus0/target1/lun0/part36"},
	{  67, 101, "/dev/scsi/host4/bus0/target1/lun0/part37"},
	{  67, 102, "/dev/scsi/host4/bus0/target1/lun0/part38"},
	{  67, 103, "/dev/scsi/host4/bus0/target1/lun0/part39"},
	{  67, 104, "/dev/scsi/host4/bus0/target1/lun0/part40"},
	{  67, 105, "/dev/scsi/host4/bus0/target1/lun0/part41"},
	{  67, 106, "/dev/scsi/host4/bus0/target1/lun0/part42"},
	{  67, 107, "/dev/scsi/host4/bus0/target1/lun0/part43"},
	{  67, 108, "/dev/scsi/host4/bus0/target1/lun0/part44"},
	{  67, 109, "/dev/scsi/host4/bus0/target1/lun0/part45"},
	{  67, 110, "/dev/scsi/host4/bus0/target1/lun0/part46"},
	{  67, 111, "/dev/scsi/host4/bus0/target1/lun0/part47"},
	{  67, 112, "/dev/scsi/host4/bus0/target1/lun0/part48"},
	{  67, 113, "/dev/scsi/host4/bus0/target1/lun0/part49"},
	{  67, 114, "/dev/scsi/host4/bus0/target1/lun0/part50"},
	{  67, 115, "/dev/scsi/host4/bus0/target1/lun0/part51"},
	{  67, 116, "/dev/scsi/host4/bus0/target1/lun0/part52"},
	{  67, 117, "/dev/scsi/host4/bus0/target1/lun0/part53"},
	{  67, 118, "/dev/scsi/host4/bus0/target1/lun0/part54"},
	{  67, 119, "/dev/scsi/host4/bus0/target1/lun0/part55"},
	{  67, 120, "/dev/scsi/host4/bus0/target1/lun0/part56"},
	{  67, 121, "/dev/scsi/host4/bus0/target1/lun0/part57"},
	{  67, 122, "/dev/scsi/host4/bus0/target1/lun0/part58"},
	{  67, 123, "/dev/scsi/host4/bus0/target1/lun0/part59"},
	{  67, 124, "/dev/scsi/host4/bus0/target1/lun0/part60"},
	{  67, 125, "/dev/scsi/host4/bus0/target1/lun0/part61"},
	{  67, 126, "/dev/scsi/host4/bus0/target1/lun0/part62"},
	{  67, 127, "/dev/scsi/host4/bus0/target1/lun0/part63"},
	{  67, 128, "/dev/scsi/host4/bus0/target2/lun0/disc"},
	{  67, 129, "/dev/scsi/host4/bus0/target2/lun0/part1"},
	{  67, 130, "/dev/scsi/host4/bus0/target2/lun0/part2"},
	{  67, 131, "/dev/scsi/host4/bus0/target2/lun0/part3"},
	{  67, 132, "/dev/scsi/host4/bus0/target2/lun0/part4"},
	{  67, 133, "/dev/scsi/host4/bus0/target2/lun0/part5"},
	{  67, 134, "/dev/scsi/host4/bus0/target2/lun0/part6"},
	{  67, 135, "/dev/scsi/host4/bus0/target2/lun0/part7"},
	{  67, 136, "/dev/scsi/host4/bus0/target2/lun0/part8"},
	{  67, 137, "/dev/scsi/host4/bus0/target2/lun0/part9"},
	{  67, 138, "/dev/scsi/host4/bus0/target2/lun0/part10"},
	{  67, 139, "/dev/scsi/host4/bus0/target2/lun0/part11"},
	{  67, 140, "/dev/scsi/host4/bus0/target2/lun0/part12"},
	{  67, 141, "/dev/scsi/host4/bus0/target2/lun0/part13"},
	{  67, 142, "/dev/scsi/host4/bus0/target2/lun0/part14"},
	{  67, 143, "/dev/scsi/host4/bus0/target2/lun0/part15"},
	{  67, 144, "/dev/scsi/host4/bus0/target2/lun0/part16"},
	{  67, 145, "/dev/scsi/host4/bus0/target2/lun0/part17"},
	{  67, 146, "/dev/scsi/host4/bus0/target2/lun0/part18"},
	{  67, 147, "/dev/scsi/host4/bus0/target2/lun0/part19"},
	{  67, 148, "/dev/scsi/host4/bus0/target2/lun0/part20"},
	{  67, 149, "/dev/scsi/host4/bus0/target2/lun0/part21"},
	{  67, 150, "/dev/scsi/host4/bus0/target2/lun0/part22"},
	{  67, 151, "/dev/scsi/host4/bus0/target2/lun0/part23"},
	{  67, 152, "/dev/scsi/host4/bus0/target2/lun0/part24"},
	{  67, 153, "/dev/scsi/host4/bus0/target2/lun0/part25"},
	{  67, 154, "/dev/scsi/host4/bus0/target2/lun0/part26"},
	{  67, 155, "/dev/scsi/host4/bus0/target2/lun0/part27"},
	{  67, 156, "/dev/scsi/host4/bus0/target2/lun0/part28"},
	{  67, 157, "/dev/scsi/host4/bus0/target2/lun0/part29"},
	{  67, 158, "/dev/scsi/host4/bus0/target2/lun0/part30"},
	{  67, 159, "/dev/scsi/host4/bus0/target2/lun0/part31"},
	{  67, 160, "/dev/scsi/host4/bus0/target2/lun0/part32"},
	{  67, 161, "/dev/scsi/host4/bus0/target2/lun0/part33"},
	{  67, 162, "/dev/scsi/host4/bus0/target2/lun0/part34"},
	{  67, 163, "/dev/scsi/host4/bus0/target2/lun0/part35"},
	{  67, 164, "/dev/scsi/host4/bus0/target2/lun0/part36"},
	{  67, 165, "/dev/scsi/host4/bus0/target2/lun0/part37"},
	{  67, 166, "/dev/scsi/host4/bus0/target2/lun0/part38"},
	{  67, 167, "/dev/scsi/host4/bus0/target2/lun0/part39"},
	{  67, 168, "/dev/scsi/host4/bus0/target2/lun0/part40"},
	{  67, 169, "/dev/scsi/host4/bus0/target2/lun0/part41"},
	{  67, 170, "/dev/scsi/host4/bus0/target2/lun0/part42"},
	{  67, 171, "/dev/scsi/host4/bus0/target2/lun0/part43"},
	{  67, 172, "/dev/scsi/host4/bus0/target2/lun0/part44"},
	{  67, 173, "/dev/scsi/host4/bus0/target2/lun0/part45"},
	{  67, 174, "/dev/scsi/host4/bus0/target2/lun0/part46"},
	{  67, 175, "/dev/scsi/host4/bus0/target2/lun0/part47"},
	{  67, 176, "/dev/scsi/host4/bus0/target2/lun0/part48"},
	{  67, 177, "/dev/scsi/host4/bus0/target2/lun0/part49"},
	{  67, 178, "/dev/scsi/host4/bus0/target2/lun0/part50"},
	{  67, 179, "/dev/scsi/host4/bus0/target2/lun0/part51"},
	{  67, 180, "/dev/scsi/host4/bus0/target2/lun0/part52"},
	{  67, 181, "/dev/scsi/host4/bus0/target2/lun0/part53"},
	{  67, 182, "/dev/scsi/host4/bus0/target2/lun0/part54"},
	{  67, 183, "/dev/scsi/host4/bus0/target2/lun0/part55"},
	{  67, 184, "/dev/scsi/host4/bus0/target2/lun0/part56"},
	{  67, 185, "/dev/scsi/host4/bus0/target2/lun0/part57"},
	{  67, 186, "/dev/scsi/host4/bus0/target2/lun0/part58"},
	{  67, 187, "/dev/scsi/host4/bus0/target2/lun0/part59"},
	{  67, 188, "/dev/scsi/host4/bus0/target2/lun0/part60"},
	{  67, 189, "/dev/scsi/host4/bus0/target2/lun0/part61"},
	{  67, 190, "/dev/scsi/host4/bus0/target2/lun0/part62"},
	{  67, 191, "/dev/scsi/host4/bus0/target2/lun0/part63"},
	{  67, 192, "/dev/scsi/host4/bus0/target3/lun0/disc"},
	{  67, 193, "/dev/scsi/host4/bus0/target3/lun0/part1"},
	{  67, 194, "/dev/scsi/host4/bus0/target3/lun0/part2"},
	{  67, 195, "/dev/scsi/host4/bus0/target3/lun0/part3"},
	{  67, 196, "/dev/scsi/host4/bus0/target3/lun0/part4"},
	{  67, 197, "/dev/scsi/host4/bus0/target3/lun0/part5"},
	{  67, 198, "/dev/scsi/host4/bus0/target3/lun0/part6"},
	{  67, 199, "/dev/scsi/host4/bus0/target3/lun0/part7"},
	{  67, 200, "/dev/scsi/host4/bus0/target3/lun0/part8"},
	{  67, 201, "/dev/scsi/host4/bus0/target3/lun0/part9"},
	{  67, 202, "/dev/scsi/host4/bus0/target3/lun0/part10"},
	{  67, 203, "/dev/scsi/host4/bus0/target3/lun0/part11"},
	{  67, 204, "/dev/scsi/host4/bus0/target3/lun0/part12"},
	{  67, 205, "/dev/scsi/host4/bus0/target3/lun0/part13"},
	{  67, 206, "/dev/scsi/host4/bus0/target3/lun0/part14"},
	{  67, 207, "/dev/scsi/host4/bus0/target3/lun0/part15"},
	{  67, 208, "/dev/scsi/host4/bus0/target3/lun0/part16"},
	{  67, 209, "/dev/scsi/host4/bus0/target3/lun0/part17"},
	{  67, 210, "/dev/scsi/host4/bus0/target3/lun0/part18"},
	{  67, 211, "/dev/scsi/host4/bus0/target3/lun0/part19"},
	{  67, 212, "/dev/scsi/host4/bus0/target3/lun0/part20"},
	{  67, 213, "/dev/scsi/host4/bus0/target3/lun0/part21"},
	{  67, 214, "/dev/scsi/host4/bus0/target3/lun0/part22"},
	{  67, 215, "/dev/scsi/host4/bus0/target3/lun0/part23"},
	{  67, 216, "/dev/scsi/host4/bus0/target3/lun0/part24"},
	{  67, 217, "/dev/scsi/host4/bus0/target3/lun0/part25"},
	{  67, 218, "/dev/scsi/host4/bus0/target3/lun0/part26"},
	{  67, 219, "/dev/scsi/host4/bus0/target3/lun0/part27"},
	{  67, 220, "/dev/scsi/host4/bus0/target3/lun0/part28"},
	{  67, 221, "/dev/scsi/host4/bus0/target3/lun0/part29"},
	{  67, 222, "/dev/scsi/host4/bus0/target3/lun0/part30"},
	{  67, 223, "/dev/scsi/host4/bus0/target3/lun0/part31"},
	{  67, 224, "/dev/scsi/host4/bus0/target3/lun0/part32"},
	{  67, 225, "/dev/scsi/host4/bus0/target3/lun0/part33"},
	{  67, 226, "/dev/scsi/host4/bus0/target3/lun0/part34"},
	{  67, 227, "/dev/scsi/host4/bus0/target3/lun0/part35"},
	{  67, 228, "/dev/scsi/host4/bus0/target3/lun0/part36"},
	{  67, 229, "/dev/scsi/host4/bus0/target3/lun0/part37"},
	{  67, 230, "/dev/scsi/host4/bus0/target3/lun0/part38"},
	{  67, 231, "/dev/scsi/host4/bus0/target3/lun0/part39"},
	{  67, 232, "/dev/scsi/host4/bus0/target3/lun0/part40"},
	{  67, 233, "/dev/scsi/host4/bus0/target3/lun0/part41"},
	{  67, 234, "/dev/scsi/host4/bus0/target3/lun0/part42"},
	{  67, 235, "/dev/scsi/host4/bus0/target3/lun0/part43"},
	{  67, 236, "/dev/scsi/host4/bus0/target3/lun0/part44"},
	{  67, 237, "/dev/scsi/host4/bus0/target3/lun0/part45"},
	{  67, 238, "/dev/scsi/host4/bus0/target3/lun0/part46"},
	{  67, 239, "/dev/scsi/host4/bus0/target3/lun0/part47"},
	{  67, 240, "/dev/scsi/host4/bus0/target3/lun0/part48"},
	{  67, 241, "/dev/scsi/host4/bus0/target3/lun0/part49"},
	{  67, 242, "/dev/scsi/host4/bus0/target3/lun0/part50"},
	{  67, 243, "/dev/scsi/host4/bus0/target3/lun0/part51"},
	{  67, 244, "/dev/scsi/host4/bus0/target3/lun0/part52"},
	{  67, 245, "/dev/scsi/host4/bus0/target3/lun0/part53"},
	{  67, 246, "/dev/scsi/host4/bus0/target3/lun0/part54"},
	{  67, 247, "/dev/scsi/host4/bus0/target3/lun0/part55"},
	{  67, 248, "/dev/scsi/host4/bus0/target3/lun0/part56"},
	{  67, 249, "/dev/scsi/host4/bus0/target3/lun0/part57"},
	{  67, 250, "/dev/scsi/host4/bus0/target3/lun0/part58"},
	{  67, 251, "/dev/scsi/host4/bus0/target3/lun0/part59"},
	{  67, 252, "/dev/scsi/host4/bus0/target3/lun0/part60"},
	{  67, 253, "/dev/scsi/host4/bus0/target3/lun0/part61"},
	{  67, 254, "/dev/scsi/host4/bus0/target3/lun0/part62"},
	{  67, 255, "/dev/scsi/host4/bus0/target3/lun0/part63"},

	{  68,   0, "/dev/scsi/host5/bus0/target0/lun0/disc"},
	{  68,   1, "/dev/scsi/host5/bus0/target0/lun0/part1"},
	{  68,   2, "/dev/scsi/host5/bus0/target0/lun0/part2"},
	{  68,   3, "/dev/scsi/host5/bus0/target0/lun0/part3"},
	{  68,   4, "/dev/scsi/host5/bus0/target0/lun0/part4"},
	{  68,   5, "/dev/scsi/host5/bus0/target0/lun0/part5"},
	{  68,   6, "/dev/scsi/host5/bus0/target0/lun0/part6"},
	{  68,   7, "/dev/scsi/host5/bus0/target0/lun0/part7"},
	{  68,   8, "/dev/scsi/host5/bus0/target0/lun0/part8"},
	{  68,   9, "/dev/scsi/host5/bus0/target0/lun0/part9"},
	{  68,  10, "/dev/scsi/host5/bus0/target0/lun0/part10"},
	{  68,  11, "/dev/scsi/host5/bus0/target0/lun0/part11"},
	{  68,  12, "/dev/scsi/host5/bus0/target0/lun0/part12"},
	{  68,  13, "/dev/scsi/host5/bus0/target0/lun0/part13"},
	{  68,  14, "/dev/scsi/host5/bus0/target0/lun0/part14"},
	{  68,  15, "/dev/scsi/host5/bus0/target0/lun0/part15"},
	{  68,  16, "/dev/scsi/host5/bus0/target0/lun0/part16"},
	{  68,  17, "/dev/scsi/host5/bus0/target0/lun0/part17"},
	{  68,  18, "/dev/scsi/host5/bus0/target0/lun0/part18"},
	{  68,  19, "/dev/scsi/host5/bus0/target0/lun0/part19"},
	{  68,  20, "/dev/scsi/host5/bus0/target0/lun0/part20"},
	{  68,  21, "/dev/scsi/host5/bus0/target0/lun0/part21"},
	{  68,  22, "/dev/scsi/host5/bus0/target0/lun0/part22"},
	{  68,  23, "/dev/scsi/host5/bus0/target0/lun0/part23"},
	{  68,  24, "/dev/scsi/host5/bus0/target0/lun0/part24"},
	{  68,  25, "/dev/scsi/host5/bus0/target0/lun0/part25"},
	{  68,  26, "/dev/scsi/host5/bus0/target0/lun0/part26"},
	{  68,  27, "/dev/scsi/host5/bus0/target0/lun0/part27"},
	{  68,  28, "/dev/scsi/host5/bus0/target0/lun0/part28"},
	{  68,  29, "/dev/scsi/host5/bus0/target0/lun0/part29"},
	{  68,  30, "/dev/scsi/host5/bus0/target0/lun0/part30"},
	{  68,  31, "/dev/scsi/host5/bus0/target0/lun0/part31"},
	{  68,  32, "/dev/scsi/host5/bus0/target0/lun0/part32"},
	{  68,  33, "/dev/scsi/host5/bus0/target0/lun0/part33"},
	{  68,  34, "/dev/scsi/host5/bus0/target0/lun0/part34"},
	{  68,  35, "/dev/scsi/host5/bus0/target0/lun0/part35"},
	{  68,  36, "/dev/scsi/host5/bus0/target0/lun0/part36"},
	{  68,  37, "/dev/scsi/host5/bus0/target0/lun0/part37"},
	{  68,  38, "/dev/scsi/host5/bus0/target0/lun0/part38"},
	{  68,  39, "/dev/scsi/host5/bus0/target0/lun0/part39"},
	{  68,  40, "/dev/scsi/host5/bus0/target0/lun0/part40"},
	{  68,  41, "/dev/scsi/host5/bus0/target0/lun0/part41"},
	{  68,  42, "/dev/scsi/host5/bus0/target0/lun0/part42"},
	{  68,  43, "/dev/scsi/host5/bus0/target0/lun0/part43"},
	{  68,  44, "/dev/scsi/host5/bus0/target0/lun0/part44"},
	{  68,  45, "/dev/scsi/host5/bus0/target0/lun0/part45"},
	{  68,  46, "/dev/scsi/host5/bus0/target0/lun0/part46"},
	{  68,  47, "/dev/scsi/host5/bus0/target0/lun0/part47"},
	{  68,  48, "/dev/scsi/host5/bus0/target0/lun0/part48"},
	{  68,  49, "/dev/scsi/host5/bus0/target0/lun0/part49"},
	{  68,  50, "/dev/scsi/host5/bus0/target0/lun0/part50"},
	{  68,  51, "/dev/scsi/host5/bus0/target0/lun0/part51"},
	{  68,  52, "/dev/scsi/host5/bus0/target0/lun0/part52"},
	{  68,  53, "/dev/scsi/host5/bus0/target0/lun0/part53"},
	{  68,  54, "/dev/scsi/host5/bus0/target0/lun0/part54"},
	{  68,  55, "/dev/scsi/host5/bus0/target0/lun0/part55"},
	{  68,  56, "/dev/scsi/host5/bus0/target0/lun0/part56"},
	{  68,  57, "/dev/scsi/host5/bus0/target0/lun0/part57"},
	{  68,  58, "/dev/scsi/host5/bus0/target0/lun0/part58"},
	{  68,  59, "/dev/scsi/host5/bus0/target0/lun0/part59"},
	{  68,  60, "/dev/scsi/host5/bus0/target0/lun0/part60"},
	{  68,  61, "/dev/scsi/host5/bus0/target0/lun0/part61"},
	{  68,  62, "/dev/scsi/host5/bus0/target0/lun0/part62"},
	{  68,  63, "/dev/scsi/host5/bus0/target0/lun0/part63"},
	{  68,  64, "/dev/scsi/host5/bus0/target1/lun0/disc"},
	{  68,  65, "/dev/scsi/host5/bus0/target1/lun0/part1"},
	{  68,  66, "/dev/scsi/host5/bus0/target1/lun0/part2"},
	{  68,  67, "/dev/scsi/host5/bus0/target1/lun0/part3"},
	{  68,  68, "/dev/scsi/host5/bus0/target1/lun0/part4"},
	{  68,  69, "/dev/scsi/host5/bus0/target1/lun0/part5"},
	{  68,  70, "/dev/scsi/host5/bus0/target1/lun0/part6"},
	{  68,  71, "/dev/scsi/host5/bus0/target1/lun0/part7"},
	{  68,  72, "/dev/scsi/host5/bus0/target1/lun0/part8"},
	{  68,  73, "/dev/scsi/host5/bus0/target1/lun0/part9"},
	{  68,  74, "/dev/scsi/host5/bus0/target1/lun0/part10"},
	{  68,  75, "/dev/scsi/host5/bus0/target1/lun0/part11"},
	{  68,  76, "/dev/scsi/host5/bus0/target1/lun0/part12"},
	{  68,  77, "/dev/scsi/host5/bus0/target1/lun0/part13"},
	{  68,  78, "/dev/scsi/host5/bus0/target1/lun0/part14"},
	{  68,  79, "/dev/scsi/host5/bus0/target1/lun0/part15"},
	{  68,  80, "/dev/scsi/host5/bus0/target1/lun0/part16"},
	{  68,  81, "/dev/scsi/host5/bus0/target1/lun0/part17"},
	{  68,  82, "/dev/scsi/host5/bus0/target1/lun0/part18"},
	{  68,  83, "/dev/scsi/host5/bus0/target1/lun0/part19"},
	{  68,  84, "/dev/scsi/host5/bus0/target1/lun0/part20"},
	{  68,  85, "/dev/scsi/host5/bus0/target1/lun0/part21"},
	{  68,  86, "/dev/scsi/host5/bus0/target1/lun0/part22"},
	{  68,  87, "/dev/scsi/host5/bus0/target1/lun0/part23"},
	{  68,  88, "/dev/scsi/host5/bus0/target1/lun0/part24"},
	{  68,  89, "/dev/scsi/host5/bus0/target1/lun0/part25"},
	{  68,  90, "/dev/scsi/host5/bus0/target1/lun0/part26"},
	{  68,  91, "/dev/scsi/host5/bus0/target1/lun0/part27"},
	{  68,  92, "/dev/scsi/host5/bus0/target1/lun0/part28"},
	{  68,  93, "/dev/scsi/host5/bus0/target1/lun0/part29"},
	{  68,  94, "/dev/scsi/host5/bus0/target1/lun0/part30"},
	{  68,  95, "/dev/scsi/host5/bus0/target1/lun0/part31"},
	{  68,  96, "/dev/scsi/host5/bus0/target1/lun0/part32"},
	{  68,  97, "/dev/scsi/host5/bus0/target1/lun0/part33"},
	{  68,  98, "/dev/scsi/host5/bus0/target1/lun0/part34"},
	{  68,  99, "/dev/scsi/host5/bus0/target1/lun0/part35"},
	{  68, 100, "/dev/scsi/host5/bus0/target1/lun0/part36"},
	{  68, 101, "/dev/scsi/host5/bus0/target1/lun0/part37"},
	{  68, 102, "/dev/scsi/host5/bus0/target1/lun0/part38"},
	{  68, 103, "/dev/scsi/host5/bus0/target1/lun0/part39"},
	{  68, 104, "/dev/scsi/host5/bus0/target1/lun0/part40"},
	{  68, 105, "/dev/scsi/host5/bus0/target1/lun0/part41"},
	{  68, 106, "/dev/scsi/host5/bus0/target1/lun0/part42"},
	{  68, 107, "/dev/scsi/host5/bus0/target1/lun0/part43"},
	{  68, 108, "/dev/scsi/host5/bus0/target1/lun0/part44"},
	{  68, 109, "/dev/scsi/host5/bus0/target1/lun0/part45"},
	{  68, 110, "/dev/scsi/host5/bus0/target1/lun0/part46"},
	{  68, 111, "/dev/scsi/host5/bus0/target1/lun0/part47"},
	{  68, 112, "/dev/scsi/host5/bus0/target1/lun0/part48"},
	{  68, 113, "/dev/scsi/host5/bus0/target1/lun0/part49"},
	{  68, 114, "/dev/scsi/host5/bus0/target1/lun0/part50"},
	{  68, 115, "/dev/scsi/host5/bus0/target1/lun0/part51"},
	{  68, 116, "/dev/scsi/host5/bus0/target1/lun0/part52"},
	{  68, 117, "/dev/scsi/host5/bus0/target1/lun0/part53"},
	{  68, 118, "/dev/scsi/host5/bus0/target1/lun0/part54"},
	{  68, 119, "/dev/scsi/host5/bus0/target1/lun0/part55"},
	{  68, 120, "/dev/scsi/host5/bus0/target1/lun0/part56"},
	{  68, 121, "/dev/scsi/host5/bus0/target1/lun0/part57"},
	{  68, 122, "/dev/scsi/host5/bus0/target1/lun0/part58"},
	{  68, 123, "/dev/scsi/host5/bus0/target1/lun0/part59"},
	{  68, 124, "/dev/scsi/host5/bus0/target1/lun0/part60"},
	{  68, 125, "/dev/scsi/host5/bus0/target1/lun0/part61"},
	{  68, 126, "/dev/scsi/host5/bus0/target1/lun0/part62"},
	{  68, 127, "/dev/scsi/host5/bus0/target1/lun0/part63"},
	{  68, 128, "/dev/scsi/host5/bus0/target2/lun0/disc"},
	{  68, 129, "/dev/scsi/host5/bus0/target2/lun0/part1"},
	{  68, 130, "/dev/scsi/host5/bus0/target2/lun0/part2"},
	{  68, 131, "/dev/scsi/host5/bus0/target2/lun0/part3"},
	{  68, 132, "/dev/scsi/host5/bus0/target2/lun0/part4"},
	{  68, 133, "/dev/scsi/host5/bus0/target2/lun0/part5"},
	{  68, 134, "/dev/scsi/host5/bus0/target2/lun0/part6"},
	{  68, 135, "/dev/scsi/host5/bus0/target2/lun0/part7"},
	{  68, 136, "/dev/scsi/host5/bus0/target2/lun0/part8"},
	{  68, 137, "/dev/scsi/host5/bus0/target2/lun0/part9"},
	{  68, 138, "/dev/scsi/host5/bus0/target2/lun0/part10"},
	{  68, 139, "/dev/scsi/host5/bus0/target2/lun0/part11"},
	{  68, 140, "/dev/scsi/host5/bus0/target2/lun0/part12"},
	{  68, 141, "/dev/scsi/host5/bus0/target2/lun0/part13"},
	{  68, 142, "/dev/scsi/host5/bus0/target2/lun0/part14"},
	{  68, 143, "/dev/scsi/host5/bus0/target2/lun0/part15"},
	{  68, 144, "/dev/scsi/host5/bus0/target2/lun0/part16"},
	{  68, 145, "/dev/scsi/host5/bus0/target2/lun0/part17"},
	{  68, 146, "/dev/scsi/host5/bus0/target2/lun0/part18"},
	{  68, 147, "/dev/scsi/host5/bus0/target2/lun0/part19"},
	{  68, 148, "/dev/scsi/host5/bus0/target2/lun0/part20"},
	{  68, 149, "/dev/scsi/host5/bus0/target2/lun0/part21"},
	{  68, 150, "/dev/scsi/host5/bus0/target2/lun0/part22"},
	{  68, 151, "/dev/scsi/host5/bus0/target2/lun0/part23"},
	{  68, 152, "/dev/scsi/host5/bus0/target2/lun0/part24"},
	{  68, 153, "/dev/scsi/host5/bus0/target2/lun0/part25"},
	{  68, 154, "/dev/scsi/host5/bus0/target2/lun0/part26"},
	{  68, 155, "/dev/scsi/host5/bus0/target2/lun0/part27"},
	{  68, 156, "/dev/scsi/host5/bus0/target2/lun0/part28"},
	{  68, 157, "/dev/scsi/host5/bus0/target2/lun0/part29"},
	{  68, 158, "/dev/scsi/host5/bus0/target2/lun0/part30"},
	{  68, 159, "/dev/scsi/host5/bus0/target2/lun0/part31"},
	{  68, 160, "/dev/scsi/host5/bus0/target2/lun0/part32"},
	{  68, 161, "/dev/scsi/host5/bus0/target2/lun0/part33"},
	{  68, 162, "/dev/scsi/host5/bus0/target2/lun0/part34"},
	{  68, 163, "/dev/scsi/host5/bus0/target2/lun0/part35"},
	{  68, 164, "/dev/scsi/host5/bus0/target2/lun0/part36"},
	{  68, 165, "/dev/scsi/host5/bus0/target2/lun0/part37"},
	{  68, 166, "/dev/scsi/host5/bus0/target2/lun0/part38"},
	{  68, 167, "/dev/scsi/host5/bus0/target2/lun0/part39"},
	{  68, 168, "/dev/scsi/host5/bus0/target2/lun0/part40"},
	{  68, 169, "/dev/scsi/host5/bus0/target2/lun0/part41"},
	{  68, 170, "/dev/scsi/host5/bus0/target2/lun0/part42"},
	{  68, 171, "/dev/scsi/host5/bus0/target2/lun0/part43"},
	{  68, 172, "/dev/scsi/host5/bus0/target2/lun0/part44"},
	{  68, 173, "/dev/scsi/host5/bus0/target2/lun0/part45"},
	{  68, 174, "/dev/scsi/host5/bus0/target2/lun0/part46"},
	{  68, 175, "/dev/scsi/host5/bus0/target2/lun0/part47"},
	{  68, 176, "/dev/scsi/host5/bus0/target2/lun0/part48"},
	{  68, 177, "/dev/scsi/host5/bus0/target2/lun0/part49"},
	{  68, 178, "/dev/scsi/host5/bus0/target2/lun0/part50"},
	{  68, 179, "/dev/scsi/host5/bus0/target2/lun0/part51"},
	{  68, 180, "/dev/scsi/host5/bus0/target2/lun0/part52"},
	{  68, 181, "/dev/scsi/host5/bus0/target2/lun0/part53"},
	{  68, 182, "/dev/scsi/host5/bus0/target2/lun0/part54"},
	{  68, 183, "/dev/scsi/host5/bus0/target2/lun0/part55"},
	{  68, 184, "/dev/scsi/host5/bus0/target2/lun0/part56"},
	{  68, 185, "/dev/scsi/host5/bus0/target2/lun0/part57"},
	{  68, 186, "/dev/scsi/host5/bus0/target2/lun0/part58"},
	{  68, 187, "/dev/scsi/host5/bus0/target2/lun0/part59"},
	{  68, 188, "/dev/scsi/host5/bus0/target2/lun0/part60"},
	{  68, 189, "/dev/scsi/host5/bus0/target2/lun0/part61"},
	{  68, 190, "/dev/scsi/host5/bus0/target2/lun0/part62"},
	{  68, 191, "/dev/scsi/host5/bus0/target2/lun0/part63"},
	{  68, 192, "/dev/scsi/host5/bus0/target3/lun0/disc"},
	{  68, 193, "/dev/scsi/host5/bus0/target3/lun0/part1"},
	{  68, 194, "/dev/scsi/host5/bus0/target3/lun0/part2"},
	{  68, 195, "/dev/scsi/host5/bus0/target3/lun0/part3"},
	{  68, 196, "/dev/scsi/host5/bus0/target3/lun0/part4"},
	{  68, 197, "/dev/scsi/host5/bus0/target3/lun0/part5"},
	{  68, 198, "/dev/scsi/host5/bus0/target3/lun0/part6"},
	{  68, 199, "/dev/scsi/host5/bus0/target3/lun0/part7"},
	{  68, 200, "/dev/scsi/host5/bus0/target3/lun0/part8"},
	{  68, 201, "/dev/scsi/host5/bus0/target3/lun0/part9"},
	{  68, 202, "/dev/scsi/host5/bus0/target3/lun0/part10"},
	{  68, 203, "/dev/scsi/host5/bus0/target3/lun0/part11"},
	{  68, 204, "/dev/scsi/host5/bus0/target3/lun0/part12"},
	{  68, 205, "/dev/scsi/host5/bus0/target3/lun0/part13"},
	{  68, 206, "/dev/scsi/host5/bus0/target3/lun0/part14"},
	{  68, 207, "/dev/scsi/host5/bus0/target3/lun0/part15"},
	{  68, 208, "/dev/scsi/host5/bus0/target3/lun0/part16"},
	{  68, 209, "/dev/scsi/host5/bus0/target3/lun0/part17"},
	{  68, 210, "/dev/scsi/host5/bus0/target3/lun0/part18"},
	{  68, 211, "/dev/scsi/host5/bus0/target3/lun0/part19"},
	{  68, 212, "/dev/scsi/host5/bus0/target3/lun0/part20"},
	{  68, 213, "/dev/scsi/host5/bus0/target3/lun0/part21"},
	{  68, 214, "/dev/scsi/host5/bus0/target3/lun0/part22"},
	{  68, 215, "/dev/scsi/host5/bus0/target3/lun0/part23"},
	{  68, 216, "/dev/scsi/host5/bus0/target3/lun0/part24"},
	{  68, 217, "/dev/scsi/host5/bus0/target3/lun0/part25"},
	{  68, 218, "/dev/scsi/host5/bus0/target3/lun0/part26"},
	{  68, 219, "/dev/scsi/host5/bus0/target3/lun0/part27"},
	{  68, 220, "/dev/scsi/host5/bus0/target3/lun0/part28"},
	{  68, 221, "/dev/scsi/host5/bus0/target3/lun0/part29"},
	{  68, 222, "/dev/scsi/host5/bus0/target3/lun0/part30"},
	{  68, 223, "/dev/scsi/host5/bus0/target3/lun0/part31"},
	{  68, 224, "/dev/scsi/host5/bus0/target3/lun0/part32"},
	{  68, 225, "/dev/scsi/host5/bus0/target3/lun0/part33"},
	{  68, 226, "/dev/scsi/host5/bus0/target3/lun0/part34"},
	{  68, 227, "/dev/scsi/host5/bus0/target3/lun0/part35"},
	{  68, 228, "/dev/scsi/host5/bus0/target3/lun0/part36"},
	{  68, 229, "/dev/scsi/host5/bus0/target3/lun0/part37"},
	{  68, 230, "/dev/scsi/host5/bus0/target3/lun0/part38"},
	{  68, 231, "/dev/scsi/host5/bus0/target3/lun0/part39"},
	{  68, 232, "/dev/scsi/host5/bus0/target3/lun0/part40"},
	{  68, 233, "/dev/scsi/host5/bus0/target3/lun0/part41"},
	{  68, 234, "/dev/scsi/host5/bus0/target3/lun0/part42"},
	{  68, 235, "/dev/scsi/host5/bus0/target3/lun0/part43"},
	{  68, 236, "/dev/scsi/host5/bus0/target3/lun0/part44"},
	{  68, 237, "/dev/scsi/host5/bus0/target3/lun0/part45"},
	{  68, 238, "/dev/scsi/host5/bus0/target3/lun0/part46"},
	{  68, 239, "/dev/scsi/host5/bus0/target3/lun0/part47"},
	{  68, 240, "/dev/scsi/host5/bus0/target3/lun0/part48"},
	{  68, 241, "/dev/scsi/host5/bus0/target3/lun0/part49"},
	{  68, 242, "/dev/scsi/host5/bus0/target3/lun0/part50"},
	{  68, 243, "/dev/scsi/host5/bus0/target3/lun0/part51"},
	{  68, 244, "/dev/scsi/host5/bus0/target3/lun0/part52"},
	{  68, 245, "/dev/scsi/host5/bus0/target3/lun0/part53"},
	{  68, 246, "/dev/scsi/host5/bus0/target3/lun0/part54"},
	{  68, 247, "/dev/scsi/host5/bus0/target3/lun0/part55"},
	{  68, 248, "/dev/scsi/host5/bus0/target3/lun0/part56"},
	{  68, 249, "/dev/scsi/host5/bus0/target3/lun0/part57"},
	{  68, 250, "/dev/scsi/host5/bus0/target3/lun0/part58"},
	{  68, 251, "/dev/scsi/host5/bus0/target3/lun0/part59"},
	{  68, 252, "/dev/scsi/host5/bus0/target3/lun0/part60"},
	{  68, 253, "/dev/scsi/host5/bus0/target3/lun0/part61"},
	{  68, 254, "/dev/scsi/host5/bus0/target3/lun0/part62"},
	{  68, 255, "/dev/scsi/host5/bus0/target3/lun0/part63"},

	{  69,   0, "/dev/scsi/host6/bus0/target0/lun0/disc"},
	{  69,   1, "/dev/scsi/host6/bus0/target0/lun0/part1"},
	{  69,   2, "/dev/scsi/host6/bus0/target0/lun0/part2"},
	{  69,   3, "/dev/scsi/host6/bus0/target0/lun0/part3"},
	{  69,   4, "/dev/scsi/host6/bus0/target0/lun0/part4"},
	{  69,   5, "/dev/scsi/host6/bus0/target0/lun0/part5"},
	{  69,   6, "/dev/scsi/host6/bus0/target0/lun0/part6"},
	{  69,   7, "/dev/scsi/host6/bus0/target0/lun0/part7"},
	{  69,   8, "/dev/scsi/host6/bus0/target0/lun0/part8"},
	{  69,   9, "/dev/scsi/host6/bus0/target0/lun0/part9"},
	{  69,  10, "/dev/scsi/host6/bus0/target0/lun0/part10"},
	{  69,  11, "/dev/scsi/host6/bus0/target0/lun0/part11"},
	{  69,  12, "/dev/scsi/host6/bus0/target0/lun0/part12"},
	{  69,  13, "/dev/scsi/host6/bus0/target0/lun0/part13"},
	{  69,  14, "/dev/scsi/host6/bus0/target0/lun0/part14"},
	{  69,  15, "/dev/scsi/host6/bus0/target0/lun0/part15"},
	{  69,  16, "/dev/scsi/host6/bus0/target0/lun0/part16"},
	{  69,  17, "/dev/scsi/host6/bus0/target0/lun0/part17"},
	{  69,  18, "/dev/scsi/host6/bus0/target0/lun0/part18"},
	{  69,  19, "/dev/scsi/host6/bus0/target0/lun0/part19"},
	{  69,  20, "/dev/scsi/host6/bus0/target0/lun0/part20"},
	{  69,  21, "/dev/scsi/host6/bus0/target0/lun0/part21"},
	{  69,  22, "/dev/scsi/host6/bus0/target0/lun0/part22"},
	{  69,  23, "/dev/scsi/host6/bus0/target0/lun0/part23"},
	{  69,  24, "/dev/scsi/host6/bus0/target0/lun0/part24"},
	{  69,  25, "/dev/scsi/host6/bus0/target0/lun0/part25"},
	{  69,  26, "/dev/scsi/host6/bus0/target0/lun0/part26"},
	{  69,  27, "/dev/scsi/host6/bus0/target0/lun0/part27"},
	{  69,  28, "/dev/scsi/host6/bus0/target0/lun0/part28"},
	{  69,  29, "/dev/scsi/host6/bus0/target0/lun0/part29"},
	{  69,  30, "/dev/scsi/host6/bus0/target0/lun0/part30"},
	{  69,  31, "/dev/scsi/host6/bus0/target0/lun0/part31"},
	{  69,  32, "/dev/scsi/host6/bus0/target0/lun0/part32"},
	{  69,  33, "/dev/scsi/host6/bus0/target0/lun0/part33"},
	{  69,  34, "/dev/scsi/host6/bus0/target0/lun0/part34"},
	{  69,  35, "/dev/scsi/host6/bus0/target0/lun0/part35"},
	{  69,  36, "/dev/scsi/host6/bus0/target0/lun0/part36"},
	{  69,  37, "/dev/scsi/host6/bus0/target0/lun0/part37"},
	{  69,  38, "/dev/scsi/host6/bus0/target0/lun0/part38"},
	{  69,  39, "/dev/scsi/host6/bus0/target0/lun0/part39"},
	{  69,  40, "/dev/scsi/host6/bus0/target0/lun0/part40"},
	{  69,  41, "/dev/scsi/host6/bus0/target0/lun0/part41"},
	{  69,  42, "/dev/scsi/host6/bus0/target0/lun0/part42"},
	{  69,  43, "/dev/scsi/host6/bus0/target0/lun0/part43"},
	{  69,  44, "/dev/scsi/host6/bus0/target0/lun0/part44"},
	{  69,  45, "/dev/scsi/host6/bus0/target0/lun0/part45"},
	{  69,  46, "/dev/scsi/host6/bus0/target0/lun0/part46"},
	{  69,  47, "/dev/scsi/host6/bus0/target0/lun0/part47"},
	{  69,  48, "/dev/scsi/host6/bus0/target0/lun0/part48"},
	{  69,  49, "/dev/scsi/host6/bus0/target0/lun0/part49"},
	{  69,  50, "/dev/scsi/host6/bus0/target0/lun0/part50"},
	{  69,  51, "/dev/scsi/host6/bus0/target0/lun0/part51"},
	{  69,  52, "/dev/scsi/host6/bus0/target0/lun0/part52"},
	{  69,  53, "/dev/scsi/host6/bus0/target0/lun0/part53"},
	{  69,  54, "/dev/scsi/host6/bus0/target0/lun0/part54"},
	{  69,  55, "/dev/scsi/host6/bus0/target0/lun0/part55"},
	{  69,  56, "/dev/scsi/host6/bus0/target0/lun0/part56"},
	{  69,  57, "/dev/scsi/host6/bus0/target0/lun0/part57"},
	{  69,  58, "/dev/scsi/host6/bus0/target0/lun0/part58"},
	{  69,  59, "/dev/scsi/host6/bus0/target0/lun0/part59"},
	{  69,  60, "/dev/scsi/host6/bus0/target0/lun0/part60"},
	{  69,  61, "/dev/scsi/host6/bus0/target0/lun0/part61"},
	{  69,  62, "/dev/scsi/host6/bus0/target0/lun0/part62"},
	{  69,  63, "/dev/scsi/host6/bus0/target0/lun0/part63"},
	{  69,  64, "/dev/scsi/host6/bus0/target1/lun0/disc"},
	{  69,  65, "/dev/scsi/host6/bus0/target1/lun0/part1"},
	{  69,  66, "/dev/scsi/host6/bus0/target1/lun0/part2"},
	{  69,  67, "/dev/scsi/host6/bus0/target1/lun0/part3"},
	{  69,  68, "/dev/scsi/host6/bus0/target1/lun0/part4"},
	{  69,  69, "/dev/scsi/host6/bus0/target1/lun0/part5"},
	{  69,  70, "/dev/scsi/host6/bus0/target1/lun0/part6"},
	{  69,  71, "/dev/scsi/host6/bus0/target1/lun0/part7"},
	{  69,  72, "/dev/scsi/host6/bus0/target1/lun0/part8"},
	{  69,  73, "/dev/scsi/host6/bus0/target1/lun0/part9"},
	{  69,  74, "/dev/scsi/host6/bus0/target1/lun0/part10"},
	{  69,  75, "/dev/scsi/host6/bus0/target1/lun0/part11"},
	{  69,  76, "/dev/scsi/host6/bus0/target1/lun0/part12"},
	{  69,  77, "/dev/scsi/host6/bus0/target1/lun0/part13"},
	{  69,  78, "/dev/scsi/host6/bus0/target1/lun0/part14"},
	{  69,  79, "/dev/scsi/host6/bus0/target1/lun0/part15"},
	{  69,  80, "/dev/scsi/host6/bus0/target1/lun0/part16"},
	{  69,  81, "/dev/scsi/host6/bus0/target1/lun0/part17"},
	{  69,  82, "/dev/scsi/host6/bus0/target1/lun0/part18"},
	{  69,  83, "/dev/scsi/host6/bus0/target1/lun0/part19"},
	{  69,  84, "/dev/scsi/host6/bus0/target1/lun0/part20"},
	{  69,  85, "/dev/scsi/host6/bus0/target1/lun0/part21"},
	{  69,  86, "/dev/scsi/host6/bus0/target1/lun0/part22"},
	{  69,  87, "/dev/scsi/host6/bus0/target1/lun0/part23"},
	{  69,  88, "/dev/scsi/host6/bus0/target1/lun0/part24"},
	{  69,  89, "/dev/scsi/host6/bus0/target1/lun0/part25"},
	{  69,  90, "/dev/scsi/host6/bus0/target1/lun0/part26"},
	{  69,  91, "/dev/scsi/host6/bus0/target1/lun0/part27"},
	{  69,  92, "/dev/scsi/host6/bus0/target1/lun0/part28"},
	{  69,  93, "/dev/scsi/host6/bus0/target1/lun0/part29"},
	{  69,  94, "/dev/scsi/host6/bus0/target1/lun0/part30"},
	{  69,  95, "/dev/scsi/host6/bus0/target1/lun0/part31"},
	{  69,  96, "/dev/scsi/host6/bus0/target1/lun0/part32"},
	{  69,  97, "/dev/scsi/host6/bus0/target1/lun0/part33"},
	{  69,  98, "/dev/scsi/host6/bus0/target1/lun0/part34"},
	{  69,  99, "/dev/scsi/host6/bus0/target1/lun0/part35"},
	{  69, 100, "/dev/scsi/host6/bus0/target1/lun0/part36"},
	{  69, 101, "/dev/scsi/host6/bus0/target1/lun0/part37"},
	{  69, 102, "/dev/scsi/host6/bus0/target1/lun0/part38"},
	{  69, 103, "/dev/scsi/host6/bus0/target1/lun0/part39"},
	{  69, 104, "/dev/scsi/host6/bus0/target1/lun0/part40"},
	{  69, 105, "/dev/scsi/host6/bus0/target1/lun0/part41"},
	{  69, 106, "/dev/scsi/host6/bus0/target1/lun0/part42"},
	{  69, 107, "/dev/scsi/host6/bus0/target1/lun0/part43"},
	{  69, 108, "/dev/scsi/host6/bus0/target1/lun0/part44"},
	{  69, 109, "/dev/scsi/host6/bus0/target1/lun0/part45"},
	{  69, 110, "/dev/scsi/host6/bus0/target1/lun0/part46"},
	{  69, 111, "/dev/scsi/host6/bus0/target1/lun0/part47"},
	{  69, 112, "/dev/scsi/host6/bus0/target1/lun0/part48"},
	{  69, 113, "/dev/scsi/host6/bus0/target1/lun0/part49"},
	{  69, 114, "/dev/scsi/host6/bus0/target1/lun0/part50"},
	{  69, 115, "/dev/scsi/host6/bus0/target1/lun0/part51"},
	{  69, 116, "/dev/scsi/host6/bus0/target1/lun0/part52"},
	{  69, 117, "/dev/scsi/host6/bus0/target1/lun0/part53"},
	{  69, 118, "/dev/scsi/host6/bus0/target1/lun0/part54"},
	{  69, 119, "/dev/scsi/host6/bus0/target1/lun0/part55"},
	{  69, 120, "/dev/scsi/host6/bus0/target1/lun0/part56"},
	{  69, 121, "/dev/scsi/host6/bus0/target1/lun0/part57"},
	{  69, 122, "/dev/scsi/host6/bus0/target1/lun0/part58"},
	{  69, 123, "/dev/scsi/host6/bus0/target1/lun0/part59"},
	{  69, 124, "/dev/scsi/host6/bus0/target1/lun0/part60"},
	{  69, 125, "/dev/scsi/host6/bus0/target1/lun0/part61"},
	{  69, 126, "/dev/scsi/host6/bus0/target1/lun0/part62"},
	{  69, 127, "/dev/scsi/host6/bus0/target1/lun0/part63"},
	{  69, 128, "/dev/scsi/host6/bus0/target2/lun0/disc"},
	{  69, 129, "/dev/scsi/host6/bus0/target2/lun0/part1"},
	{  69, 130, "/dev/scsi/host6/bus0/target2/lun0/part2"},
	{  69, 131, "/dev/scsi/host6/bus0/target2/lun0/part3"},
	{  69, 132, "/dev/scsi/host6/bus0/target2/lun0/part4"},
	{  69, 133, "/dev/scsi/host6/bus0/target2/lun0/part5"},
	{  69, 134, "/dev/scsi/host6/bus0/target2/lun0/part6"},
	{  69, 135, "/dev/scsi/host6/bus0/target2/lun0/part7"},
	{  69, 136, "/dev/scsi/host6/bus0/target2/lun0/part8"},
	{  69, 137, "/dev/scsi/host6/bus0/target2/lun0/part9"},
	{  69, 138, "/dev/scsi/host6/bus0/target2/lun0/part10"},
	{  69, 139, "/dev/scsi/host6/bus0/target2/lun0/part11"},
	{  69, 140, "/dev/scsi/host6/bus0/target2/lun0/part12"},
	{  69, 141, "/dev/scsi/host6/bus0/target2/lun0/part13"},
	{  69, 142, "/dev/scsi/host6/bus0/target2/lun0/part14"},
	{  69, 143, "/dev/scsi/host6/bus0/target2/lun0/part15"},
	{  69, 144, "/dev/scsi/host6/bus0/target2/lun0/part16"},
	{  69, 145, "/dev/scsi/host6/bus0/target2/lun0/part17"},
	{  69, 146, "/dev/scsi/host6/bus0/target2/lun0/part18"},
	{  69, 147, "/dev/scsi/host6/bus0/target2/lun0/part19"},
	{  69, 148, "/dev/scsi/host6/bus0/target2/lun0/part20"},
	{  69, 149, "/dev/scsi/host6/bus0/target2/lun0/part21"},
	{  69, 150, "/dev/scsi/host6/bus0/target2/lun0/part22"},
	{  69, 151, "/dev/scsi/host6/bus0/target2/lun0/part23"},
	{  69, 152, "/dev/scsi/host6/bus0/target2/lun0/part24"},
	{  69, 153, "/dev/scsi/host6/bus0/target2/lun0/part25"},
	{  69, 154, "/dev/scsi/host6/bus0/target2/lun0/part26"},
	{  69, 155, "/dev/scsi/host6/bus0/target2/lun0/part27"},
	{  69, 156, "/dev/scsi/host6/bus0/target2/lun0/part28"},
	{  69, 157, "/dev/scsi/host6/bus0/target2/lun0/part29"},
	{  69, 158, "/dev/scsi/host6/bus0/target2/lun0/part30"},
	{  69, 159, "/dev/scsi/host6/bus0/target2/lun0/part31"},
	{  69, 160, "/dev/scsi/host6/bus0/target2/lun0/part32"},
	{  69, 161, "/dev/scsi/host6/bus0/target2/lun0/part33"},
	{  69, 162, "/dev/scsi/host6/bus0/target2/lun0/part34"},
	{  69, 163, "/dev/scsi/host6/bus0/target2/lun0/part35"},
	{  69, 164, "/dev/scsi/host6/bus0/target2/lun0/part36"},
	{  69, 165, "/dev/scsi/host6/bus0/target2/lun0/part37"},
	{  69, 166, "/dev/scsi/host6/bus0/target2/lun0/part38"},
	{  69, 167, "/dev/scsi/host6/bus0/target2/lun0/part39"},
	{  69, 168, "/dev/scsi/host6/bus0/target2/lun0/part40"},
	{  69, 169, "/dev/scsi/host6/bus0/target2/lun0/part41"},
	{  69, 170, "/dev/scsi/host6/bus0/target2/lun0/part42"},
	{  69, 171, "/dev/scsi/host6/bus0/target2/lun0/part43"},
	{  69, 172, "/dev/scsi/host6/bus0/target2/lun0/part44"},
	{  69, 173, "/dev/scsi/host6/bus0/target2/lun0/part45"},
	{  69, 174, "/dev/scsi/host6/bus0/target2/lun0/part46"},
	{  69, 175, "/dev/scsi/host6/bus0/target2/lun0/part47"},
	{  69, 176, "/dev/scsi/host6/bus0/target2/lun0/part48"},
	{  69, 177, "/dev/scsi/host6/bus0/target2/lun0/part49"},
	{  69, 178, "/dev/scsi/host6/bus0/target2/lun0/part50"},
	{  69, 179, "/dev/scsi/host6/bus0/target2/lun0/part51"},
	{  69, 180, "/dev/scsi/host6/bus0/target2/lun0/part52"},
	{  69, 181, "/dev/scsi/host6/bus0/target2/lun0/part53"},
	{  69, 182, "/dev/scsi/host6/bus0/target2/lun0/part54"},
	{  69, 183, "/dev/scsi/host6/bus0/target2/lun0/part55"},
	{  69, 184, "/dev/scsi/host6/bus0/target2/lun0/part56"},
	{  69, 185, "/dev/scsi/host6/bus0/target2/lun0/part57"},
	{  69, 186, "/dev/scsi/host6/bus0/target2/lun0/part58"},
	{  69, 187, "/dev/scsi/host6/bus0/target2/lun0/part59"},
	{  69, 188, "/dev/scsi/host6/bus0/target2/lun0/part60"},
	{  69, 189, "/dev/scsi/host6/bus0/target2/lun0/part61"},
	{  69, 190, "/dev/scsi/host6/bus0/target2/lun0/part62"},
	{  69, 191, "/dev/scsi/host6/bus0/target2/lun0/part63"},
	{  69, 192, "/dev/scsi/host6/bus0/target3/lun0/disc"},
	{  69, 193, "/dev/scsi/host6/bus0/target3/lun0/part1"},
	{  69, 194, "/dev/scsi/host6/bus0/target3/lun0/part2"},
	{  69, 195, "/dev/scsi/host6/bus0/target3/lun0/part3"},
	{  69, 196, "/dev/scsi/host6/bus0/target3/lun0/part4"},
	{  69, 197, "/dev/scsi/host6/bus0/target3/lun0/part5"},
	{  69, 198, "/dev/scsi/host6/bus0/target3/lun0/part6"},
	{  69, 199, "/dev/scsi/host6/bus0/target3/lun0/part7"},
	{  69, 200, "/dev/scsi/host6/bus0/target3/lun0/part8"},
	{  69, 201, "/dev/scsi/host6/bus0/target3/lun0/part9"},
	{  69, 202, "/dev/scsi/host6/bus0/target3/lun0/part10"},
	{  69, 203, "/dev/scsi/host6/bus0/target3/lun0/part11"},
	{  69, 204, "/dev/scsi/host6/bus0/target3/lun0/part12"},
	{  69, 205, "/dev/scsi/host6/bus0/target3/lun0/part13"},
	{  69, 206, "/dev/scsi/host6/bus0/target3/lun0/part14"},
	{  69, 207, "/dev/scsi/host6/bus0/target3/lun0/part15"},
	{  69, 208, "/dev/scsi/host6/bus0/target3/lun0/part16"},
	{  69, 209, "/dev/scsi/host6/bus0/target3/lun0/part17"},
	{  69, 210, "/dev/scsi/host6/bus0/target3/lun0/part18"},
	{  69, 211, "/dev/scsi/host6/bus0/target3/lun0/part19"},
	{  69, 212, "/dev/scsi/host6/bus0/target3/lun0/part20"},
	{  69, 213, "/dev/scsi/host6/bus0/target3/lun0/part21"},
	{  69, 214, "/dev/scsi/host6/bus0/target3/lun0/part22"},
	{  69, 215, "/dev/scsi/host6/bus0/target3/lun0/part23"},
	{  69, 216, "/dev/scsi/host6/bus0/target3/lun0/part24"},
	{  69, 217, "/dev/scsi/host6/bus0/target3/lun0/part25"},
	{  69, 218, "/dev/scsi/host6/bus0/target3/lun0/part26"},
	{  69, 219, "/dev/scsi/host6/bus0/target3/lun0/part27"},
	{  69, 220, "/dev/scsi/host6/bus0/target3/lun0/part28"},
	{  69, 221, "/dev/scsi/host6/bus0/target3/lun0/part29"},
	{  69, 222, "/dev/scsi/host6/bus0/target3/lun0/part30"},
	{  69, 223, "/dev/scsi/host6/bus0/target3/lun0/part31"},
	{  69, 224, "/dev/scsi/host6/bus0/target3/lun0/part32"},
	{  69, 225, "/dev/scsi/host6/bus0/target3/lun0/part33"},
	{  69, 226, "/dev/scsi/host6/bus0/target3/lun0/part34"},
	{  69, 227, "/dev/scsi/host6/bus0/target3/lun0/part35"},
	{  69, 228, "/dev/scsi/host6/bus0/target3/lun0/part36"},
	{  69, 229, "/dev/scsi/host6/bus0/target3/lun0/part37"},
	{  69, 230, "/dev/scsi/host6/bus0/target3/lun0/part38"},
	{  69, 231, "/dev/scsi/host6/bus0/target3/lun0/part39"},
	{  69, 232, "/dev/scsi/host6/bus0/target3/lun0/part40"},
	{  69, 233, "/dev/scsi/host6/bus0/target3/lun0/part41"},
	{  69, 234, "/dev/scsi/host6/bus0/target3/lun0/part42"},
	{  69, 235, "/dev/scsi/host6/bus0/target3/lun0/part43"},
	{  69, 236, "/dev/scsi/host6/bus0/target3/lun0/part44"},
	{  69, 237, "/dev/scsi/host6/bus0/target3/lun0/part45"},
	{  69, 238, "/dev/scsi/host6/bus0/target3/lun0/part46"},
	{  69, 239, "/dev/scsi/host6/bus0/target3/lun0/part47"},
	{  69, 240, "/dev/scsi/host6/bus0/target3/lun0/part48"},
	{  69, 241, "/dev/scsi/host6/bus0/target3/lun0/part49"},
	{  69, 242, "/dev/scsi/host6/bus0/target3/lun0/part50"},
	{  69, 243, "/dev/scsi/host6/bus0/target3/lun0/part51"},
	{  69, 244, "/dev/scsi/host6/bus0/target3/lun0/part52"},
	{  69, 245, "/dev/scsi/host6/bus0/target3/lun0/part53"},
	{  69, 246, "/dev/scsi/host6/bus0/target3/lun0/part54"},
	{  69, 247, "/dev/scsi/host6/bus0/target3/lun0/part55"},
	{  69, 248, "/dev/scsi/host6/bus0/target3/lun0/part56"},
	{  69, 249, "/dev/scsi/host6/bus0/target3/lun0/part57"},
	{  69, 250, "/dev/scsi/host6/bus0/target3/lun0/part58"},
	{  69, 251, "/dev/scsi/host6/bus0/target3/lun0/part59"},
	{  69, 252, "/dev/scsi/host6/bus0/target3/lun0/part60"},
	{  69, 253, "/dev/scsi/host6/bus0/target3/lun0/part61"},
	{  69, 254, "/dev/scsi/host6/bus0/target3/lun0/part62"},
	{  69, 255, "/dev/scsi/host6/bus0/target3/lun0/part63"},

	{  70,   0, "/dev/scsi/host7/bus0/target0/lun0/disc"},
	{  70,   1, "/dev/scsi/host7/bus0/target0/lun0/part1"},
	{  70,   2, "/dev/scsi/host7/bus0/target0/lun0/part2"},
	{  70,   3, "/dev/scsi/host7/bus0/target0/lun0/part3"},
	{  70,   4, "/dev/scsi/host7/bus0/target0/lun0/part4"},
	{  70,   5, "/dev/scsi/host7/bus0/target0/lun0/part5"},
	{  70,   6, "/dev/scsi/host7/bus0/target0/lun0/part6"},
	{  70,   7, "/dev/scsi/host7/bus0/target0/lun0/part7"},
	{  70,   8, "/dev/scsi/host7/bus0/target0/lun0/part8"},
	{  70,   9, "/dev/scsi/host7/bus0/target0/lun0/part9"},
	{  70,  10, "/dev/scsi/host7/bus0/target0/lun0/part10"},
	{  70,  11, "/dev/scsi/host7/bus0/target0/lun0/part11"},
	{  70,  12, "/dev/scsi/host7/bus0/target0/lun0/part12"},
	{  70,  13, "/dev/scsi/host7/bus0/target0/lun0/part13"},
	{  70,  14, "/dev/scsi/host7/bus0/target0/lun0/part14"},
	{  70,  15, "/dev/scsi/host7/bus0/target0/lun0/part15"},
	{  70,  16, "/dev/scsi/host7/bus0/target0/lun0/part16"},
	{  70,  17, "/dev/scsi/host7/bus0/target0/lun0/part17"},
	{  70,  18, "/dev/scsi/host7/bus0/target0/lun0/part18"},
	{  70,  19, "/dev/scsi/host7/bus0/target0/lun0/part19"},
	{  70,  20, "/dev/scsi/host7/bus0/target0/lun0/part20"},
	{  70,  21, "/dev/scsi/host7/bus0/target0/lun0/part21"},
	{  70,  22, "/dev/scsi/host7/bus0/target0/lun0/part22"},
	{  70,  23, "/dev/scsi/host7/bus0/target0/lun0/part23"},
	{  70,  24, "/dev/scsi/host7/bus0/target0/lun0/part24"},
	{  70,  25, "/dev/scsi/host7/bus0/target0/lun0/part25"},
	{  70,  26, "/dev/scsi/host7/bus0/target0/lun0/part26"},
	{  70,  27, "/dev/scsi/host7/bus0/target0/lun0/part27"},
	{  70,  28, "/dev/scsi/host7/bus0/target0/lun0/part28"},
	{  70,  29, "/dev/scsi/host7/bus0/target0/lun0/part29"},
	{  70,  30, "/dev/scsi/host7/bus0/target0/lun0/part30"},
	{  70,  31, "/dev/scsi/host7/bus0/target0/lun0/part31"},
	{  70,  32, "/dev/scsi/host7/bus0/target0/lun0/part32"},
	{  70,  33, "/dev/scsi/host7/bus0/target0/lun0/part33"},
	{  70,  34, "/dev/scsi/host7/bus0/target0/lun0/part34"},
	{  70,  35, "/dev/scsi/host7/bus0/target0/lun0/part35"},
	{  70,  36, "/dev/scsi/host7/bus0/target0/lun0/part36"},
	{  70,  37, "/dev/scsi/host7/bus0/target0/lun0/part37"},
	{  70,  38, "/dev/scsi/host7/bus0/target0/lun0/part38"},
	{  70,  39, "/dev/scsi/host7/bus0/target0/lun0/part39"},
	{  70,  40, "/dev/scsi/host7/bus0/target0/lun0/part40"},
	{  70,  41, "/dev/scsi/host7/bus0/target0/lun0/part41"},
	{  70,  42, "/dev/scsi/host7/bus0/target0/lun0/part42"},
	{  70,  43, "/dev/scsi/host7/bus0/target0/lun0/part43"},
	{  70,  44, "/dev/scsi/host7/bus0/target0/lun0/part44"},
	{  70,  45, "/dev/scsi/host7/bus0/target0/lun0/part45"},
	{  70,  46, "/dev/scsi/host7/bus0/target0/lun0/part46"},
	{  70,  47, "/dev/scsi/host7/bus0/target0/lun0/part47"},
	{  70,  48, "/dev/scsi/host7/bus0/target0/lun0/part48"},
	{  70,  49, "/dev/scsi/host7/bus0/target0/lun0/part49"},
	{  70,  50, "/dev/scsi/host7/bus0/target0/lun0/part50"},
	{  70,  51, "/dev/scsi/host7/bus0/target0/lun0/part51"},
	{  70,  52, "/dev/scsi/host7/bus0/target0/lun0/part52"},
	{  70,  53, "/dev/scsi/host7/bus0/target0/lun0/part53"},
	{  70,  54, "/dev/scsi/host7/bus0/target0/lun0/part54"},
	{  70,  55, "/dev/scsi/host7/bus0/target0/lun0/part55"},
	{  70,  56, "/dev/scsi/host7/bus0/target0/lun0/part56"},
	{  70,  57, "/dev/scsi/host7/bus0/target0/lun0/part57"},
	{  70,  58, "/dev/scsi/host7/bus0/target0/lun0/part58"},
	{  70,  59, "/dev/scsi/host7/bus0/target0/lun0/part59"},
	{  70,  60, "/dev/scsi/host7/bus0/target0/lun0/part60"},
	{  70,  61, "/dev/scsi/host7/bus0/target0/lun0/part61"},
	{  70,  62, "/dev/scsi/host7/bus0/target0/lun0/part62"},
	{  70,  63, "/dev/scsi/host7/bus0/target0/lun0/part63"},
	{  70,  64, "/dev/scsi/host7/bus0/target1/lun0/disc"},
	{  70,  65, "/dev/scsi/host7/bus0/target1/lun0/part1"},
	{  70,  66, "/dev/scsi/host7/bus0/target1/lun0/part2"},
	{  70,  67, "/dev/scsi/host7/bus0/target1/lun0/part3"},
	{  70,  68, "/dev/scsi/host7/bus0/target1/lun0/part4"},
	{  70,  69, "/dev/scsi/host7/bus0/target1/lun0/part5"},
	{  70,  70, "/dev/scsi/host7/bus0/target1/lun0/part6"},
	{  70,  71, "/dev/scsi/host7/bus0/target1/lun0/part7"},
	{  70,  72, "/dev/scsi/host7/bus0/target1/lun0/part8"},
	{  70,  73, "/dev/scsi/host7/bus0/target1/lun0/part9"},
	{  70,  74, "/dev/scsi/host7/bus0/target1/lun0/part10"},
	{  70,  75, "/dev/scsi/host7/bus0/target1/lun0/part11"},
	{  70,  76, "/dev/scsi/host7/bus0/target1/lun0/part12"},
	{  70,  77, "/dev/scsi/host7/bus0/target1/lun0/part13"},
	{  70,  78, "/dev/scsi/host7/bus0/target1/lun0/part14"},
	{  70,  79, "/dev/scsi/host7/bus0/target1/lun0/part15"},
	{  70,  80, "/dev/scsi/host7/bus0/target1/lun0/part16"},
	{  70,  81, "/dev/scsi/host7/bus0/target1/lun0/part17"},
	{  70,  82, "/dev/scsi/host7/bus0/target1/lun0/part18"},
	{  70,  83, "/dev/scsi/host7/bus0/target1/lun0/part19"},
	{  70,  84, "/dev/scsi/host7/bus0/target1/lun0/part20"},
	{  70,  85, "/dev/scsi/host7/bus0/target1/lun0/part21"},
	{  70,  86, "/dev/scsi/host7/bus0/target1/lun0/part22"},
	{  70,  87, "/dev/scsi/host7/bus0/target1/lun0/part23"},
	{  70,  88, "/dev/scsi/host7/bus0/target1/lun0/part24"},
	{  70,  89, "/dev/scsi/host7/bus0/target1/lun0/part25"},
	{  70,  90, "/dev/scsi/host7/bus0/target1/lun0/part26"},
	{  70,  91, "/dev/scsi/host7/bus0/target1/lun0/part27"},
	{  70,  92, "/dev/scsi/host7/bus0/target1/lun0/part28"},
	{  70,  93, "/dev/scsi/host7/bus0/target1/lun0/part29"},
	{  70,  94, "/dev/scsi/host7/bus0/target1/lun0/part30"},
	{  70,  95, "/dev/scsi/host7/bus0/target1/lun0/part31"},
	{  70,  96, "/dev/scsi/host7/bus0/target1/lun0/part32"},
	{  70,  97, "/dev/scsi/host7/bus0/target1/lun0/part33"},
	{  70,  98, "/dev/scsi/host7/bus0/target1/lun0/part34"},
	{  70,  99, "/dev/scsi/host7/bus0/target1/lun0/part35"},
	{  70, 100, "/dev/scsi/host7/bus0/target1/lun0/part36"},
	{  70, 101, "/dev/scsi/host7/bus0/target1/lun0/part37"},
	{  70, 102, "/dev/scsi/host7/bus0/target1/lun0/part38"},
	{  70, 103, "/dev/scsi/host7/bus0/target1/lun0/part39"},
	{  70, 104, "/dev/scsi/host7/bus0/target1/lun0/part40"},
	{  70, 105, "/dev/scsi/host7/bus0/target1/lun0/part41"},
	{  70, 106, "/dev/scsi/host7/bus0/target1/lun0/part42"},
	{  70, 107, "/dev/scsi/host7/bus0/target1/lun0/part43"},
	{  70, 108, "/dev/scsi/host7/bus0/target1/lun0/part44"},
	{  70, 109, "/dev/scsi/host7/bus0/target1/lun0/part45"},
	{  70, 110, "/dev/scsi/host7/bus0/target1/lun0/part46"},
	{  70, 111, "/dev/scsi/host7/bus0/target1/lun0/part47"},
	{  70, 112, "/dev/scsi/host7/bus0/target1/lun0/part48"},
	{  70, 113, "/dev/scsi/host7/bus0/target1/lun0/part49"},
	{  70, 114, "/dev/scsi/host7/bus0/target1/lun0/part50"},
	{  70, 115, "/dev/scsi/host7/bus0/target1/lun0/part51"},
	{  70, 116, "/dev/scsi/host7/bus0/target1/lun0/part52"},
	{  70, 117, "/dev/scsi/host7/bus0/target1/lun0/part53"},
	{  70, 118, "/dev/scsi/host7/bus0/target1/lun0/part54"},
	{  70, 119, "/dev/scsi/host7/bus0/target1/lun0/part55"},
	{  70, 120, "/dev/scsi/host7/bus0/target1/lun0/part56"},
	{  70, 121, "/dev/scsi/host7/bus0/target1/lun0/part57"},
	{  70, 122, "/dev/scsi/host7/bus0/target1/lun0/part58"},
	{  70, 123, "/dev/scsi/host7/bus0/target1/lun0/part59"},
	{  70, 124, "/dev/scsi/host7/bus0/target1/lun0/part60"},
	{  70, 125, "/dev/scsi/host7/bus0/target1/lun0/part61"},
	{  70, 126, "/dev/scsi/host7/bus0/target1/lun0/part62"},
	{  70, 127, "/dev/scsi/host7/bus0/target1/lun0/part63"},
	{  70, 128, "/dev/scsi/host7/bus0/target2/lun0/disc"},
	{  70, 129, "/dev/scsi/host7/bus0/target2/lun0/part1"},
	{  70, 130, "/dev/scsi/host7/bus0/target2/lun0/part2"},
	{  70, 131, "/dev/scsi/host7/bus0/target2/lun0/part3"},
	{  70, 132, "/dev/scsi/host7/bus0/target2/lun0/part4"},
	{  70, 133, "/dev/scsi/host7/bus0/target2/lun0/part5"},
	{  70, 134, "/dev/scsi/host7/bus0/target2/lun0/part6"},
	{  70, 135, "/dev/scsi/host7/bus0/target2/lun0/part7"},
	{  70, 136, "/dev/scsi/host7/bus0/target2/lun0/part8"},
	{  70, 137, "/dev/scsi/host7/bus0/target2/lun0/part9"},
	{  70, 138, "/dev/scsi/host7/bus0/target2/lun0/part10"},
	{  70, 139, "/dev/scsi/host7/bus0/target2/lun0/part11"},
	{  70, 140, "/dev/scsi/host7/bus0/target2/lun0/part12"},
	{  70, 141, "/dev/scsi/host7/bus0/target2/lun0/part13"},
	{  70, 142, "/dev/scsi/host7/bus0/target2/lun0/part14"},
	{  70, 143, "/dev/scsi/host7/bus0/target2/lun0/part15"},
	{  70, 144, "/dev/scsi/host7/bus0/target2/lun0/part16"},
	{  70, 145, "/dev/scsi/host7/bus0/target2/lun0/part17"},
	{  70, 146, "/dev/scsi/host7/bus0/target2/lun0/part18"},
	{  70, 147, "/dev/scsi/host7/bus0/target2/lun0/part19"},
	{  70, 148, "/dev/scsi/host7/bus0/target2/lun0/part20"},
	{  70, 149, "/dev/scsi/host7/bus0/target2/lun0/part21"},
	{  70, 150, "/dev/scsi/host7/bus0/target2/lun0/part22"},
	{  70, 151, "/dev/scsi/host7/bus0/target2/lun0/part23"},
	{  70, 152, "/dev/scsi/host7/bus0/target2/lun0/part24"},
	{  70, 153, "/dev/scsi/host7/bus0/target2/lun0/part25"},
	{  70, 154, "/dev/scsi/host7/bus0/target2/lun0/part26"},
	{  70, 155, "/dev/scsi/host7/bus0/target2/lun0/part27"},
	{  70, 156, "/dev/scsi/host7/bus0/target2/lun0/part28"},
	{  70, 157, "/dev/scsi/host7/bus0/target2/lun0/part29"},
	{  70, 158, "/dev/scsi/host7/bus0/target2/lun0/part30"},
	{  70, 159, "/dev/scsi/host7/bus0/target2/lun0/part31"},
	{  70, 160, "/dev/scsi/host7/bus0/target2/lun0/part32"},
	{  70, 161, "/dev/scsi/host7/bus0/target2/lun0/part33"},
	{  70, 162, "/dev/scsi/host7/bus0/target2/lun0/part34"},
	{  70, 163, "/dev/scsi/host7/bus0/target2/lun0/part35"},
	{  70, 164, "/dev/scsi/host7/bus0/target2/lun0/part36"},
	{  70, 165, "/dev/scsi/host7/bus0/target2/lun0/part37"},
	{  70, 166, "/dev/scsi/host7/bus0/target2/lun0/part38"},
	{  70, 167, "/dev/scsi/host7/bus0/target2/lun0/part39"},
	{  70, 168, "/dev/scsi/host7/bus0/target2/lun0/part40"},
	{  70, 169, "/dev/scsi/host7/bus0/target2/lun0/part41"},
	{  70, 170, "/dev/scsi/host7/bus0/target2/lun0/part42"},
	{  70, 171, "/dev/scsi/host7/bus0/target2/lun0/part43"},
	{  70, 172, "/dev/scsi/host7/bus0/target2/lun0/part44"},
	{  70, 173, "/dev/scsi/host7/bus0/target2/lun0/part45"},
	{  70, 174, "/dev/scsi/host7/bus0/target2/lun0/part46"},
	{  70, 175, "/dev/scsi/host7/bus0/target2/lun0/part47"},
	{  70, 176, "/dev/scsi/host7/bus0/target2/lun0/part48"},
	{  70, 177, "/dev/scsi/host7/bus0/target2/lun0/part49"},
	{  70, 178, "/dev/scsi/host7/bus0/target2/lun0/part50"},
	{  70, 179, "/dev/scsi/host7/bus0/target2/lun0/part51"},
	{  70, 180, "/dev/scsi/host7/bus0/target2/lun0/part52"},
	{  70, 181, "/dev/scsi/host7/bus0/target2/lun0/part53"},
	{  70, 182, "/dev/scsi/host7/bus0/target2/lun0/part54"},
	{  70, 183, "/dev/scsi/host7/bus0/target2/lun0/part55"},
	{  70, 184, "/dev/scsi/host7/bus0/target2/lun0/part56"},
	{  70, 185, "/dev/scsi/host7/bus0/target2/lun0/part57"},
	{  70, 186, "/dev/scsi/host7/bus0/target2/lun0/part58"},
	{  70, 187, "/dev/scsi/host7/bus0/target2/lun0/part59"},
	{  70, 188, "/dev/scsi/host7/bus0/target2/lun0/part60"},
	{  70, 189, "/dev/scsi/host7/bus0/target2/lun0/part61"},
	{  70, 190, "/dev/scsi/host7/bus0/target2/lun0/part62"},
	{  70, 191, "/dev/scsi/host7/bus0/target2/lun0/part63"},
	{  70, 192, "/dev/scsi/host7/bus0/target3/lun0/disc"},
	{  70, 193, "/dev/scsi/host7/bus0/target3/lun0/part1"},
	{  70, 194, "/dev/scsi/host7/bus0/target3/lun0/part2"},
	{  70, 195, "/dev/scsi/host7/bus0/target3/lun0/part3"},
	{  70, 196, "/dev/scsi/host7/bus0/target3/lun0/part4"},
	{  70, 197, "/dev/scsi/host7/bus0/target3/lun0/part5"},
	{  70, 198, "/dev/scsi/host7/bus0/target3/lun0/part6"},
	{  70, 199, "/dev/scsi/host7/bus0/target3/lun0/part7"},
	{  70, 200, "/dev/scsi/host7/bus0/target3/lun0/part8"},
	{  70, 201, "/dev/scsi/host7/bus0/target3/lun0/part9"},
	{  70, 202, "/dev/scsi/host7/bus0/target3/lun0/part10"},
	{  70, 203, "/dev/scsi/host7/bus0/target3/lun0/part11"},
	{  70, 204, "/dev/scsi/host7/bus0/target3/lun0/part12"},
	{  70, 205, "/dev/scsi/host7/bus0/target3/lun0/part13"},
	{  70, 206, "/dev/scsi/host7/bus0/target3/lun0/part14"},
	{  70, 207, "/dev/scsi/host7/bus0/target3/lun0/part15"},
	{  70, 208, "/dev/scsi/host7/bus0/target3/lun0/part16"},
	{  70, 209, "/dev/scsi/host7/bus0/target3/lun0/part17"},
	{  70, 210, "/dev/scsi/host7/bus0/target3/lun0/part18"},
	{  70, 211, "/dev/scsi/host7/bus0/target3/lun0/part19"},
	{  70, 212, "/dev/scsi/host7/bus0/target3/lun0/part20"},
	{  70, 213, "/dev/scsi/host7/bus0/target3/lun0/part21"},
	{  70, 214, "/dev/scsi/host7/bus0/target3/lun0/part22"},
	{  70, 215, "/dev/scsi/host7/bus0/target3/lun0/part23"},
	{  70, 216, "/dev/scsi/host7/bus0/target3/lun0/part24"},
	{  70, 217, "/dev/scsi/host7/bus0/target3/lun0/part25"},
	{  70, 218, "/dev/scsi/host7/bus0/target3/lun0/part26"},
	{  70, 219, "/dev/scsi/host7/bus0/target3/lun0/part27"},
	{  70, 220, "/dev/scsi/host7/bus0/target3/lun0/part28"},
	{  70, 221, "/dev/scsi/host7/bus0/target3/lun0/part29"},
	{  70, 222, "/dev/scsi/host7/bus0/target3/lun0/part30"},
	{  70, 223, "/dev/scsi/host7/bus0/target3/lun0/part31"},
	{  70, 224, "/dev/scsi/host7/bus0/target3/lun0/part32"},
	{  70, 225, "/dev/scsi/host7/bus0/target3/lun0/part33"},
	{  70, 226, "/dev/scsi/host7/bus0/target3/lun0/part34"},
	{  70, 227, "/dev/scsi/host7/bus0/target3/lun0/part35"},
	{  70, 228, "/dev/scsi/host7/bus0/target3/lun0/part36"},
	{  70, 229, "/dev/scsi/host7/bus0/target3/lun0/part37"},
	{  70, 230, "/dev/scsi/host7/bus0/target3/lun0/part38"},
	{  70, 231, "/dev/scsi/host7/bus0/target3/lun0/part39"},
	{  70, 232, "/dev/scsi/host7/bus0/target3/lun0/part40"},
	{  70, 233, "/dev/scsi/host7/bus0/target3/lun0/part41"},
	{  70, 234, "/dev/scsi/host7/bus0/target3/lun0/part42"},
	{  70, 235, "/dev/scsi/host7/bus0/target3/lun0/part43"},
	{  70, 236, "/dev/scsi/host7/bus0/target3/lun0/part44"},
	{  70, 237, "/dev/scsi/host7/bus0/target3/lun0/part45"},
	{  70, 238, "/dev/scsi/host7/bus0/target3/lun0/part46"},
	{  70, 239, "/dev/scsi/host7/bus0/target3/lun0/part47"},
	{  70, 240, "/dev/scsi/host7/bus0/target3/lun0/part48"},
	{  70, 241, "/dev/scsi/host7/bus0/target3/lun0/part49"},
	{  70, 242, "/dev/scsi/host7/bus0/target3/lun0/part50"},
	{  70, 243, "/dev/scsi/host7/bus0/target3/lun0/part51"},
	{  70, 244, "/dev/scsi/host7/bus0/target3/lun0/part52"},
	{  70, 245, "/dev/scsi/host7/bus0/target3/lun0/part53"},
	{  70, 246, "/dev/scsi/host7/bus0/target3/lun0/part54"},
	{  70, 247, "/dev/scsi/host7/bus0/target3/lun0/part55"},
	{  70, 248, "/dev/scsi/host7/bus0/target3/lun0/part56"},
	{  70, 249, "/dev/scsi/host7/bus0/target3/lun0/part57"},
	{  70, 250, "/dev/scsi/host7/bus0/target3/lun0/part58"},
	{  70, 251, "/dev/scsi/host7/bus0/target3/lun0/part59"},
	{  70, 252, "/dev/scsi/host7/bus0/target3/lun0/part60"},
	{  70, 253, "/dev/scsi/host7/bus0/target3/lun0/part61"},
	{  70, 254, "/dev/scsi/host7/bus0/target3/lun0/part62"},
	{  70, 255, "/dev/scsi/host7/bus0/target3/lun0/part63"},

	{  71,   0, "/dev/scsi/host8/bus0/target0/lun0/disc"},
	{  71,   1, "/dev/scsi/host8/bus0/target0/lun0/part1"},
	{  71,   2, "/dev/scsi/host8/bus0/target0/lun0/part2"},
	{  71,   3, "/dev/scsi/host8/bus0/target0/lun0/part3"},
	{  71,   4, "/dev/scsi/host8/bus0/target0/lun0/part4"},
	{  71,   5, "/dev/scsi/host8/bus0/target0/lun0/part5"},
	{  71,   6, "/dev/scsi/host8/bus0/target0/lun0/part6"},
	{  71,   7, "/dev/scsi/host8/bus0/target0/lun0/part7"},
	{  71,   8, "/dev/scsi/host8/bus0/target0/lun0/part8"},
	{  71,   9, "/dev/scsi/host8/bus0/target0/lun0/part9"},
	{  71,  10, "/dev/scsi/host8/bus0/target0/lun0/part10"},
	{  71,  11, "/dev/scsi/host8/bus0/target0/lun0/part11"},
	{  71,  12, "/dev/scsi/host8/bus0/target0/lun0/part12"},
	{  71,  13, "/dev/scsi/host8/bus0/target0/lun0/part13"},
	{  71,  14, "/dev/scsi/host8/bus0/target0/lun0/part14"},
	{  71,  15, "/dev/scsi/host8/bus0/target0/lun0/part15"},
	{  71,  16, "/dev/scsi/host8/bus0/target0/lun0/part16"},
	{  71,  17, "/dev/scsi/host8/bus0/target0/lun0/part17"},
	{  71,  18, "/dev/scsi/host8/bus0/target0/lun0/part18"},
	{  71,  19, "/dev/scsi/host8/bus0/target0/lun0/part19"},
	{  71,  20, "/dev/scsi/host8/bus0/target0/lun0/part20"},
	{  71,  21, "/dev/scsi/host8/bus0/target0/lun0/part21"},
	{  71,  22, "/dev/scsi/host8/bus0/target0/lun0/part22"},
	{  71,  23, "/dev/scsi/host8/bus0/target0/lun0/part23"},
	{  71,  24, "/dev/scsi/host8/bus0/target0/lun0/part24"},
	{  71,  25, "/dev/scsi/host8/bus0/target0/lun0/part25"},
	{  71,  26, "/dev/scsi/host8/bus0/target0/lun0/part26"},
	{  71,  27, "/dev/scsi/host8/bus0/target0/lun0/part27"},
	{  71,  28, "/dev/scsi/host8/bus0/target0/lun0/part28"},
	{  71,  29, "/dev/scsi/host8/bus0/target0/lun0/part29"},
	{  71,  30, "/dev/scsi/host8/bus0/target0/lun0/part30"},
	{  71,  31, "/dev/scsi/host8/bus0/target0/lun0/part31"},
	{  71,  32, "/dev/scsi/host8/bus0/target0/lun0/part32"},
	{  71,  33, "/dev/scsi/host8/bus0/target0/lun0/part33"},
	{  71,  34, "/dev/scsi/host8/bus0/target0/lun0/part34"},
	{  71,  35, "/dev/scsi/host8/bus0/target0/lun0/part35"},
	{  71,  36, "/dev/scsi/host8/bus0/target0/lun0/part36"},
	{  71,  37, "/dev/scsi/host8/bus0/target0/lun0/part37"},
	{  71,  38, "/dev/scsi/host8/bus0/target0/lun0/part38"},
	{  71,  39, "/dev/scsi/host8/bus0/target0/lun0/part39"},
	{  71,  40, "/dev/scsi/host8/bus0/target0/lun0/part40"},
	{  71,  41, "/dev/scsi/host8/bus0/target0/lun0/part41"},
	{  71,  42, "/dev/scsi/host8/bus0/target0/lun0/part42"},
	{  71,  43, "/dev/scsi/host8/bus0/target0/lun0/part43"},
	{  71,  44, "/dev/scsi/host8/bus0/target0/lun0/part44"},
	{  71,  45, "/dev/scsi/host8/bus0/target0/lun0/part45"},
	{  71,  46, "/dev/scsi/host8/bus0/target0/lun0/part46"},
	{  71,  47, "/dev/scsi/host8/bus0/target0/lun0/part47"},
	{  71,  48, "/dev/scsi/host8/bus0/target0/lun0/part48"},
	{  71,  49, "/dev/scsi/host8/bus0/target0/lun0/part49"},
	{  71,  50, "/dev/scsi/host8/bus0/target0/lun0/part50"},
	{  71,  51, "/dev/scsi/host8/bus0/target0/lun0/part51"},
	{  71,  52, "/dev/scsi/host8/bus0/target0/lun0/part52"},
	{  71,  53, "/dev/scsi/host8/bus0/target0/lun0/part53"},
	{  71,  54, "/dev/scsi/host8/bus0/target0/lun0/part54"},
	{  71,  55, "/dev/scsi/host8/bus0/target0/lun0/part55"},
	{  71,  56, "/dev/scsi/host8/bus0/target0/lun0/part56"},
	{  71,  57, "/dev/scsi/host8/bus0/target0/lun0/part57"},
	{  71,  58, "/dev/scsi/host8/bus0/target0/lun0/part58"},
	{  71,  59, "/dev/scsi/host8/bus0/target0/lun0/part59"},
	{  71,  60, "/dev/scsi/host8/bus0/target0/lun0/part60"},
	{  71,  61, "/dev/scsi/host8/bus0/target0/lun0/part61"},
	{  71,  62, "/dev/scsi/host8/bus0/target0/lun0/part62"},
	{  71,  63, "/dev/scsi/host8/bus0/target0/lun0/part63"},
	{  71,  64, "/dev/scsi/host8/bus0/target1/lun0/disc"},
	{  71,  65, "/dev/scsi/host8/bus0/target1/lun0/part1"},
	{  71,  66, "/dev/scsi/host8/bus0/target1/lun0/part2"},
	{  71,  67, "/dev/scsi/host8/bus0/target1/lun0/part3"},
	{  71,  68, "/dev/scsi/host8/bus0/target1/lun0/part4"},
	{  71,  69, "/dev/scsi/host8/bus0/target1/lun0/part5"},
	{  71,  70, "/dev/scsi/host8/bus0/target1/lun0/part6"},
	{  71,  71, "/dev/scsi/host8/bus0/target1/lun0/part7"},
	{  71,  72, "/dev/scsi/host8/bus0/target1/lun0/part8"},
	{  71,  73, "/dev/scsi/host8/bus0/target1/lun0/part9"},
	{  71,  74, "/dev/scsi/host8/bus0/target1/lun0/part10"},
	{  71,  75, "/dev/scsi/host8/bus0/target1/lun0/part11"},
	{  71,  76, "/dev/scsi/host8/bus0/target1/lun0/part12"},
	{  71,  77, "/dev/scsi/host8/bus0/target1/lun0/part13"},
	{  71,  78, "/dev/scsi/host8/bus0/target1/lun0/part14"},
	{  71,  79, "/dev/scsi/host8/bus0/target1/lun0/part15"},
	{  71,  80, "/dev/scsi/host8/bus0/target1/lun0/part16"},
	{  71,  81, "/dev/scsi/host8/bus0/target1/lun0/part17"},
	{  71,  82, "/dev/scsi/host8/bus0/target1/lun0/part18"},
	{  71,  83, "/dev/scsi/host8/bus0/target1/lun0/part19"},
	{  71,  84, "/dev/scsi/host8/bus0/target1/lun0/part20"},
	{  71,  85, "/dev/scsi/host8/bus0/target1/lun0/part21"},
	{  71,  86, "/dev/scsi/host8/bus0/target1/lun0/part22"},
	{  71,  87, "/dev/scsi/host8/bus0/target1/lun0/part23"},
	{  71,  88, "/dev/scsi/host8/bus0/target1/lun0/part24"},
	{  71,  89, "/dev/scsi/host8/bus0/target1/lun0/part25"},
	{  71,  90, "/dev/scsi/host8/bus0/target1/lun0/part26"},
	{  71,  91, "/dev/scsi/host8/bus0/target1/lun0/part27"},
	{  71,  92, "/dev/scsi/host8/bus0/target1/lun0/part28"},
	{  71,  93, "/dev/scsi/host8/bus0/target1/lun0/part29"},
	{  71,  94, "/dev/scsi/host8/bus0/target1/lun0/part30"},
	{  71,  95, "/dev/scsi/host8/bus0/target1/lun0/part31"},
	{  71,  96, "/dev/scsi/host8/bus0/target1/lun0/part32"},
	{  71,  97, "/dev/scsi/host8/bus0/target1/lun0/part33"},
	{  71,  98, "/dev/scsi/host8/bus0/target1/lun0/part34"},
	{  71,  99, "/dev/scsi/host8/bus0/target1/lun0/part35"},
	{  71, 100, "/dev/scsi/host8/bus0/target1/lun0/part36"},
	{  71, 101, "/dev/scsi/host8/bus0/target1/lun0/part37"},
	{  71, 102, "/dev/scsi/host8/bus0/target1/lun0/part38"},
	{  71, 103, "/dev/scsi/host8/bus0/target1/lun0/part39"},
	{  71, 104, "/dev/scsi/host8/bus0/target1/lun0/part40"},
	{  71, 105, "/dev/scsi/host8/bus0/target1/lun0/part41"},
	{  71, 106, "/dev/scsi/host8/bus0/target1/lun0/part42"},
	{  71, 107, "/dev/scsi/host8/bus0/target1/lun0/part43"},
	{  71, 108, "/dev/scsi/host8/bus0/target1/lun0/part44"},
	{  71, 109, "/dev/scsi/host8/bus0/target1/lun0/part45"},
	{  71, 110, "/dev/scsi/host8/bus0/target1/lun0/part46"},
	{  71, 111, "/dev/scsi/host8/bus0/target1/lun0/part47"},
	{  71, 112, "/dev/scsi/host8/bus0/target1/lun0/part48"},
	{  71, 113, "/dev/scsi/host8/bus0/target1/lun0/part49"},
	{  71, 114, "/dev/scsi/host8/bus0/target1/lun0/part50"},
	{  71, 115, "/dev/scsi/host8/bus0/target1/lun0/part51"},
	{  71, 116, "/dev/scsi/host8/bus0/target1/lun0/part52"},
	{  71, 117, "/dev/scsi/host8/bus0/target1/lun0/part53"},
	{  71, 118, "/dev/scsi/host8/bus0/target1/lun0/part54"},
	{  71, 119, "/dev/scsi/host8/bus0/target1/lun0/part55"},
	{  71, 120, "/dev/scsi/host8/bus0/target1/lun0/part56"},
	{  71, 121, "/dev/scsi/host8/bus0/target1/lun0/part57"},
	{  71, 122, "/dev/scsi/host8/bus0/target1/lun0/part58"},
	{  71, 123, "/dev/scsi/host8/bus0/target1/lun0/part59"},
	{  71, 124, "/dev/scsi/host8/bus0/target1/lun0/part60"},
	{  71, 125, "/dev/scsi/host8/bus0/target1/lun0/part61"},
	{  71, 126, "/dev/scsi/host8/bus0/target1/lun0/part62"},
	{  71, 127, "/dev/scsi/host8/bus0/target1/lun0/part63"},
	{  71, 128, "/dev/scsi/host8/bus0/target2/lun0/disc"},
	{  71, 129, "/dev/scsi/host8/bus0/target2/lun0/part1"},
	{  71, 130, "/dev/scsi/host8/bus0/target2/lun0/part2"},
	{  71, 131, "/dev/scsi/host8/bus0/target2/lun0/part3"},
	{  71, 132, "/dev/scsi/host8/bus0/target2/lun0/part4"},
	{  71, 133, "/dev/scsi/host8/bus0/target2/lun0/part5"},
	{  71, 134, "/dev/scsi/host8/bus0/target2/lun0/part6"},
	{  71, 135, "/dev/scsi/host8/bus0/target2/lun0/part7"},
	{  71, 136, "/dev/scsi/host8/bus0/target2/lun0/part8"},
	{  71, 137, "/dev/scsi/host8/bus0/target2/lun0/part9"},
	{  71, 138, "/dev/scsi/host8/bus0/target2/lun0/part10"},
	{  71, 139, "/dev/scsi/host8/bus0/target2/lun0/part11"},
	{  71, 140, "/dev/scsi/host8/bus0/target2/lun0/part12"},
	{  71, 141, "/dev/scsi/host8/bus0/target2/lun0/part13"},
	{  71, 142, "/dev/scsi/host8/bus0/target2/lun0/part14"},
	{  71, 143, "/dev/scsi/host8/bus0/target2/lun0/part15"},
	{  71, 144, "/dev/scsi/host8/bus0/target2/lun0/part16"},
	{  71, 145, "/dev/scsi/host8/bus0/target2/lun0/part17"},
	{  71, 146, "/dev/scsi/host8/bus0/target2/lun0/part18"},
	{  71, 147, "/dev/scsi/host8/bus0/target2/lun0/part19"},
	{  71, 148, "/dev/scsi/host8/bus0/target2/lun0/part20"},
	{  71, 149, "/dev/scsi/host8/bus0/target2/lun0/part21"},
	{  71, 150, "/dev/scsi/host8/bus0/target2/lun0/part22"},
	{  71, 151, "/dev/scsi/host8/bus0/target2/lun0/part23"},
	{  71, 152, "/dev/scsi/host8/bus0/target2/lun0/part24"},
	{  71, 153, "/dev/scsi/host8/bus0/target2/lun0/part25"},
	{  71, 154, "/dev/scsi/host8/bus0/target2/lun0/part26"},
	{  71, 155, "/dev/scsi/host8/bus0/target2/lun0/part27"},
	{  71, 156, "/dev/scsi/host8/bus0/target2/lun0/part28"},
	{  71, 157, "/dev/scsi/host8/bus0/target2/lun0/part29"},
	{  71, 158, "/dev/scsi/host8/bus0/target2/lun0/part30"},
	{  71, 159, "/dev/scsi/host8/bus0/target2/lun0/part31"},
	{  71, 160, "/dev/scsi/host8/bus0/target2/lun0/part32"},
	{  71, 161, "/dev/scsi/host8/bus0/target2/lun0/part33"},
	{  71, 162, "/dev/scsi/host8/bus0/target2/lun0/part34"},
	{  71, 163, "/dev/scsi/host8/bus0/target2/lun0/part35"},
	{  71, 164, "/dev/scsi/host8/bus0/target2/lun0/part36"},
	{  71, 165, "/dev/scsi/host8/bus0/target2/lun0/part37"},
	{  71, 166, "/dev/scsi/host8/bus0/target2/lun0/part38"},
	{  71, 167, "/dev/scsi/host8/bus0/target2/lun0/part39"},
	{  71, 168, "/dev/scsi/host8/bus0/target2/lun0/part40"},
	{  71, 169, "/dev/scsi/host8/bus0/target2/lun0/part41"},
	{  71, 170, "/dev/scsi/host8/bus0/target2/lun0/part42"},
	{  71, 171, "/dev/scsi/host8/bus0/target2/lun0/part43"},
	{  71, 172, "/dev/scsi/host8/bus0/target2/lun0/part44"},
	{  71, 173, "/dev/scsi/host8/bus0/target2/lun0/part45"},
	{  71, 174, "/dev/scsi/host8/bus0/target2/lun0/part46"},
	{  71, 175, "/dev/scsi/host8/bus0/target2/lun0/part47"},
	{  71, 176, "/dev/scsi/host8/bus0/target2/lun0/part48"},
	{  71, 177, "/dev/scsi/host8/bus0/target2/lun0/part49"},
	{  71, 178, "/dev/scsi/host8/bus0/target2/lun0/part50"},
	{  71, 179, "/dev/scsi/host8/bus0/target2/lun0/part51"},
	{  71, 180, "/dev/scsi/host8/bus0/target2/lun0/part52"},
	{  71, 181, "/dev/scsi/host8/bus0/target2/lun0/part53"},
	{  71, 182, "/dev/scsi/host8/bus0/target2/lun0/part54"},
	{  71, 183, "/dev/scsi/host8/bus0/target2/lun0/part55"},
	{  71, 184, "/dev/scsi/host8/bus0/target2/lun0/part56"},
	{  71, 185, "/dev/scsi/host8/bus0/target2/lun0/part57"},
	{  71, 186, "/dev/scsi/host8/bus0/target2/lun0/part58"},
	{  71, 187, "/dev/scsi/host8/bus0/target2/lun0/part59"},
	{  71, 188, "/dev/scsi/host8/bus0/target2/lun0/part60"},
	{  71, 189, "/dev/scsi/host8/bus0/target2/lun0/part61"},
	{  71, 190, "/dev/scsi/host8/bus0/target2/lun0/part62"},
	{  71, 191, "/dev/scsi/host8/bus0/target2/lun0/part63"},
	{  71, 192, "/dev/scsi/host8/bus0/target3/lun0/disc"},
	{  71, 193, "/dev/scsi/host8/bus0/target3/lun0/part1"},
	{  71, 194, "/dev/scsi/host8/bus0/target3/lun0/part2"},
	{  71, 195, "/dev/scsi/host8/bus0/target3/lun0/part3"},
	{  71, 196, "/dev/scsi/host8/bus0/target3/lun0/part4"},
	{  71, 197, "/dev/scsi/host8/bus0/target3/lun0/part5"},
	{  71, 198, "/dev/scsi/host8/bus0/target3/lun0/part6"},
	{  71, 199, "/dev/scsi/host8/bus0/target3/lun0/part7"},
	{  71, 200, "/dev/scsi/host8/bus0/target3/lun0/part8"},
	{  71, 201, "/dev/scsi/host8/bus0/target3/lun0/part9"},
	{  71, 202, "/dev/scsi/host8/bus0/target3/lun0/part10"},
	{  71, 203, "/dev/scsi/host8/bus0/target3/lun0/part11"},
	{  71, 204, "/dev/scsi/host8/bus0/target3/lun0/part12"},
	{  71, 205, "/dev/scsi/host8/bus0/target3/lun0/part13"},
	{  71, 206, "/dev/scsi/host8/bus0/target3/lun0/part14"},
	{  71, 207, "/dev/scsi/host8/bus0/target3/lun0/part15"},
	{  71, 208, "/dev/scsi/host8/bus0/target3/lun0/part16"},
	{  71, 209, "/dev/scsi/host8/bus0/target3/lun0/part17"},
	{  71, 210, "/dev/scsi/host8/bus0/target3/lun0/part18"},
	{  71, 211, "/dev/scsi/host8/bus0/target3/lun0/part19"},
	{  71, 212, "/dev/scsi/host8/bus0/target3/lun0/part20"},
	{  71, 213, "/dev/scsi/host8/bus0/target3/lun0/part21"},
	{  71, 214, "/dev/scsi/host8/bus0/target3/lun0/part22"},
	{  71, 215, "/dev/scsi/host8/bus0/target3/lun0/part23"},
	{  71, 216, "/dev/scsi/host8/bus0/target3/lun0/part24"},
	{  71, 217, "/dev/scsi/host8/bus0/target3/lun0/part25"},
	{  71, 218, "/dev/scsi/host8/bus0/target3/lun0/part26"},
	{  71, 219, "/dev/scsi/host8/bus0/target3/lun0/part27"},
	{  71, 220, "/dev/scsi/host8/bus0/target3/lun0/part28"},
	{  71, 221, "/dev/scsi/host8/bus0/target3/lun0/part29"},
	{  71, 222, "/dev/scsi/host8/bus0/target3/lun0/part30"},
	{  71, 223, "/dev/scsi/host8/bus0/target3/lun0/part31"},
	{  71, 224, "/dev/scsi/host8/bus0/target3/lun0/part32"},
	{  71, 225, "/dev/scsi/host8/bus0/target3/lun0/part33"},
	{  71, 226, "/dev/scsi/host8/bus0/target3/lun0/part34"},
	{  71, 227, "/dev/scsi/host8/bus0/target3/lun0/part35"},
	{  71, 228, "/dev/scsi/host8/bus0/target3/lun0/part36"},
	{  71, 229, "/dev/scsi/host8/bus0/target3/lun0/part37"},
	{  71, 230, "/dev/scsi/host8/bus0/target3/lun0/part38"},
	{  71, 231, "/dev/scsi/host8/bus0/target3/lun0/part39"},
	{  71, 232, "/dev/scsi/host8/bus0/target3/lun0/part40"},
	{  71, 233, "/dev/scsi/host8/bus0/target3/lun0/part41"},
	{  71, 234, "/dev/scsi/host8/bus0/target3/lun0/part42"},
	{  71, 235, "/dev/scsi/host8/bus0/target3/lun0/part43"},
	{  71, 236, "/dev/scsi/host8/bus0/target3/lun0/part44"},
	{  71, 237, "/dev/scsi/host8/bus0/target3/lun0/part45"},
	{  71, 238, "/dev/scsi/host8/bus0/target3/lun0/part46"},
	{  71, 239, "/dev/scsi/host8/bus0/target3/lun0/part47"},
	{  71, 240, "/dev/scsi/host8/bus0/target3/lun0/part48"},
	{  71, 241, "/dev/scsi/host8/bus0/target3/lun0/part49"},
	{  71, 242, "/dev/scsi/host8/bus0/target3/lun0/part50"},
	{  71, 243, "/dev/scsi/host8/bus0/target3/lun0/part51"},
	{  71, 244, "/dev/scsi/host8/bus0/target3/lun0/part52"},
	{  71, 245, "/dev/scsi/host8/bus0/target3/lun0/part53"},
	{  71, 246, "/dev/scsi/host8/bus0/target3/lun0/part54"},
	{  71, 247, "/dev/scsi/host8/bus0/target3/lun0/part55"},
	{  71, 248, "/dev/scsi/host8/bus0/target3/lun0/part56"},
	{  71, 249, "/dev/scsi/host8/bus0/target3/lun0/part57"},
	{  71, 250, "/dev/scsi/host8/bus0/target3/lun0/part58"},
	{  71, 251, "/dev/scsi/host8/bus0/target3/lun0/part59"},
	{  71, 252, "/dev/scsi/host8/bus0/target3/lun0/part60"},
	{  71, 253, "/dev/scsi/host8/bus0/target3/lun0/part61"},
	{  71, 254, "/dev/scsi/host8/bus0/target3/lun0/part62"},
	{  71, 255, "/dev/scsi/host8/bus0/target3/lun0/part63"},

	{  33,   0, "/dev/ide/host2/bus0/target0/lun0/disc"},
	{  33,   1, "/dev/ide/host2/bus0/target0/lun0/part1"},
	{  33,   2, "/dev/ide/host2/bus0/target0/lun0/part2"},
	{  33,   3, "/dev/ide/host2/bus0/target0/lun0/part3"},
	{  33,   4, "/dev/ide/host2/bus0/target0/lun0/part4"},
	{  33,   5, "/dev/ide/host2/bus0/target0/lun0/part5"},
	{  33,   6, "/dev/ide/host2/bus0/target0/lun0/part6"},
	{  33,   7, "/dev/ide/host2/bus0/target0/lun0/part7"},
	{  33,   8, "/dev/ide/host2/bus0/target0/lun0/part8"},
	{  33,   9, "/dev/ide/host2/bus0/target0/lun0/part9"},
	{  33,  10, "/dev/ide/host2/bus0/target0/lun0/part10"},
	{  33,  11, "/dev/ide/host2/bus0/target0/lun0/part11"},
	{  33,  12, "/dev/ide/host2/bus0/target0/lun0/part12"},
	{  33,  13, "/dev/ide/host2/bus0/target0/lun0/part13"},
	{  33,  14, "/dev/ide/host2/bus0/target0/lun0/part14"},
	{  33,  15, "/dev/ide/host2/bus0/target0/lun0/part15"},
	{  33,  16, "/dev/ide/host2/bus0/target0/lun0/part16"},
	{  33,  17, "/dev/ide/host2/bus0/target0/lun0/part17"},
	{  33,  18, "/dev/ide/host2/bus0/target0/lun0/part18"},
	{  33,  19, "/dev/ide/host2/bus0/target0/lun0/part19"},
	{  33,  20, "/dev/ide/host2/bus0/target0/lun0/part20"},
	{  33,  21, "/dev/ide/host2/bus0/target0/lun0/part21"},
	{  33,  22, "/dev/ide/host2/bus0/target0/lun0/part22"},
	{  33,  23, "/dev/ide/host2/bus0/target0/lun0/part23"},
	{  33,  24, "/dev/ide/host2/bus0/target0/lun0/part24"},
	{  33,  25, "/dev/ide/host2/bus0/target0/lun0/part25"},
	{  33,  26, "/dev/ide/host2/bus0/target0/lun0/part26"},
	{  33,  27, "/dev/ide/host2/bus0/target0/lun0/part27"},
	{  33,  28, "/dev/ide/host2/bus0/target0/lun0/part28"},
	{  33,  29, "/dev/ide/host2/bus0/target0/lun0/part29"},
	{  33,  30, "/dev/ide/host2/bus0/target0/lun0/part30"},
	{  33,  31, "/dev/ide/host2/bus0/target0/lun0/part31"},
	{  33,  32, "/dev/ide/host2/bus0/target0/lun0/part32"},
	{  33,  33, "/dev/ide/host2/bus0/target0/lun0/part33"},
	{  33,  34, "/dev/ide/host2/bus0/target0/lun0/part34"},
	{  33,  35, "/dev/ide/host2/bus0/target0/lun0/part35"},
	{  33,  36, "/dev/ide/host2/bus0/target0/lun0/part36"},
	{  33,  37, "/dev/ide/host2/bus0/target0/lun0/part37"},
	{  33,  38, "/dev/ide/host2/bus0/target0/lun0/part38"},
	{  33,  39, "/dev/ide/host2/bus0/target0/lun0/part39"},
	{  33,  40, "/dev/ide/host2/bus0/target0/lun0/part40"},
	{  33,  41, "/dev/ide/host2/bus0/target0/lun0/part41"},
	{  33,  42, "/dev/ide/host2/bus0/target0/lun0/part42"},
	{  33,  43, "/dev/ide/host2/bus0/target0/lun0/part43"},
	{  33,  44, "/dev/ide/host2/bus0/target0/lun0/part44"},
	{  33,  45, "/dev/ide/host2/bus0/target0/lun0/part45"},
	{  33,  46, "/dev/ide/host2/bus0/target0/lun0/part46"},
	{  33,  47, "/dev/ide/host2/bus0/target0/lun0/part47"},
	{  33,  48, "/dev/ide/host2/bus0/target0/lun0/part48"},
	{  33,  49, "/dev/ide/host2/bus0/target0/lun0/part49"},
	{  33,  50, "/dev/ide/host2/bus0/target0/lun0/part50"},
	{  33,  51, "/dev/ide/host2/bus0/target0/lun0/part51"},
	{  33,  52, "/dev/ide/host2/bus0/target0/lun0/part52"},
	{  33,  53, "/dev/ide/host2/bus0/target0/lun0/part53"},
	{  33,  54, "/dev/ide/host2/bus0/target0/lun0/part54"},
	{  33,  55, "/dev/ide/host2/bus0/target0/lun0/part55"},
	{  33,  56, "/dev/ide/host2/bus0/target0/lun0/part56"},
	{  33,  57, "/dev/ide/host2/bus0/target0/lun0/part57"},
	{  33,  58, "/dev/ide/host2/bus0/target0/lun0/part58"},
	{  33,  59, "/dev/ide/host2/bus0/target0/lun0/part59"},
	{  33,  60, "/dev/ide/host2/bus0/target0/lun0/part60"},
	{  33,  61, "/dev/ide/host2/bus0/target0/lun0/part61"},
	{  33,  62, "/dev/ide/host2/bus0/target0/lun0/part62"},
	{  33,  63, "/dev/ide/host2/bus0/target0/lun0/part63"},
	{  33,  64, "/dev/ide/host2/bus0/target1/lun0/disc"},
	{  33,  65, "/dev/ide/host2/bus0/target1/lun0/part1"},
	{  33,  66, "/dev/ide/host2/bus0/target1/lun0/part2"},
	{  33,  67, "/dev/ide/host2/bus0/target1/lun0/part3"},
	{  33,  68, "/dev/ide/host2/bus0/target1/lun0/part4"},
	{  33,  69, "/dev/ide/host2/bus0/target1/lun0/part5"},
	{  33,  70, "/dev/ide/host2/bus0/target1/lun0/part6"},
	{  33,  71, "/dev/ide/host2/bus0/target1/lun0/part7"},
	{  33,  72, "/dev/ide/host2/bus0/target1/lun0/part8"},
	{  33,  73, "/dev/ide/host2/bus0/target1/lun0/part9"},
	{  33,  74, "/dev/ide/host2/bus0/target1/lun0/part10"},
	{  33,  75, "/dev/ide/host2/bus0/target1/lun0/part11"},
	{  33,  76, "/dev/ide/host2/bus0/target1/lun0/part12"},
	{  33,  77, "/dev/ide/host2/bus0/target1/lun0/part13"},
	{  33,  78, "/dev/ide/host2/bus0/target1/lun0/part14"},
	{  33,  79, "/dev/ide/host2/bus0/target1/lun0/part15"},
	{  33,  80, "/dev/ide/host2/bus0/target1/lun0/part16"},
	{  33,  81, "/dev/ide/host2/bus0/target1/lun0/part17"},
	{  33,  82, "/dev/ide/host2/bus0/target1/lun0/part18"},
	{  33,  83, "/dev/ide/host2/bus0/target1/lun0/part19"},
	{  33,  84, "/dev/ide/host2/bus0/target1/lun0/part20"},
	{  33,  85, "/dev/ide/host2/bus0/target1/lun0/part21"},
	{  33,  86, "/dev/ide/host2/bus0/target1/lun0/part22"},
	{  33,  87, "/dev/ide/host2/bus0/target1/lun0/part23"},
	{  33,  88, "/dev/ide/host2/bus0/target1/lun0/part24"},
	{  33,  89, "/dev/ide/host2/bus0/target1/lun0/part25"},
	{  33,  90, "/dev/ide/host2/bus0/target1/lun0/part26"},
	{  33,  91, "/dev/ide/host2/bus0/target1/lun0/part27"},
	{  33,  92, "/dev/ide/host2/bus0/target1/lun0/part28"},
	{  33,  93, "/dev/ide/host2/bus0/target1/lun0/part29"},
	{  33,  94, "/dev/ide/host2/bus0/target1/lun0/part30"},
	{  33,  95, "/dev/ide/host2/bus0/target1/lun0/part31"},
	{  33,  96, "/dev/ide/host2/bus0/target1/lun0/part32"},
	{  33,  97, "/dev/ide/host2/bus0/target1/lun0/part33"},
	{  33,  98, "/dev/ide/host2/bus0/target1/lun0/part34"},
	{  33,  99, "/dev/ide/host2/bus0/target1/lun0/part35"},
	{  33, 100, "/dev/ide/host2/bus0/target1/lun0/part36"},
	{  33, 101, "/dev/ide/host2/bus0/target1/lun0/part37"},
	{  33, 102, "/dev/ide/host2/bus0/target1/lun0/part38"},
	{  33, 103, "/dev/ide/host2/bus0/target1/lun0/part39"},
	{  33, 104, "/dev/ide/host2/bus0/target1/lun0/part40"},
	{  33, 105, "/dev/ide/host2/bus0/target1/lun0/part41"},
	{  33, 106, "/dev/ide/host2/bus0/target1/lun0/part42"},
	{  33, 107, "/dev/ide/host2/bus0/target1/lun0/part43"},
	{  33, 108, "/dev/ide/host2/bus0/target1/lun0/part44"},
	{  33, 109, "/dev/ide/host2/bus0/target1/lun0/part45"},
	{  33, 110, "/dev/ide/host2/bus0/target1/lun0/part46"},
	{  33, 111, "/dev/ide/host2/bus0/target1/lun0/part47"},
	{  33, 112, "/dev/ide/host2/bus0/target1/lun0/part48"},
	{  33, 113, "/dev/ide/host2/bus0/target1/lun0/part49"},
	{  33, 114, "/dev/ide/host2/bus0/target1/lun0/part50"},
	{  33, 115, "/dev/ide/host2/bus0/target1/lun0/part51"},
	{  33, 116, "/dev/ide/host2/bus0/target1/lun0/part52"},
	{  33, 117, "/dev/ide/host2/bus0/target1/lun0/part53"},
	{  33, 118, "/dev/ide/host2/bus0/target1/lun0/part54"},
	{  33, 119, "/dev/ide/host2/bus0/target1/lun0/part55"},
	{  33, 120, "/dev/ide/host2/bus0/target1/lun0/part56"},
	{  33, 121, "/dev/ide/host2/bus0/target1/lun0/part57"},
	{  33, 122, "/dev/ide/host2/bus0/target1/lun0/part58"},
	{  33, 123, "/dev/ide/host2/bus0/target1/lun0/part59"},
	{  33, 124, "/dev/ide/host2/bus0/target1/lun0/part60"},
	{  33, 125, "/dev/ide/host2/bus0/target1/lun0/part61"},
	{  33, 126, "/dev/ide/host2/bus0/target1/lun0/part62"},
	{  33, 127, "/dev/ide/host2/bus0/target1/lun0/part63"},
	{  33, 128, "/dev/ide/host2/bus0/target2/lun0/disc"},
	{  33, 129, "/dev/ide/host2/bus0/target2/lun0/part1"},
	{  33, 130, "/dev/ide/host2/bus0/target2/lun0/part2"},
	{  33, 131, "/dev/ide/host2/bus0/target2/lun0/part3"},
	{  33, 132, "/dev/ide/host2/bus0/target2/lun0/part4"},
	{  33, 133, "/dev/ide/host2/bus0/target2/lun0/part5"},
	{  33, 134, "/dev/ide/host2/bus0/target2/lun0/part6"},
	{  33, 135, "/dev/ide/host2/bus0/target2/lun0/part7"},
	{  33, 136, "/dev/ide/host2/bus0/target2/lun0/part8"},
	{  33, 137, "/dev/ide/host2/bus0/target2/lun0/part9"},
	{  33, 138, "/dev/ide/host2/bus0/target2/lun0/part10"},
	{  33, 139, "/dev/ide/host2/bus0/target2/lun0/part11"},
	{  33, 140, "/dev/ide/host2/bus0/target2/lun0/part12"},
	{  33, 141, "/dev/ide/host2/bus0/target2/lun0/part13"},
	{  33, 142, "/dev/ide/host2/bus0/target2/lun0/part14"},
	{  33, 143, "/dev/ide/host2/bus0/target2/lun0/part15"},
	{  33, 144, "/dev/ide/host2/bus0/target2/lun0/part16"},
	{  33, 145, "/dev/ide/host2/bus0/target2/lun0/part17"},
	{  33, 146, "/dev/ide/host2/bus0/target2/lun0/part18"},
	{  33, 147, "/dev/ide/host2/bus0/target2/lun0/part19"},
	{  33, 148, "/dev/ide/host2/bus0/target2/lun0/part20"},
	{  33, 149, "/dev/ide/host2/bus0/target2/lun0/part21"},
	{  33, 150, "/dev/ide/host2/bus0/target2/lun0/part22"},
	{  33, 151, "/dev/ide/host2/bus0/target2/lun0/part23"},
	{  33, 152, "/dev/ide/host2/bus0/target2/lun0/part24"},
	{  33, 153, "/dev/ide/host2/bus0/target2/lun0/part25"},
	{  33, 154, "/dev/ide/host2/bus0/target2/lun0/part26"},
	{  33, 155, "/dev/ide/host2/bus0/target2/lun0/part27"},
	{  33, 156, "/dev/ide/host2/bus0/target2/lun0/part28"},
	{  33, 157, "/dev/ide/host2/bus0/target2/lun0/part29"},
	{  33, 158, "/dev/ide/host2/bus0/target2/lun0/part30"},
	{  33, 159, "/dev/ide/host2/bus0/target2/lun0/part31"},
	{  33, 160, "/dev/ide/host2/bus0/target2/lun0/part32"},
	{  33, 161, "/dev/ide/host2/bus0/target2/lun0/part33"},
	{  33, 162, "/dev/ide/host2/bus0/target2/lun0/part34"},
	{  33, 163, "/dev/ide/host2/bus0/target2/lun0/part35"},
	{  33, 164, "/dev/ide/host2/bus0/target2/lun0/part36"},
	{  33, 165, "/dev/ide/host2/bus0/target2/lun0/part37"},
	{  33, 166, "/dev/ide/host2/bus0/target2/lun0/part38"},
	{  33, 167, "/dev/ide/host2/bus0/target2/lun0/part39"},
	{  33, 168, "/dev/ide/host2/bus0/target2/lun0/part40"},
	{  33, 169, "/dev/ide/host2/bus0/target2/lun0/part41"},
	{  33, 170, "/dev/ide/host2/bus0/target2/lun0/part42"},
	{  33, 171, "/dev/ide/host2/bus0/target2/lun0/part43"},
	{  33, 172, "/dev/ide/host2/bus0/target2/lun0/part44"},
	{  33, 173, "/dev/ide/host2/bus0/target2/lun0/part45"},
	{  33, 174, "/dev/ide/host2/bus0/target2/lun0/part46"},
	{  33, 175, "/dev/ide/host2/bus0/target2/lun0/part47"},
	{  33, 176, "/dev/ide/host2/bus0/target2/lun0/part48"},
	{  33, 177, "/dev/ide/host2/bus0/target2/lun0/part49"},
	{  33, 178, "/dev/ide/host2/bus0/target2/lun0/part50"},
	{  33, 179, "/dev/ide/host2/bus0/target2/lun0/part51"},
	{  33, 180, "/dev/ide/host2/bus0/target2/lun0/part52"},
	{  33, 181, "/dev/ide/host2/bus0/target2/lun0/part53"},
	{  33, 182, "/dev/ide/host2/bus0/target2/lun0/part54"},
	{  33, 183, "/dev/ide/host2/bus0/target2/lun0/part55"},
	{  33, 184, "/dev/ide/host2/bus0/target2/lun0/part56"},
	{  33, 185, "/dev/ide/host2/bus0/target2/lun0/part57"},
	{  33, 186, "/dev/ide/host2/bus0/target2/lun0/part58"},
	{  33, 187, "/dev/ide/host2/bus0/target2/lun0/part59"},
	{  33, 188, "/dev/ide/host2/bus0/target2/lun0/part60"},
	{  33, 189, "/dev/ide/host2/bus0/target2/lun0/part61"},
	{  33, 190, "/dev/ide/host2/bus0/target2/lun0/part62"},
	{  33, 191, "/dev/ide/host2/bus0/target2/lun0/part63"},
	{  33, 192, "/dev/ide/host2/bus0/target3/lun0/disc"},
	{  33, 193, "/dev/ide/host2/bus0/target3/lun0/part1"},
	{  33, 194, "/dev/ide/host2/bus0/target3/lun0/part2"},
	{  33, 195, "/dev/ide/host2/bus0/target3/lun0/part3"},
	{  33, 196, "/dev/ide/host2/bus0/target3/lun0/part4"},
	{  33, 197, "/dev/ide/host2/bus0/target3/lun0/part5"},
	{  33, 198, "/dev/ide/host2/bus0/target3/lun0/part6"},
	{  33, 199, "/dev/ide/host2/bus0/target3/lun0/part7"},
	{  33, 200, "/dev/ide/host2/bus0/target3/lun0/part8"},
	{  33, 201, "/dev/ide/host2/bus0/target3/lun0/part9"},
	{  33, 202, "/dev/ide/host2/bus0/target3/lun0/part10"},
	{  33, 203, "/dev/ide/host2/bus0/target3/lun0/part11"},
	{  33, 204, "/dev/ide/host2/bus0/target3/lun0/part12"},
	{  33, 205, "/dev/ide/host2/bus0/target3/lun0/part13"},
	{  33, 206, "/dev/ide/host2/bus0/target3/lun0/part14"},
	{  33, 207, "/dev/ide/host2/bus0/target3/lun0/part15"},
	{  33, 208, "/dev/ide/host2/bus0/target3/lun0/part16"},
	{  33, 209, "/dev/ide/host2/bus0/target3/lun0/part17"},
	{  33, 210, "/dev/ide/host2/bus0/target3/lun0/part18"},
	{  33, 211, "/dev/ide/host2/bus0/target3/lun0/part19"},
	{  33, 212, "/dev/ide/host2/bus0/target3/lun0/part20"},
	{  33, 213, "/dev/ide/host2/bus0/target3/lun0/part21"},
	{  33, 214, "/dev/ide/host2/bus0/target3/lun0/part22"},
	{  33, 215, "/dev/ide/host2/bus0/target3/lun0/part23"},
	{  33, 216, "/dev/ide/host2/bus0/target3/lun0/part24"},
	{  33, 217, "/dev/ide/host2/bus0/target3/lun0/part25"},
	{  33, 218, "/dev/ide/host2/bus0/target3/lun0/part26"},
	{  33, 219, "/dev/ide/host2/bus0/target3/lun0/part27"},
	{  33, 220, "/dev/ide/host2/bus0/target3/lun0/part28"},
	{  33, 221, "/dev/ide/host2/bus0/target3/lun0/part29"},
	{  33, 222, "/dev/ide/host2/bus0/target3/lun0/part30"},
	{  33, 223, "/dev/ide/host2/bus0/target3/lun0/part31"},
	{  33, 224, "/dev/ide/host2/bus0/target3/lun0/part32"},
	{  33, 225, "/dev/ide/host2/bus0/target3/lun0/part33"},
	{  33, 226, "/dev/ide/host2/bus0/target3/lun0/part34"},
	{  33, 227, "/dev/ide/host2/bus0/target3/lun0/part35"},
	{  33, 228, "/dev/ide/host2/bus0/target3/lun0/part36"},
	{  33, 229, "/dev/ide/host2/bus0/target3/lun0/part37"},
	{  33, 230, "/dev/ide/host2/bus0/target3/lun0/part38"},
	{  33, 231, "/dev/ide/host2/bus0/target3/lun0/part39"},
	{  33, 232, "/dev/ide/host2/bus0/target3/lun0/part40"},
	{  33, 233, "/dev/ide/host2/bus0/target3/lun0/part41"},
	{  33, 234, "/dev/ide/host2/bus0/target3/lun0/part42"},
	{  33, 235, "/dev/ide/host2/bus0/target3/lun0/part43"},
	{  33, 236, "/dev/ide/host2/bus0/target3/lun0/part44"},
	{  33, 237, "/dev/ide/host2/bus0/target3/lun0/part45"},
	{  33, 238, "/dev/ide/host2/bus0/target3/lun0/part46"},
	{  33, 239, "/dev/ide/host2/bus0/target3/lun0/part47"},
	{  33, 240, "/dev/ide/host2/bus0/target3/lun0/part48"},
	{  33, 241, "/dev/ide/host2/bus0/target3/lun0/part49"},
	{  33, 242, "/dev/ide/host2/bus0/target3/lun0/part50"},
	{  33, 243, "/dev/ide/host2/bus0/target3/lun0/part51"},
	{  33, 244, "/dev/ide/host2/bus0/target3/lun0/part52"},
	{  33, 245, "/dev/ide/host2/bus0/target3/lun0/part53"},
	{  33, 246, "/dev/ide/host2/bus0/target3/lun0/part54"},
	{  33, 247, "/dev/ide/host2/bus0/target3/lun0/part55"},
	{  33, 248, "/dev/ide/host2/bus0/target3/lun0/part56"},
	{  33, 249, "/dev/ide/host2/bus0/target3/lun0/part57"},
	{  33, 250, "/dev/ide/host2/bus0/target3/lun0/part58"},
	{  33, 251, "/dev/ide/host2/bus0/target3/lun0/part59"},
	{  33, 252, "/dev/ide/host2/bus0/target3/lun0/part60"},
	{  33, 253, "/dev/ide/host2/bus0/target3/lun0/part61"},
	{  33, 254, "/dev/ide/host2/bus0/target3/lun0/part62"},
	{  33, 255, "/dev/ide/host2/bus0/target3/lun0/part63"},

	{  34,   0, "/dev/ide/host2/bus1/target0/lun0/disc"},
	{  34,   1, "/dev/ide/host2/bus1/target0/lun0/part1"},
	{  34,   2, "/dev/ide/host2/bus1/target0/lun0/part2"},
	{  34,   3, "/dev/ide/host2/bus1/target0/lun0/part3"},
	{  34,   4, "/dev/ide/host2/bus1/target0/lun0/part4"},
	{  34,   5, "/dev/ide/host2/bus1/target0/lun0/part5"},
	{  34,   6, "/dev/ide/host2/bus1/target0/lun0/part6"},
	{  34,   7, "/dev/ide/host2/bus1/target0/lun0/part7"},
	{  34,   8, "/dev/ide/host2/bus1/target0/lun0/part8"},
	{  34,   9, "/dev/ide/host2/bus1/target0/lun0/part9"},
	{  34,  10, "/dev/ide/host2/bus1/target0/lun0/part10"},
	{  34,  11, "/dev/ide/host2/bus1/target0/lun0/part11"},
	{  34,  12, "/dev/ide/host2/bus1/target0/lun0/part12"},
	{  34,  13, "/dev/ide/host2/bus1/target0/lun0/part13"},
	{  34,  14, "/dev/ide/host2/bus1/target0/lun0/part14"},
	{  34,  15, "/dev/ide/host2/bus1/target0/lun0/part15"},
	{  34,  16, "/dev/ide/host2/bus1/target0/lun0/part16"},
	{  34,  17, "/dev/ide/host2/bus1/target0/lun0/part17"},
	{  34,  18, "/dev/ide/host2/bus1/target0/lun0/part18"},
	{  34,  19, "/dev/ide/host2/bus1/target0/lun0/part19"},
	{  34,  20, "/dev/ide/host2/bus1/target0/lun0/part20"},
	{  34,  21, "/dev/ide/host2/bus1/target0/lun0/part21"},
	{  34,  22, "/dev/ide/host2/bus1/target0/lun0/part22"},
	{  34,  23, "/dev/ide/host2/bus1/target0/lun0/part23"},
	{  34,  24, "/dev/ide/host2/bus1/target0/lun0/part24"},
	{  34,  25, "/dev/ide/host2/bus1/target0/lun0/part25"},
	{  34,  26, "/dev/ide/host2/bus1/target0/lun0/part26"},
	{  34,  27, "/dev/ide/host2/bus1/target0/lun0/part27"},
	{  34,  28, "/dev/ide/host2/bus1/target0/lun0/part28"},
	{  34,  29, "/dev/ide/host2/bus1/target0/lun0/part29"},
	{  34,  30, "/dev/ide/host2/bus1/target0/lun0/part30"},
	{  34,  31, "/dev/ide/host2/bus1/target0/lun0/part31"},
	{  34,  32, "/dev/ide/host2/bus1/target0/lun0/part32"},
	{  34,  33, "/dev/ide/host2/bus1/target0/lun0/part33"},
	{  34,  34, "/dev/ide/host2/bus1/target0/lun0/part34"},
	{  34,  35, "/dev/ide/host2/bus1/target0/lun0/part35"},
	{  34,  36, "/dev/ide/host2/bus1/target0/lun0/part36"},
	{  34,  37, "/dev/ide/host2/bus1/target0/lun0/part37"},
	{  34,  38, "/dev/ide/host2/bus1/target0/lun0/part38"},
	{  34,  39, "/dev/ide/host2/bus1/target0/lun0/part39"},
	{  34,  40, "/dev/ide/host2/bus1/target0/lun0/part40"},
	{  34,  41, "/dev/ide/host2/bus1/target0/lun0/part41"},
	{  34,  42, "/dev/ide/host2/bus1/target0/lun0/part42"},
	{  34,  43, "/dev/ide/host2/bus1/target0/lun0/part43"},
	{  34,  44, "/dev/ide/host2/bus1/target0/lun0/part44"},
	{  34,  45, "/dev/ide/host2/bus1/target0/lun0/part45"},
	{  34,  46, "/dev/ide/host2/bus1/target0/lun0/part46"},
	{  34,  47, "/dev/ide/host2/bus1/target0/lun0/part47"},
	{  34,  48, "/dev/ide/host2/bus1/target0/lun0/part48"},
	{  34,  49, "/dev/ide/host2/bus1/target0/lun0/part49"},
	{  34,  50, "/dev/ide/host2/bus1/target0/lun0/part50"},
	{  34,  51, "/dev/ide/host2/bus1/target0/lun0/part51"},
	{  34,  52, "/dev/ide/host2/bus1/target0/lun0/part52"},
	{  34,  53, "/dev/ide/host2/bus1/target0/lun0/part53"},
	{  34,  54, "/dev/ide/host2/bus1/target0/lun0/part54"},
	{  34,  55, "/dev/ide/host2/bus1/target0/lun0/part55"},
	{  34,  56, "/dev/ide/host2/bus1/target0/lun0/part56"},
	{  34,  57, "/dev/ide/host2/bus1/target0/lun0/part57"},
	{  34,  58, "/dev/ide/host2/bus1/target0/lun0/part58"},
	{  34,  59, "/dev/ide/host2/bus1/target0/lun0/part59"},
	{  34,  60, "/dev/ide/host2/bus1/target0/lun0/part60"},
	{  34,  61, "/dev/ide/host2/bus1/target0/lun0/part61"},
	{  34,  62, "/dev/ide/host2/bus1/target0/lun0/part62"},
	{  34,  63, "/dev/ide/host2/bus1/target0/lun0/part63"},
	{  34,  64, "/dev/ide/host2/bus1/target1/lun0/disc"},
	{  34,  65, "/dev/ide/host2/bus1/target1/lun0/part1"},
	{  34,  66, "/dev/ide/host2/bus1/target1/lun0/part2"},
	{  34,  67, "/dev/ide/host2/bus1/target1/lun0/part3"},
	{  34,  68, "/dev/ide/host2/bus1/target1/lun0/part4"},
	{  34,  69, "/dev/ide/host2/bus1/target1/lun0/part5"},
	{  34,  70, "/dev/ide/host2/bus1/target1/lun0/part6"},
	{  34,  71, "/dev/ide/host2/bus1/target1/lun0/part7"},
	{  34,  72, "/dev/ide/host2/bus1/target1/lun0/part8"},
	{  34,  73, "/dev/ide/host2/bus1/target1/lun0/part9"},
	{  34,  74, "/dev/ide/host2/bus1/target1/lun0/part10"},
	{  34,  75, "/dev/ide/host2/bus1/target1/lun0/part11"},
	{  34,  76, "/dev/ide/host2/bus1/target1/lun0/part12"},
	{  34,  77, "/dev/ide/host2/bus1/target1/lun0/part13"},
	{  34,  78, "/dev/ide/host2/bus1/target1/lun0/part14"},
	{  34,  79, "/dev/ide/host2/bus1/target1/lun0/part15"},
	{  34,  80, "/dev/ide/host2/bus1/target1/lun0/part16"},
	{  34,  81, "/dev/ide/host2/bus1/target1/lun0/part17"},
	{  34,  82, "/dev/ide/host2/bus1/target1/lun0/part18"},
	{  34,  83, "/dev/ide/host2/bus1/target1/lun0/part19"},
	{  34,  84, "/dev/ide/host2/bus1/target1/lun0/part20"},
	{  34,  85, "/dev/ide/host2/bus1/target1/lun0/part21"},
	{  34,  86, "/dev/ide/host2/bus1/target1/lun0/part22"},
	{  34,  87, "/dev/ide/host2/bus1/target1/lun0/part23"},
	{  34,  88, "/dev/ide/host2/bus1/target1/lun0/part24"},
	{  34,  89, "/dev/ide/host2/bus1/target1/lun0/part25"},
	{  34,  90, "/dev/ide/host2/bus1/target1/lun0/part26"},
	{  34,  91, "/dev/ide/host2/bus1/target1/lun0/part27"},
	{  34,  92, "/dev/ide/host2/bus1/target1/lun0/part28"},
	{  34,  93, "/dev/ide/host2/bus1/target1/lun0/part29"},
	{  34,  94, "/dev/ide/host2/bus1/target1/lun0/part30"},
	{  34,  95, "/dev/ide/host2/bus1/target1/lun0/part31"},
	{  34,  96, "/dev/ide/host2/bus1/target1/lun0/part32"},
	{  34,  97, "/dev/ide/host2/bus1/target1/lun0/part33"},
	{  34,  98, "/dev/ide/host2/bus1/target1/lun0/part34"},
	{  34,  99, "/dev/ide/host2/bus1/target1/lun0/part35"},
	{  34, 100, "/dev/ide/host2/bus1/target1/lun0/part36"},
	{  34, 101, "/dev/ide/host2/bus1/target1/lun0/part37"},
	{  34, 102, "/dev/ide/host2/bus1/target1/lun0/part38"},
	{  34, 103, "/dev/ide/host2/bus1/target1/lun0/part39"},
	{  34, 104, "/dev/ide/host2/bus1/target1/lun0/part40"},
	{  34, 105, "/dev/ide/host2/bus1/target1/lun0/part41"},
	{  34, 106, "/dev/ide/host2/bus1/target1/lun0/part42"},
	{  34, 107, "/dev/ide/host2/bus1/target1/lun0/part43"},
	{  34, 108, "/dev/ide/host2/bus1/target1/lun0/part44"},
	{  34, 109, "/dev/ide/host2/bus1/target1/lun0/part45"},
	{  34, 110, "/dev/ide/host2/bus1/target1/lun0/part46"},
	{  34, 111, "/dev/ide/host2/bus1/target1/lun0/part47"},
	{  34, 112, "/dev/ide/host2/bus1/target1/lun0/part48"},
	{  34, 113, "/dev/ide/host2/bus1/target1/lun0/part49"},
	{  34, 114, "/dev/ide/host2/bus1/target1/lun0/part50"},
	{  34, 115, "/dev/ide/host2/bus1/target1/lun0/part51"},
	{  34, 116, "/dev/ide/host2/bus1/target1/lun0/part52"},
	{  34, 117, "/dev/ide/host2/bus1/target1/lun0/part53"},
	{  34, 118, "/dev/ide/host2/bus1/target1/lun0/part54"},
	{  34, 119, "/dev/ide/host2/bus1/target1/lun0/part55"},
	{  34, 120, "/dev/ide/host2/bus1/target1/lun0/part56"},
	{  34, 121, "/dev/ide/host2/bus1/target1/lun0/part57"},
	{  34, 122, "/dev/ide/host2/bus1/target1/lun0/part58"},
	{  34, 123, "/dev/ide/host2/bus1/target1/lun0/part59"},
	{  34, 124, "/dev/ide/host2/bus1/target1/lun0/part60"},
	{  34, 125, "/dev/ide/host2/bus1/target1/lun0/part61"},
	{  34, 126, "/dev/ide/host2/bus1/target1/lun0/part62"},
	{  34, 127, "/dev/ide/host2/bus1/target1/lun0/part63"},
	{  34, 128, "/dev/ide/host2/bus1/target2/lun0/disc"},
	{  34, 129, "/dev/ide/host2/bus1/target2/lun0/part1"},
	{  34, 130, "/dev/ide/host2/bus1/target2/lun0/part2"},
	{  34, 131, "/dev/ide/host2/bus1/target2/lun0/part3"},
	{  34, 132, "/dev/ide/host2/bus1/target2/lun0/part4"},
	{  34, 133, "/dev/ide/host2/bus1/target2/lun0/part5"},
	{  34, 134, "/dev/ide/host2/bus1/target2/lun0/part6"},
	{  34, 135, "/dev/ide/host2/bus1/target2/lun0/part7"},
	{  34, 136, "/dev/ide/host2/bus1/target2/lun0/part8"},
	{  34, 137, "/dev/ide/host2/bus1/target2/lun0/part9"},
	{  34, 138, "/dev/ide/host2/bus1/target2/lun0/part10"},
	{  34, 139, "/dev/ide/host2/bus1/target2/lun0/part11"},
	{  34, 140, "/dev/ide/host2/bus1/target2/lun0/part12"},
	{  34, 141, "/dev/ide/host2/bus1/target2/lun0/part13"},
	{  34, 142, "/dev/ide/host2/bus1/target2/lun0/part14"},
	{  34, 143, "/dev/ide/host2/bus1/target2/lun0/part15"},
	{  34, 144, "/dev/ide/host2/bus1/target2/lun0/part16"},
	{  34, 145, "/dev/ide/host2/bus1/target2/lun0/part17"},
	{  34, 146, "/dev/ide/host2/bus1/target2/lun0/part18"},
	{  34, 147, "/dev/ide/host2/bus1/target2/lun0/part19"},
	{  34, 148, "/dev/ide/host2/bus1/target2/lun0/part20"},
	{  34, 149, "/dev/ide/host2/bus1/target2/lun0/part21"},
	{  34, 150, "/dev/ide/host2/bus1/target2/lun0/part22"},
	{  34, 151, "/dev/ide/host2/bus1/target2/lun0/part23"},
	{  34, 152, "/dev/ide/host2/bus1/target2/lun0/part24"},
	{  34, 153, "/dev/ide/host2/bus1/target2/lun0/part25"},
	{  34, 154, "/dev/ide/host2/bus1/target2/lun0/part26"},
	{  34, 155, "/dev/ide/host2/bus1/target2/lun0/part27"},
	{  34, 156, "/dev/ide/host2/bus1/target2/lun0/part28"},
	{  34, 157, "/dev/ide/host2/bus1/target2/lun0/part29"},
	{  34, 158, "/dev/ide/host2/bus1/target2/lun0/part30"},
	{  34, 159, "/dev/ide/host2/bus1/target2/lun0/part31"},
	{  34, 160, "/dev/ide/host2/bus1/target2/lun0/part32"},
	{  34, 161, "/dev/ide/host2/bus1/target2/lun0/part33"},
	{  34, 162, "/dev/ide/host2/bus1/target2/lun0/part34"},
	{  34, 163, "/dev/ide/host2/bus1/target2/lun0/part35"},
	{  34, 164, "/dev/ide/host2/bus1/target2/lun0/part36"},
	{  34, 165, "/dev/ide/host2/bus1/target2/lun0/part37"},
	{  34, 166, "/dev/ide/host2/bus1/target2/lun0/part38"},
	{  34, 167, "/dev/ide/host2/bus1/target2/lun0/part39"},
	{  34, 168, "/dev/ide/host2/bus1/target2/lun0/part40"},
	{  34, 169, "/dev/ide/host2/bus1/target2/lun0/part41"},
	{  34, 170, "/dev/ide/host2/bus1/target2/lun0/part42"},
	{  34, 171, "/dev/ide/host2/bus1/target2/lun0/part43"},
	{  34, 172, "/dev/ide/host2/bus1/target2/lun0/part44"},
	{  34, 173, "/dev/ide/host2/bus1/target2/lun0/part45"},
	{  34, 174, "/dev/ide/host2/bus1/target2/lun0/part46"},
	{  34, 175, "/dev/ide/host2/bus1/target2/lun0/part47"},
	{  34, 176, "/dev/ide/host2/bus1/target2/lun0/part48"},
	{  34, 177, "/dev/ide/host2/bus1/target2/lun0/part49"},
	{  34, 178, "/dev/ide/host2/bus1/target2/lun0/part50"},
	{  34, 179, "/dev/ide/host2/bus1/target2/lun0/part51"},
	{  34, 180, "/dev/ide/host2/bus1/target2/lun0/part52"},
	{  34, 181, "/dev/ide/host2/bus1/target2/lun0/part53"},
	{  34, 182, "/dev/ide/host2/bus1/target2/lun0/part54"},
	{  34, 183, "/dev/ide/host2/bus1/target2/lun0/part55"},
	{  34, 184, "/dev/ide/host2/bus1/target2/lun0/part56"},
	{  34, 185, "/dev/ide/host2/bus1/target2/lun0/part57"},
	{  34, 186, "/dev/ide/host2/bus1/target2/lun0/part58"},
	{  34, 187, "/dev/ide/host2/bus1/target2/lun0/part59"},
	{  34, 188, "/dev/ide/host2/bus1/target2/lun0/part60"},
	{  34, 189, "/dev/ide/host2/bus1/target2/lun0/part61"},
	{  34, 190, "/dev/ide/host2/bus1/target2/lun0/part62"},
	{  34, 191, "/dev/ide/host2/bus1/target2/lun0/part63"},
	{  34, 192, "/dev/ide/host2/bus1/target3/lun0/disc"},
	{  34, 193, "/dev/ide/host2/bus1/target3/lun0/part1"},
	{  34, 194, "/dev/ide/host2/bus1/target3/lun0/part2"},
	{  34, 195, "/dev/ide/host2/bus1/target3/lun0/part3"},
	{  34, 196, "/dev/ide/host2/bus1/target3/lun0/part4"},
	{  34, 197, "/dev/ide/host2/bus1/target3/lun0/part5"},
	{  34, 198, "/dev/ide/host2/bus1/target3/lun0/part6"},
	{  34, 199, "/dev/ide/host2/bus1/target3/lun0/part7"},
	{  34, 200, "/dev/ide/host2/bus1/target3/lun0/part8"},
	{  34, 201, "/dev/ide/host2/bus1/target3/lun0/part9"},
	{  34, 202, "/dev/ide/host2/bus1/target3/lun0/part10"},
	{  34, 203, "/dev/ide/host2/bus1/target3/lun0/part11"},
	{  34, 204, "/dev/ide/host2/bus1/target3/lun0/part12"},
	{  34, 205, "/dev/ide/host2/bus1/target3/lun0/part13"},
	{  34, 206, "/dev/ide/host2/bus1/target3/lun0/part14"},
	{  34, 207, "/dev/ide/host2/bus1/target3/lun0/part15"},
	{  34, 208, "/dev/ide/host2/bus1/target3/lun0/part16"},
	{  34, 209, "/dev/ide/host2/bus1/target3/lun0/part17"},
	{  34, 210, "/dev/ide/host2/bus1/target3/lun0/part18"},
	{  34, 211, "/dev/ide/host2/bus1/target3/lun0/part19"},
	{  34, 212, "/dev/ide/host2/bus1/target3/lun0/part20"},
	{  34, 213, "/dev/ide/host2/bus1/target3/lun0/part21"},
	{  34, 214, "/dev/ide/host2/bus1/target3/lun0/part22"},
	{  34, 215, "/dev/ide/host2/bus1/target3/lun0/part23"},
	{  34, 216, "/dev/ide/host2/bus1/target3/lun0/part24"},
	{  34, 217, "/dev/ide/host2/bus1/target3/lun0/part25"},
	{  34, 218, "/dev/ide/host2/bus1/target3/lun0/part26"},
	{  34, 219, "/dev/ide/host2/bus1/target3/lun0/part27"},
	{  34, 220, "/dev/ide/host2/bus1/target3/lun0/part28"},
	{  34, 221, "/dev/ide/host2/bus1/target3/lun0/part29"},
	{  34, 222, "/dev/ide/host2/bus1/target3/lun0/part30"},
	{  34, 223, "/dev/ide/host2/bus1/target3/lun0/part31"},
	{  34, 224, "/dev/ide/host2/bus1/target3/lun0/part32"},
	{  34, 225, "/dev/ide/host2/bus1/target3/lun0/part33"},
	{  34, 226, "/dev/ide/host2/bus1/target3/lun0/part34"},
	{  34, 227, "/dev/ide/host2/bus1/target3/lun0/part35"},
	{  34, 228, "/dev/ide/host2/bus1/target3/lun0/part36"},
	{  34, 229, "/dev/ide/host2/bus1/target3/lun0/part37"},
	{  34, 230, "/dev/ide/host2/bus1/target3/lun0/part38"},
	{  34, 231, "/dev/ide/host2/bus1/target3/lun0/part39"},
	{  34, 232, "/dev/ide/host2/bus1/target3/lun0/part40"},
	{  34, 233, "/dev/ide/host2/bus1/target3/lun0/part41"},
	{  34, 234, "/dev/ide/host2/bus1/target3/lun0/part42"},
	{  34, 235, "/dev/ide/host2/bus1/target3/lun0/part43"},
	{  34, 236, "/dev/ide/host2/bus1/target3/lun0/part44"},
	{  34, 237, "/dev/ide/host2/bus1/target3/lun0/part45"},
	{  34, 238, "/dev/ide/host2/bus1/target3/lun0/part46"},
	{  34, 239, "/dev/ide/host2/bus1/target3/lun0/part47"},
	{  34, 240, "/dev/ide/host2/bus1/target3/lun0/part48"},
	{  34, 241, "/dev/ide/host2/bus1/target3/lun0/part49"},
	{  34, 242, "/dev/ide/host2/bus1/target3/lun0/part50"},
	{  34, 243, "/dev/ide/host2/bus1/target3/lun0/part51"},
	{  34, 244, "/dev/ide/host2/bus1/target3/lun0/part52"},
	{  34, 245, "/dev/ide/host2/bus1/target3/lun0/part53"},
	{  34, 246, "/dev/ide/host2/bus1/target3/lun0/part54"},
	{  34, 247, "/dev/ide/host2/bus1/target3/lun0/part55"},
	{  34, 248, "/dev/ide/host2/bus1/target3/lun0/part56"},
	{  34, 249, "/dev/ide/host2/bus1/target3/lun0/part57"},
	{  34, 250, "/dev/ide/host2/bus1/target3/lun0/part58"},
	{  34, 251, "/dev/ide/host2/bus1/target3/lun0/part59"},
	{  34, 252, "/dev/ide/host2/bus1/target3/lun0/part60"},
	{  34, 253, "/dev/ide/host2/bus1/target3/lun0/part61"},
	{  34, 254, "/dev/ide/host2/bus1/target3/lun0/part62"},
	{  34, 255, "/dev/ide/host2/bus1/target3/lun0/part63"},

	{   9,   0, "/dev/md/0"},
	{   9,   1, "/dev/md/1"},
	{   9,   2, "/dev/md/2"},
	{   9,   3, "/dev/md/3"},
	{   9,   4, "/dev/md/4"},
	{   9,   5, "/dev/md/5"},
	{   9,   6, "/dev/md/6"},
	{   9,   7, "/dev/md/7"},
	{   9,   8, "/dev/md/8"},
	{   9,   9, "/dev/md/9"},
	{   9,  10, "/dev/md/10"},
	{   9,  11, "/dev/md/11"},
	{   9,  12, "/dev/md/12"},
	{   9,  13, "/dev/md/13"},
	{   9,  14, "/dev/md/14"},
	{   9,  15, "/dev/md/15"},
	{   9,  16, "/dev/md/16"},
	{   9,  17, "/dev/md/17"},
	{   9,  18, "/dev/md/18"},
	{   9,  19, "/dev/md/19"},
	{   9,  20, "/dev/md/20"},
	{   9,  21, "/dev/md/21"},
	{   9,  22, "/dev/md/22"},
	{   9,  23, "/dev/md/23"},
	{   9,  24, "/dev/md/24"},
	{   9,  25, "/dev/md/25"},
	{   9,  26, "/dev/md/26"},
	{   9,  27, "/dev/md/27"},
	{   9,  28, "/dev/md/28"},
	{   9,  29, "/dev/md/29"},
	{   9,  30, "/dev/md/30"},
	{   9,  31, "/dev/md/31"},
	{   9,  32, "/dev/md/32"},
	{   9,  33, "/dev/md/33"},
	{   9,  34, "/dev/md/34"},
	{   9,  35, "/dev/md/35"},
	{   9,  36, "/dev/md/36"},
	{   9,  37, "/dev/md/37"},
	{   9,  38, "/dev/md/38"},
	{   9,  39, "/dev/md/39"},
	{   9,  40, "/dev/md/40"},
	{   9,  41, "/dev/md/41"},
	{   9,  42, "/dev/md/42"},
	{   9,  43, "/dev/md/43"},
	{   9,  44, "/dev/md/44"},
	{   9,  45, "/dev/md/45"},
	{   9,  46, "/dev/md/46"},
	{   9,  47, "/dev/md/47"},
	{   9,  48, "/dev/md/48"},
	{   9,  49, "/dev/md/49"},
	{   9,  50, "/dev/md/50"},
	{   9,  51, "/dev/md/51"},
	{   9,  52, "/dev/md/52"},
	{   9,  53, "/dev/md/53"},
	{   9,  54, "/dev/md/54"},
	{   9,  55, "/dev/md/55"},
	{   9,  56, "/dev/md/56"},
	{   9,  57, "/dev/md/57"},
	{   9,  58, "/dev/md/58"},
	{   9,  59, "/dev/md/59"},
	{   9,  60, "/dev/md/60"},
	{   9,  61, "/dev/md/61"},
	{   9,  62, "/dev/md/62"},
	{   9,  63, "/dev/md/63"},
	{   9,  64, "/dev/md/64"},
	{   9,  65, "/dev/md/65"},
	{   9,  66, "/dev/md/66"},
	{   9,  67, "/dev/md/67"},
	{   9,  68, "/dev/md/68"},
	{   9,  69, "/dev/md/69"},
	{   9,  70, "/dev/md/70"},
	{   9,  71, "/dev/md/71"},
	{   9,  72, "/dev/md/72"},
	{   9,  73, "/dev/md/73"},
	{   9,  74, "/dev/md/74"},
	{   9,  75, "/dev/md/75"},
	{   9,  76, "/dev/md/76"},
	{   9,  77, "/dev/md/77"},
	{   9,  78, "/dev/md/78"},
	{   9,  79, "/dev/md/79"},
	{   9,  80, "/dev/md/80"},
	{   9,  81, "/dev/md/81"},
	{   9,  82, "/dev/md/82"},
	{   9,  83, "/dev/md/83"},
	{   9,  84, "/dev/md/84"},
	{   9,  85, "/dev/md/85"},
	{   9,  86, "/dev/md/86"},
	{   9,  87, "/dev/md/87"},
	{   9,  88, "/dev/md/88"},
	{   9,  89, "/dev/md/89"},
	{   9,  90, "/dev/md/90"},
	{   9,  91, "/dev/md/91"},
	{   9,  92, "/dev/md/92"},
	{   9,  93, "/dev/md/93"},
	{   9,  94, "/dev/md/94"},
	{   9,  95, "/dev/md/95"},
	{   9,  96, "/dev/md/96"},
	{   9,  97, "/dev/md/97"},
	{   9,  98, "/dev/md/98"},
	{   9,  99, "/dev/md/99"},
	{   9, 100, "/dev/md/100"},
	{   9, 101, "/dev/md/101"},
	{   9, 102, "/dev/md/102"},
	{   9, 103, "/dev/md/103"},
	{   9, 104, "/dev/md/104"},
	{   9, 105, "/dev/md/105"},
	{   9, 106, "/dev/md/106"},
	{   9, 107, "/dev/md/107"},
	{   9, 108, "/dev/md/108"},
	{   9, 109, "/dev/md/109"},
	{   9, 110, "/dev/md/110"},
	{   9, 111, "/dev/md/111"},
	{   9, 112, "/dev/md/112"},
	{   9, 113, "/dev/md/113"},
	{   9, 114, "/dev/md/114"},
	{   9, 115, "/dev/md/115"},
	{   9, 116, "/dev/md/116"},
	{   9, 117, "/dev/md/117"},
	{   9, 118, "/dev/md/118"},
	{   9, 119, "/dev/md/119"},
	{   9, 120, "/dev/md/120"},
	{   9, 121, "/dev/md/121"},
	{   9, 122, "/dev/md/122"},
	{   9, 123, "/dev/md/123"},
	{   9, 124, "/dev/md/124"},
	{   9, 125, "/dev/md/125"},
	{   9, 126, "/dev/md/126"},
	{   9, 127, "/dev/md/127"},
	{   9, 128, "/dev/md/128"},
	{   9, 129, "/dev/md/129"},
	{   9, 130, "/dev/md/130"},
	{   9, 131, "/dev/md/131"},
	{   9, 132, "/dev/md/132"},
	{   9, 133, "/dev/md/133"},
	{   9, 134, "/dev/md/134"},
	{   9, 135, "/dev/md/135"},
	{   9, 136, "/dev/md/136"},
	{   9, 137, "/dev/md/137"},
	{   9, 138, "/dev/md/138"},
	{   9, 139, "/dev/md/139"},
	{   9, 140, "/dev/md/140"},
	{   9, 141, "/dev/md/141"},
	{   9, 142, "/dev/md/142"},
	{   9, 143, "/dev/md/143"},
	{   9, 144, "/dev/md/144"},
	{   9, 145, "/dev/md/145"},
	{   9, 146, "/dev/md/146"},
	{   9, 147, "/dev/md/147"},
	{   9, 148, "/dev/md/148"},
	{   9, 149, "/dev/md/149"},
	{   9, 150, "/dev/md/150"},
	{   9, 151, "/dev/md/151"},
	{   9, 152, "/dev/md/152"},
	{   9, 153, "/dev/md/153"},
	{   9, 154, "/dev/md/154"},
	{   9, 155, "/dev/md/155"},
	{   9, 156, "/dev/md/156"},
	{   9, 157, "/dev/md/157"},
	{   9, 158, "/dev/md/158"},
	{   9, 159, "/dev/md/159"},
	{   9, 160, "/dev/md/160"},
	{   9, 161, "/dev/md/161"},
	{   9, 162, "/dev/md/162"},
	{   9, 163, "/dev/md/163"},
	{   9, 164, "/dev/md/164"},
	{   9, 165, "/dev/md/165"},
	{   9, 166, "/dev/md/166"},
	{   9, 167, "/dev/md/167"},
	{   9, 168, "/dev/md/168"},
	{   9, 169, "/dev/md/169"},
	{   9, 170, "/dev/md/170"},
	{   9, 171, "/dev/md/171"},
	{   9, 172, "/dev/md/172"},
	{   9, 173, "/dev/md/173"},
	{   9, 174, "/dev/md/174"},
	{   9, 175, "/dev/md/175"},
	{   9, 176, "/dev/md/176"},
	{   9, 177, "/dev/md/177"},
	{   9, 178, "/dev/md/178"},
	{   9, 179, "/dev/md/179"},
	{   9, 180, "/dev/md/180"},
	{   9, 181, "/dev/md/181"},
	{   9, 182, "/dev/md/182"},
	{   9, 183, "/dev/md/183"},
	{   9, 184, "/dev/md/184"},
	{   9, 185, "/dev/md/185"},
	{   9, 186, "/dev/md/186"},
	{   9, 187, "/dev/md/187"},
	{   9, 188, "/dev/md/188"},
	{   9, 189, "/dev/md/189"},
	{   9, 190, "/dev/md/190"},
	{   9, 191, "/dev/md/191"},
	{   9, 192, "/dev/md/192"},
	{   9, 193, "/dev/md/193"},
	{   9, 194, "/dev/md/194"},
	{   9, 195, "/dev/md/195"},
	{   9, 196, "/dev/md/196"},
	{   9, 197, "/dev/md/197"},
	{   9, 198, "/dev/md/198"},
	{   9, 199, "/dev/md/199"},
	{   9, 200, "/dev/md/200"},
	{   9, 201, "/dev/md/201"},
	{   9, 202, "/dev/md/202"},
	{   9, 203, "/dev/md/203"},
	{   9, 204, "/dev/md/204"},
	{   9, 205, "/dev/md/205"},
	{   9, 206, "/dev/md/206"},
	{   9, 207, "/dev/md/207"},
	{   9, 208, "/dev/md/208"},
	{   9, 209, "/dev/md/209"},
	{   9, 210, "/dev/md/210"},
	{   9, 211, "/dev/md/211"},
	{   9, 212, "/dev/md/212"},
	{   9, 213, "/dev/md/213"},
	{   9, 214, "/dev/md/214"},
	{   9, 215, "/dev/md/215"},
	{   9, 216, "/dev/md/216"},
	{   9, 217, "/dev/md/217"},
	{   9, 218, "/dev/md/218"},
	{   9, 219, "/dev/md/219"},
	{   9, 220, "/dev/md/220"},
	{   9, 221, "/dev/md/221"},
	{   9, 222, "/dev/md/222"},
	{   9, 223, "/dev/md/223"},
	{   9, 224, "/dev/md/224"},
	{   9, 225, "/dev/md/225"},
	{   9, 226, "/dev/md/226"},
	{   9, 227, "/dev/md/227"},
	{   9, 228, "/dev/md/228"},
	{   9, 229, "/dev/md/229"},
	{   9, 230, "/dev/md/230"},
	{   9, 231, "/dev/md/231"},
	{   9, 232, "/dev/md/232"},
	{   9, 233, "/dev/md/233"},
	{   9, 234, "/dev/md/234"},
	{   9, 235, "/dev/md/235"},
	{   9, 236, "/dev/md/236"},
	{   9, 237, "/dev/md/237"},
	{   9, 238, "/dev/md/238"},
	{   9, 239, "/dev/md/239"},
	{   9, 240, "/dev/md/240"},
	{   9, 241, "/dev/md/241"},
	{   9, 242, "/dev/md/242"},
	{   9, 243, "/dev/md/243"},
	{   9, 244, "/dev/md/244"},
	{   9, 245, "/dev/md/245"},
	{   9, 246, "/dev/md/246"},
	{   9, 247, "/dev/md/247"},
	{   9, 248, "/dev/md/248"},
	{   9, 249, "/dev/md/249"},
	{   9, 250, "/dev/md/250"},
	{   9, 251, "/dev/md/251"},
	{   9, 252, "/dev/md/252"},
	{   9, 253, "/dev/md/253"},
	{   9, 254, "/dev/md/254"},
	{   9, 255, "/dev/md/255"},

	{ 0, 0, "/dev/null"}
    };
    int table_index;

    for (table_index = 0; 1; ++table_index)
    {
	if (strcmp(file_name, path_table[table_index].name) == 0)
	{
	    buf->st_rdev = makedev(path_table[table_index].major,
				    path_table[table_index].minor);
	    return 0;
	}
	if ((path_table[table_index].major == 0) &&
	    (path_table[table_index].minor == 0))
	{
	    return -1;
	}
    }
}
