/*
 * linux/drivers/i2c/bcm2wire-bit.c
 *
 * Copyright 2001-2004, Broadcom Corporation
 * All Rights Reserved.
 *
 * THIS SOFTWARE IS OFFERED "AS IS", AND BROADCOM GRANTS NO WARRANTIES OF ANY
 * KIND, EXPRESS OR IMPLIED, BY STATUTE, COMMUNICATION OR OTHERWISE. BROADCOM
 * SPECIFICALLY DISCLAIMS ANY IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
 * FOR A SPECIFIC PURPOSE OR NONINFRINGEMENT CONCERNING THIS SOFTWARE.
 *
 *  On BCM947xx systems, the following 2-wire serial interface devices 
 *  are on the bus:
 *	- DS1337 real time clock
 *	- LM63 temperature sensor
 *
 * NOTE: 
 * This uses Linux I2C routines, as the 2-wire serial interface used 
 * on BCM94780 is very similar to I2C.
 *
 * $Id$
*/

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/i2c.h>
#include <linux/i2c-algo-bit.h>

#include <asm/io.h>
#include <asm/system.h>
#include <asm/time.h>

#include <typedefs.h>
#include <osl.h>
#include <bcmendian.h>
#include <bcmutils.h>
#include <bcmboard.h>
#include <serbus.h>

/* ------------------------------------------------------------------------- */
/* Local Variables 															 */
/* ------------------------------------------------------------------------- */
static struct gpio_bb_hwinfo gpio2WireHwInfo = 
			{ GPIO_BB_2WIRE, GPIO_2WIRE_SCL, GPIO_2WIRE_SDA, 
				GPIO_INVALID_PIN, GPIO_INVALID_PIN, GPIO_INVALID_PIN };

extern void(*bcmSetSDAdirIn)(void);
extern void(*bcmSetSDAdirOut)(void);


/* ------------------------------------------------------------------------- */
/* Functions																	 */
/* ------------------------------------------------------------------------- */
static int bcm2wire_bit_reg(struct i2c_client *client)
{
	return 0;
}

static int bcm2wire_bit_unreg(struct i2c_client *client)
{
	return 0;
}

static void bcm2wire_bit_inc_use(struct i2c_adapter *adap)
{
#ifdef MODULE
	MOD_INC_USE_COUNT;
#endif
}

static void bcm2wire_bit_dec_use(struct i2c_adapter *adap)
{
#ifdef MODULE
	MOD_DEC_USE_COUNT;
#endif
}


#define SER_HW_B_BCM	0x20	/* BCM bit-wiggling	*/

static struct i2c_algo_bit_data bcm2wire_data = {
	(void *)&gpio2WireHwInfo,
	bcm_serbus_setsda,
	bcm_serbus_setscl,
	bcm_serbus_getsda,
	NULL,	/* bcm_serbus_getscl. There is no check required on BCM94780
				to confirm if SCL is set, hence this is NULL. */
	10, 10, 100,		/*	waits, timeout */
};

static struct i2c_adapter bcm2wire_bit_ops = {
	name:			    "Broadcom 2-wire serial bus",
	id:			        SER_HW_B_BCM,
	algo_data:		    &bcm2wire_data,
	inc_use:			bcm2wire_bit_inc_use,
	dec_use:			bcm2wire_bit_dec_use,
	client_register:	bcm2wire_bit_reg,
	client_unregister:	bcm2wire_bit_unreg
};

int __init bcm2wire_init(void)
{
	int gpio_pin;

	/* If SCL and SDA GPIO pins are set by NVRAM, overwrite defaults. */

	gpio_pin = getvartoint(NULL, "sda_gpio");
	if((gpio_pin >= 0) && (gpio_pin < BCM_MAX_GPIO)) {
		/* GPIO pin used for SDA is set by NVRAM */
		gpio2WireHwInfo.sdaPinNum = (unsigned int)gpio_pin;
	}

	gpio_pin = getvartoint(NULL, "scl_gpio");
	if((gpio_pin >= 0) && (gpio_pin < BCM_MAX_GPIO)) {
		/* GPIO pin used for SCL is set by NVRAM */
		gpio2WireHwInfo.sclPinNum = (unsigned int)gpio_pin;
	}

	if (bcm_serbus_init(&gpio2WireHwInfo) != 0) {
        return -ENODEV;
	}

	bcmSetSDAdirIn = bcm_serbus_setsdadirin;
	bcmSetSDAdirOut = bcm_serbus_setsdadirout;

	return i2c_bit_add_bus(&bcm2wire_bit_ops);
}

#ifdef MODULE
int init_module(void) 
{
	return bcm2wire_init();
}

void cleanup_module(void) 
{
	bcmSetSDAdirIn = NULL;
	bcmSetSDAdirOut = NULL;

	i2c_bit_del_bus(&bcm2wire_bit_ops);
}

#endif

