/*
    LM63.c - Device driver for LM63 temperature sensor.
             The code in this module is modeled from other
             sensor drivers in the lm_sensors package.

    ---------

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/


#include <linux/version.h>
#include <linux/module.h>
#include <linux/slab.h>
#include <linux/proc_fs.h>
#include <linux/ioport.h>
#include <linux/sysctl.h>
#include <asm-mips/errno.h>
#include <asm-mips/io.h>
#include <linux/types.h>
#include <linux/i2c.h>
#include <linux/init.h>
#include <linux/delay.h>
#include <linux/i2c-proc.h>
#include <serbus.h>
#include <bcmnvram.h>


#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,2,18)) || \
    (LINUX_VERSION_CODE == KERNEL_VERSION(2,3,0))
#define init_MUTEX(s) do { *(s) = MUTEX; } while(0)
#endif

#ifndef THIS_MODULE
#define THIS_MODULE NULL
#endif

#ifndef I2C_DRIVERID_LM63   /* should live in i2c-id.h */
#define I2C_DRIVERID_LM63                1999
#endif

#ifndef LM_DATE
#define LM_DATE "20020915"
#endif

#ifndef LM_VERSION
#define LM_VERSION "2.6.5"
#endif

// #define BCMDBG 1
#if BCMDBG
#define	LM63_TRACE(s)	printk s
#else
#define	LM63_TRACE(s)
#endif
#define ASSERT(x)	((x) ? (void)0 : printk("LM63 assert failed - line = %d\n", __LINE__ ))


/* Addresses to scan */
static unsigned short normal_i2c[] = { 0x4c };
static unsigned short normal_i2c_range[] = { SENSORS_I2C_END };
static unsigned int normal_isa[] = { SENSORS_ISA_END };
static unsigned int normal_isa_range[] = { SENSORS_ISA_END };

/* Insmod parameters */
SENSORS_INSMOD_1(lm63);


/*
 * LM63 register definition
 */

#define LM63_REG_MAN_ID                         0xFE 
#define LM63_REG_DIE_REV                        0xFF

#define LM63_REG_CONFIG                         0x03
#define LM63_REG_STATUS                         0x02
#define LM63_REG_MASK                           0x16

#define LM63_REG_LOCAL_TEMP                     0x00
#define LM63_REG_LOCAL_TEMP_HI_SETPOINT         0x05
#define LM63_REG_REMOTE_TEMP_MSB                0x01
#define LM63_REG_REMOTE_TEMP_LSB                0x10
#define LM63_REG_REMOTE_TEMP_OFFSET_MSB         0x11
#define LM63_REG_REMOTE_TEMP_OFFSET_LSB         0x12
#define LM63_REG_REMOTE_TEMP_HI_SETPOINT_MSB    0x07
#define LM63_REG_REMOTE_TEMP_HI_SETPOINT_LSB    0x13
#define LM63_REG_REMOTE_TEMP_HI_HYST            0x4F
#define LM63_REG_REMOTE_TEMP_LOW_SETPOINT_MSB   0x08
#define LM63_REG_REMOTE_TEMP_LOW_SETPOINT_LSB   0x14
#define LM63_REG_REMOTE_TEMP_TCRIT_SETPOINT     0x19
#define LM63_REG_REMOTE_TEMP_TCRIT_HYST         0x21

#define LM63_REG_REMOTE_TEMP_PWM_LUT            0x50 /* 8 temp & pwm pairs */

#define LM63_REG_FAN_PWMRPM_CONFIG              0x4A
#define LM63_REG_FAN_SPINUP_CONFIG              0x4B
#define LM63_REG_FAN_PWM_FREQ                   0x4D
#define LM63_REG_FAN_PWM_VALUE                  0x4C


/*
 * LM63_REG_STATUS bits
 */
#define STATUS_BUSY 							0x80
#define STATUS_LOCAL_HIGH_ALARM 				0x40
#define STATUS_UNUSED_B5 						0x20
#define STATUS_REMOTE_HIGH_ALARM 				0x10
#define STATUS_REMOTE_LOW_ALARM 				0x08
#define STATUS_REMOTE_DIODE_FAULT_ALARM 		0x04
#define STATUS_REMOTE_TCRIT_ALARM 				0x02
#define STATUS_TACH_ALARM 						0x01

/*
 * LM63_REG_MASK bits
 */
#define MASK_UNUSED_B7 							0x80
#define MASK_LOCAL_HIGH_ALARM 					0x40
#define MASK_UNUSED_B5 							0x20
#define MASK_REMOTE_HIGH_ALARM 					0x10
#define MASK_REMOTE_LOW_ALARM 					0x08
#define MASK_UNUSED_B2 							0x04
#define MASK_REMOTE_TCRIT_ALARM 				0x02
#define MASK_TACH_ALARM 						0x01

/*
 * LM63_REG_CONFIG bits
 */
#define CONFIG_ALERT_MASK 0x80



/* -------- */

#if BCMDBG

typedef struct  
{
	u8 			reg_addr;
	const char 	*reg_name;
} reg_name_table_entry;

static reg_name_table_entry reg_name_table[] =
{
	{LM63_REG_MAN_ID, "LM63_REG_MAN_ID"},
	{LM63_REG_DIE_REV, "LM63_REG_DIE_REV"},
	{LM63_REG_CONFIG, "LM63_REG_CONFIG"},
	{LM63_REG_STATUS, "LM63_REG_STATUS"},
	{LM63_REG_MASK, "LM63_REG_MASK"},
	{LM63_REG_LOCAL_TEMP, "LM63_REG_LOCAL_TEMP"},
	{LM63_REG_LOCAL_TEMP_HI_SETPOINT, "LM63_REG_LOCAL_TEMP_HI_SETPOINT"},
	{LM63_REG_REMOTE_TEMP_MSB, "LM63_REG_REMOTE_TEMP_MSB"},
	{LM63_REG_REMOTE_TEMP_LSB, "LM63_REG_REMOTE_TEMP_LSB"},
	{LM63_REG_REMOTE_TEMP_OFFSET_MSB, "LM63_REG_REMOTE_TEMP_OFFSET_MSB"},
	{LM63_REG_REMOTE_TEMP_OFFSET_LSB, "LM63_REG_REMOTE_TEMP_OFFSET_LSB"},
	{LM63_REG_REMOTE_TEMP_HI_SETPOINT_MSB, "LM63_REG_REMOTE_TEMP_HI_SETPOINT_MSB"},
	{LM63_REG_REMOTE_TEMP_HI_SETPOINT_LSB, "LM63_REG_REMOTE_TEMP_HI_SETPOINT_LSB"},
	{LM63_REG_REMOTE_TEMP_HI_HYST, "LM63_REG_REMOTE_TEMP_HI_HYST"},
	{LM63_REG_REMOTE_TEMP_LOW_SETPOINT_MSB, "LM63_REG_REMOTE_TEMP_LOW_SETPOINT_MSB"},
	{LM63_REG_REMOTE_TEMP_LOW_SETPOINT_LSB, "LM63_REG_REMOTE_TEMP_LOW_SETPOINT_LSB"},
	{LM63_REG_REMOTE_TEMP_TCRIT_SETPOINT, "LM63_REG_REMOTE_TEMP_TCRIT_SETPOINT"},
	{LM63_REG_REMOTE_TEMP_TCRIT_HYST, "LM63_REG_REMOTE_TEMP_TCRIT_HYST"},
	{LM63_REG_REMOTE_TEMP_PWM_LUT, "LM63_REG_REMOTE_TEMP_PWM_LUT"},
	{LM63_REG_FAN_PWMRPM_CONFIG, "LM63_REG_FAN_PWMRPM_CONFIG"},
	{LM63_REG_FAN_SPINUP_CONFIG, "LM63_REG_FAN_SPINUP_CONFIG"},
	{LM63_REG_FAN_PWM_FREQ, "LM63_REG_FAN_PWM_FREQ"},
	{LM63_REG_FAN_PWM_VALUE, "LM63_REG_FAN_PWM_VALUE"}
};

static const char* unknown = "unknown";

static const char* lm63_get_reg_name(u8 reg_addr);

static const char* lm63_get_reg_name(u8 reg_addr)
{
	int i;

	for (i=0; i < sizeof(reg_name_table); i++)
	{
		if (reg_name_table[i].reg_addr == reg_addr)
			return reg_name_table[i].reg_name;
	}
	return unknown;
}

#endif

/* -------- */



/*
 * Conversions and various macros
 * The LM63 uses signed 8-bit values for the local temperatures,
 * and signed 11-bit values for the remote temperatures (except
 * T_CRIT). The 11-bit conversion formulas may not round negative
 * numbers perfectly, but who cares?
 */

#define TEMP1_FROM_REG(val)  (val & 0x80 ? val-0x100 : val)
#define TEMP1_TO_REG(val)    (val < 0 ? val+0x100 : val)
#define TEMP2_FROM_REG(val)  (((val & 0x8000 ? val-0x10000 : val) \
                             * 10 + 128) >> 8)
#define TEMP2_TO_REG(val)    (((val << 8) / 10 + (val < 0 ? \
                             0x10000 : 0)) & 0xFFE0)
#define DUTYCYCLE_TO_PWM(dc, freq) \
							 ((dc * 2 * freq) / 100)
#define PWM_TO_DUTYCYCLE(pwm, freq) \
							 (freq == 0 ? 0 : ((pwm * 100) / (2 * freq)))

// Macro to get a random integer with a specified range 
#define getrandom(min, max) \
    ((rand()%(int)(((max) + 1)-(min)))+ (min))


#ifdef MODULE
extern int init_module(void);
extern int cleanup_module(void);
#endif				/* MODULE */

/* For each registered LM63, we need to keep some data in memory. The 
   structure itself is dynamically allocated, at the same time when 
   a new LM63 client is allocated. */
struct lm63_data {
	int sysctl_id;
	enum chips type;

	struct semaphore update_lock;
	char valid;		/* !=0 if following fields are valid */
	unsigned long last_updated;	/* In jiffies */

	u8  raw_read_register;   /* for debugging */

	u8  man_id;
	u8  die_rev;

	u8  config;
	u8  status;
	u8  mask;
	
	u8  local_temp;
	u8  local_temp_hi_setpoint;
	u16 remote_temp;
	u16 remote_temp_offset;
	u16 remote_temp_hi_setpoint;
	u16 remote_temp_low_setpoint;
	u8  remote_temp_hi_hyst;
	u8  remote_temp_tcrit_setpoint;
	u8  remote_temp_tcrit_hyst;

	u8  remote_temp_pwm_lut_temp[8];
	u8  remote_temp_pwm_lut_pwm[8];

	u8  fan_pwmrpm_config;
	u8  fan_spinup_config;
	u8  fan_pwm_freq;
	u8  fan_pwm_value;
};

#ifdef MODULE
static
#else
extern
#endif
int __init sensors_lm63_init(void);
static int __init lm63_cleanup(void);

static int lm63_attach_adapter(struct i2c_adapter *adapter);
static int lm63_detect(struct i2c_adapter *adapter, int address,
			  unsigned short flags, int kind);
static int lm63_detach_client(struct i2c_client *client);
static int lm63_command(struct i2c_client *client, unsigned int cmd,
			   void *arg);
static void lm63_inc_use(struct i2c_client *client);
static void lm63_dec_use(struct i2c_client *client);

static int lm63_read_value(struct i2c_client *client, u8 register);
static int lm63_write_value(struct i2c_client *client, u8 register,
			       u8 value);
static void lm63_update_client(struct i2c_client *client);
static void lm63_init_client(struct i2c_client *client);

/* Register access functions */
static void lm63_config(struct i2c_client *client, int operation,
			 int ctl_name, int *nrels_mag, long *results);
static void lm63_status(struct i2c_client *client, int operation,
			 int ctl_name, int *nrels_mag, long *results);
static void lm63_mask(struct i2c_client *client, int operation,
			 int ctl_name, int *nrels_mag, long *results);

static void lm63_use_remote_temp(struct i2c_client *client, int operation,
			 int ctl_name, int *nrels_mag, long *results);
static void lm63_temperature(struct i2c_client *client, int operation,
			 int ctl_name, int *nrels_mag, long *results);
static void lm63_local_temp(struct i2c_client *client, int operation,
			 int ctl_name, int *nrels_mag, long *results);
static void lm63_remote_temp(struct i2c_client *client, int operation,
			 int ctl_name, int *nrels_mag, long *results);
static void lm63_remote_temp_offset(struct i2c_client *client, int operation,
			 int ctl_name, int *nrels_mag, long *results);
static void lm63_remote_temp_tcrit_setpoint(struct i2c_client *client, int operation,
			 int ctl_name, int *nrels_mag, long *results);

static void lm63_remote_temp_pwm_lut(struct i2c_client *client, int operation,
			 int ctl_name, int *nrels_mag, long *results);

static void lm63_fan_pwmrpm_config(struct i2c_client *client, int operation,
			 int ctl_name, int *nrels_mag, long *results);
static void lm63_fan_spinup_config(struct i2c_client *client, int operation,
			 int ctl_name, int *nrels_mag, long *results);
static void lm63_fan_pwm_freq(struct i2c_client *client, int operation,
			 int ctl_name, int *nrels_mag, long *results);
static void lm63_fan_pwm_value(struct i2c_client *client, int operation,
			 int ctl_name, int *nrels_mag, long *results);
static void lm63_fan_duty_cycle(struct i2c_client *client, int operation,
			 int ctl_name, int *nrels_mag, long *results);

static void lm63_raw_read(struct i2c_client *client, int operation,
			 int ctl_name, int *nrels_mag, long *results);
static void lm63_raw_write(struct i2c_client *client, int operation,
			 int ctl_name, int *nrels_mag, long *results);


/* I choose here for semi-static LM63 allocation. Complete dynamic
   allocation could also be used; the code needed for this would probably
   take more memory than the datastructure takes now. */
static int lm63_id = 0;

static struct i2c_driver LM63_driver = {
	/* name */          "LM63 sensor driver",
	/* id */             I2C_DRIVERID_LM63,
	/* flags */          I2C_DF_NOTIFY,
	/* attach_adapter */ &lm63_attach_adapter,
	/* detach_client */  &lm63_detach_client,
	/* command */        &lm63_command,
	/* inc_use */        &lm63_inc_use,
	/* dec_use */        &lm63_dec_use
};

/* Used by LM63_init/cleanup */
static int __initdata lm63_initialized = 0;

/* 
	Which temperature (local or remote) to report in the "temperature" proc entry.
	This is controlled by the NVRAM variable sensor_use_remote.
 */
static bool use_remote_temp = TRUE;

/* -- SENSORS SYSCTL START -- */

#define LM63_SYSCTL_CONFIG							1200
#define LM63_SYSCTL_STATUS							1201
#define LM63_SYSCTL_MASK							1202

#define LM63_SYSCTL_LOCAL_TEMP						1300
#define LM63_SYSCTL_REMOTE_TEMP						1301
#define LM63_SYSCTL_REMOTE_TEMP_OFFSET				1302
#define LM63_SYSCTL_REMOTE_TEMP_TCRIT_SETPOINT		1303
#define LM63_SYSCTL_TEMPERATURE						1304
#define LM63_SYSCTL_USE_REMOTE_TEMP					1305

#define LM63_SYSCTL_REMOTE_TEMP_PWM_LUT				1306

#define LM63_SYSCTL_FAN_PWMRPM_CONFIG				1400
#define LM63_SYSCTL_FAN_SPINUP_CONFIG				1401
#define LM63_SYSCTL_FAN_PWM_FREQ					1402
#define LM63_SYSCTL_FAN_PWM_VALUE					1403
#define LM63_SYSCTL_FAN_DUTY_CYCLE					1404

#define LM63_SYSCTL_RAW_READ						1500
#define LM63_SYSCTL_RAW_WRITE						1501

/* -- SENSORS SYSCTL END -- */


/* The /proc/sys entries */
/* These files are created for each detected LM63. This is just a template;
   though at first sight, you might think we could use a statically
   allocated list, we need some way to get back to the parent - which
   is done through one of the 'extra' fields which are initialized 
   when a new copy is allocated. */

static ctl_table LM63_dir_table_template[] = {
	{LM63_SYSCTL_CONFIG, "config", NULL, 0, 0644, NULL, &i2c_proc_real,
	  &i2c_sysctl_real, NULL, &lm63_config},
	{LM63_SYSCTL_STATUS, "alarms", NULL, 0, 0644, NULL, &i2c_proc_real,
	  &i2c_sysctl_real, NULL, &lm63_status},
	{LM63_SYSCTL_MASK, "alarm_mask", NULL, 0, 0644, NULL, &i2c_proc_real,
	  &i2c_sysctl_real, NULL, &lm63_mask},

	{LM63_SYSCTL_LOCAL_TEMP, "local_temp", NULL, 0, 0644, NULL, &i2c_proc_real,
	  &i2c_sysctl_real, NULL, &lm63_local_temp},   /* die temperature */
	  
	{LM63_SYSCTL_USE_REMOTE_TEMP, "use_remote_temp", NULL, 0, 0644, NULL, &i2c_proc_real,
	  &i2c_sysctl_real, NULL, &lm63_use_remote_temp},
	{LM63_SYSCTL_REMOTE_TEMP, "temperature", NULL, 0, 0644, NULL, &i2c_proc_real,
	  &i2c_sysctl_real, NULL, &lm63_temperature},
	{LM63_SYSCTL_REMOTE_TEMP, "temp", NULL, 0, 0644, NULL, &i2c_proc_real,
	  &i2c_sysctl_real, NULL, &lm63_remote_temp},
	{LM63_SYSCTL_REMOTE_TEMP_OFFSET, "temp_offset", NULL, 0, 0644, NULL, &i2c_proc_real,
	  &i2c_sysctl_real, NULL, &lm63_remote_temp_offset},
	{LM63_SYSCTL_REMOTE_TEMP_TCRIT_SETPOINT, "tcrit", NULL, 0, 0644, NULL, &i2c_proc_real,
	  &i2c_sysctl_real, NULL, &lm63_remote_temp_tcrit_setpoint},

	{LM63_SYSCTL_REMOTE_TEMP_PWM_LUT, "temp_pwm_lut", NULL, 0, 0644, NULL, &i2c_proc_real,
	  &i2c_sysctl_real, NULL, &lm63_remote_temp_pwm_lut},

	{LM63_SYSCTL_FAN_PWMRPM_CONFIG, "fan_pwmrpm_config", NULL, 0, 0644, NULL, &i2c_proc_real,
	  &i2c_sysctl_real, NULL, &lm63_fan_pwmrpm_config},
	{LM63_SYSCTL_FAN_SPINUP_CONFIG, "fan_spinup_config", NULL, 0, 0644, NULL, &i2c_proc_real,
	  &i2c_sysctl_real, NULL, &lm63_fan_spinup_config},
	{LM63_SYSCTL_FAN_PWM_FREQ, "fan_pwm_freq", NULL, 0, 0644, NULL, &i2c_proc_real,
	  &i2c_sysctl_real, NULL, &lm63_fan_pwm_freq},
	{LM63_SYSCTL_FAN_PWM_VALUE, "pwm", NULL, 0, 0644, NULL, &i2c_proc_real,
	  &i2c_sysctl_real, NULL, &lm63_fan_pwm_value},
	{LM63_SYSCTL_FAN_PWM_VALUE, "fan_duty_cycle", NULL, 0, 0644, NULL, &i2c_proc_real,
	  &i2c_sysctl_real, NULL, &lm63_fan_duty_cycle},

	/* For debugging */
	{LM63_SYSCTL_RAW_READ, "raw_read", NULL, 0, 0644, NULL, &i2c_proc_real,
	  &i2c_sysctl_real, NULL, &lm63_raw_read},
	{LM63_SYSCTL_RAW_WRITE, "raw_write", NULL, 0, 0644, NULL, &i2c_proc_real,
	  &i2c_sysctl_real, NULL, &lm63_raw_write},

	{0}
};

int lm63_attach_adapter(struct i2c_adapter *adapter)
{
	int error;
	struct i2c_client_address_data  lm63_client_data;

	LM63_TRACE(("lm63.c : lm63_attach_adapter() : Entering function...\n"));

	lm63_client_data.normal_i2c       = addr_data.normal_i2c;
	lm63_client_data.normal_i2c_range = addr_data.normal_i2c_range;
	lm63_client_data.probe            = addr_data.probe;
	lm63_client_data.probe_range      = addr_data.probe_range;
	lm63_client_data.ignore           = addr_data.ignore;
	lm63_client_data.ignore_range     = addr_data.ignore_range;
	lm63_client_data.force            = addr_data.forces->force;

	error = i2c_probe(adapter, &lm63_client_data, lm63_detect);
	i2c_detect(adapter, &addr_data, lm63_detect);

        return error;
}

static int lm63_detect(struct i2c_adapter *adapter, int address,
			  unsigned short flags, int kind)
{
	int i;
	struct i2c_client *new_client;
	struct lm63_data *data;
	int err = 0;
	const char *type_name = "";
	const char *client_name = "";
	u8 man_id = 0;
	u8 die_rev = 0;
//	int lockFlag;

	LM63_TRACE(("lm63.c: lm63_detect(): Entering function.  flags=%i, kind=%i.\n", flags, kind));

	if (!i2c_check_functionality(adapter, I2C_FUNC_SMBUS_BYTE_DATA))
		goto ERROR0;

	/* OK. For now, we presume we have a valid client. We now create the
	   client structure, even though we cannot fill it completely yet.
	   But it allows us to access LM63_{read,write}_value. */

	if (!(new_client = kmalloc(sizeof(struct i2c_client) +
				   sizeof(struct lm63_data),
				   GFP_KERNEL))) {
		err = -ENOMEM;
		goto ERROR0;
	}

	data = (struct lm63_data *) (new_client + 1);
	new_client->addr = address;
	new_client->data = data;
	new_client->adapter = adapter;
	new_client->driver = &LM63_driver;
	new_client->flags = 0;

	/* Now, we do the remaining detection. */

	if (kind < 0) {
		man_id = lm63_read_value(new_client, LM63_REG_MAN_ID);
		die_rev = lm63_read_value(new_client, LM63_REG_DIE_REV);

		if (man_id != 0x01 ||
			die_rev != 0x41)  /* LM63 (Does this die revision increment?) */
			goto ERROR1;
	}


	/* Fill in the remaining client fields and put into the global list */
    type_name = "lm63";
    client_name = "LM63 chip";
	strcpy(new_client->name, client_name);
	data->type = kind;

	new_client->id = lm63_id++;
	data->valid = 0;
	init_MUTEX(&data->update_lock);

	/* Tell the I2C layer a new client has arrived */
	if ((err = i2c_attach_client(new_client)))
		goto ERROR3;

	/* Register a new directory entry with module sensors */
	if ((i = i2c_register_entry(new_client,
					type_name,
					LM63_dir_table_template,
					THIS_MODULE)) < 0) {
		err = i;
		goto ERROR4;
	}
	data->sysctl_id = i;

	data->man_id = man_id;
	data->die_rev = die_rev;
	data->raw_read_register = LM63_REG_DIE_REV;

	/* Initialize the LM63 chip */
	lm63_init_client(new_client);

	return 0;

/* OK, this is not exactly good programming practice, usually. But it is
   very code-efficient in this case. */

      ERROR4:
	i2c_detach_client(new_client);
      ERROR3:
      ERROR1:
	kfree(new_client);
      ERROR0:
	return err;
}

int lm63_detach_client(struct i2c_client *client)
{
	int err;

	LM63_TRACE(("lm63.c : lm63_detach_client() : Entering function...\n"));

	i2c_deregister_entry(((struct lm63_data *) (client->data))->
				 sysctl_id);

	if ((err = i2c_detach_client(client))) {
		printk
		    ("bcmlm63.o: Client deregistration failed, client not detached.\n");
		return err;
	}

	kfree(client);

	return 0;

}

/* No commands defined yet */
int lm63_command(struct i2c_client *client, unsigned int cmd, void *arg)
{
	LM63_TRACE(("lm63.c : lm63_command() : Entering function...\n"));

	return 0;
}

void lm63_inc_use(struct i2c_client *client)
{
	LM63_TRACE(("lm63.c : lm63_inc_use() : Entering function...\n"));
#ifdef MODULE
	MOD_INC_USE_COUNT;
#endif
}

void lm63_dec_use(struct i2c_client *client)
{
	LM63_TRACE(("lm63.c : lm63_dec_use() : Entering function...\n"));
#ifdef MODULE
	MOD_DEC_USE_COUNT;
#endif
}

int lm63_read_value(struct i2c_client *client, u8 reg)
{
	s32 value;
	int lockFlag;

	if((lockFlag = bcm_serbus_lock(NULL)) < 0)
		return -ERESTARTSYS;   /* FIXME @dtk return error? */

	value = i2c_smbus_read_byte_data(client, reg);

	bcm_serbus_unlock(NULL, lockFlag);

	LM63_TRACE(("lm63.c : lm63_read_value() : Read register=%s(0x%x) value=0x%x.\n", 
				lm63_get_reg_name(reg), reg, value));

	return 0xff & value;

//	return 0xFF & i2c_smbus_read_byte_data(client, reg);
}

int lm63_write_value(struct i2c_client *client, u8 reg, u8 value)
{
	int lockFlag;
	int rc;

	if((lockFlag = bcm_serbus_lock(NULL)) < 0)
		return -ERESTARTSYS;   /* FIXME @dtk return error? */

	rc = i2c_smbus_write_byte_data(client, reg, value);

	bcm_serbus_unlock(NULL, lockFlag);

	LM63_TRACE(("lm63.c : lm63_write_value() : Wrote register=%s(0x%x) value=0x%x.\n", 
				lm63_get_reg_name(reg), reg, value));

	return rc;
	
//	return i2c_smbus_write_byte_data(client, reg, value);
}

/* Called when we have found a new LM63. It should set limits, etc. */
void lm63_init_client(struct i2c_client *client)
{
	struct lm63_data *data = client->data;

	LM63_TRACE(("lm63.c : lm63_init_client() : Setting init values...\n"));
	lm63_update_client(client);

	/* Determine which temperature to use */
	if (strcmp(nvram_safe_get("sensor_use_remote_temp"), "disabled") == 0)
		use_remote_temp = FALSE;

	/*
		If remote diode fault alarm is raised (which can happen if 
		a remote sensor is not attached to the chip), then mask the 
		remote alarms and default to use local temperature.
	 */
	if (data->status & STATUS_REMOTE_DIODE_FAULT_ALARM)
	{
		data->mask |= (MASK_REMOTE_HIGH_ALARM | 
					   MASK_REMOTE_LOW_ALARM | 
					   MASK_REMOTE_TCRIT_ALARM);
		lm63_write_value(client, LM63_REG_MASK, data->mask);
		printk("bcmlm63.o: Remote sensor diode is not connected "
			   "or it is not operational.  Using internal/local "
			   "temperature of LM63.\n");
		use_remote_temp = FALSE;
	}

	/* Clears old status */
	lm63_update_client(client);

	/* Turn on fan to full speed at startup */
	data->fan_pwm_value = DUTYCYCLE_TO_PWM(100, data->fan_pwm_freq);
	lm63_write_value(client, LM63_REG_FAN_PWM_VALUE,
				    data->fan_pwm_value);
	
	/* Enable ALERT interrupts by clearing the mask */
	data->config &= ~CONFIG_ALERT_MASK;
	lm63_write_value(client, LM63_REG_CONFIG, data->config);

}

void lm63_update_client(struct i2c_client *client)
{
	struct lm63_data *data = client->data;
	int i;

	LM63_TRACE(("lm63.c : lm63_update_client() : Entering function...\n"));

	down(&data->update_lock);

	if ((jiffies - data->last_updated > HZ) ||  /* 1 sec cache */
            (jiffies < data->last_updated)      || 
             !data->valid) {

#if 0
		data->man_id = 
			lm63_read_value(client, LM63_REG_MAN_ID);
		data->die_rev = 
			lm63_read_value(client, LM63_REG_DIE_REV);
#endif
		data->config = 
			lm63_read_value(client, LM63_REG_CONFIG);
		data->status = 
			lm63_read_value(client, LM63_REG_STATUS);
		data->mask = 
			lm63_read_value(client, LM63_REG_MASK);

		data->local_temp =
			lm63_read_value(client, LM63_REG_LOCAL_TEMP);
		data->local_temp_hi_setpoint =
			lm63_read_value(client, LM63_REG_LOCAL_TEMP_HI_SETPOINT);
		data->remote_temp =
			(lm63_read_value(client, LM63_REG_REMOTE_TEMP_MSB) << 8)
			+ lm63_read_value(client, LM63_REG_REMOTE_TEMP_LSB);
		data->remote_temp_offset =
			(lm63_read_value(client, LM63_REG_REMOTE_TEMP_OFFSET_MSB) << 8)
			+ lm63_read_value(client, LM63_REG_REMOTE_TEMP_OFFSET_LSB);
		data->remote_temp_hi_setpoint =
			(lm63_read_value(client, LM63_REG_REMOTE_TEMP_HI_SETPOINT_MSB) << 8)
			+ lm63_read_value(client, LM63_REG_REMOTE_TEMP_HI_SETPOINT_LSB);
		data->remote_temp_hi_hyst =
			lm63_read_value(client, LM63_REG_REMOTE_TEMP_HI_HYST);
		data->remote_temp_low_setpoint =
			(lm63_read_value(client, LM63_REG_REMOTE_TEMP_LOW_SETPOINT_MSB) << 8)
			+ lm63_read_value(client, LM63_REG_REMOTE_TEMP_LOW_SETPOINT_LSB);
		data->remote_temp_tcrit_setpoint =
			lm63_read_value(client, LM63_REG_REMOTE_TEMP_TCRIT_SETPOINT);
		data->remote_temp_tcrit_hyst =
			lm63_read_value(client, LM63_REG_REMOTE_TEMP_TCRIT_HYST);

		data->fan_pwmrpm_config = 
			lm63_read_value(client, LM63_REG_FAN_PWMRPM_CONFIG);
		data->fan_spinup_config = 
			lm63_read_value(client, LM63_REG_FAN_SPINUP_CONFIG);
		data->fan_pwm_freq = 
			lm63_read_value(client, LM63_REG_FAN_PWM_FREQ);
		data->fan_pwm_value = 
			lm63_read_value(client, LM63_REG_FAN_PWM_VALUE);
		
		for (i = 0; i < 8; i++) {
			data->remote_temp_pwm_lut_temp[i] = 
				lm63_read_value(client, LM63_REG_REMOTE_TEMP_PWM_LUT + (2 * i));
			data->remote_temp_pwm_lut_pwm[i] = 
				lm63_read_value(client, LM63_REG_REMOTE_TEMP_PWM_LUT + (2 * i) + 1);
		}


		data->last_updated = jiffies;
		data->valid = 1;
	}
	up(&data->update_lock);
}


/* The next few functions are the call-back functions of the /proc/sys and
   sysctl files. Which function is used is defined in the ctl_table in
   the extra1 field.
   Each function must return the magnitude (power of 10 to divide the date
   with) if it is called with operation==SENSORS_PROC_REAL_INFO. It must
   put a maximum of *nrels elements in results reflecting the data of this
   file, and set *nrels to the number it actually put in it, if operation==
   SENSORS_PROC_REAL_READ. Finally, it must get upto *nrels elements from
   results and write them to the chip, if operations==SENSORS_PROC_REAL_WRITE.
   Note that on SENSORS_PROC_REAL_READ, I do not check whether results is
   large enough (by checking the incoming value of *nrels). This is not very
   good practice, but as long as you put less than about 5 values in results,
   you can assume it is large enough. */

static void lm63_config(struct i2c_client *client, int operation,
			 int ctl_name, int *nrels_mag, long *results)
{
	struct lm63_data *data = client->data;

	LM63_TRACE(("lm63.c : lm63_config() : Entering function. operation=%d.\n", operation));

	if (operation == SENSORS_PROC_REAL_INFO)
		*nrels_mag = 0;
	else if (operation == SENSORS_PROC_REAL_READ) {
		lm63_update_client(client);
		results[0] = data->config;
		*nrels_mag = 1;
	} else if (operation == SENSORS_PROC_REAL_WRITE) {
		if (*nrels_mag >= 1) {
			data->config = results[0];
			lm63_write_value(client, LM63_REG_CONFIG,
					    data->config);
		}
	}
}

static void lm63_status(struct i2c_client *client, int operation,
			 int ctl_name, int *nrels_mag, long *results)
{
	struct lm63_data *data = client->data;

	LM63_TRACE(("lm63.c : lm63_status() : Entering function. operation=%d.\n", operation));

	if (operation == SENSORS_PROC_REAL_INFO)
		*nrels_mag = 0;
	else if (operation == SENSORS_PROC_REAL_READ) {
		lm63_update_client(client);
		results[0] = data->status;
		*nrels_mag = 1;
	} else if (operation == SENSORS_PROC_REAL_WRITE) {
		if (*nrels_mag >= 1) {
			data->status = results[0];
			lm63_write_value(client, LM63_REG_STATUS,
					    data->status);
		}
	}
}

static void lm63_mask(struct i2c_client *client, int operation,
			 int ctl_name, int *nrels_mag, long *results)
{
	struct lm63_data *data = client->data;

	LM63_TRACE(("lm63.c : lm63_mask() : Entering function. operation=%d.\n", operation));

	if (operation == SENSORS_PROC_REAL_INFO)
		*nrels_mag = 0;
	else if (operation == SENSORS_PROC_REAL_READ) {
		lm63_update_client(client);
		results[0] = data->mask;
		*nrels_mag = 1;
	} else if (operation == SENSORS_PROC_REAL_WRITE) {
		if (*nrels_mag >= 1) {
			data->mask = results[0];
			lm63_write_value(client, LM63_REG_MASK,
					    data->mask);
		}
	}
}

static void lm63_local_temp(struct i2c_client *client, int operation,
	int ctl_name, int *nrels_mag, long *results)
{
	struct lm63_data *data = client->data;

	LM63_TRACE(("lm63.c : lm63_local_temp() : Entering function. operation=%d.\n", operation));

	if (operation == SENSORS_PROC_REAL_INFO)
		*nrels_mag = 0; 
	else if (operation == SENSORS_PROC_REAL_READ)
	{
		lm63_update_client(client);
		results[0] = TEMP1_FROM_REG(data->local_temp_hi_setpoint);
 		results[1] = TEMP1_FROM_REG(data->local_temp);  
		*nrels_mag = 2;
		/* lm63 does not have local temp low setpoint */
	}
	else if (operation == SENSORS_PROC_REAL_WRITE)
	{
		if (*nrels_mag >= 1)
		{
			data->local_temp_hi_setpoint = TEMP1_TO_REG(results[0]);
			lm63_write_value(client, LM63_REG_LOCAL_TEMP_HI_SETPOINT,
				data->local_temp_hi_setpoint);
		}
	}
}

static void lm63_use_remote_temp(struct i2c_client *client, int operation,
	int ctl_name, int *nrels_mag, long *results)
{
	LM63_TRACE(("lm63.c : lm63_use_remote_temp() : Entering function. operation=%d.\n", operation));

	if (operation == SENSORS_PROC_REAL_INFO)
		*nrels_mag = 0; 
	else if (operation == SENSORS_PROC_REAL_READ)
	{
		/* lm63_update_client(client); */
		if (use_remote_temp)
			results[0] = 1; 
		else 
			results[0] = 0;
		*nrels_mag = 1;
	}
}

static void lm63_remote_temp_offset(struct i2c_client *client, int operation,
	int ctl_name, int *nrels_mag, long *results)
{
	struct lm63_data *data = client->data;

	LM63_TRACE(("lm63.c : lm63_remote_temp_offset() : Entering function. operation=%d.\n", operation));

	if (operation == SENSORS_PROC_REAL_INFO)
		*nrels_mag = 1; 
	else if (operation == SENSORS_PROC_REAL_READ)
	{
		lm63_update_client(client);
		results[0] = TEMP2_FROM_REG(data->remote_temp_offset);
		*nrels_mag = 1;
	}
	else if (operation == SENSORS_PROC_REAL_WRITE)
	{
		if (*nrels_mag >= 1)
		{
			data->remote_temp_offset = TEMP2_TO_REG(results[0]);
			lm63_write_value(client, LM63_REG_REMOTE_TEMP_OFFSET_MSB,
				data->remote_temp_offset >> 8);
			lm63_write_value(client, LM63_REG_REMOTE_TEMP_OFFSET_LSB,
				data->remote_temp_offset & 0xFF);
		}
	}
}

static void lm63_temperature(struct i2c_client *client, int operation,
	int ctl_name, int *nrels_mag, long *results)
{
	struct lm63_data *data = client->data;

	LM63_TRACE(("lm63.c : lm63_temperature() : Entering function. operation=%d.\n", operation));

	if (operation == SENSORS_PROC_REAL_INFO)
	{
		if (use_remote_temp)
			*nrels_mag = 1; 
		else
			*nrels_mag = 0; 
	}
	else if (operation == SENSORS_PROC_REAL_READ)
	{
		lm63_update_client(client);
		if (use_remote_temp)
			results[0] = TEMP2_FROM_REG(data->remote_temp); 
		else 
			results[0] = TEMP1_FROM_REG(data->local_temp);
		*nrels_mag = 1;
	}
}

static void lm63_remote_temp(struct i2c_client *client, int operation,
	int ctl_name, int *nrels_mag, long *results)
{
	struct lm63_data *data = client->data;

	LM63_TRACE(("lm63.c : lm63_remote_temp() : Entering function. operation=%d.\n", operation));

	if (operation == SENSORS_PROC_REAL_INFO)
		*nrels_mag = 1; 
	else if (operation == SENSORS_PROC_REAL_READ)
	{
		lm63_update_client(client);
		results[0] = TEMP2_FROM_REG(data->remote_temp_hi_setpoint);
/*		results[1] = TEMP2_FROM_REG(data->remote_temp_low_setpoint);  */
		results[1] = data->remote_temp_hi_hyst * 10;
		results[2] = TEMP2_FROM_REG(data->remote_temp);
		*nrels_mag = 3;
	}
	else if (operation == SENSORS_PROC_REAL_WRITE)
	{
		if (*nrels_mag >= 1)
		{
			data->remote_temp_hi_setpoint = TEMP2_TO_REG(results[0]);
			lm63_write_value(client, LM63_REG_REMOTE_TEMP_HI_SETPOINT_MSB,
				data->remote_temp_hi_setpoint >> 8);
			lm63_write_value(client, LM63_REG_REMOTE_TEMP_HI_SETPOINT_LSB,
				data->remote_temp_hi_setpoint & 0xFF);
		}
		if (*nrels_mag >= 2)
		{
			data->remote_temp_hi_hyst = results[1];
			lm63_write_value(client, LM63_REG_REMOTE_TEMP_HI_HYST,
				data->remote_temp_hi_hyst / 10);
/*			
			lm63_write_value(client, LM63_REG_REMOTE_TEMP_LOW_SETPOINT_MSB,
				data->remote_temp_low_setpoint >> 8);
			lm63_write_value(client, LM63_REG_REMOTE_TEMP_LOW_SETPOINT_LSB,
				data->remote_temp_low_setpoint & 0xFF);
 */
		}
	}
}

static void lm63_remote_temp_tcrit_setpoint(struct i2c_client *client, int operation,
			 int ctl_name, int *nrels_mag, long *results)
{
	struct lm63_data *data = client->data;

	LM63_TRACE(("lm63.c : lm63_remote_temp_tcrit_setpoint() : Entering function. operation=%d.\n", operation));

	if (operation == SENSORS_PROC_REAL_INFO)
		*nrels_mag = 0;
	else if (operation == SENSORS_PROC_REAL_READ) {
		lm63_update_client(client);
		results[0] = TEMP1_FROM_REG(data->remote_temp_tcrit_setpoint);
		results[1] = data->remote_temp_tcrit_hyst;
 		*nrels_mag = 2;
	} else if (operation == SENSORS_PROC_REAL_WRITE) {
		if (*nrels_mag >= 1) {
			data->remote_temp_tcrit_setpoint = TEMP1_TO_REG(results[0]);
 			lm63_write_value(client, LM63_REG_REMOTE_TEMP_TCRIT_SETPOINT,
					    data->remote_temp_tcrit_setpoint);
		}
		if (*nrels_mag >= 2) {
			data->remote_temp_tcrit_hyst = results[1];
 			lm63_write_value(client, LM63_REG_REMOTE_TEMP_TCRIT_HYST,
					    data->remote_temp_tcrit_hyst);
		}
	}
}

static void lm63_remote_temp_pwm_lut(struct i2c_client *client, int operation,
			 int ctl_name, int *nrels_mag, long *results)
{
	struct lm63_data *data = client->data;
	int i, nPairs;

	LM63_TRACE(("lm63.c : lm63_remote_temp_pwm_lut() : Entering function. operation=%d.\n", operation));


	if (operation == SENSORS_PROC_REAL_INFO)
		*nrels_mag = 0;
	else if (operation == SENSORS_PROC_REAL_READ) {
		lm63_update_client(client);
		for (i = 0; i < 8; i++)
		{
			results[2*i] = data->remote_temp_pwm_lut_temp[i];
			results[2*i+1] = data->remote_temp_pwm_lut_pwm[i];
		}
		*nrels_mag = 16;
	} else if (operation == SENSORS_PROC_REAL_WRITE) {
		/* Write up to 8 pairs of values */
		nPairs = *nrels_mag / 2;
		if (nPairs > 8)
			nPairs = 8;

		for (i = 0; i < nPairs; i++) {
			data->remote_temp_pwm_lut_temp[i] = results[i*2];
 			lm63_write_value(client, LM63_REG_REMOTE_TEMP_PWM_LUT + (2 * i),
					    data->remote_temp_pwm_lut_temp[i]);
			data->remote_temp_pwm_lut_pwm[i] = results[i*2+1];
 			lm63_write_value(client, LM63_REG_REMOTE_TEMP_PWM_LUT + (2 * i) + 1,
					    data->remote_temp_pwm_lut_pwm[i]);
		}
	}
	
}


void lm63_fan_pwmrpm_config(struct i2c_client *client, int operation, int ctl_name,
	      int *nrels_mag, long *results)
{
	struct lm63_data *data = client->data;

	LM63_TRACE(("lm63.c : lm63_fan_pwmrpm_config() : Entering function. operation=%d.\n", operation));

	if (operation == SENSORS_PROC_REAL_INFO)
		*nrels_mag = 0;
	else if (operation == SENSORS_PROC_REAL_READ) {
		lm63_update_client(client);
		results[0] = data->fan_pwmrpm_config;
		*nrels_mag = 1;
	} else if (operation == SENSORS_PROC_REAL_WRITE) {
		if (*nrels_mag >= 1) {
			data->fan_pwmrpm_config = results[0];
			lm63_write_value(client, LM63_REG_FAN_PWMRPM_CONFIG,
					    data->fan_pwmrpm_config);
		}
	}
}

void lm63_fan_spinup_config(struct i2c_client *client, int operation, int ctl_name,
	      int *nrels_mag, long *results)
{
	struct lm63_data *data = client->data;

	LM63_TRACE(("lm63.c : lm63_fan_spinup_config() : Entering function. operation=%d.\n", operation));

	if (operation == SENSORS_PROC_REAL_INFO)
		*nrels_mag = 0;
	else if (operation == SENSORS_PROC_REAL_READ) {
		lm63_update_client(client);
		results[0] = data->fan_spinup_config;
		*nrels_mag = 1;
	} else if (operation == SENSORS_PROC_REAL_WRITE) {
		if (*nrels_mag >= 1) {
			data->fan_spinup_config = results[0];
			lm63_write_value(client, LM63_REG_FAN_SPINUP_CONFIG,
					    data->fan_spinup_config);
		}
	}
}

void lm63_fan_pwm_freq(struct i2c_client *client, int operation, int ctl_name,
	      int *nrels_mag, long *results)
{
	struct lm63_data *data = client->data;

	LM63_TRACE(("lm63.c : lm63_fan_pwm_freq() : Entering function. operation=%d.\n", operation));

	if (operation == SENSORS_PROC_REAL_INFO)
		*nrels_mag = 0;
	else if (operation == SENSORS_PROC_REAL_READ) {
		lm63_update_client(client);
		results[0] = data->fan_pwm_freq;
		*nrels_mag = 1;
	} else if (operation == SENSORS_PROC_REAL_WRITE) {
		if (*nrels_mag >= 1) {
			data->fan_pwm_freq = results[0];
			lm63_write_value(client, LM63_REG_FAN_PWM_FREQ,
					    data->fan_pwm_freq);
		}
	}
}

void lm63_fan_pwm_value(struct i2c_client *client, int operation, int ctl_name,
	      int *nrels_mag, long *results)
{
	struct lm63_data *data = client->data;

	LM63_TRACE(("lm63.c : lm63_fan_pwm_value() : Entering function. operation=%d.\n", operation));

	if (operation == SENSORS_PROC_REAL_INFO)
		*nrels_mag = 0;
	else if (operation == SENSORS_PROC_REAL_READ) {
		lm63_update_client(client);
		results[0] = data->fan_pwm_value;
		*nrels_mag = 1;
	} else if (operation == SENSORS_PROC_REAL_WRITE) {
		if (*nrels_mag >= 1) {
			data->fan_pwm_value = results[0];
			lm63_write_value(client, LM63_REG_FAN_PWM_VALUE,
					    data->fan_pwm_value);
		}
	}
}

/*
	Fan duty cycle is calculated from pwm value.
 */
void lm63_fan_duty_cycle(struct i2c_client *client, int operation, int ctl_name,
	      int *nrels_mag, long *results)
{
	struct lm63_data *data = client->data;

	LM63_TRACE(("lm63.c : lm63_fan_duty_cycle() : Entering function. operation=%d.\n", operation));

	if (operation == SENSORS_PROC_REAL_INFO)
		*nrels_mag = 0;
	else if (operation == SENSORS_PROC_REAL_READ) {
		lm63_update_client(client);
		results[0] = PWM_TO_DUTYCYCLE(data->fan_pwm_value, data->fan_pwm_freq);
		*nrels_mag = 1;
	} else if (operation == SENSORS_PROC_REAL_WRITE) {
		if (*nrels_mag >= 1) {
			data->fan_pwm_value = DUTYCYCLE_TO_PWM(results[0], data->fan_pwm_freq);
			lm63_write_value(client, LM63_REG_FAN_PWM_VALUE,
					    data->fan_pwm_value);
		}
	}
}

/* 
	Usage: Write register address then Read.
 */
void lm63_raw_read(struct i2c_client *client, int operation, int ctl_name,
	      int *nrels_mag, long *results)
{
	struct lm63_data *data = client->data;

	LM63_TRACE(("lm63.c : lm63_raw_write() : Entering function. operation=%d.\n", operation));

	if (operation == SENSORS_PROC_REAL_INFO)
		*nrels_mag = 0;
	else if (operation == SENSORS_PROC_REAL_READ) {
		results[0] = data->raw_read_register;
		down(&data->update_lock);
		results[1] = lm63_read_value(client, data->raw_read_register);
		up(&data->update_lock);
		*nrels_mag = 2;
	} else if (operation == SENSORS_PROC_REAL_WRITE) {
		if (*nrels_mag >= 1) {
			data->raw_read_register = results[0];
		}
	}
}

/* Usage: register_address, value */
void lm63_raw_write(struct i2c_client *client, int operation, int ctl_name,
	      int *nrels_mag, long *results)
{
/*	struct lm63_data *data = client->data; */

	LM63_TRACE(("lm63.c : lm63_raw_write() : Entering function. operation=%d.\n", operation));

	if (operation == SENSORS_PROC_REAL_INFO)
		*nrels_mag = 0;
	else if (operation == SENSORS_PROC_REAL_READ) {
		results[0] = 0;
		results[1] = 0;
		*nrels_mag = 2;
	} else if (operation == SENSORS_PROC_REAL_WRITE) {
		if (*nrels_mag >= 2) {
			lm63_write_value(client, results[0], results[1]);
		}
	}
}

int __init sensors_lm63_init(void)
{
	int res;

	printk("bcmlm63.o version %s (%s)\n", LM_VERSION, LM_DATE);
	lm63_initialized = 0;

	if ((res = i2c_add_driver(&LM63_driver))) {
		printk
		    ("bcmlm63.o: Driver registration failed, module not inserted.\n");
		lm63_cleanup();
		return res;
	}
	lm63_initialized++;
	return 0;
}

int __init lm63_cleanup(void)
{
	int res;

	LM63_TRACE(("lm63.c : lm63_cleanup() : Entering function...\n"));

	if (lm63_initialized >= 1) {
		if ((res = i2c_del_driver(&LM63_driver))) {
			printk
			    ("bcmlm63.o: Driver deregistration failed, module not removed.\n");
			return res;
		}
		lm63_initialized--;
	}
	return 0;
}

EXPORT_NO_SYMBOLS;

#ifdef MODULE
#ifdef MODULE_LICENSE
MODULE_LICENSE("GPL");
#endif

MODULE_AUTHOR
    ("David Kwan <dkwan@broadcom.com>");


MODULE_DESCRIPTION("LM63 driver");

int init_module(void)
{
	LM63_TRACE(("lm63.c : init_module() : Entering function...\n"));
	return sensors_lm63_init();
}

int cleanup_module(void)
{
	LM63_TRACE(("lm63.c : cleanup_module() : Entering function...\n"));
	return lm63_cleanup();
}

#endif				/* MODULE */
