/*
 * NVRAM variable manipulation
 *
 * Copyright 2001-2004, Broadcom Corporation
 * All Rights Reserved.
 *
 * THIS SOFTWARE IS OFFERED "AS IS", AND BROADCOM GRANTS NO WARRANTIES OF ANY
 * KIND, EXPRESS OR IMPLIED, BY STATUTE, COMMUNICATION OR OTHERWISE. BROADCOM
 * SPECIFICALLY DISCLAIMS ANY IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
 * FOR A SPECIFIC PURPOSE OR NONINFRINGEMENT CONCERNING THIS SOFTWARE.
 */


#ifndef _bcmnvram_h_
#define _bcmnvram_h_

#if (((!defined(linux)) || (!defined(__KERNEL__))) && (!defined(_CFE_)))
#include <string.h>
#endif
#include <bcmboard.h>

#ifndef _LANGUAGE_ASSEMBLY

#include <typedefs.h>

struct nvram_header {
	uint32 magic;
	uint32 len;
	uint32 crc_ver_init;	/* 0:7 crc, 8:15 ver, 16:27 init, mem. test 28, 29-31 reserved */
	uint32 config_refresh;	/* 0:15 config, 16:31 refresh */
	uint32 config_ncdl;	/* ncdl values for memc */
};

struct nvram_tuple {
	char *name;
	char *value;
	struct nvram_tuple *next;
};

extern void nvram_close(void);

/*
 * Initialize NVRAM access. May be unnecessary or undefined on certain
 * platforms.
 */
extern int nvram_init(void *sbh);

/*
 * Disable NVRAM access. May be unnecessary or undefined on certain
 * platforms.
 */
extern void nvram_exit(void);

/*
 * Get the value of an NVRAM variable. The pointer returned may be
 * invalid after a set.
 * @param	name	name of variable to get
 * @return	value of variable or NULL if undefined
 */
extern char * nvram_get(const char *name);

/* 
 * Get the value of an NVRAM variable.
 * @param	name	name of variable to get
 * @return	value of variable or NUL if undefined
 */
#define nvram_safe_get(name) (nvram_get(name) ? : "")

/*
 * Match an NVRAM variable.
 * @param	name	name of variable to match
 * @param	match	value to compare against value of variable
 * @return	TRUE if variable is defined and its value is string equal
 *		to match or FALSE otherwise
 */
static INLINE int
nvram_match(const char *name, const char *match) {
	const char *value = nvram_get(name);
	return (value && !strcmp(value, match));
}

/*
 * Inversely match an NVRAM variable.
 * @param	name	name of variable to match
 * @param	match	value to compare against value of variable
 * @return	TRUE if variable is defined and its value is not string
 *		equal to invmatch or FALSE otherwise
 */
static INLINE int
nvram_invmatch(const char *name, const char *invmatch) {
	const char *value = nvram_get(name);
	return (value && strcmp(value, invmatch));
}

/*
 * Set the value of an NVRAM variable. The name and value strings are
 * copied into private storage. Pointers to previously set values
 * may become invalid. The new value may be immediately
 * retrieved but will not be permanently stored until a commit.
 * @param	name	name of variable to set
 * @param	value	value of variable
 * @return	0 on success and errno on failure
 */
extern int nvram_set(const char *name, const char *value);

/*
 * Unset an NVRAM variable. Pointers to previously set values
 * remain valid until a set.
 * @param	name	name of variable to unset
 * @return	0 on success and errno on failure
 * NOTE: use nvram_commit to commit this change to flash.
 */
extern int nvram_unset(const char *name);

/*
 * Commit NVRAM variables to permanent storage. All pointers to values
 * may be invalid after a commit.
 * NVRAM values are undefined after a commit.
 * @return	0 on success and errno on failure
 */
extern int nvram_commit(void);

/*
 * Get all NVRAM variables (format name=value\0 ... \0\0).
 * @param	buf	buffer to store variables
 * @param	count	size of buffer in bytes
 * @return	0 on success and errno on failure
 */
extern int nvram_getall(char *buf, int count);

#endif /* _LANGUAGE_ASSEMBLY */

#define NVRAM_MAGIC			0x48534C46	/* 'FLSH' */
/* NVRAM copy support added in version 2 */
#define NVRAM_VERSION		2
#define NVRAM_HEADER_SIZE	20

#define NVRAM_SPACE			(32 * 1024)
#define NVRAM_SEC_SIZE		(64 * 1024)
#define NVRAM_SPACE_OFFSET	(32 * 1024)	/* NVRAM at 32K in 64K region */

/* NVRAM regions */
#define	BCM_FLASH_DEVICE			"flash0"
#define	BCM_BOOT_DEVICE				"flash0.boot"
#define	BCM_NVRAM_DEVICE			"flash0.nvram"
#define	BCM_NVRAMCOPY_DEVICE		"flash0.nvramcopy"
#define BCM_TRX_HDR_DEVICE			"flash0.trx"
#define BCM_OS_DEVICE				"flash0.os"
#define BCM_TRX_DEVICE				"flash1.trx"

/* NVRAM init return values and values that the CFE init code will look for */
#define NV_INIT_RESET		1	/* Reset button pressed, restore factory defaults */
#define NV_INIT_NONV		2	/* No run-time NVRAM variables, program run-time NVRAM with factory defaults */
#define NV_INIT_MFGSTATE	3	/* Run-time NVRAM variables region has pre-defined pattern; enter mfg test state */
#define NV_INIT_RESMAIN		4	/* Restore from NVRAM copy */
#define NV_INIT_RESCOPY 	5	/* Backup NVRAM to copy */
#define NV_INIT_NOREBOOT	6	/* Just do a commit without rebooting */




#endif /* _bcmnvram_h_ */
