/*
 * GPIO char driver
 *
 * Copyright (C) 2002 Broadcom Corporation
 *
 * $Id: gpio.c,v 1.2 2003/02/03 17:41:01 mhuang Exp $
 */

#include <linux/module.h>
#include <linux/init.h>
#include <linux/fs.h>
#include <linux/proc_fs.h>
#include <linux/miscdevice.h>
#include <asm/uaccess.h>
#include <asm/signal.h>

#include <typedefs.h>
#include <osl.h>
#include <bcmendian.h>
#include <bcmutils.h>
#include <bcmdevs.h>
#include <bcmnvram.h>
#include <bcmboard.h>
#include <sbutils.h>
#include <sbchipc.h>
#include <sbconfig.h>

static void *gpio_sbh;
static int gpio_major;
static devfs_handle_t gpio_dir;
static struct {
	char *name;
	devfs_handle_t handle;
} gpio_file[] = {
	{ "in", NULL },
	{ "out", NULL },
	{ "outen", NULL },
	{ "control", NULL }
};

typedef struct gpiodevstruct
{
	int devOpenCount;
} gpiodev_t;

static gpiodev_t gpioDevStruct;


#define GPIO_DIR_IN			0
#define GPIO_DIR_OUT		1
#define GPIO_DIR_IN_OUT		2
#define GPIO_MAX_NAME_SIZE	80

#define CC_GPIO_INT			1
#define GPIO_IRQ			3

/* 
Default configration is all generic IO and all inputs with
not interrupt enabled and no handler.
*/
static struct {
	char name[GPIO_MAX_NAME_SIZE];		/* name used in the proc filesystem */
	u32 pinNum;		/* GPIO pin number */
	u32 direction;	/* Pin direction 0=in (default), 1=out */
	u32 polarity;	/* Output polarity 0=normal (default), 1=reverse
						LEDs might be reverse polarity
					*/
	u32 interrupt;	/* If direction is in, is it interrupt
						drive. 0=no interrupt (default), 1=interrupt */
	u32 intr_level;	/* signal level to be interrupted at, 
						0=low, 1=high. Default is 'high'. */
	pid_t handler;	/* If interrupt, handler associated to
						be signalled. If NULL, value will be
						latched and user app will poll it. */
	int handler_sig;/* Signal to be sent to the handler if 
						handler is defined. Default is SIGIO.*/
} gpio_proc_config[] = {
	{ "gpio_0", 0, 0, 0, 0, 1, 0, SIGIO },
	{ "gpio_1", 1, 0, 0, 0, 1, 0, SIGIO },
	{ "gpio_2", 2, 0, 0, 0, 1, 0, SIGIO },
	{ "gpio_3", 3, 0, 0, 0, 1, 0, SIGIO },
	{ "gpio_4", 4, 0, 0, 0, 1, 0, SIGIO },
	{ "gpio_5", 5, 0, 0, 0, 1, 0, SIGIO },
	{ "gpio_6", 6, 0, 0, 0, 1, 0, SIGIO },
	{ "gpio_7", 7, 0, 0, 0, 1, 0, SIGIO },
};

static u32 latchedInput = 0;

/* Outputs enabled for pulse (eg. blinking LEDs) */
static int gpio_out_pulse_mask = 0;

/* Fixed pulse timing */
#define OUT_PULSE_TIME		((500 * HZ) / 1000)	/* 500 msecs */

DECLARE_WAIT_QUEUE_HEAD(outPulseQ);

static void gpio_input_tasklet(unsigned long arg);
DECLARE_TASKLET(gpioInputTasklet, gpio_input_tasklet, 0);
static void gpio_out_pulse_tasklet(unsigned long arg);
DECLARE_TASKLET(gpioOutPulseTasklet, gpio_out_pulse_tasklet, 0);

static DECLARE_MUTEX(gpio_sem);

static void cc_gpio_enableintr(void)
{
	/* Enable the CC GPIO interrupt. */
	chipcregs_t *cc;
	void *regs;
	
	if ((regs = sb_setcore(gpio_sbh, SB_CC, 0))) {
		cc = (chipcregs_t *) regs;
		W_REG(&cc->intmask, 1);
	}
}

static void cc_gpio_disableintr(void)
{
	/* Disable the CC GPIO interrupt. */
	chipcregs_t *cc;
	void *regs;
	
	if ((regs = sb_setcore(gpio_sbh, SB_CC, 0))) {
		cc = (chipcregs_t *) regs;
		W_REG(&cc->intmask, 0);
	}
}

static void gpio_pin_enableintr(int pinNum, int intrLevel)
{
	/* Enable GPIO interrupt for the pin. */
	if(intrLevel) {
		sb_gpiointpolarity(gpio_sbh, (1 << pinNum), 0);
	} else {
		sb_gpiointpolarity(gpio_sbh, (1 << pinNum), (1 << pinNum));
	}
	sb_gpiointmask(gpio_sbh,  (1 << pinNum), (1 << pinNum));
}

static void gpio_pin_disableintr(pinNum)
{
	/* Disable GPIO interrupt for the pin. */
	sb_gpiointmask(gpio_sbh, (1 << pinNum), 0);
}

static void
gpio_input_tasklet(unsigned long arg)
{
	static u32 debounceTime = 0;
	
	/*
	Debounce the inputs to avoid continuous interrupts. If
	debounce timer is clear, set the timer and schedule
	tasklet again.  Next time tasklet will check for timer 
	elapsed and reads inputs only after timer has elapsed.  
	It enables the interrupts also only after it has read 
	in the inputs.
	*/
	if (debounceTime == 0) {
		debounceTime = jiffies + (10 * HZ) / 1000; /* 10msec */
		/* Reschedule self */
		tasklet_schedule(&gpioInputTasklet);
	} else if (time_after(jiffies, debounceTime)) {
		latchedInput |= sb_gpioin(gpio_sbh);

		/* Since we just read the inputs, reset the debounce timer. */
		debounceTime = 0;

		/* ISR handled - enable interrupts */
		cc_gpio_enableintr();
	} else {
		/* Debounce timer not yet elapsed, reschedule self */
		tasklet_schedule(&gpioInputTasklet);
	}
}

static void
gpio_isr(int irq, void *dev_id, struct pt_regs *ptregs)
{
	chipcregs_t *cc;
	void *regs;
	sbconfig_t *sb;
	u32 ccIntStatus;

	if ((regs = sb_setcore(gpio_sbh, SB_CC, 0))) {
		cc = (chipcregs_t *) regs;

		sb = (sbconfig_t *)((ulong) regs + SBCONFIGOFF);

		ccIntStatus = R_REG(&cc->intstatus);

		if(ccIntStatus & CC_GPIO_INT) {
			cc_gpio_disableintr();
			tasklet_schedule(&gpioInputTasklet);
		}
	}
}

static void
gpio_out_pulse_tasklet(unsigned long arg)
{
	u32 outVal, i;
	u32 d;
	
	for (i=0; i<BCM_MAX_GPIO; i++) {
		if ((gpio_out_pulse_mask >> i) & 1) {
			d = (1 << i);
			outVal = (~(sb_gpioout(gpio_sbh, 0, 0)) & d);
			sb_gpioout(gpio_sbh, d, outVal);
		}
	}
}

static int gpio_out_pulse(void *unused)
{
	daemonize();
	strcpy(current->comm, "gpio_out_pulse");
	sigfillset(&current->blocked);

	for (;;) {
		if(gpio_out_pulse_mask) {
			tasklet_schedule(&gpioOutPulseTasklet);
			set_current_state(TASK_INTERRUPTIBLE);
			schedule_timeout(OUT_PULSE_TIME);
		} else {
			interruptible_sleep_on(&outPulseQ);
		}
	}
}

static int
gpio_open(struct inode *inode, struct file * file)
{
	if (MINOR(inode->i_rdev) > ARRAYSIZE(gpio_file))
		return -ENODEV;

	MOD_INC_USE_COUNT;
	return 0;
}

static int
gpio_release(struct inode *inode, struct file * file)
{
	MOD_DEC_USE_COUNT;
	return 0;
}

static ssize_t
gpio_read(struct file *file, char *buf, size_t count, loff_t *ppos)
{
	u32 val;
	int retVal;

	if (down_interruptible(&gpio_sem))
		return -ERESTARTSYS;

	switch ((unsigned int)MINOR(file->f_dentry->d_inode->i_rdev)) {
	case 0:
		val = sb_gpioin(gpio_sbh);
		break;
	case 1:
		val = sb_gpioout(gpio_sbh, 0, 0);
		break;
	case 2:
		val = sb_gpioouten(gpio_sbh, 0, 0);
		break;
	case 3:
		val = sb_gpiocontrol(gpio_sbh, 0, 0);
		break;
	default:
		retVal = -ENODEV;
		goto err;
	}

	if (put_user(val, (u32 *) buf))
		retVal = -EFAULT;

err:
	up(&gpio_sem);
	return sizeof(val);
}

static ssize_t
gpio_write(struct file *file, const char *buf, size_t count, loff_t *ppos)
{
	u32 val;
	int retVal;

	if (down_interruptible(&gpio_sem))
		return -ERESTARTSYS;

	if (get_user(val, (u32 *) buf)) {
		retVal = -EFAULT;
		goto err;
	}

	switch ((unsigned int)MINOR(file->f_dentry->d_inode->i_rdev)) {
	case 0:
		retVal = -EACCES;
		goto err;
	case 1:
		sb_gpioout(gpio_sbh, ~0, val);
		break;
	case 2:
		sb_gpioouten(gpio_sbh, ~0, val);
		break;
	case 3:
		sb_gpiocontrol(gpio_sbh, ~0, val);
		break;
	default:
		retVal = -ENODEV;
	}

err:
	up(&gpio_sem);
	return sizeof(val);
}

static struct file_operations gpio_fops = {
#ifdef MODULE
	owner:		THIS_MODULE,
#endif
	open:		gpio_open,
	release:	gpio_release,
	read:		gpio_read,
	write:		gpio_write,
};


#ifdef CONFIG_PROC_FS
static char proc_gpio_root_name[] = "miscio";

static int gpio_proc_read(char *page, char **start, off_t off, int count, 
	int *eof, void *data)
{
	char *out = page;
	u32 val;
	u32 d;
	int len;
	u32 index = (u32)data;

	if (down_interruptible(&gpio_sem))
		return -ERESTARTSYS;

	d = (1 << gpio_proc_config[index].pinNum);

	/* Distinguish reading input vs output */
	if (gpio_proc_config[index].direction == 1) {
		/* Reading current output setting */
		val = sb_gpioout(gpio_sbh, 0, 0);
	} else {
		/* Reading input */
		if (gpio_proc_config[index].interrupt) {
			/* If pin has interrupt enabled, read latched value */
			val = latchedInput;
		} else {
			/* Read current state of input pin. */
			val = sb_gpioin(gpio_sbh);
		}
	}

	if(gpio_proc_config[index].polarity) {
		/* Output is reverse polarity */
		val = ~val;
	}

	val &= d;
	out += sprintf(out, "%d\n", val ? 1 : 0);
	len = out - page - off;
	if (len < count) {
		*eof = 1;
		if (len <= 0) {
			up(&gpio_sem);
			return 0;
		}
	} else {
		len = count;
	}
	*start = page + off;

	up(&gpio_sem);
	return len;
}

/*
Set or clear an output GPIO pin or clear the latched value of
the input pin which was configured to be set with interrupt.
*/
static int gpio_proc_write(struct file *file, const char *buf, 
	unsigned long count, void *data)
{
	char *cur, lbuf[count];
	u32 d;
	u32 val;
	u32 index = (u32)data;
	int retVal;

	if (!capable(CAP_SYS_ADMIN))
		return -EACCES;

	if (down_interruptible(&gpio_sem))
		return -ERESTARTSYS;

	memset(lbuf, 0, count);
	copy_from_user(lbuf, buf, count);
	cur = lbuf;
	d = (1 << gpio_proc_config[index].pinNum);

	val = bcm_strtoul(cur, NULL, 0);

	/* Make sure value is legal (0, 1, or 2 for pulse output) */
	if ((val < 0) || (val > 2)) {
		retVal = -EINVAL;
		goto err;
	}

	/* Make sure the pin is configured to be output */
	if (gpio_proc_config[index].direction == 1) {
		if (val == 2) {
			/* Output needs to be a periodic pulse */
			gpio_out_pulse_mask |= d;
			val = 0;
			if (gpio_out_pulse_mask == d) {
				/* First blink */
				wake_up_interruptible(&outPulseQ);
			}
		} else {
			if(gpio_proc_config[index].polarity) {
				/* Output is reverse polarity */
				val = !val;
			}
			/* Also, unmask pulse output if enabled. */
			gpio_out_pulse_mask &= ~d;
		}
		sb_gpioout(gpio_sbh, d, (val << gpio_proc_config[index].pinNum));
	} else {
		/* 
		Clearing the latched value of the input which
		is set by the interrupt.
		*/
		if ((gpio_proc_config[index].direction == 0) &&
				(gpio_proc_config[index].interrupt) &&
				(val == 0)) {
			latchedInput &= ~d;
		}
	}
	
err:
	up(&gpio_sem);
	return count;
}

static int gpio_config_proc_read(char *page, char **start, off_t off, int count, 
	int *eof, void *data)
{
	char *out = page;
	int len;
	u32 index = (u32)data;

	if (down_interruptible(&gpio_sem))
		return -ERESTARTSYS;

/* Since interrupt handler is not yes supported, that information is not shown */
#if 0
	out += sprintf(out, "name=%s\nPin number=%d\ndirection=%s\npolarity=%s\ninterrupt=%s\ninterrupt level=%d\nHandler PID=%d\nSignal for Handler=%d\n",
				gpio_proc_config[index].name, gpio_proc_config[index].pinNum, 
				(gpio_proc_config[index].direction ? "out":"in"), 
				(gpio_proc_config[index].polarity ? "reverse":"normal"), 
				(gpio_proc_config[index].interrupt ? "yes":"no"), 
				gpio_proc_config[index].intr_level,
				gpio_proc_config[index].handler,
				gpio_proc_config[index].handler_sig);
#else
	out += sprintf(out, "name=%s\npin number=%d\ndirection=%s\npolarity=%s\ninterrupt=%s\ninterrupt level=%d\n",
			gpio_proc_config[index].name, gpio_proc_config[index].pinNum, 
			(gpio_proc_config[index].direction ? "out":"in"), 
			(gpio_proc_config[index].polarity ? "reverse":"normal"), 
			(gpio_proc_config[index].interrupt ? "yes":"no"), 
			gpio_proc_config[index].intr_level);
#endif
	len = out - page - off;
	if (len < count) {
		*eof = 1;
		if (len <= 0) {
			up(&gpio_sem);
			return 0;
		}
	} else {
		len = count;
	}
	*start = page + off;

	up(&gpio_sem);
	return len;
}

static int gpio_config_proc_write(struct file *file, const char *buf, 
	unsigned long count, void *data)
{
	char *cur, lbuf[count];
	char *value;
	u32 index = (u32)data;
	u32 d;

	if (!capable(CAP_SYS_ADMIN))
		return -EACCES;

	if (down_interruptible(&gpio_sem))
		return -ERESTARTSYS;

	memset(lbuf, 0, count);
	copy_from_user(lbuf, buf, count);
	lbuf[count-1] = '\0';
	cur = lbuf;

	if ((value = strchr(cur, '=')) != NULL) {
		*value = '\0';
		value++;
	}

	if (strcmp(cur, "direction") == 0) {
		if (value && *value) {
			if(strcmp(value, "out") == 0)
				gpio_proc_config[index].direction = 1;
			else
				gpio_proc_config[index].direction = 0;
		}
	} else if (strcmp(cur, "polarity") == 0) {
		if (value && *value) {
			if(strcmp(value, "reverse") == 0)
				gpio_proc_config[index].polarity = 1;
			else
				gpio_proc_config[index].polarity = 0;
		}
	} else if (strcmp(cur, "interrupt") == 0) {
		if (value && *value) {
			if(strcmp(value, "yes") == 0)
				gpio_proc_config[index].interrupt = 1;
			else
				gpio_proc_config[index].interrupt = 0;
		}
	} else if (strcmp(cur, "interrupt level") == 0) {
		if (value && *value)
			gpio_proc_config[index].intr_level = (bcm_strtoul(value, NULL, 0) ? 1 : 0);
	} else if (strcmp(cur, "handler") == 0) {
		if (value && *value)
			gpio_proc_config[index].handler = bcm_strtoul(value, NULL, 0);
	} 	else if (strcmp(cur, "handler signal") == 0) {
		if (value && *value)
			gpio_proc_config[index].handler_sig = bcm_strtoul(value, NULL, 0);
	} else {
		printk("GPIO configuration error: Unknown configuration parameter (%s)\n", cur);
	}
	
	d = (1 << gpio_proc_config[index].pinNum);
	/* Setup direction */
	if (gpio_proc_config[index].direction) {
		/* output */
		sb_gpioouten(gpio_sbh, d, d);
	} else {
		/* input */
		sb_gpioouten(gpio_sbh, d, 0);
	}
	
	/* Setup interrupt */
	if (gpio_proc_config[index].interrupt) {
		/* Enable interrupt for this GPIO pin for specified level if required. */
		gpio_pin_enableintr(gpio_proc_config[index].pinNum, 
						gpio_proc_config[index].intr_level);
	} else {
		gpio_pin_disableintr(gpio_proc_config[index].pinNum);
	}
	
	up(&gpio_sem);
	return count;
}

static void gpio_configure_name(int pinNum)
{
	char gpioNameVarStr[GPIO_MAX_NAME_SIZE];
	char *gpioName;

	strcpy(gpioNameVarStr, gpio_proc_config[pinNum].name);
	strcat(gpioNameVarStr, "_name");
	gpioName = nvram_safe_get(gpioNameVarStr);
	if (strcmp(gpioName, "") != 0) {
		strcpy(gpio_proc_config[pinNum].name, gpioName);
	}
}

static int __init gpio_create_procfs(void)
{
	int i;
	struct proc_dir_entry *ent;
	char buf[256];

	for (i = 0; i < ARRAYSIZE(gpio_proc_config); i++) {
		gpio_configure_name(i);
		snprintf(buf, sizeof(buf), "%s/%s", 
				proc_gpio_root_name, gpio_proc_config[i].name);
		ent = create_proc_entry(buf, S_IFREG|S_IRUGO|S_IWUSR, 0);
		if (!ent) {
			/* 
			Maybe the directory /proc/miscio does not exist, 
			create and try again.
			*/
			if(!proc_mkdir(proc_gpio_root_name, 0)) return -1;

			ent = create_proc_entry(buf, S_IFREG|S_IRUGO|S_IWUSR, 0);
			if (!ent) return -1;
		}
		ent->nlink = 1;
		ent->data = (void *)i;
		ent->read_proc = gpio_proc_read;
		ent->write_proc = gpio_proc_write;
#ifdef MODULE
		ent->owner = THIS_MODULE;
#endif
		/* Create entry for configuration of this GPIO pin */
		strcat(buf, "_config");
		ent = create_proc_entry(buf, S_IFREG|S_IRUGO|S_IWUSR, 0);
		ent->nlink = 1;
		ent->data = (void *)i;
		ent->read_proc = gpio_config_proc_read;
		ent->write_proc = gpio_config_proc_write;
#ifdef MODULE
		ent->owner = THIS_MODULE;
#endif
	}

	return 0;
}

int gpio_remove_procfs(void)
{
	int i;
	char buf[256];

	for (i = 0; i < ARRAYSIZE(gpio_proc_config); i++) {
		snprintf(buf, sizeof(buf), "%s/%s", 
				proc_gpio_root_name, gpio_proc_config[i].name);
		remove_proc_entry(buf, 0);
		strcat(buf, "_config");
		remove_proc_entry(buf, 0);
	}

	return 0;
}
#endif


static int __init
gpio_init(void)
{
	int i;

	if (!(gpio_sbh = sb_kattach()))
		return -ENODEV;

	sb_gpiosetcore(gpio_sbh);

	if ((gpio_major = devfs_register_chrdev(0, "gpio", &gpio_fops)) < 0)
		return gpio_major;

	gpio_dir = devfs_mk_dir(NULL, "gpio", NULL);

	for (i = 0; i < ARRAYSIZE(gpio_file); i++) {
		gpio_file[i].handle = devfs_register(gpio_dir,
						     gpio_file[i].name,
						     DEVFS_FL_DEFAULT, gpio_major, i,
						     S_IFCHR | S_IRUGO | S_IWUGO,
						     &gpio_fops, NULL);
	}

	/*
	Mask off all interrupts, it will be enabled by the applications
	while configuring the GPIOs.
	*/
	sb_gpiointmask(gpio_sbh, ~0, 0);
	/* register our interrupt handler */
	if (request_irq(GPIO_IRQ, gpio_isr, SA_SHIRQ, 
							"gpio", &gpioDevStruct)) {
		printk("GPIO init error: Failed to setup IRQ.\n");
		return -EPERM;
	}

#ifdef CONFIG_PROC_FS
	gpio_create_procfs();
#endif

	/* Start kernel thread for blinking LEDs */
	kernel_thread(gpio_out_pulse, NULL, 0);

	/* Enable chip common interrupts for inputs */
	cc_gpio_enableintr();

	return 0;
}

static void __exit
gpio_exit(void)
{
	int i;

	/* Disable chip common GPIO interrupts */
	cc_gpio_disableintr();

	/* Disable all GPIO pin interrupts */
	sb_gpiointmask(gpio_sbh, ~0, 0);

	for (i = 0; i < ARRAYSIZE(gpio_file); i++)
		devfs_unregister(gpio_file[i].handle);
	devfs_unregister(gpio_dir);
	devfs_unregister_chrdev(gpio_major, "gpio");
	sb_detach(gpio_sbh);

#ifdef CONFIG_PROC_FS
	gpio_remove_procfs();
#endif
}

module_init(gpio_init);
module_exit(gpio_exit);

