/*
 * $Id: mtdchar.c,v 1.1.1.1 2003/02/03 22:37:45 mhuang Exp $
 *
 * Character-device access to raw MTD devices.
 * Pure 2.4 version - compatibility cruft removed to mtdchar-compat.c
 *
 */

#include <linux/config.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/mtd/mtd.h>
#include <linux/slab.h>

#ifdef CONFIG_DEVFS_FS
#include <linux/devfs_fs_kernel.h>
static void mtd_notify_add(struct mtd_info* mtd);
static void mtd_notify_remove(struct mtd_info* mtd);

static struct mtd_notifier notifier = {
	add:	mtd_notify_add,
	remove:	mtd_notify_remove,
};

static devfs_handle_t devfs_dir_handle;
static devfs_handle_t devfs_rw_handle[MAX_MTD_DEVICES];
static devfs_handle_t devfs_ro_handle[MAX_MTD_DEVICES];
#endif

#define MAX_KMALLOC_SIZE 0x20000

#ifdef NEW_FLASH_WRITE
#include <linux/proc_fs.h>

#ifndef ROUNDUP
#define	ROUNDUP(x, y)		((((ulong)(x)+((y)-1))/(y))*(y))
#endif

#define MTD_FIRMWARE_PARTITION	"linux"
#define MTD_ROOTFS_PARTITION	"rootfs"
static void mtd_erase_callback (struct erase_info *instr);

int mtd_debug_level = 0;

#define DPRINTF(args...)                       \
        if (mtd_debug_level != 0) {   \
                printk(KERN_INFO args); \
        }

struct brcm_write_buffer {
	volatile char *startPtr;
	char *endPtr;
	char *copyPtr;
	char *commitPtr;
	int length;
	char bytesPrgmed[24];
} static volatile brcmWriteBuffer = {NULL, NULL, NULL, NULL, 0, "None"};

DECLARE_MUTEX(flash_update);

static void mtd_write_commit(void *param)
{
	struct file *file = (struct file *)param;
	struct mtd_info *mtd = (struct mtd_info *)file->private_data;
	struct erase_info_user commitInfo;
	char *commitBuffer = NULL;
	int ret = 0;
	int retLen = 0;
	int val;
	struct erase_info erase;
	wait_queue_head_t waitq;
	int bytesLeft = 0;

	if (mtd && mtd->module)
		try_inc_mod_count(mtd->module);

	daemonize();
	strcpy(current->comm, "mtd_write_commit");
	sigfillset(&current->blocked);

	if (brcmWriteBuffer.startPtr == NULL) {
		/* There is nothing to commit, return */
		ret = -ENODATA;
		goto err;
	}

	DECLARE_WAITQUEUE(wait, current);
	init_waitqueue_head(&waitq);

	commitInfo.length = ROUNDUP(MAX_KMALLOC_SIZE, mtd->erasesize);
	commitInfo.start = 0;
	commitBuffer = kmalloc(commitInfo.length,GFP_KERNEL);
	
	if (!commitBuffer) {
		ret= -ENOMEM;
		goto err;
	}
	
	/* endPtr is always at erase size boundary */
	bytesLeft = (brcmWriteBuffer.endPtr - brcmWriteBuffer.startPtr);
	while (bytesLeft > 0) {
		/* correct commit length. bytesLeft should be at erase size boundary */
		commitInfo.length = ((commitInfo.length) < (bytesLeft) ? (commitInfo.length) : (bytesLeft));

		/* Write the block to be committed to commit buffer */
		memcpy(commitBuffer, brcmWriteBuffer.commitPtr, commitInfo.length);
		
		/* Read and save the region to be programmed */
		ret = mtd->read(mtd, commitInfo.start, commitInfo.length, &retLen, brcmWriteBuffer.commitPtr);
		if ((ret) || (retLen != commitInfo.length)) {
			ret = -EIO;
			printk("mtd_write_commit: read failed.\n");
			goto err;
		} else {
			/*
			Since we have read the block from Flash, move the commitPtr so 
			that reads for that region will be done from the buffer and 
			not Flash.
			*/
			brcmWriteBuffer.commitPtr += commitInfo.length;
			down_interruptible(&flash_update);
			up(&flash_update);
		}

		/* Unlock region to be programmed */
		if (mtd->unlock) {
			ret = mtd->unlock(mtd, commitInfo.start, commitInfo.length);
		}

		if (!ret) {

			/* Erase the region to be programmed */
			erase.addr = commitInfo.start;
			erase.len = commitInfo.length;
			erase.mtd = mtd;
			erase.callback = mtd_erase_callback;
			erase.priv = (unsigned long)&waitq;
		
			ret = mtd->erase(mtd, &erase);
			if (!ret) {
				set_current_state(TASK_UNINTERRUPTIBLE);
				add_wait_queue(&waitq, &wait);
				if (erase.state != MTD_ERASE_DONE &&
					erase.state != MTD_ERASE_FAILED)
					schedule();
				remove_wait_queue(&waitq, &wait);
				set_current_state(TASK_RUNNING);
				ret = (erase.state == MTD_ERASE_FAILED)?-EIO:0;
				if (!ret) {
					ret = (*(mtd->write))(mtd, commitInfo.start, commitInfo.length, &retLen, commitBuffer);
					if ((!ret) && (retLen == commitInfo.length)) {
						commitInfo.start += retLen;
						bytesLeft -= retLen;
						val = (brcmWriteBuffer.commitPtr - brcmWriteBuffer.startPtr);
						if (val > brcmWriteBuffer.length)
							val = brcmWriteBuffer.length;
						sprintf(brcmWriteBuffer.bytesPrgmed, "%d\n", val);
					} else {
						/* Something went wrong while writing to Flash. Exit with error. */
						ret = -EIO;
						break;
					}
				}
			}
		}
		if (ret) goto err;
	}

err:
	kfree (commitBuffer);

	if (mtd->sync)
		mtd->sync(mtd);

	if (ret) {
		printk("Error - Flash commit of firmware image failed.\n");
		strcpy(brcmWriteBuffer.bytesPrgmed, "Error");
	}

	put_mtd_device(mtd);
}

int mtd_read_new(struct mtd_info *mtd, loff_t from, size_t count, 
											size_t *retLen, char *kbuf)
{
	int ret = 0;
	unsigned int bytesFromBuffer = 0;
	unsigned int bytesInBuffer = 0;
	unsigned int bytesFromFlash = 0;
	loff_t rootfsOffset = 0;
	
	/*
	If we have started the programming for the Flash for the firmware image, then
	read the bytes from the local buffer of what is already programmed and read from
	Flash of what is not.
	*/
	DEBUG(MTD_DEBUG_LEVEL0,"MTD_read_new");
	DPRINTF("MTD_read_new: mtd = %s, from = 0x%08llx, count = %lu\n", mtd->name, from, count);


	down_interruptible(&flash_update);
	if((!strcmp(mtd->name, MTD_ROOTFS_PARTITION)) ||
			(!strcmp(mtd->name, MTD_FIRMWARE_PARTITION))) {
		if((bytesInBuffer = (brcmWriteBuffer.commitPtr - brcmWriteBuffer.startPtr)) > 0) {
			/*
			Flash programming has started and hence for the programmed
			portion, read from the buffer.
			*/
			/* 
			Write to the Flash is done in the "linux" region, wheras the read maybe
			from "linux" or "rootfs".  If it is from "rootfs" region, the offset needs
			to be corrected so that it is the offset in the linux region and not the
			rootfs region.
			*/
			if(!strcmp(mtd->name, MTD_ROOTFS_PARTITION)) {
				struct mtd_info *fwMTD = NULL;
				/* Need to get "rootfs" offset within "linux" */
				/*
				Get the size of "linux". This requires index of "linux" to
				be 1 less than index of "rootfs" that is the 2 are consecutive
				regions which is a safe assumption as that is how the firmware
				is built and programmed.
				*/
				fwMTD = __get_mtd_device(NULL, (mtd->index - 1));
				if(!strcmp(fwMTD->name, MTD_FIRMWARE_PARTITION)) {
					/* Confirmed that fwMTD is the firmware (linux) partition. */
					/* Calculate the offset of "rootfs" within "linux" */
					rootfsOffset = (loff_t)(fwMTD->size - mtd->size);
				}
			}
			/* Corrected offset. */
			rootfsOffset += from;
			if (rootfsOffset < bytesInBuffer) {
				/*
				Read from buffer upto what is in buffer and read the 
				rest from Flash.
				*/
				if ((rootfsOffset + count) > bytesInBuffer)
					bytesFromBuffer = (bytesInBuffer - rootfsOffset);
				else
					bytesFromBuffer = count;
			
				memcpy(kbuf, (brcmWriteBuffer.startPtr + rootfsOffset), bytesFromBuffer);
			
				count -= bytesFromBuffer;
				from += bytesFromBuffer;
				kbuf += bytesFromBuffer;
			}
		}
	}

	if(count) {
		/* Need to read from Flash */
		ret = mtd->read(mtd, from, count, &bytesFromFlash, kbuf);
		if (ret) {
			bytesFromFlash = 0;
		}
	}
	up(&flash_update);
	
	DPRINTF("MTD_read_new: read finished - bytesFromFlash = %lu, bytesFromBuffer = %lu, ret = %d\n",
		bytesFromFlash, bytesFromBuffer, ret);
	*retLen = (bytesFromBuffer + bytesFromFlash);
	return ret;
} /* mtd_read_new */
#endif

static loff_t mtd_lseek (struct file *file, loff_t offset, int orig)
{
	struct mtd_info *mtd=(struct mtd_info *)file->private_data;

	switch (orig) {
	case 0:
		/* SEEK_SET */
		file->f_pos = offset;
		break;
	case 1:
		/* SEEK_CUR */
		file->f_pos += offset;
		break;
	case 2:
		/* SEEK_END */
		file->f_pos =mtd->size + offset;
		break;
	default:
		return -EINVAL;
	}

	if (file->f_pos < 0)
		file->f_pos = 0;
	else if (file->f_pos >= mtd->size)
		file->f_pos = mtd->size - 1;

	return file->f_pos;
}



static int mtd_open(struct inode *inode, struct file *file)
{
	int minor = MINOR(inode->i_rdev);
	int devnum = minor >> 1;
	struct mtd_info *mtd;

	DEBUG(MTD_DEBUG_LEVEL0, "MTD_open\n");

	if (devnum >= MAX_MTD_DEVICES)
		return -ENODEV;

	/* You can't open the RO devices RW */
	if ((file->f_mode & 2) && (minor & 1))
		return -EACCES;

	mtd = get_mtd_device(NULL, devnum);
	
	if (!mtd)
		return -ENODEV;
	
	if (MTD_ABSENT == mtd->type) {
		put_mtd_device(mtd);
		return -ENODEV;
	}

	file->private_data = mtd;
		
	/* You can't open it RW if it's not a writeable device */
	if ((file->f_mode & 2) && !(mtd->flags & MTD_WRITEABLE)) {
		put_mtd_device(mtd);
		return -EACCES;
	}
		
	return 0;
} /* mtd_open */

/*====================================================================*/

static int mtd_close(struct inode *inode, struct file *file)
{
	struct mtd_info *mtd;

	DEBUG(MTD_DEBUG_LEVEL0, "MTD_close\n");

	mtd = (struct mtd_info *)file->private_data;
	
	if((!strcmp(mtd->name, MTD_ROOTFS_PARTITION)) ||
			(!strcmp(mtd->name, MTD_FIRMWARE_PARTITION))) {
		if((brcmWriteBuffer.commitPtr - brcmWriteBuffer.startPtr) > 0) {
			/* The device is in the middle of a buffered write, so
			 * skip trying to sync it.  It's not necessary because
			 * the buffered write will sync when it's done, and if
			 * we tried to sync it during the buffered write we'd
			 * just hang until the buffered write was done. */
			put_mtd_device(mtd);
			return 0;
		}
	}

	if (mtd->sync)
		mtd->sync(mtd);
	
	put_mtd_device(mtd);

	return 0;
} /* mtd_close */

static ssize_t mtd_read(struct file *file, char *buf, size_t count,loff_t *ppos)
{
	struct mtd_info *mtd = (struct mtd_info *)file->private_data;
	size_t retlen=0;
	size_t total_retlen=0;
	int ret=0;
	int len;
	char *kbuf;
	
	DEBUG(MTD_DEBUG_LEVEL0,"MTD_read\n");

	if (*ppos + count > mtd->size)
		count = mtd->size - *ppos;

	if (!count)
		return 0;
	
	while (count) {
		if (count > MAX_KMALLOC_SIZE) 
			len = MAX_KMALLOC_SIZE;
		else
			len = count;

		kbuf=kmalloc(len,GFP_KERNEL);
		if (!kbuf)
			return -ENOMEM;
		
		ret = MTD_READ(mtd, *ppos, len, &retlen, kbuf);
		if (!ret) {
			*ppos += retlen;
			if (copy_to_user(buf, kbuf, retlen)) {
			        kfree(kbuf);
				return -EFAULT;
			}
			else
				total_retlen += retlen;

			count -= retlen;
			buf += retlen;
		}
		else {
			kfree(kbuf);
			return ret;
		}
		
		kfree(kbuf);
	}
	
	return total_retlen;
} /* mtd_read */

static ssize_t mtd_write(struct file *file, const char *buf, size_t count,loff_t *ppos)
{
	struct mtd_info *mtd = (struct mtd_info *)file->private_data;
	char *kbuf;
	size_t retlen;
	size_t total_retlen=0;
	int ret=0;
	int len;

	DEBUG(MTD_DEBUG_LEVEL0,"MTD_write\n");
	
#ifdef NEW_FLASH_WRITE
	/*
	If writing to the linux partition, we are writing
	the firmware.  In the new Flash write scheme, the
	write actually just copies the data in local buffer
	and the MEMWRITECOMMIT actually writes it to the 
	Flash device.
	*/
	if(!strcmp(mtd->name, MTD_FIRMWARE_PARTITION)) {
		if (count > brcmWriteBuffer.length) {
			ret = -EINVAL;
		} else {
			if (copy_from_user(brcmWriteBuffer.copyPtr, buf, count))
				return -EFAULT;
			brcmWriteBuffer.copyPtr += count;
			ret = count;
		}
		return ret;
	}
#endif
	if (*ppos == mtd->size)
		return -ENOSPC;
	
	if (*ppos + count > mtd->size)
		count = mtd->size - *ppos;

	if (!count)
		return 0;

	while (count) {
		if (count > MAX_KMALLOC_SIZE) 
			len = MAX_KMALLOC_SIZE;
		else
			len = count;

		kbuf=kmalloc(len,GFP_KERNEL);
		if (!kbuf) {
			printk("kmalloc is null\n");
			return -ENOMEM;
		}

		if (copy_from_user(kbuf, buf, len)) {
			kfree(kbuf);
			return -EFAULT;
		}
		
		ret = (*(mtd->write))(mtd, *ppos, len, &retlen, kbuf);
		if (!ret) {
			*ppos += retlen;
			total_retlen += retlen;
			count -= retlen;
			buf += retlen;
		}
		else {
			kfree(kbuf);
			return ret;
		}
		
		kfree(kbuf);
	}

	return total_retlen;
} /* mtd_write */

/*======================================================================

    IOCTL calls for getting device parameters.

======================================================================*/
static void mtd_erase_callback (struct erase_info *instr)
{
	wake_up((wait_queue_head_t *)instr->priv);
}

static int mtd_ioctl(struct inode *inode, struct file *file,
		     u_int cmd, u_long arg)
{
	struct mtd_info *mtd = (struct mtd_info *)file->private_data;
	int ret = 0;
	u_long size;
	
	DEBUG(MTD_DEBUG_LEVEL0, "MTD_ioctl\n");

	size = (cmd & IOCSIZE_MASK) >> IOCSIZE_SHIFT;
	if (cmd & IOC_IN) {
		ret = verify_area(VERIFY_READ, (char *)arg, size);
		if (ret) return ret;
	}
	if (cmd & IOC_OUT) {
		ret = verify_area(VERIFY_WRITE, (char *)arg, size);
		if (ret) return ret;
	}
	
	switch (cmd) {
	case MEMGETREGIONCOUNT:
		if (copy_to_user((int *) arg, &(mtd->numeraseregions), sizeof(int)))
			return -EFAULT;
		break;

	case MEMGETREGIONINFO:
	{
		struct region_info_user ur;

		if (copy_from_user(	&ur, 
					(struct region_info_user *)arg, 
					sizeof(struct region_info_user))) {
			return -EFAULT;
		}

		if (ur.regionindex >= mtd->numeraseregions)
			return -EINVAL;
		if (copy_to_user((struct mtd_erase_region_info *) arg, 
				&(mtd->eraseregions[ur.regionindex]),
				sizeof(struct mtd_erase_region_info)))
			return -EFAULT;
		break;
	}

	case MEMGETINFO:
		if (copy_to_user((struct mtd_info *)arg, mtd,
				 sizeof(struct mtd_info_user)))
			return -EFAULT;
		break;

	case MEMERASE:
	{
		struct erase_info *erase=kmalloc(sizeof(struct erase_info),GFP_KERNEL);
		if (!erase)
			ret = -ENOMEM;
		else {
			wait_queue_head_t waitq;
			DECLARE_WAITQUEUE(wait, current);

			init_waitqueue_head(&waitq);

			memset (erase,0,sizeof(struct erase_info));
			if (copy_from_user(&erase->addr, (u_long *)arg,
					   2 * sizeof(u_long))) {
				kfree(erase);
				return -EFAULT;
			}
			erase->mtd = mtd;
			erase->callback = mtd_erase_callback;
			erase->priv = (unsigned long)&waitq;
			
			ret = mtd->erase(mtd, erase);
			if (!ret) {
				set_current_state(TASK_UNINTERRUPTIBLE);
				add_wait_queue(&waitq, &wait);
				if (erase->state != MTD_ERASE_DONE &&
				    erase->state != MTD_ERASE_FAILED)
					schedule();
				remove_wait_queue(&waitq, &wait);
				set_current_state(TASK_RUNNING);

				ret = (erase->state == MTD_ERASE_FAILED)?-EIO:0;
			}
			kfree(erase);
		}
		break;
	}

	case MEMWRITEOOB:
	{
		struct mtd_oob_buf buf;
		void *databuf;
		ssize_t retlen;
		
		if (copy_from_user(&buf, (struct mtd_oob_buf *)arg, sizeof(struct mtd_oob_buf)))
			return -EFAULT;
		
		if (buf.length > 0x4096)
			return -EINVAL;

		if (!mtd->write_oob)
			ret = -EOPNOTSUPP;
		else
			ret = verify_area(VERIFY_READ, (char *)buf.ptr, buf.length);

		if (ret)
			return ret;

		databuf = kmalloc(buf.length, GFP_KERNEL);
		if (!databuf)
			return -ENOMEM;
		
		if (copy_from_user(databuf, buf.ptr, buf.length)) {
			kfree(databuf);
			return -EFAULT;
		}

		ret = (mtd->write_oob)(mtd, buf.start, buf.length, &retlen, databuf);

		if (copy_to_user((void *)arg + sizeof(u_int32_t), &retlen, sizeof(u_int32_t)))
			ret = -EFAULT;

		kfree(databuf);
		break;

	}

	case MEMREADOOB:
	{
		struct mtd_oob_buf buf;
		void *databuf;
		ssize_t retlen;

		if (copy_from_user(&buf, (struct mtd_oob_buf *)arg, sizeof(struct mtd_oob_buf)))
			return -EFAULT;
		
		if (buf.length > 0x4096)
			return -EINVAL;

		if (!mtd->read_oob)
			ret = -EOPNOTSUPP;
		else
			ret = verify_area(VERIFY_WRITE, (char *)buf.ptr, buf.length);

		if (ret)
			return ret;

		databuf = kmalloc(buf.length, GFP_KERNEL);
		if (!databuf)
			return -ENOMEM;
		
		ret = (mtd->read_oob)(mtd, buf.start, buf.length, &retlen, databuf);

		if (copy_to_user((void *)arg + sizeof(u_int32_t), &retlen, sizeof(u_int32_t)))
			ret = -EFAULT;
		else if (retlen && copy_to_user(buf.ptr, databuf, retlen))
			ret = -EFAULT;
		
		kfree(databuf);
		break;
	}

	case MEMLOCK:
	{
		unsigned long adrs[2];

		if (copy_from_user(adrs ,(void *)arg, 2* sizeof(unsigned long)))
			return -EFAULT;

		if (!mtd->lock)
			ret = -EOPNOTSUPP;
		else
			ret = mtd->lock(mtd, adrs[0], adrs[1]);
		break;
	}

	case MEMUNLOCK:
	{
		unsigned long adrs[2];

		if (copy_from_user(adrs, (void *)arg, 2* sizeof(unsigned long)))
			return -EFAULT;

		if (!mtd->unlock)
			ret = -EOPNOTSUPP;
		else
			ret = mtd->unlock(mtd, adrs[0], adrs[1]);
		break;
	}

#ifdef NEW_FLASH_WRITE
	case MEMWRITESET:
	{
		/* Allocate buffer to hold the new image. */
		unsigned int size;
		unsigned int roundedSize;

		if (copy_from_user(&size, (void *)arg, sizeof(unsigned int)))
			return -EFAULT;
		roundedSize = ROUNDUP(size, mtd->erasesize);
		brcmWriteBuffer.startPtr = vmalloc(roundedSize);
		if (!(brcmWriteBuffer.startPtr))
			return -ENOMEM;
		brcmWriteBuffer.copyPtr = brcmWriteBuffer.startPtr;
		brcmWriteBuffer.commitPtr = brcmWriteBuffer.startPtr;
		brcmWriteBuffer.endPtr = brcmWriteBuffer.startPtr + roundedSize;
		brcmWriteBuffer.length = size;
		break;
	}

	case MEMWRITECOMMIT:
	{
		/* Start kernel thread for commiting the writes. */
		kernel_thread((void *)mtd_write_commit, (void *)file, 0);
		ret = 0;
		break;
	}

	case MEMWRITEABORT:
	{
		if(brcmWriteBuffer.startPtr != NULL) {
			vfree((void *)brcmWriteBuffer.startPtr);
		}
		break;
	}

#endif
	case MEMSETDEBUGLEVEL:
	{
		mtd_debug_level = 1;
		ret = 0;
		break;
	}
	
	default:
		DEBUG(MTD_DEBUG_LEVEL0, "Invalid ioctl %x (MEMGETINFO = %x)\n", cmd, MEMGETINFO);
		ret = -ENOTTY;
	}

	return ret;
} /* memory_ioctl */

static struct file_operations mtd_fops = {
	owner:		THIS_MODULE,
	llseek:		mtd_lseek,     	/* lseek */
	read:		mtd_read,	/* read */
	write: 		mtd_write, 	/* write */
	ioctl:		mtd_ioctl,	/* ioctl */
	open:		mtd_open,	/* open */
	release:	mtd_close,	/* release */
};


#ifdef CONFIG_DEVFS_FS
/* Notification that a new device has been added. Create the devfs entry for
 * it. */

static void mtd_notify_add(struct mtd_info* mtd)
{
	char name[8];

	if (!mtd)
		return;

	sprintf(name, "%d", mtd->index);
	devfs_rw_handle[mtd->index] = devfs_register(devfs_dir_handle, name,
			DEVFS_FL_DEFAULT, MTD_CHAR_MAJOR, mtd->index*2,
			S_IFCHR | S_IRUGO | S_IWUGO,
			&mtd_fops, NULL);

	sprintf(name, "%dro", mtd->index);
	devfs_ro_handle[mtd->index] = devfs_register(devfs_dir_handle, name,
			DEVFS_FL_DEFAULT, MTD_CHAR_MAJOR, mtd->index*2+1,
			S_IFCHR | S_IRUGO | S_IWUGO,
			&mtd_fops, NULL);
}

static void mtd_notify_remove(struct mtd_info* mtd)
{
	if (!mtd)
		return;

	devfs_unregister(devfs_rw_handle[mtd->index]);
	devfs_unregister(devfs_ro_handle[mtd->index]);
}
#endif

#ifdef NEW_FLASH_WRITE
#ifdef CONFIG_PROC_FS
static int mtd_write_proc_read(char *page, char **start, off_t off, int count, 
	int *eof, void *data)
{
	char *out = page;
	int len;
	int retVal = 0;

	strcpy(out, brcmWriteBuffer.bytesPrgmed);
	out += strlen(brcmWriteBuffer.bytesPrgmed);
	len = out - page - off;
	if (len < count) {
		*eof = 1;
		if (len <= 0) {
			retVal = 0;
			goto err;
		}
	} else {
		len = count;
	}

	*start = page + off;
	retVal = len;
	
err:
	return retVal;
}

static int __init mtd_write_create_procfs(void)
{
	struct proc_dir_entry *ent;

	ent = create_proc_entry("mtd_bytes_written", S_IFREG|S_IRUGO|S_IWUSR, 0);
	if (!ent) {
		return -1;
	}
	ent->nlink = 1;
	ent->read_proc = mtd_write_proc_read;
#ifdef MODULE
	ent->owner = THIS_MODULE;
#endif

	return 0;
}

int mtd_write_remove_procfs(void)
{
	remove_proc_entry("mtd_bytes_written", 0);
	return 0;
}
#endif
#endif

static int __init init_mtdchar(void)
{
#ifdef CONFIG_DEVFS_FS
	if (devfs_register_chrdev(MTD_CHAR_MAJOR, "mtd", &mtd_fops))
	{
		printk(KERN_NOTICE "Can't allocate major number %d for Memory Technology Devices.\n",
		       MTD_CHAR_MAJOR);
		return -EAGAIN;
	}

	devfs_dir_handle = devfs_mk_dir(NULL, "mtd", NULL);

	register_mtd_user(&notifier);
#else
	if (register_chrdev(MTD_CHAR_MAJOR, "mtd", &mtd_fops))
	{
		printk(KERN_NOTICE "Can't allocate major number %d for Memory Technology Devices.\n",
		       MTD_CHAR_MAJOR);
		return -EAGAIN;
	}
#endif

#ifdef NEW_FLASH_WRITE
#ifdef CONFIG_PROC_FS
	mtd_write_create_procfs();
#endif
#endif
	return 0;
}

static void __exit cleanup_mtdchar(void)
{
#ifdef CONFIG_DEVFS_FS
	unregister_mtd_user(&notifier);
	devfs_unregister(devfs_dir_handle);
	devfs_unregister_chrdev(MTD_CHAR_MAJOR, "mtd");
#else
	unregister_chrdev(MTD_CHAR_MAJOR, "mtd");
#endif

#ifdef NEW_FLASH_WRITE
#ifdef CONFIG_PROC_FS
	mtd_write_remove_procfs();
#endif
#endif
}

module_init(init_mtdchar);
module_exit(cleanup_mtdchar);


MODULE_LICENSE("GPL");
MODULE_AUTHOR("David Woodhouse <dwmw2@infradead.org>");
MODULE_DESCRIPTION("Direct character-device access to MTD devices");
