/*
 * This is the interface to the Broadcom natural language translation
 * management mechanism.
 *
 * Copyright 2004, Broadcom Corporation
 * All Rights Reserved.
 *
 * This is UNPUBLISHED PROPRIETARY SOURCE CODE of Broadcom Corporation;
 * the contents of this file may not be disclosed to third parties, copied
 * or duplicated in any form, in whole or in part, without the prior
 * written permission of Broadcom Corporation.
 */

#ifndef TRANSLATION_H
#define TRANSLATION_H

#include <stddef.h>


/*
 *  This is a part of the Broadcom natural language translation management
 *  mechanism.  This mechanism assists in allowing software to be configurable
 *  to use different natural languages for different end users.  The mechanism
 *  allows text embedded in C code to be replaced with text in different
 *  languages at run time, without requiring the C code to be changed when
 *  support for new languages is added.  Instead, a list of strings that need
 *  translation is provided to allow the translation into multiple languages,
 *  and files containing those translations are compiled together into a
 *  translation resource library to be used at run time.
 *
 *  If a shared library or executable is to use this system, a modification is
 *  required in the Makefile to build that shared library or executable.  That
 *  modification is to run the shell command
 *
 *      build_translation_array_start <modulename> > translation_array_start.c
 *
 *  where <modulename> is the name of the shared library or executable and then
 *  compile translation_array_start.c to translation_array_start.o using the
 *  normal build rules for C programs, then link translation_array_start.o into
 *  the shared library or executable before any of the other object files.
 *  Also, on the command line for compiling each of the other object files
 *  -DTRANSLATION_MODULE_NAME=<modulename> should be added.  Finally, the
 *  module should be linked against libtranslation.so.
 *
 *  Once the module is being built this way, any string literals in the code
 *  that are to be translated should be wrapped in the TRANSLATE() macro.  Any
 *  C files that use the TRANSLATE() macro will have to include
 *  "translation/translation.h" (this file), directly or indirectly, to get
 *  access to this macro.
 *
 *  That's all it takes.  The rest is automatic.
 */

/*
 *  Note that the current_language_code variable, the language_reference_t
 *  structure, the setup_translation_pointers() function, and the
 *  TRANSLATION_STRINGIZE(), TRANSLATION_CONCAT(), and TRANSLATE_INTERNAL()
 *  macros should not be used directly.  They are declared here so that they
 *  can be used in the TRANSLATE() macro.
 */

extern int current_language_code;

struct language_reference_t
{
	char *text;
	char ***list_pointer;
};

extern void setup_translation_pointers(const char *module_name,
		ptrdiff_t array_element_offset, char ***translation_list_location);

#define TRANSLATION_STRINGIZE(arg)  #arg

#define TRANSLATION_CONCAT(arg1, arg2) arg1 ## arg2

#define TRANSLATE_INTERNAL(text, module_name) \
	({ \
		static char **translation_list = NULL; \
		static char reference[] __attribute__\
				((section(".translation_characters"))) = text; \
		static struct language_reference_t element __attribute__\
				((section(".translation_array"))) = \
		{ reference, &translation_list }; \
		extern struct language_reference_t \
				TRANSLATION_CONCAT(module_name, _translation_array_start)[]; \
		\
		if (translation_list == NULL) \
		{ \
			setup_translation_pointers( \
					TRANSLATION_STRINGIZE(module_name), \
					((&element) - \
					 TRANSLATION_CONCAT(module_name, \
										_translation_array_start) - 1), \
					&translation_list); \
		} \
		translation_list[current_language_code]; \
	})

#define TRANSLATE(text) TRANSLATE_INTERNAL(text, TRANSLATION_MODULE_NAME)


#endif /* TRANSLATION_H */
