/*
 * QoS helper functions for Samba
 *
 * Copyright 2003, Broadcom Corporation
 * All Rights Reserved.
 *
 *
 * $Id: qos.c,v 1.10 2003/11/03 $
 */

#include "includes.h"


typedef struct _file_ext
{
	char extension[MAX_FILE_EXT_LEN+1]; /* Add one byte for '\0' */
} FILE_EXT;


typedef struct _file_ext_group
{
    int NumOfTypes;
	FILE_EXT FileExtTypes[MAX_FILE_TYPES];

} FILE_EXT_GROUP;


typedef struct _file_ext_table
{
    FILE_EXT_GROUP  LevelOneFileExts;
	FILE_EXT_GROUP  LevelTwoFileExts;
} FILE_EXT_TABLE;


static FILE_EXT_TABLE QoSFileExtTable;

BOOL qosEnabled;


/****************************************************************************
 Parses the input string to search for the file extensions and store them
 into the caller supplied fileGroup structure.
****************************************************************************/
static void parse_file_ext_types(char *str, FILE_EXT_GROUP *fileGroup)
{
   int i = 0;
   int ext_len = 0;
   int numOfType = 0;


   fileGroup->NumOfTypes = 0;

   if (str == NULL)
   {
       /* NULL pointer, return immediately
        */
	   return;
   }

   /* search thru the string
    */
   while(1)
   {
       switch(str[i])
       {
       case '\0':
           if (ext_len > 0)
           {
               fileGroup->FileExtTypes[numOfType].extension[ext_len] = '\0';
               numOfType++;
           }

           /* Set number of file types
            */
           fileGroup->NumOfTypes = numOfType;
           return;
           break;

       case ' ':
           if (ext_len > 0)
           {
               /* end of one file extension type
                */
               fileGroup->FileExtTypes[numOfType].extension[ext_len] = '\0';
               numOfType++;

               /* prepare for searching next file type
                */
               ext_len = 0;
           }
           break;

       default:
           if (ext_len <= 0)
           {
               /* beginning of a new file type
                */
               if (numOfType >= MAX_FILE_TYPES)
               {
                    /* printf("Too many file extension types \n"); */

                    /* Set number of file types
                     */
                    fileGroup->NumOfTypes = numOfType;
                    return;
               }
           }

           /* turn letter to uppercase and save it.
            */
           if (ext_len < MAX_FILE_EXT_LEN)  /* reserve 1 byte for '\0' */
           {
                fileGroup->FileExtTypes[numOfType].extension[ext_len] = toupper(str[i]);
                ext_len ++;
           }
           else
           {
                DEBUG(0, ("parse_file_ext_types: File extension too long\n"));

           }

           break;

       }

       i++;
   }
}



/****************************************************************************
 Initialise the QoS file extension look up table.
****************************************************************************/
void qos_init(void)
{
    qosEnabled = lp_qos_enable();

	if (!qosEnabled)
	{
         /* no need to prepare the file extension table
          */
         DEBUG(0,("QoS is disabled\n"));
		 return;
	}

	DEBUG(0,("QoS is enabled\n"));


	/* Parse the level 1 file extensions and save them into the table
	 */
	parse_file_ext_types(lp_level1_file_extension(), &QoSFileExtTable.LevelOneFileExts);

	/* Parse the level 2 file extensions and save them into the table
	 */
	parse_file_ext_types(lp_level2_file_extension(), &QoSFileExtTable.LevelTwoFileExts);


#if 0
    {
        int i;

		for (i = 0; i < QoSFileExtTable.LevelOneFileExts.NumOfTypes; i++)
		{
            DEBUG(0, ("Level one file ext %d: %s\n", i, QoSFileExtTable.LevelOneFileExts.FileExtTypes[i].extension));
		}

    	for (i = 0; i < QoSFileExtTable.LevelTwoFileExts.NumOfTypes; i++)
	    {
		    DEBUG(0, ("Level two file ext %d: %s\n", i, QoSFileExtTable.LevelTwoFileExts.FileExtTypes[i].extension));
	    }


    }
#endif

    return;
}


/****************************************************************************
 Parse the file name to get the extension string.
****************************************************************************/

static int get_file_extension(char *fileName, char *extension, int extBufSize)
{
    int dotPos = -1;
    int fileNameLen = 0;

    /* find the last . character
     */
    while(fileName[fileNameLen] != '\0')
    {
         if (fileName[fileNameLen] == '.')
         {
             dotPos = fileNameLen;
         }

         fileNameLen ++;
    }

    if(dotPos == -1)
    {
        /* no extension
         */
        extension[0] = '\0';
        return (0);
    }

    if ((fileNameLen - dotPos) > extBufSize)
    {
         /* file extension too long
          */
         return (-1);
    }

    memcpy(extension, &fileName[dotPos+1], (fileNameLen - dotPos));

    return(0);
}


/****************************************************************************
 Map the file extension string to file type.
****************************************************************************/

static file_ext_type_enum file_extension_type(char *extension)
{
    file_ext_type_enum fileType = OTHER_FILE_EXT;
	int found = 0;
	int i;


        /* Compare the caller supplied file extension with types stored in our
         * file extension table
         */
    /* Compare with level one file types
     */
    for(i = 0; i < QoSFileExtTable.LevelOneFileExts.NumOfTypes; i++)
    {
        if(StrCaseCmp (extension, QoSFileExtTable.LevelOneFileExts.FileExtTypes[i].extension) == 0)
        {
            /* it is a level one type file
             */
            fileType = LEVEL_ONE_FILE_EXT;
			found = 1;
			break;
        }

    }

	if (found == 0)
	{
        /* Compare with level two file types
         */
        for(i = 0; i < QoSFileExtTable.LevelTwoFileExts.NumOfTypes; i++)
        {
            if(StrCaseCmp (extension, QoSFileExtTable.LevelTwoFileExts.FileExtTypes[i].extension) == 0)
            {
                /* it is a level two type file
                 */
                fileType = LEVEL_TWO_FILE_EXT;
			    found = 1;
			    break;
            }
		}
	}

    return (fileType);

}



/****************************************************************************
  Update the qos relevant openned file count
****************************************************************************/

void qos_increment_file_count(char *fileName, connection_struct *conn)
{
    char extension[MAX_FILE_EXT_LEN+1];
    file_ext_type_enum extType;

	/* figureout the file extension
     */
    DEBUG(10, (" qos_increment_file_count: open file %s\n", fileName));
	if (get_file_extension(fileName, extension, sizeof(extension)) ==0)
	{
		DEBUG(10, (" qos_increment_file_count: file extension %s\n", extension));

        extType = file_extension_type(extension);

		switch(extType)
		{
			case LEVEL_ONE_FILE_EXT:
				DEBUG(10, ("qos_increment_file_count: open a level one file \n"));
				conn->num_level1_files_open++;
				break;


            case LEVEL_TWO_FILE_EXT:
	            DEBUG(10, ("qos_increment_file_count: open a level two file \n"));
	            conn->num_level2_files_open++;
	            break;

            case OTHER_FILE_EXT:
			default:
				break;
		}

	}
}


/****************************************************************************
  Update the qos relevant openned file count
****************************************************************************/

void qos_decrement_file_count(char *fileName, connection_struct *conn)
{

    char extension[MAX_FILE_EXT_LEN+1];
    file_ext_type_enum extType;


    DEBUG(10, (" qos_decrement_file_count: close file %s\n", fileName));

    /* figureout the file extension
     */
	if (get_file_extension(fileName, extension, sizeof(extension)) ==0)
	{
		DEBUG(10, (" qos_decrement_file_count: file extension %s\n", extension));

        extType = file_extension_type(extension);

		switch(extType)
		{
			case LEVEL_ONE_FILE_EXT:
				DEBUG(10, ("qos_decrement_file_count: close a level one file \n"));
				conn->num_level1_files_open--;
				break;


            case LEVEL_TWO_FILE_EXT:
	            DEBUG(10, ("qos_decrement_file_count: close a level two file \n"));
	            conn->num_level2_files_open--;
	            break;

            case OTHER_FILE_EXT:
			default:
				break;
		}
	}
}

