/*
   Use Samba mechanisms to authenticate a domain user.

   This program is written to cooperate with the Unix SMB/CIFS implementation.

   Copyright (C) Broadcom Corporation              2004

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "includes.h"


#define max(a,b) ((a)>(b)?(a):(b))

static void
setup()
{
	pstring configfile;
	pstrcpy(configfile, dyn_CONFIGFILE);

	if (!lp_load(configfile,True,False,False)) {
		fprintf(stderr, "Can't load %s - run testparm to debug it\n", 
			dyn_CONFIGFILE);
		exit(2);
	}

	setup_logging("smbauth", True);
}


/*********************************************************
 Start here.
**********************************************************/
int main(int argc, char **argv)
{
	int result = 0;
	const char * user_name;
	const char * password;
	const char * domain_name;
	fstring pdc_name;
	const char * local_machine_name;
	size_t pdc_name_len;
	size_t machine_name_len;
	size_t max_name_len;
	size_t len;
	const char * format = "/usr/local/samba/bin/smbclient -L %s -U%s%%%s -c quit > /dev/null 2>/dev/null";
	struct in_addr pdc_addr;
	const struct passwd * pwent;


	/*
		Handle command line arguments.
	 */
	if (argc != 3) 
	{
		printf("Usage: %s <user> <password>\n", argv[0]);
		printf("\tUse a colon (':') to indicate a null password.\n");
		exit(1);
	}
	user_name = argv[1];
	password = argv[2];
	if (strcmp(password, ":") == 0)
		password = NULL;


	/*
		Do some Samba related initialization.
	 */
	setup();
	load_interfaces();

	/*
	 * Set the machine NETBIOS name if not already
	 * set from the config file. 
	 */ 
	if (!init_names())
		return 1;


	/*
		Get information about the PDC.
	 */
	domain_name = lp_workgroup();
	if (!get_dc_name(domain_name, NULL, pdc_name, &pdc_addr))
	{
		fprintf(stderr, "ERROR: Cannot get the PDC name for domain %s.\n", domain_name);
		return 2;	
	}

	//printf("smbauth: domain=%s pdc=%s\n", domain_name, pdc_name);

//	local_machine_name = get_local_machine_name();
//	if (local_machine_name == NULL)
//	{
//		fprintf(stderr, "ERROR: Cannot get the local machine name.\n");
//		return 3;	
//	}


	/*
		Allocate buffer to hold the command.
	 */
	pdc_name_len = strlen(pdc_name);
	machine_name_len = strlen(local_machine_name);
	max_name_len = 16; //max(pdc_name_len, machine_name_len);

	len = strlen(format) + max_name_len + strlen(user_name) + strlen(password) + 32;
	char *cmd = (char *)malloc(len);
	if (cmd == NULL)
	{
		fprintf(stderr, "ERROR: Cannot allocate memory for command string.\n");
		free(cmd);
		return 4;	
	}


	/*
		Authenticate the user by going to the domain server.

		For unknown reasons, smbclient run against the local machine does not
		properly authenticate users.  So, running it against the domain controller
		does.
	 */
	smb_snprintf(cmd, len, format, inet_ntoa(pdc_addr), user_name, password);
	//printf("smbauth: cmd='%s'\n", cmd);
	result = smbrun(cmd, NULL);
	if (result != 0)
	{
		fprintf(stderr, "ERROR: Could not authenticate %s on PDC %s.\n", user_name, pdc_name);
		free(cmd);
		return 5;
	}


	/*
		This domain member keeps an entry in its own /etc/passwd file.  This
		is not for authentication.  It merely holds the uid and gid for this 
		user; Samba always authenticates against the domain controller.
		Because the calls to smbclient for the purposes of creating this
		/etc/passwd entry are numerous and slow, we short-circuit this
		procedure if the user is already in our local password file.
	 */
	pwent = getpwnam(user_name);
	if (pwent != NULL)
	{
		//printf("smbauth: short-circuited caching for %s\n", pwent->pw_name);
		free(cmd);
		return 0;
	}
	//printf("smbauth: caching uid/gid for %s\n", user_name);


	/*
		Cache the account information from the domain server to this domain 
		member.  This is important because the caching process assigns the
		local UID and GID to this user.

		For unknown reasons, the password must be null.
	 */
	smb_snprintf(cmd, len, format, "127.0.0.1"/*local_machine_name*/, user_name, "" /*password*/);
	//printf("smbauth: cmd='%s'\n", cmd);
	result = smbrun(cmd, NULL);
	if (result != 0)
	{
		fprintf(stderr, "ERROR: Could not cache %s on local machine.\n", user_name);
		free(cmd);
		return 6;
	}


	/*
		Oh, this just keeps getting better.  Sometimes, smbcontrol won't work
		with a password, and sometimes it won't work without one.   So,
		we're tring all the options.  To make this fun, smbcontrol never gives
		me an error code, so I won't know which if any actually works.  Grrrr.
		
		Try a second time to cache the account information from the domain 
		server to this domain member.  This is important because the caching 
		process assigns the local UID and GID to this user.
	 */
	smb_snprintf(cmd, len, format, "127.0.0.1"/*local_machine_name*/, user_name, password);
	//printf("smbauth: cmd='%s'\n", cmd);
	result = smbrun(cmd, NULL);
	if (result != 0)
	{
		fprintf(stderr, "ERROR: Could not cache %s on local machine (2nd try).\n", user_name);
		free(cmd);
		return 7;
	}


	free(cmd);
	return result;
}
