/*
 * Driver to control the Gout general purpose output for the Acard ATP865 IDE
 * controller chip.
 *
 * Copyright 2001-2004, Broadcom Corporation
 * All Rights Reserved.
 *
 * THIS SOFTWARE IS OFFERED "AS IS", AND BROADCOM GRANTS NO WARRANTIES OF ANY
 * KIND, EXPRESS OR IMPLIED, BY STATUTE, COMMUNICATION OR OTHERWISE. BROADCOM
 * SPECIFICALLY DISCLAIMS ANY IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
 * FOR A SPECIFIC PURPOSE OR NONINFRINGEMENT CONCERNING THIS SOFTWARE.
 *
 * $Id$
*/

#include <linux/module.h>
#include <linux/init.h>
#include <linux/fs.h>
#include <linux/proc_fs.h>
#include <linux/pci.h>
#include <linux/ide.h>
#include <linux/config.h>
#include <linux/sched.h>
#include <linux/miscdevice.h>
#include <asm/uaccess.h>
#include <asm/signal.h>

#include <typedefs.h>
#include <osl.h>
#include <bcmendian.h>
#include <bcmutils.h>
#include <bcmdevs.h>
#include <bcmnvram.h>
#include <bcmboard.h>
#include <sbutils.h>
#include <sbchipc.h>
#include <sbconfig.h>



#define AECGPO_MAX_NAME_SIZE	80

static char aecgpoName[] = "gpio_aec865";
static int aecgpoPolarity = 0;	/* 0 - normal, 1 - reverse */
static struct pci_dev aecPCIdev;

static int scan_bus_for_aec(struct pci_bus *bus)
{
    u32	devfn, device_no;

    devfn = 0;
    for (device_no=0; device_no < (0x100/8); device_no++) {
	struct pci_dev dev0;
	u16 vendor_0, device_0;

	memset(&dev0, 0, sizeof(dev0));
	dev0.bus = bus;
	dev0.sysdata = bus->sysdata;

	dev0.devfn = devfn;
	pci_read_config_word(&dev0, PCI_VENDOR_ID, &vendor_0);
	pci_read_config_word(&dev0, PCI_DEVICE_ID, &device_0);

	if((vendor_0 == PCI_VENDOR_ID_ARTOP) && 
		(device_0 = PCI_DEVICE_ID_ARTOP_ATP865)) {
		/* Found AEC device with GPO */
		memcpy(&aecPCIdev, &dev0, sizeof(struct pci_dev));
		return 1;
	}
	devfn += 8;
    }

    {
	const struct list_head *l;
	// now scan any children
	for (l = bus->children.next; l != &bus->children; l = l->next) {
	    if (scan_bus_for_aec(pci_bus_b(l)))
			return 1;
	}
    }

	return 0;
}

static int pci_scan_for_aec(void)
{
    struct pci_bus *bus;

    if (!pcibios_present()) {
        printk("No PCI bios present\n");
		return (-1);
    }

    pci_for_each_bus(bus) {
	if (scan_bus_for_aec(bus))
		return 1;
    }

	return 0;
}


#ifdef CONFIG_PROC_FS
static char proc_aecgpo_root_name[] = "miscio";

static int aecgpo_proc_read(char *page, char **start, off_t off, int count, 
	int *eof, void *data)
{
	char *out = page;
	int len;
	char readstr[] = "Reading of this GPO is not supported";

	strcpy(out, readstr);
	out += strlen(readstr);
	len = out - page - off;
	if (len < count) {
		*eof = 1;
		if (len <= 0) {
			return 0;
		}
	} else {
		len = count;
	}
	*start = page + off;

	return len;
}

static int aecgpo_proc_write(struct file *file, const char *buf, 
	unsigned long count, void *data)
{
	char *cur, lbuf[count];
	u32 val;
	int retVal;
	unsigned int addrValue;
	char value;

	if (!capable(CAP_SYS_ADMIN))
		return -EACCES;

	memset(lbuf, 0, count);
	copy_from_user(lbuf, buf, count);
	cur = lbuf;

	val = bcm_strtoul(cur, NULL, 0);

	/* Make sure value is legal (0, 1) */
	if ((val < 0) || (val > 1)) {
		retVal = -EINVAL;
		goto err;
	}

	/* Set the output value. */
	if(aecgpoPolarity) {
		/* Output is reverse polarity */
		val = !val;
	}
	
	pci_read_config_dword(&aecPCIdev, 0x20, &addrValue);
	
	addrValue &= ~0x01;
	addrValue += 2;
	
	value = IN_BYTE(addrValue);
	if (val)
		value |= 0x10;
	else
		value &= ~0x10;
	
	OUT_BYTE(value, addrValue);
	
err:
	return count;
}

static int aecgpo_config_proc_read(char *page, char **start, off_t off, int count, 
	int *eof, void *data)
{
	char *out = page;
	int len;

	out += sprintf(out, "Name=%s\nPolarity=%s\n",
			aecgpoName, (aecgpoPolarity ? "reverse":"normal"));

	len = out - page - off;
	if (len < count) {
		*eof = 1;
		if (len <= 0) {
			return 0;
		}
	} else {
		len = count;
	}
	*start = page + off;

	return len;
}

static int aecgpo_config_proc_write(struct file *file, const char *buf, 
	unsigned long count, void *data)
{
	char *cur, lbuf[count];
	char *value;

	if (!capable(CAP_SYS_ADMIN))
		return -EACCES;

	memset(lbuf, 0, count);
	copy_from_user(lbuf, buf, count);
	lbuf[count-1] = '\0';
	cur = lbuf;

	if ((value = strchr(cur, '=')) != NULL) {
		*value = '\0';
		value++;
	}

	if (strcmp(cur, "polarity") == 0) {
		if (value && *value) {
			if(strcmp(value, "reverse") == 0)
				aecgpoPolarity = 1;
			else
				aecgpoPolarity = 0;
		}
	} else {
		printk("aecgpo configuration error: Unknown configuration parameter (%s)\n", cur);
	}
	
	return count;
}

static void aecgpo_configure_name(void)
{
	char aecgpoNameVarStr[AECGPO_MAX_NAME_SIZE];
	char *aecgpoMapName;

	strcpy(aecgpoNameVarStr, aecgpoName);
	strcat(aecgpoNameVarStr, "_name");
	aecgpoMapName = nvram_get(aecgpoNameVarStr);
	if ((aecgpoMapName) && (strcmp(aecgpoMapName, "") != 0)) {
		strcpy(aecgpoName, aecgpoMapName);
	}
}

static int __init aecgpo_create_procfs(void)
{
	struct proc_dir_entry *ent;
	char buf[256];

	aecgpo_configure_name();
	snprintf(buf, sizeof(buf), "%s/%s", proc_aecgpo_root_name, aecgpoName);
	ent = create_proc_entry(buf, S_IFREG|S_IRUGO|S_IWUSR, 0);
	if (!ent) {
		/* 
		Maybe the directory /proc/miscio does not exist, 
		create and try again.
		*/
		if(!proc_mkdir(proc_aecgpo_root_name, 0)) return -1;

		ent = create_proc_entry(buf, S_IFREG|S_IRUGO|S_IWUSR, 0);
		if (!ent) return -1;
	}
	ent->nlink = 1;
	ent->read_proc = aecgpo_proc_read;
	ent->write_proc = aecgpo_proc_write;
	ent->owner = THIS_MODULE;
	/* Create entry for configuration of this aecgpo pin */
	strcat(buf, "_config");
	ent = create_proc_entry(buf, S_IFREG|S_IRUGO|S_IWUSR, 0);
	ent->nlink = 1;
	ent->read_proc = aecgpo_config_proc_read;
	ent->write_proc = aecgpo_config_proc_write;
	ent->owner = THIS_MODULE;

	return 0;
}

static int aecgpo_remove_procfs(void)
{
	char buf[256];

	snprintf(buf, sizeof(buf), "%s/%s", proc_aecgpo_root_name, aecgpoName);
	remove_proc_entry(buf, 0);
	strcat(buf, "_config");
	remove_proc_entry(buf, 0);

	return 0;
}
#endif

static int __init
aecgpo_init(void)
{
    if (pci_scan_for_aec() == 1) {
		printk(KERN_INFO 
			"Acard ATP865 device found. Adding proc entries for GPOut.\n");
		/* Found AEC device with GPO, continue */
#ifdef CONFIG_PROC_FS
		aecgpo_create_procfs();
#endif
    } 
	/* else do nothing */
    return 0;
}

static void __exit
aecgpo_exit(void)
{
#ifdef CONFIG_PROC_FS
	aecgpo_remove_procfs();
#endif
    return;
}

module_init(aecgpo_init);
module_exit(aecgpo_exit);


