/*
 * Serial bus (2-wire and 3-wire) protocol via GPIO bit banging.
 *
 * Copyright 2001-2004, Broadcom Corporation
 * All Rights Reserved.
 *
 * THIS SOFTWARE IS OFFERED "AS IS", AND BROADCOM GRANTS NO WARRANTIES OF ANY
 * KIND, EXPRESS OR IMPLIED, BY STATUTE, COMMUNICATION OR OTHERWISE. BROADCOM
 * SPECIFICALLY DISCLAIMS ANY IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
 * FOR A SPECIFIC PURPOSE OR NONINFRINGEMENT CONCERNING THIS SOFTWARE.
 *
 * On BCM947xx systems, the following serial interface devices 
 * are supported on the bus:
 * - CPLD - 3 wire serial protocol for miscellaneous IO
 * - DS1337 real time clock - 2-wire protocol
 * - LM63 temperature sensor - 2-wire SMBUS protocol
 *
 * $Id$ $DateTime$ $Author$
*/

#include <typedefs.h>
#include <osl.h>
#include <bcmendian.h>
#include <bcmutils.h>
#include <bcmdevs.h>
#include <bcmboard.h>
#include <bcmnvram.h>
#include <sbutils.h>
#include <sbchipc.h>
#include <sbconfig.h>

#include <serbus.h>


/**************************************************************
*	IMPORTANT
***************************************************************/
/*
The SDIO GPIO pin is shared between the 3-wire and the 2-wire 
interface used on the BCM94780 board.  Also, in the case of the 
3-wire interface, this is just a SDO (output), whereas for the 
2-wire interface it is SDIO (input and output).  This makes 
the logic a bit complex.  GPIO states required for 2-wire are 
default.
*/

/* ------------------------------------------------------------------------- */
/* Error and Trace Print Macros 											 */
/* ------------------------------------------------------------------------- */
#define	BCMSERBUS_ERR(s)	printk s

// #define BCMDBG 1
#if BCMDBG
#define	BCMSERBUS_TRACE(s)	printk s
#else
#define	BCMSERBUS_TRACE(s)
#endif
#define ASSERT(x)	((x) ? (void)0 : printk("BCMSERBUS_BIT: assert failed - line = %d\n", __LINE__ ))


/* ------------------------------------------------------------------------- */
/* Global Variables (shared only with the device drivers) 					 */
/* ------------------------------------------------------------------------- */
struct tasklet_struct *rtcSetTimeTaskletPtr = NULL;
struct tasklet_struct *miocpldTaskletPtr = NULL;
struct tasklet_struct *LEDBlinkTaskletPtr = NULL;


/* ------------------------------------------------------------------------- */
/* Local Variables 															 */
/* ------------------------------------------------------------------------- */
static void *serbus_sbh = NULL;
static DECLARE_MUTEX(gpio_bb_sem);

/*
BCM94780NAS hardware shares SDA and hence some locking is required. This
flag is set if CPLD is not present or not used to skip that locking.
*/
static bool cpldPresent=TRUE;	/* Do the locks by default */
/* GPIO pins used for SCL and SDA for 2-wire get from NVRAM variables */
static unsigned int gpio_2wire_sda = GPIO_2WIRE_SDA;	/* Default */
static unsigned int gpio_2wire_scl = GPIO_2WIRE_SCL;	/* Default */


/* ------------------------------------------------------------------------- */
/* Functions																	 */
/* ------------------------------------------------------------------------- */
int set3WireInitState (struct gpio_bb_hwinfo *gpiohwInfo)
{
	int sdaOrig = 0;
	int gpioVal;

	ASSERT(serbus_sbh != NULL);

	/* 
	If 3-wire, save and return the original SDA value and force 
	SDA to 0 if SDA is 1. This is seen as START by 2-wire.
	*/
	gpioVal = sb_gpioin(serbus_sbh);
	sdaOrig = (0 != (gpioVal & (1 << gpiohwInfo->sdoPinNum)));
	
	if (sdaOrig) {
		/* If SDA is high, set it to low. */
		gpioVal &= ~(1 << gpiohwInfo->sdoPinNum);
		sb_gpioout(serbus_sbh, ~0, gpioVal);
	}

	return sdaOrig;
}

void clear3WireInitState (struct gpio_bb_hwinfo *gpiohwInfo, int sdaOrig)
{
	int gpioVal;

	ASSERT(serbus_sbh != NULL);

	/* If 3-wire, restore the original SDA value. If it was 1 and 
	we are changing it to 1, this is seen as STOP by 2-wire.
	*/
	gpioVal = sb_gpioin(serbus_sbh);
	
	if (sdaOrig) {
		/* If SDA was high, set it to high. */
		gpioVal |= (1 << gpiohwInfo->sdoPinNum);
		sb_gpioout(serbus_sbh, ~0, gpioVal);
	} else {
		/* If SDA was low, set it to high. */
		gpioVal &= ~(1 << gpiohwInfo->sdoPinNum);
		sb_gpioout(serbus_sbh, ~0, gpioVal);
	}
}

void bcm_serbus_setsdadirin(void)
{
	int outenValue;

	ASSERT(serbus_sbh != NULL);
	outenValue = sb_gpioouten(serbus_sbh, 0, 0);
	BCMSERBUS_TRACE(("BCMSERBUS_BIT: GPIO OutEn Value = %x\n", outenValue));
	outenValue &= ~(1 << gpio_2wire_sda);
	sb_gpioouten(serbus_sbh, ~0, outenValue);
	outenValue = sb_gpioouten(serbus_sbh, 0, 0);
	BCMSERBUS_TRACE(("BCMSERBUS_BIT: GPIO OutEn Value after updating = %x\n", outenValue));
	OSL_DELAY(10);
}

void bcm_serbus_setsdadirout(void)
{
	int outenValue;

	ASSERT(serbus_sbh != NULL);
	outenValue = sb_gpioouten(serbus_sbh, 0, 0);
	BCMSERBUS_TRACE(("BCMSERBUS_BIT: GPIO OutEn Value = %x\n", outenValue));
	outenValue |= (1 << gpio_2wire_sda);
	sb_gpioouten(serbus_sbh, ~0, outenValue);
	outenValue = sb_gpioouten(serbus_sbh, 0, 0);
	BCMSERBUS_TRACE(("BCMSERBUS_BIT: GPIO OutEn Value after updating = %x\n", outenValue));
	OSL_DELAY(10);
}

int bcm_serbus_lock(struct gpio_bb_hwinfo *gpiohwInfo)
{
	/*
	2-wire SDA and 3-wire SDO are shared.  When a device on either the
	2-wire bus or 3-wire bus needs to access SDA(SDO) it needs to call 
	this function to get control of SDA and then call bcm_serbus_unlock 
	to relinquish control.
	*/
	int sdaOrig = 0;

	if (cpldPresent) {
		/* Disable device tasklets. */
		if (rtcSetTimeTaskletPtr != NULL) {
			tasklet_disable(rtcSetTimeTaskletPtr);
		}
		if (miocpldTaskletPtr != NULL) {
			tasklet_disable(miocpldTaskletPtr);
		}
		if (LEDBlinkTaskletPtr != NULL) {
			tasklet_disable(LEDBlinkTaskletPtr);
		}
	}
	
	/* Take device shared mutex */
	if (down_interruptible(&gpio_bb_sem)) {
		if (cpldPresent) {
			if (rtcSetTimeTaskletPtr != NULL) {
				tasklet_enable(rtcSetTimeTaskletPtr);
			}
			if (miocpldTaskletPtr != NULL) {
				tasklet_enable(miocpldTaskletPtr);
			}
			if (LEDBlinkTaskletPtr != NULL) {
				tasklet_enable(LEDBlinkTaskletPtr);
			}
		}
		return -ERESTARTSYS;
	}

	if ((gpiohwInfo != NULL) && (gpiohwInfo->device == GPIO_BB_3WIRE)) {
		sdaOrig = set3WireInitState(gpiohwInfo);
	}

	return (sdaOrig);
}

void bcm_serbus_unlock(struct gpio_bb_hwinfo *gpiohwInfo, int sdaOrig)
{
	if ((gpiohwInfo != NULL) && (gpiohwInfo->device == GPIO_BB_3WIRE)) {
		clear3WireInitState(gpiohwInfo, sdaOrig);
	}

	/* Give device shared mutex */
	up(&gpio_bb_sem);

	if (cpldPresent) {
		/* Enable device tasklets. */
		if (rtcSetTimeTaskletPtr != NULL) {
			tasklet_enable(rtcSetTimeTaskletPtr);
		}
		if (miocpldTaskletPtr != NULL) {
			tasklet_enable(miocpldTaskletPtr);
		}
		if (LEDBlinkTaskletPtr != NULL) {
			tasklet_enable(LEDBlinkTaskletPtr);
		}
	}
}

void bcm_serbus_setscl(void *data, int state)
{
	struct gpio_bb_hwinfo *gpiohwInfo = (struct gpio_bb_hwinfo *)data;
	int gpioVal = 0;

	ASSERT(serbus_sbh != NULL);

	gpioVal = sb_gpioin(serbus_sbh);

	if ((gpiohwInfo == NULL) || (gpiohwInfo->sclPinNum == GPIO_INVALID_PIN))
		return;
	
	if (state)
	{
		/* Set the clock bit */
		gpioVal |= (1 << gpiohwInfo->sclPinNum);
	}
	else
	{
		/* Clear the clock bit */
		gpioVal &= ~(1 << gpiohwInfo->sclPinNum);
	}

	/* clock */
	sb_gpioout(serbus_sbh, ~0, gpioVal);

	BCMSERBUS_TRACE(("BCMSERBUS_BIT: Clocked %d\n", state));
}

void bcm_serbus_setsda(void *data, int state)
{
	struct gpio_bb_hwinfo *gpiohwInfo = (struct gpio_bb_hwinfo *)data;
	int gpioVal = 0;

	ASSERT(serbus_sbh != NULL);

	gpioVal = sb_gpioin(serbus_sbh);

	if ((gpiohwInfo == NULL) || (gpiohwInfo->sdaPinNum == GPIO_INVALID_PIN))
		return;

	if (state)
	{
		/* Set the clock bit */
		gpioVal |= (1 << gpiohwInfo->sdaPinNum);
	}
	else
	{
		/* Clear the clock bit */
		gpioVal &= ~(1 << gpiohwInfo->sdaPinNum);
	}

	sb_gpioout(serbus_sbh, ~0, gpioVal);
}

void bcm_serbus_setscs(void *data, int state)
{
	struct gpio_bb_hwinfo *gpiohwInfo = (struct gpio_bb_hwinfo *)data;
	int gpioVal = 0;

	ASSERT(serbus_sbh != NULL);

	gpioVal = sb_gpioin(serbus_sbh);

	if ((gpiohwInfo == NULL) || (gpiohwInfo->scsPinNum == GPIO_INVALID_PIN))
		return;
	
	if (state)
	{
		/* Set the clock bit */
		gpioVal |= (1 << gpiohwInfo->scsPinNum);
	}
	else
	{
		/* Clear the clock bit */
		gpioVal &= ~(1 << gpiohwInfo->scsPinNum);
	}

	sb_gpioout(serbus_sbh, ~0, gpioVal);
}

void bcm_serbus_setsdo(void *data, int state)
{
	struct gpio_bb_hwinfo *gpiohwInfo = (struct gpio_bb_hwinfo *)data;

	if ((gpiohwInfo == NULL) || (gpiohwInfo->sdoPinNum == GPIO_INVALID_PIN))
		return;

	/* 2-wire SDA is shared with 3-wire SDO */
	gpiohwInfo->sdaPinNum = gpiohwInfo->sdoPinNum;
	bcm_serbus_setsda(data, state);
}

int bcm_serbus_getscl(void *data)
{
	struct gpio_bb_hwinfo *gpiohwInfo = (struct gpio_bb_hwinfo *)data;
	int gpioVal = 0;

	ASSERT(serbus_sbh != NULL);

	gpioVal = sb_gpioin(serbus_sbh);

	if ((gpiohwInfo == NULL) || (gpiohwInfo->sclPinNum == GPIO_INVALID_PIN))
		return 0;

	return (0 != (gpioVal & gpiohwInfo->sclPinNum));
}

int bcm_serbus_getsda(void *data)
{
	struct gpio_bb_hwinfo *gpiohwInfo = (struct gpio_bb_hwinfo *)data;
	int gpioVal = 0;

	ASSERT(serbus_sbh != NULL);

	gpioVal = sb_gpioin(serbus_sbh);

	if ((gpiohwInfo == NULL) || (gpiohwInfo->sdaPinNum == GPIO_INVALID_PIN))
		return 0;

	return (0 != (gpioVal & (1 << gpiohwInfo->sdaPinNum)));
}

int bcm_serbus_getsdi(void *data)
{
	struct gpio_bb_hwinfo *gpiohwInfo = (struct gpio_bb_hwinfo *)data;
	int gpioVal = 0;

	ASSERT(serbus_sbh != NULL);

	gpioVal = sb_gpioin(serbus_sbh);

	if ((gpiohwInfo == NULL) || (gpiohwInfo->sdiPinNum == GPIO_INVALID_PIN))
		return 0;

	return (0 != (gpioVal & (1 << gpiohwInfo->sdiPinNum)));
}

int bcm_serbus_init(struct gpio_bb_hwinfo *gpioHwInfo)
{
	int outenValue;

	if (!(serbus_sbh = sb_kattach()))
		return -ENODEV;

	sb_gpiosetcore(serbus_sbh);

	if (gpioHwInfo->device == GPIO_BB_2WIRE) {
		/* Setup the GPIO pins required for 2-wire bit-banging. */
		outenValue = sb_gpioouten(serbus_sbh, 0, 0);
		BCMSERBUS_TRACE(("BCMSERBUS_BIT: GPIO OutEn Value = %x\n", outenValue));
		outenValue |= GPIO_2WIRE_OUTMASK;
		sb_gpioouten(serbus_sbh, ~0, outenValue);
		outenValue = sb_gpioouten(serbus_sbh, 0, 0);
		BCMSERBUS_TRACE(("BCMSERBUS_BIT: GPIO OutEn Value after updating = %x\n", outenValue));

		/* Set default values for all GPIO pins used for bit banging. */
		bcm_serbus_setsda((void *)gpioHwInfo, 1);
		bcm_serbus_setscl((void *)gpioHwInfo, 1);
	} else if (gpioHwInfo->device == GPIO_BB_3WIRE) {
		/*
		Setup the GPIO pins required to communicate with CPLD controlling
		the misc I/O.
		*/
		outenValue = sb_gpioouten(serbus_sbh, 0, 0);
		BCMSERBUS_TRACE(("BCMSERBUS_BIT: GPIO OutEn Value = %x\n", outenValue));
		/*
		NOTE: 2-wire SDA and 3-wire SDO are shared, but the default setting
		for SDA is output, hence it is okay to do this.
		*/
		outenValue |= GPIO_3WIRE_OUTMASK;
		sb_gpioouten(serbus_sbh, ~0, outenValue);
		outenValue = sb_gpioouten(serbus_sbh, 0, 0);
		BCMSERBUS_TRACE(("BCMSERBUS_BIT: GPIO OutEn Value after updating = %x\n", outenValue));
		
		/* Set default values for these GPIO pins. */
		/* 
		Set only SCL, CS1 to be 0. Since SDO is shared with 2-wire SDA, 
		don't touch that. 
		*/
		bcm_serbus_setscs((void *)gpioHwInfo, 0);
		bcm_serbus_setscl((void *)gpioHwInfo, 0);
	}

	return 0;
}

void bcm_serbus_exit(void)
{
	ASSERT(serbus_sbh != NULL);

	sb_detach(serbus_sbh);
}


int __init bcm_serbus_mod_init(void) 
{
	int gpio_pin;

	/* Setup options based on NVRAM variables */

	if(nvram_invmatch("misc_io_mode", "bcmmiocpld")) {
		/* misc_io is not using CPLD, set the flag to FALSE. */
		cpldPresent = FALSE;
	}

	gpio_pin = getvartoint(NULL, "sda_gpio");
	if((gpio_pin >= 0) && (gpio_pin < BCM_MAX_GPIO)) {
		/* GPIO pin used for SDA is set by NVRAM */
		gpio_2wire_sda = (unsigned int)gpio_pin;
	}

	gpio_pin = getvartoint(NULL, "scl_gpio");
	if((gpio_pin >= 0) && (gpio_pin < BCM_MAX_GPIO)) {
		/* GPIO pin used for SCL is set by NVRAM */
		gpio_2wire_scl = (unsigned int)gpio_pin;
	}

	return 0;
}

void bcm_serbus_mod_exit(void) 
{
	/* This is a stub function. 
	 */
	return;
}


#ifdef MODULE
 module_init(bcm_serbus_mod_init);
 module_exit(bcm_serbus_mod_exit);
#endif
