/*
 * Copyright 2003, Broadcom Corporation
 * All Rights Reserved.
 *
 * This is UNPUBLISHED PROPRIETARY SOURCE CODE of Broadcom Corporation;
 * the contents of this file may not be disclosed to third parties, copied
 * or duplicated in any form, in whole or in part, without the prior
 * written permission of Broadcom Corporation.
 *
 * THIS SOFTWARE IS OFFERED "AS IS", AND BROADCOM GRANTS NO WARRANTIES OF ANY
 * KIND, EXPRESS OR IMPLIED, BY STATUTE, COMMUNICATION OR OTHERWISE. BROADCOM
 * SPECIFICALLY DISCLAIMS ANY IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
 * FOR A SPECIFIC PURPOSE OR NONINFRINGEMENT CONCERNING THIS SOFTWARE.
 */
/*
	The values in this header file must match on both the host and
	BroadNAS sides.
 */

#if !defined(_NAS_LOAD_H_)
#define _NAS_LOAD_H_

#include <stdio.h>
#include <stdlib.h>
#ifdef _WIN32
#include "winsock2.h"
#else
#include <stdarg.h>
#include <sys/types.h>
#include <errno.h>
#endif

/* Ports over which the protocol runs. */
#define NAS_BCAST_PORT 0x4157		/* for discovery */
#define NAS_STREAM_PORT 0x4158	/* for downloading */

#define DOWNLOAD_FILENAME	"broadnas_split.nas"	/* file to be downloaded */

/*
	The communication protocol is simple and uses clear text.  It is 
	customizable via these constants.  Each packet should have a 
 */
#define PROTOCOL_TAG			"BroadNAS"
#define PROTOCOL_SEPARATOR		":"
	/* Currently CFE limits PROTOCOL_SEPARATOR to being a single character. */

#define PROTOCOL_MSG_ANNOUNCE	"ANNC"
#define PROTOCOL_MSG_IDENTIFY	"IDFY"
#define PROTOCOL_MSG_AUTHREQ	"AREQ"
#define PROTOCOL_MSG_AUTHRSP	"ARSP"
#define PROTOCOL_MSG_DATABEGIN	"DBGN"
#define PROTOCOL_MSG_DATA		"DATA"
#define PROTOCOL_MSG_ACK		"DACK"
#define PROTOCOL_MSG_ABORT		"ABRT"
#define PROTOCOL_MSG_ERROR		"EROR"

#define PROTOCOL_SERVER_VERSION	"v1.0"
#define PROTOCOL_PIVOT_CLIENT_VERSION	"v1.0"
#define PROTOCOL_CLIENT_VERSION	PROTOCOL_PIVOT_CLIENT_VERSION /* For backwards compatibility */
#define PROTOCOL_CFE_CLIENT_VERSION	"CFE v1.0" /* version of CFE client code, not CFE itself */
#define PROTOCOL_AFFIRMATIVE	"yes"
#define PROTOCOL_NEGATIVE		"no"
#define PROTOCOL_DONE			"done"


/* Thses constants are dependent on the specifics of the protocol 
	parts above. */
#define TAG_LEN					8
#define	SEP_LEN					1
#define MSG_TYPE_LEN			4
#define LEN_LEN					8 /* length of length field */
#define HDR_LEN					(TAG_LEN + MSG_TYPE_LEN + LEN_LEN + 3*SEP_LEN)


typedef struct 
{
	char	tag[TAG_LEN];
	char	sep1[SEP_LEN];
	char	msg_type[MSG_TYPE_LEN];
	char	sep2[SEP_LEN];
	char	len[LEN_LEN];
	char	sep3[SEP_LEN];
} protocol_header;


/* Interval between announcments.  The time is given in seconds. */
#define PROTOCOL_INTERVAL_SECS	10 /* 10 seconds */

/*
	All messages must fit within a buffer of this size.  This mainly
	imposes a limit on the username/password pair.
 */
#define MAX_BUF	1024


/*
	This is the maximum chuck of file data that can be sent (or
	received) by one send/recv command.  This value must be a
	little less than MAX_BUF to allow for our protocol headers.
 */
#define MAX_DATA 512

/*
	The following functions may be useful on non-Windows machines.
 */


/*
	Format and display an error message.
 */
int
errorf(const char *fmt, ...)
{
    va_list args;
	int ret;

	fprintf(stderr, "ERROR: ");
    va_start(args, fmt);
    ret = vfprintf(stderr, fmt, args);
    va_end(args);

    return ret;
}


/*
	Helper function for parsing a protocol message.  Specifically
	this is used to parse variable content fields.  It modifies its
	input.

	The input must point to part of a message.  It finds the field
	separator, overwrites it (or part of it) with a null byte, and
	finds the beginning of the next field.  If there is no next
	field it returns NULL.

	Note: It cannot indicate if the input has any field or not.
	The caller will have to check this condition.
 */

static char *
parse_field(char * msg)
{
	char * sep;

//	sep = strstr(msg, PROTOCOL_SEPARATOR);
	sep = strchr(msg, PROTOCOL_SEPARATOR[0]); /*FIXME: Limited to one char.*/
	if (sep != NULL)
	{
		*sep = '\0'; /*modify input*/
		sep += strlen(PROTOCOL_SEPARATOR);
	}
	return sep;
}


/*
	These items hide some of the Windows/Linux differences.
 */
#if _WIN32
typedef DWORD ERROR_VALUE;
#define GET_ERROR WSAGetLastError()

typedef int socklen_t;

typedef HANDLE thread_id;
#else
typedef int ERROR_VALUE;
#define GET_ERROR errno

typedef int SOCKET;
#define INVALID_SOCKET -1
#define SOCKET_ERROR -1

typedef pthread_t  thread_id;
 #endif


#define SUCCESS				0
#define PKT_SOCKET_CLOSED	1024
#define PKT_FORMAT_BAD		1025
#define PKT_ERROR_MSG		1026

ERROR_VALUE
send_msg(	int sock, 
			const struct sockaddr_in *to, /* NULL for stream sockets. */
			const char *msg_type, 
			const void *msg, 
			size_t msg_len)
{
	char buf[MAX_BUF];
	char *ptr = buf + HDR_LEN;
	unsigned pkt_len = HDR_LEN + msg_len;

	/* Build the packet. */
	/* First the header. */
	sprintf(buf, "%s%s%s%s%08d%s",
					PROTOCOL_TAG, PROTOCOL_SEPARATOR,
					msg_type, PROTOCOL_SEPARATOR,
					msg_len, PROTOCOL_SEPARATOR);
	/* Then the message. */
	memcpy(ptr, msg, msg_len);


	/* Send the packet. */
	unsigned tally = 0;
	while (tally < pkt_len)
	{
		int sent;

		/*
			sendto works with both UDP and TCP.  For TCP 'to' should be NULL.
		 */
		sent = sendto(sock, buf + tally, pkt_len - tally, 0,
						(struct sockaddr *)to, sizeof(struct sockaddr));
		if (sent == -1)
		{
			return GET_ERROR;
		}
		tally += sent;
	}
	return SUCCESS;
}


/*
	Validates the packet header.  
	
	Return codes:
	 0: No error.  Returns valid length value.
	-1: Error packet.  Returns valid length value.
	-2: Unexpected packet.  Length value undefined.
	-3: Corrupt packet.  Length value undefined.
 */
#define PKT_VALID		 0
#define PKT_ERROR		-1
#define PKT_UNEXPECTED	-2
#define PKT_CORRUPT		-3

static int
validate_pkt_hdr(	protocol_header * hdr,
					const char *expected_msg,
					unsigned *p_length )
{
	int i, len;

	/* Error check the packet header. */
	if (strncmp(hdr->tag, PROTOCOL_TAG, TAG_LEN) != 0 ||
		strncmp(hdr->sep1, PROTOCOL_SEPARATOR, SEP_LEN) != 0 ||
		strncmp(hdr->sep2, PROTOCOL_SEPARATOR, SEP_LEN) != 0 ||
		strncmp(hdr->sep3, PROTOCOL_SEPARATOR, SEP_LEN) != 0)
	{
		return PKT_CORRUPT;
	}

	for (i=0; i<LEN_LEN; i++)
		if (!isdigit(hdr->len[i]))
			return PKT_CORRUPT;
	len = atoi(hdr->len);
	if (len > MAX_DATA)
		return PKT_CORRUPT;

	assert(p_length != NULL);
	*p_length = len;

	if (strncmp(hdr->msg_type, expected_msg, MSG_TYPE_LEN) != 0)
		if (strncmp(hdr->msg_type, PROTOCOL_MSG_ERROR, MSG_TYPE_LEN) != 0)
			return PKT_UNEXPECTED;
		else
			return PKT_ERROR;

	return PKT_VALID;
}

/*
	Receives a packet on a stream socket.  This function frames the
	data and error checks the packet header.  It returns only the
	"message" bytes.

	This function only works with stream sockets;
 */
ERROR_VALUE
recv_msg(	int sock, 
			const char *expected_msg, 
			char msg[], 
			unsigned *msg_len )
{
	char buf[MAX_BUF];
	unsigned len = 0;
	int parse_result;
	unsigned pkt_len;

	assert(msg_len != NULL);


	/* Read enough of the packet to get started. */
	while (len < HDR_LEN)
	{
		int recvd;
		
		recvd = recv(sock, buf + len, HDR_LEN - len, 0);
		if (recvd == 0)
			return PKT_SOCKET_CLOSED;
		if (recvd == -1)
		{
			return GET_ERROR;
		}
		len += recvd;
	}

	parse_result = validate_pkt_hdr((protocol_header *)buf, expected_msg, msg_len);
	if (parse_result != PKT_VALID && parse_result != PKT_ERROR)
		return PKT_FORMAT_BAD;

	/* Read message portion of packet. */
	pkt_len = HDR_LEN + *msg_len;
	while (len < pkt_len)
	{
		unsigned recvd;
		recvd = recv(sock, buf + len, pkt_len - len, 0);
		if (recvd == 0)
			return PKT_SOCKET_CLOSED;
		if (recvd == (unsigned)SOCKET_ERROR)
		{
			return GET_ERROR;
		}
		len += recvd;
	}


	/* Copy message to output buffer. */
	memcpy(msg, buf+HDR_LEN, *msg_len);


	/* If we had received an error packet, display it. */
	if (parse_result == PKT_ERROR)
	{
		fprintf(stderr, "Error message from remote machine: %s\n", msg);
		return PKT_ERROR_MSG;
	}


	return SUCCESS;
}


/*
	Reads a UDP packet.  Because it is UDP the entire packet is available,
	so we don't need to do multiple reads.  Plus, we cannot just read the
	header and wait to read the rest.
 */
ERROR_VALUE
recv_msg_from(	int sock, 
				const char *expected_msg, 
				char msg[], 
				unsigned *msg_len,
				struct sockaddr *from )
{
	char buf[MAX_BUF];
	socklen_t from_len = sizeof(*from);
	unsigned recvd;
	int parse_result;

	assert(msg_len != NULL);


	/* Read enough of the packet to get started. */
	recvd = recvfrom(sock, buf, MAX_BUF, 0, from, &from_len);
	if (recvd == 0)
		return PKT_SOCKET_CLOSED;
	if (recvd == (unsigned)SOCKET_ERROR)
	{
		return GET_ERROR;
	}

	parse_result = validate_pkt_hdr((protocol_header *)buf, expected_msg, msg_len);
	if (parse_result != PKT_VALID && parse_result != PKT_ERROR)
		return PKT_FORMAT_BAD;

	assert(HDR_LEN + *msg_len == recvd);

	/* Copy message to output buffer. */
	memcpy(msg, buf+HDR_LEN, *msg_len);


	/* If we had received an error packet, display it. */
	if (parse_result == PKT_ERROR)
	{
		fprintf(stderr, "Error message from remote machine: %s\n", msg);
		return PKT_ERROR_MSG;
	}


	return SUCCESS;
}

#endif // !defined(_NAS_LOAD_H_)
