/*
 * Copyright 2001-2004, Broadcom Corporation
 * All Rights Reserved.
 *
 * THIS SOFTWARE IS OFFERED "AS IS", AND BROADCOM GRANTS NO WARRANTIES OF ANY
 * KIND, EXPRESS OR IMPLIED, BY STATUTE, COMMUNICATION OR OTHERWISE. BROADCOM
 * SPECIFICALLY DISCLAIMS ANY IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
 * FOR A SPECIFIC PURPOSE OR NONINFRINGEMENT CONCERNING THIS SOFTWARE.
 */

#include <linux/init.h>
#include <linux/proc_fs.h>
#include <asm/mipsregs.h>
#include <asm/uaccess.h>
 
#ifdef CONFIG_BCM4780

#define NUM_PERFCNT	7
#define MAX_CMD_BUFSIZE	128
#define PERFCNT_NAME	"perfcnt"

static int
perfcnt_proc_read( char *buf,
                   char **start,
                   off_t off,
                   int count,
                   int *eof,
                   void *data )
{
	unsigned long perfcnt0  = read_c0_perfcnt0();
	unsigned long perfcnt1  = read_c0_perfcnt1();
	unsigned long perfcnt2  = read_c0_perfcnt2();
	unsigned long perfcnt3  = read_c0_perfcnt3();
	unsigned long perfctrl0 = read_c0_perfctrl0();
	unsigned long perfctrl1 = read_c0_perfctrl1();
	unsigned long perfglbl  = read_c0_perfglbl();
	
	return sprintf( buf, "%08lx %08lx %08lx %08lx %08lx %08lx %08lx\n",
			~perfcnt0, ~perfcnt1, ~perfcnt2, ~perfcnt3,
			perfctrl0, perfctrl1, perfglbl );
}

static int
perfcnt_proc_write( struct file *file,
                    const char *buffer,
                    unsigned long count, 
                    void *data )
{
	char	raw_buf[ MAX_CMD_BUFSIZE ];
	char *	bufp;
	char *	bufp_end;
	
	if (count > sizeof( raw_buf ))
	{
		printk( "perfcnt: Buffer size %lu too big\n", count );
		return -ENOMEM;
	}
		
	if (copy_from_user( raw_buf, buffer, count ))
	{
		printk( "perfcnt: Copy from user space failed\n" );
		return -EFAULT;
	}
	
	bufp = raw_buf;
	bufp_end = raw_buf + count;

	while (bufp_end - bufp >= 11)	/* 11 char fixed format: n xxxxxxxx\n */
	{
		int		reg;
		unsigned long	value;
		int		ndx;
		
		if (*bufp < '0' || *bufp > '6')
		{
			printk( "perfcnt: Invalid register selection %u\n", (unsigned)*bufp );
			return -EINVAL;
		}
		reg = *bufp++ - '0';
			
		if (*bufp != ' ' && *bufp != '\t')
		{
			printk( "perfcnt: No space or tab after register selection\n" );
			return -EINVAL;
		}
		bufp++;
			
		value = 0;
		for (ndx = 0; ndx < 8; ++ndx)
		{
			value <<= 4;
			
			if (*bufp >= '0' && *bufp <= '9')
			{
				value |= (unsigned long)(*bufp - '0');
			}
			else if (*bufp >= 'a' && *bufp <= 'f')
			{
				value |= (unsigned long)(*bufp - 'a');
			}
			else if (*bufp >= 'A' && *bufp <= 'F')
			{
				value |= (unsigned long)(*bufp - 'A');
			}
			else
			{
				printk( "perfcnt: Invalid hex character %u\n", (unsigned)*bufp );
				return -EINVAL;
			}
			
			bufp++;
		}
		
		if (*bufp != '\n')
		{
			printk( "perfcnt: No newline after value\n" );
			return -EINVAL;
		}
		bufp++;
		
		switch (reg)
		{
			case 0: write_c0_perfcnt0( value ); break;
			case 1: write_c0_perfcnt1( value ); break;
			case 2: write_c0_perfcnt2( value ); break;
			case 3: write_c0_perfcnt3( value ); break;
			case 4: write_c0_perfctrl0( value ); break;
			case 5: write_c0_perfctrl1( value ); break;
			case 6: write_c0_perfglbl( value ); break;
			default: break;
		}
	}

	return count;
}

static int __init
perfcnt_init( void )
{
	struct proc_dir_entry *proc_entry;

	proc_entry = create_proc_entry(	PERFCNT_NAME,
					S_IFREG | S_IRUGO | S_IWUGO,
					NULL );
	if (proc_entry == NULL)
		return -ENOMEM;

	proc_entry->read_proc = perfcnt_proc_read;
	proc_entry->write_proc = perfcnt_proc_write;
	proc_entry->data = NULL;
	return 0;
}

static void __exit
perfcnt_exit( void )
{
	remove_proc_entry( PERFCNT_NAME, NULL );
}

module_init(perfcnt_init);
module_exit(perfcnt_exit);

#endif /* CONFIG_BCM4780 */
