
/************************************************************************
 *
 *  syscon_platform.c
 *
 *  Platform specific parts of SYSCON module (except TTY related parts)
 *
 * ######################################################################
 *
 * mips_start_of_legal_notice
 * 
 * Copyright (c) 2003 MIPS Technologies, Inc. All rights reserved.
 *
 *
 * Unpublished rights (if any) reserved under the copyright laws of the
 * United States of America and other countries.
 *
 * This code is proprietary to MIPS Technologies, Inc. ("MIPS
 * Technologies"). Any copying, reproducing, modifying or use of this code
 * (in whole or in part) that is not expressly permitted in writing by MIPS
 * Technologies or an authorized third party is strictly prohibited. At a
 * minimum, this code is protected under unfair competition and copyright
 * laws. Violations thereof may result in criminal penalties and fines.
 *
 * MIPS Technologies reserves the right to change this code to improve
 * function, design or otherwise. MIPS Technologies does not assume any
 * liability arising out of the application or use of this code, or of any
 * error or omission in such code. Any warranties, whether express,
 * statutory, implied or otherwise, including but not limited to the implied
 * warranties of merchantability or fitness for a particular purpose, are
 * excluded. Except as expressly provided in any written license agreement
 * from MIPS Technologies or an authorized third party, the furnishing of
 * this code does not give recipient any license to any intellectual
 * property rights, including any patent rights, that cover this code.
 *
 * This code shall not be exported or transferred for the purpose of
 * reexporting in violation of any U.S. or non-U.S. regulation, treaty,
 * Executive Order, law, statute, amendment or supplement thereto.
 *
 * This code constitutes one or more of the following: commercial computer
 * software, commercial computer software documentation or other commercial
 * items. If the user of this code, or any related documentation of any
 * kind, including related technical data or manuals, is an agency,
 * department, or other entity of the United States government
 * ("Government"), the use, duplication, reproduction, release,
 * modification, disclosure, or transfer of this code, or any related
 * documentation of any kind, is restricted in accordance with Federal
 * Acquisition Regulation 12.212 for civilian agencies and Defense Federal
 * Acquisition Regulation Supplement 227.7202 for military agencies. The use
 * of this code by the Government is further restricted in accordance with
 * the terms of the license agreement(s) and/or applicable contract terms
 * and conditions covering this code from MIPS Technologies or an authorized
 * third party.
 *
 * 
 * mips_end_of_legal_notice
 * 
 *
 ************************************************************************/


/************************************************************************
 *  Include files
 ************************************************************************/

#include <sysdefs.h>
#include <sysdev.h>
#include <syscon_api.h>
#include <syscon.h>
#include <sys_api.h>
#include <string.h>
#include <excep_api.h>
#include <io_api.h>
#include <product.h>
#include <malta.h>

#include <extern_param.h>


/************************************************************************
 *  Definitions
 ************************************************************************/

#define WORD0_OFS    0x00000       /* Word0:                       */
#define WORD1_OFS    0x00004       /* Word1:                       */

/************************************************************************
 *  Public variables
 ************************************************************************/
UINT32 rtl_monitorflash_size ; /* default 512K Byte */
extern t_extern_param *ext_para_ptr;
/************************************************************************
 *  Static variables
 ************************************************************************/

/* Pointer to array of objects */
static t_syscon_obj       *syscon_objects;

/* Size of ASCII display (if any) on platform */
static UINT32		  alpha_display_size;
/* Addresses of individual ASCII charactes on ASCII display */
static volatile UINT32    *alpha_io[8];

/* Mapping between hex digit and ASCII character (e.g 0-> ASCII(0x30)) */
static const UINT8        bin2char[16] = { 0x30, 0x31, 0x32, 0x33, 
					   0x34, 0x35, 0x36, 0x37,
					   0x38, 0x39, 0x41, 0x42,
					   0x43, 0x44, 0x45, 0x46  };

/* Register size */
static UINT8  size8  = sizeof(UINT8);
static UINT8  size32 = sizeof(UINT32);

/* MAC address and Serial Number */
static UINT8       amd_mac_addr[8];
static t_sn_bcd    sn_buffer;
static t_mac_addr  mac_addr_buffer;
static t_mac_addr  *mac_addr_read = NULL;
static t_sn_bcd	   *sn_read       = NULL;



/************************************************************************
 *  Static function prototypes
 ************************************************************************/


/************************************************************************
 *  Implementation : Static functions registered for handling particular 
 *  SYSCON objects for particular platform(s)
 ************************************************************************/


/************************************************************************
 *  com_en0_major_device_malta_read
 ************************************************************************/
static UINT32
com_en0_major_device_malta_read(
    void   *param,
    void   *data,
    UINT32 size )
{
   /*  On Malta, the AMD 79C973 controller, implements
    *  the LAN device; speed and duplex mode are autonegotiated 
    */
    *(UINT32 *)param = SYS_MAJOR_LAN_8139CPLUS;
    return OK;
}

 


 
/************************************************************************
 *  com_en0_intline_malta_read
 ************************************************************************/
static UINT32
com_en0_intline_malta_read(
    void   *param,
    void   *data,
    UINT32 size )
{
    /* Return interrupt line used for EN0 on Malta interrupt controller */
    *(UINT32 *)param = RTL_INTLINE_8139CPLUS;
    return OK;
}




































/************************************************************************
 *  board_monitorflash_base_atlas_malta_read
 ************************************************************************/
static UINT32
board_monitorflash_base_atlas_malta_read(
    void   *param,
    void   *data,
    UINT32 size )
{
#ifdef Board_Product_pvr
    *(void **)param = (void *) (0xbfc00000);
#else
    if (rtl_monitorflash_size <= 0x100000)
        *(void **)param = (void *) (0xbfc00000);
    else	
        *(void **)param = (void *) (0xbfd00000 - rtl_monitorflash_size);
#endif

    return OK;
}


/************************************************************************
 *  board_monitorflash_size_atlas_malta_read
 ************************************************************************/
static UINT32
board_monitorflash_size_atlas_malta_read(
    void   *param,
    void   *data,
    UINT32 size )
{
    *(UINT32 *)param = rtl_monitorflash_size;
    return OK;
}


/************************************************************************
 *  board_fileflash_base_atlas_malta_read
 ************************************************************************/
static UINT32
board_fileflash_base_atlas_malta_read(
    void   *param,
    void   *data,
    UINT32 size )
{   
    *(void **)param = (void *)ext_para_ptr->env_param_saddr;
    return OK;
}




/************************************************************************
 *  board_fileflash_size_atlas_malta_read
 ************************************************************************/
static UINT32
board_fileflash_size_atlas_malta_read(
    void   *param,
    void   *data,
    UINT32 size )
{
    *(UINT32 *)param = 0xfe00;   /* parameter   area bfbf0000--bfbffdff */
    return OK;                   /* magic #     area bfbfff00--bfbfff7f */
}                                /* image table area bfbfff80--bfbfffff */
                                 /* or parameter   area bfc40000--bfc4fdff */
                                 /* magic #     area bfc4ff00--bfc4ff7f */
                                 /* image table area bfc4ff80--bfc4ffff *  


/************************************************************************
 *  millisec
 ************************************************************************/
static UINT32
millisec(
    UINT32 cpufreq_hz )
{
    UINT32 calc, cycle_per_count, count;

    if( cpufreq_hz == 0 )
        return 0;

    SYSCON_read( SYSCON_CPU_CYCLE_PER_COUNT_ID,
                 (void *)&cycle_per_count,
                 sizeof(UINT32) );

    SYSCON_read( SYSCON_CPU_CP0_COUNT_ID,
		 (void *)&count,
		 sizeof(UINT32) );	

    calc = cpufreq_hz / ( 1000 * cycle_per_count );

    if( calc != 0 )
    {
		calc = count / calc;
    }
    else
    {
        calc = count * 1000 * cycle_per_count / cpufreq_hz;
    }

    return calc;
}


/************************************************************************
 *  board_get_millisec_atlas_malta_read
 ************************************************************************/
#if 0 //cylee marked since we never use this function
static UINT32
board_get_millisec_atlas_malta_read(
    void   *param,
    void   *data,
    UINT32 size )
{
    *(UINT32 *)param = millisec( sys_cpufreq_hz );
    return OK;
}
#endif






/************************************************************************
 * board_monitorflash_sectorsize_malta_read 
 ************************************************************************/
static UINT32
board_monitorflash_sectorsize_malta_read(
    void   *param,
    void   *data,
    UINT32 size )
{
    *(UINT32 *)param = RTL_MONITORFLASH_SECTORSIZE;
    return OK;
}




/************************************************************************
 *  board_fileflash_sectorsize_atlas_malta_read
 ************************************************************************/
static UINT32
board_fileflash_sectorsize_atlas_malta_read(
    void   *param,
    void   *data,
    UINT32 size )
{
    *(UINT32 *)param = RTL_FILEFLASH_SECTORSIZE;
    return OK;
}

















/************************************************************************
 *  board_fileflash_write_protected_atlas_malta_read
 ************************************************************************/
static UINT32
board_fileflash_write_protected_atlas_malta_read(
    void   *param,
    void   *data,
    UINT32 size )
{
/*                                                      reserved cyhuang	
    *(UINT32 *)param =
        (REGP(KSEG1BASE, ATLAS_STATUS) & ATLAS_STATUS_MFWR_MSK) ?
	    0 : 1;
*/
    return OK;
}

/************************************************************************
 *  board_monitorflash_write_protected_atlas_malta_read
 ************************************************************************/
static UINT32
board_monitorflash_write_protected_atlas_malta_read(
    void   *param,
    void   *data,
    UINT32 size )
{
/*                         reserved cyhuang	
    *(UINT32 *)param =
        (REGP(KSEG1BASE, ATLAS_STATUS) & ATLAS_STATUS_MFWR_MSK) ?
	    1 : 0;
*/
    return OK;
}


/************************************************************************
 *  board_rtc_data_malta_read
 ************************************************************************/
static UINT32
board_rtc_data_malta_read(
    void   *param,
    void   *data,
    UINT32 size )
{
//    *(UINT32 *)param = MALTA_RTCDAT; reserved cyhuang
    return OK;
}


/************************************************************************
 *  board_am79c973_base_malta_read
 ************************************************************************/
static UINT32
board_rtl8139cplus_base_malta_read(
    void   *param,
    void   *data,
    UINT32 size )
{                                          /* modified cyhuang */
   
    *(UINT32 *)param = ETHERNET_REG_BASE;
    return OK;
       
}



/************************************************************************
 *  Implementation : Public functions
 ************************************************************************/





/************************************************************************
 *
 *                          syscon_arch_board_init
 *  Description :
 *  -------------
 *
 *  Initialize platform specific part of SYSCON (except TTY related parts)
 *
 *  Return values :
 *  ---------------
 *
 *  None
 *
 ************************************************************************/
void 
syscon_arch_board_init( 
    t_syscon_obj *objects )		/* Array of SYSCON objects	*/
{
    syscon_objects = objects;

 
	syscon_arch_core_init( objects,
			           /* RAM size */
				   RTL_SYSTEMRAM_SIZE
				  );
        


    syscon_register_generic( SYSCON_BOARD_MONITORFLASH_BASE_ID,
			     
			      board_monitorflash_base_atlas_malta_read, NULL,
			      NULL,					NULL );

    syscon_register_generic( SYSCON_BOARD_MONITORFLASH_SIZE_ID,
			     
			      board_monitorflash_size_atlas_malta_read, NULL,
			      NULL,					NULL );

    syscon_register_generic( SYSCON_BOARD_FILEFLASH_BASE_ID,
			    
			      board_fileflash_base_atlas_malta_read, NULL,
			      NULL,				     NULL );

    syscon_register_generic( SYSCON_BOARD_FILEFLASH_SIZE_ID,
			     
			      board_fileflash_size_atlas_malta_read, NULL,
			      NULL,				     NULL );

 
    syscon_register_generic( SYSCON_BOARD_RTL8139CPLUS_BASE_ID,
			  
			      board_rtl8139cplus_base_malta_read, NULL,
			      NULL,			      NULL );

    
#if 0 //cylee marked since we never use this function
    syscon_register_generic( SYSCON_BOARD_GET_MILLISEC_ID,
			      
			      board_get_millisec_atlas_malta_read, NULL,
			      NULL,			     NULL );
#endif

    syscon_register_generic( SYSCON_BOARD_MONITORFLASH_SECTORSIZE_ID,
			      
			      board_monitorflash_sectorsize_malta_read, NULL,
			      NULL,				        NULL );

    syscon_register_generic( SYSCON_BOARD_FILEFLASH_SECTORSIZE_ID,
			     
			      board_fileflash_sectorsize_atlas_malta_read, NULL,
			      NULL,					   NULL );




    syscon_register_generic( SYSCON_BOARD_MONITORFLASH_WRITE_PROTECTED_ID,
			     
			      board_monitorflash_write_protected_atlas_malta_read, NULL,
			      NULL,						NULL );

    syscon_register_generic( SYSCON_BOARD_FILEFLASH_WRITE_PROTECTED_ID,
			      
			      board_fileflash_write_protected_atlas_malta_read, NULL,
			      NULL,						NULL );


    syscon_register_generic( SYSCON_COM_EN0_MAJOR_DEVICE_ID,   /*reserved cyhuang */
                            
                  com_en0_major_device_malta_read, NULL,
                  NULL,			       		NULL );
 
    syscon_register_generic( SYSCON_COM_EN0_IC_ID,             /*reserved cyhuang */
                              
                  syscon_true_read, NULL,
                  NULL,		NULL );
 
    syscon_register_generic( SYSCON_COM_EN0_INTLINE_ID,        /*reserved cyhuang */
                             
                  com_en0_intline_malta_read, NULL,
                  NULL,			  NULL );

   
}


/************************************************************************
 *
 *                          syscon_register_id_board
 *  Description :
 *  -------------
 *
 *  Function used to register platform specific SYSCON object functions.
 *
 *  A read and/or write function may be registered for each platform.
 *  A NULL function pointer indicates that the operation (read or
 *  write) is not allowed for the particular platform.
 *
 *  read_data and write_data pointers are passed to the read and write
 *  function.

 *  Return values :
 *  ---------------
 *
 *  None
 *
 ************************************************************************/
void 
syscon_register_id_board(
    t_syscon_ids   id,			/* OBJECT ID from syscon_api.h	*/
    /* Atlas */
    t_syscon_func  read_atlas,		/* Atlas read function		*/
    void	   *read_data_atlas,	/* Registered data		*/
    t_syscon_func  write_atlas,		/* Atlas write function		*/
    void	   *write_data_atlas,	/* Registered data		*/
    /* SEAD */
    t_syscon_func  read_sead,		/* SEAD read function		*/
    void	   *read_data_sead,	/* Registered data		*/
    t_syscon_func  write_sead,		/* SEAD write function		*/
    void	   *write_data_sead,	/* Registered data		*/
    /* Malta */
    t_syscon_func  read_malta,		/* Malta read function		*/
    void	   *read_data_malta,	/* Registered data		*/
    t_syscon_func  write_malta,		/* Malta write function		*/
    void	   *write_data_malta )	/* Registered data		*/
{
    t_syscon_obj *obj;

    obj = &syscon_objects[id];

    switch( sys_platform )
    {
      
      case PRODUCT_MALTA_ID :
        obj->read       = read_malta;
	obj->read_data  = read_data_malta;
	obj->write      = write_malta;
	obj->write_data = write_data_malta;
	break;
      default : /* Should not happen */
        obj->read  = NULL;
	obj->write = NULL;
	break;
    }    
}
