
/************************************************************************
 *
 *      vfd.c
 *
 *      The 'VFD' module implements the VFD driver
 *      interface to be used via 'IO' device driver services:
 *
 *        1) init  device:  configure and initialize IDE driver
 *        2) open  device:  not used
 *        3) close device:  not used
 *        4) read  device:  not used
 *        5) write device:  not used
 *        6) ctrl  device:  
 *                          
 *			    
 *
 * ######################################################################
 *
 * mips_start_of_legal_notice
 * 
 * Copyright (c) 2003 MIPS Technologies, Inc. All rights reserved.
 *
 *
 * Unpublished rights (if any) reserved under the copyright laws of the
 * United States of America and other countries.
 *
 * This code is proprietary to MIPS Technologies, Inc. ("MIPS
 * Technologies"). Any copying, reproducing, modifying or use of this code
 * (in whole or in part) that is not expressly permitted in writing by MIPS
 * Technologies or an authorized third party is strictly prohibited. At a
 * minimum, this code is protected under unfair competition and copyright
 * laws. Violations thereof may result in criminal penalties and fines.
 *
 * MIPS Technologies reserves the right to change this code to improve
 * function, design or otherwise. MIPS Technologies does not assume any
 * liability arising out of the application or use of this code, or of any
 * error or omission in such code. Any warranties, whether express,
 * statutory, implied or otherwise, including but not limited to the implied
 * warranties of merchantability or fitness for a particular purpose, are
 * excluded. Except as expressly provided in any written license agreement
 * from MIPS Technologies or an authorized third party, the furnishing of
 * this code does not give recipient any license to any intellectual
 * property rights, including any patent rights, that cover this code.
 *
 * This code shall not be exported or transferred for the purpose of
 * reexporting in violation of any U.S. or non-U.S. regulation, treaty,
 * Executive Order, law, statute, amendment or supplement thereto.
 *
 * This code constitutes one or more of the following: commercial computer
 * software, commercial computer software documentation or other commercial
 * items. If the user of this code, or any related documentation of any
 * kind, including related technical data or manuals, is an agency,
 * department, or other entity of the United States government
 * ("Government"), the use, duplication, reproduction, release,
 * modification, disclosure, or transfer of this code, or any related
 * documentation of any kind, is restricted in accordance with Federal
 * Acquisition Regulation 12.212 for civilian agencies and Defense Federal
 * Acquisition Regulation Supplement 227.7202 for military agencies. The use
 * of this code by the Government is further restricted in accordance with
 * the terms of the license agreement(s) and/or applicable contract terms
 * and conditions covering this code from MIPS Technologies or an authorized
 * third party.
 * 
 * 
 * mips_end_of_legal_notice
 * 
 *
 ************************************************************************/


/************************************************************************
 *      Include files
 ************************************************************************/

#include <sysdefs.h>

#include <sysdev.h>

#include <sys_api.h>
#include <io_api.h>

#include <stdio.h>
#include <string.h>

#include <vfd_api.h>

/************************************************************************
 *      Definitions
 ************************************************************************/
#define MIS_VFD_CTL        0x1801b700 
#define MIS_VFD_WRCTL      0x1801b704
#define MIS_VFDO           0x1801b708
#define MIS_VFD_ARDCTL     0x1801b70c
#define MIS_VFD_KPADLIE    0x1801b710
#define MIS_VFD_KPADHIE    0x1801b714
#define MIS_VFD_SWIE       0x1801b718
#define MIS_VFD_ARDKPADL   0x1801b71c
#define MIS_VFD_ARDKPADH   0x1801b720
#define MIS_VFD_ARDSW      0x1801b724
#define MIS_UMSK_ISR_KPADAH		0x1801b018
#define MIS_UMSK_ISR_KPADAL		0x1801b01c
#define MIS_UMSK_ISR_KPADDAH	0x1801b020
#define MIS_UMSK_ISR_KPADDAL	0x1801b024

/************************************************************************
 *  Macro Definitions
*************************************************************************/


/************************************************************************
 *      Public variables
 ************************************************************************/

/************************************************************************
 *      Static variables
 ************************************************************************/



/************************************************************************
 *      Static function prototypes
 ************************************************************************/
static INT32 
vfd_init(
    UINT32 major,          /* IN: major device number             */
    UINT32 minor,          /* IN: minor device number             */
    void   *p_param )      /* INOUT: device parameter block       */
;

static
UINT32 VFD_read_keypad(t_vfd_ctrl_descriptor *p_param ) ;

static
UINT32 VFD_write_lcd(t_vfd_ctrl_descriptor *p_param ) ;

static
UINT32 VFD_write_led(t_vfd_ctrl_descriptor *p_param );

static
UINT32 Count(UINT32 w);
 

/************************************************************************
 *      Implementation : Static functions
 ************************************************************************/


/************************************************************************
 *
 *                          vfd_init
 *  Description :
 *  -------------
 *
 *  This service initializes the VFD driver.
 *
 *  Parameters :
 *  ------------
 *
 *  'major',     IN,    major device number
 *  'minor',     IN,    not used
 *  'p_param',   INOUT, not used
 *
 *
 *  Return values :
 *  ---------------
 *
 *  'OK'(=0)
 *
 ************************************************************************/

 
static INT32 
vfd_init(
    UINT32 major,          /* IN: major device number             */
    UINT32 minor,          /* IN: minor device number             */
    void   *p_param )      /* INOUT: device parameter block       */
{   
    int i;
   

    /* VFD Auto Read Control Register */
    /* 48 keys, Switch HIGH active, Keypad HIGH Active, 10ms-period Auto Read */
    REG32(KSEG1(MIS_VFD_ARDCTL)) = 0x00000630;

    /* [KEYPAD] disable key1-key4, seg1-seg8 interrupt */
    REG32(KSEG1(MIS_VFD_KPADLIE))= 0x00000000;
    
    /* [KEYPAD] disable key1-key4, seg9-seg12 interrupt */
    REG32(KSEG1(MIS_VFD_KPADHIE)) = 0x00000000;
    
    /* [SWITCH] disable switch0-switch3 interrupt */
    REG32(KSEG1(MIS_VFD_SWIE)) = 0x00000000;
    
    /* Enable VFD Controller (enable kpad, enable switch, 1.037us Duty Cycle, Enable VFD) */
    REG32(KSEG1(MIS_VFD_CTL)) = 0x0000000d;
    
	/* [KEYPAD] Clear Interrupt Flags */
    REG32(KSEG1(MIS_UMSK_ISR_KPADDAH))= 0x00fffff0;
    REG32(KSEG1(MIS_UMSK_ISR_KPADDAL))=0xfffffff0;
    REG32(KSEG1(MIS_UMSK_ISR_KPADAH))=  0x00fffff0;
//    REG32(KSEG1(MIS_UMSK_ISR_KPADAL))=  0xfffffff0;
	
	
    
	/* Initialization of TP6311 */

	/* 1st step: Reset Display Memory:
	   [Byte0 : data setting command, normal operation, increament, write to display memory 
       [Byte1 : address setting command, from 0]
	*/
    REG32(KSEG1(MIS_VFDO)) = 0xc040;
    REG32(KSEG1(MIS_VFD_WRCTL)) = 0x00000330;  //1330
    
    /* wait_write_done */
    while((REG32(KSEG1(MIS_VFD_WRCTL)) & 0x02) == 0x00);
    
    for(i=0;i<11;i++) {
	REG32(KSEG1(MIS_VFDO)) = 0x00000000;
	REG32(KSEG1(MIS_VFD_WRCTL)) = 0x000000f0;  //10f0
	/* wait_write_done */
        while((REG32(KSEG1(MIS_VFD_WRCTL)) & 0x02) == 0x00);
    }
    
    

    REG32(KSEG1(MIS_VFDO)) = 0x00000000;
    REG32(KSEG1(MIS_VFD_WRCTL)) = 0x000010f0;
    
    /* wait_write_done */
    while((REG32(KSEG1(MIS_VFD_WRCTL)) & 0x02) == 0x00);

	/* 2nd step:
	   [Byte0 : display mode setting command, 11 digits, 17 segments]
	   [Byte1 : display control command, turn on display, pulse width = 14/16]
    */
    
    REG32(KSEG1(MIS_VFDO)) = 0x00008f0a;      
    REG32(KSEG1(MIS_VFD_WRCTL)) = 0x00001330 ;  //cy test
    
    /* wait_write_done */
    while((REG32(KSEG1(MIS_VFD_WRCTL)) & 0x02) == 0x00);
    
    return OK;
}


/************************************************************************
 *
 *                          vfd_ctrl
 *  Description :
 *  -------------
 *  This service comprises following specific VFD services:
 *
 *    1) VFD_READ_KEYPAD		0x01
 *    2) VFD_WRITE_LCD		    0x02
 *    3) VFD_WRITE_LED	        0x03
 *    
 *  Parameters :
 *  ------------
 *
 *  'major',     IN,    major device number
 *  'minor',     IN,    minor device number for multi device drivers
 *  'p_param',   IN,    variable of type, t_ide_ctrl_descriptor.
 *
 *  Return values :
 *  ---------------
 *
 *  OK (0x00):                   VFD service completed successfully
 *  Other :			 See error codes in vfd_api.h
 *
 ************************************************************************/
static INT32 
vfd_ctrl(
    UINT32                major,     /* IN: major device number		*/
    UINT32		  minor,     /* IN: minor device number		*/
    t_vfd_ctrl_descriptor *p_param ) /* IN: write data			*/
{
   
    
    switch( p_param->command )
    {
      
          
      case VFD_READ_KEYPAD :
      {
      	  
           return VFD_read_keypad(p_param);
      }     
               
          
          
      case VFD_WRITE_LCD  : 
           return VFD_write_lcd(p_param);
           
      case VFD_WRITE_LED :
           return VFD_write_led(p_param);  
           
      default : 
	return ERROR_VFD_INVALID_COMMAND;
    }

    return OK;
}

/************************************************************************
 *
 *                          VFD_read_keypad
 *  Description :
 *  -------------
 *  read keypad to check if assert / disassert 
 *
 *  Parameters :
 *  ------------
 * 
 *  'p_param',   IN,    variable of type, t_vfd_ctrl_descriptor.
 *
 *
 *  Return values :
 *  ---------------
 *
 * 'OK' = 0x00:                         succesfully
 *
 *
 *
 ************************************************************************/
static
UINT32 VFD_read_keypad(t_vfd_ctrl_descriptor *p_param )
{
   UINT32 l_value, h_value, x_value,  bit_count;
   
   l_value = REG32(KSEG1(MIS_UMSK_ISR_KPADAL)) ;
   h_value = REG32(KSEG1(MIS_UMSK_ISR_KPADAH)) ; 
   x_value = REG32(KSEG1(MIS_VFD_ARDKPADL)) ;
  	
   	
   
   if (l_value != 0)
     REG32(KSEG1(MIS_UMSK_ISR_KPADAL)) = l_value & (~0x1);
     
   if (h_value != 0)
     REG32(KSEG1(MIS_UMSK_ISR_KPADAH)) = h_value & (~0x1);  
     
     
   /* [KEYPAD] disable key1-key4, seg1-seg8 interrupt */
    REG32(KSEG1(MIS_VFD_KPADLIE))= 0x00000000;  
   
  
   
   bit_count = Count(x_value) ;
  
  
  
   if (bit_count !=1)
   {
      
      return ERROR_VFD_NO_PUSH ;
   }   
      
   switch (x_value)
   {
     case 0x1 :
               p_param ->data = KEY_PWR ;
               break;
     case 0x10 :
               p_param ->data = KEY_EJECT ;
               break;   
     case 0x100 :
               p_param ->data = KEY_PLAY ;
               break;   
     case 0x1000 :
               p_param ->data = KEY_SOUR ;
               break;   
     case 0x10000 :
               p_param ->data = KEY_STOP ;
               break;   
     case 0x100000 :
               p_param ->data = KEY_REC ;
               break;   
     case 0x1000000 :
               p_param ->data = KEY_CHP ; 
               break;
     case 0x10000000 :
               p_param ->data = KEY_CHM ;                                                                          
               
               break;   
     default :
                              
               return ERROR_VFD_KEYPAD_STATUS ; 	
  }               
      
  return OK; 
}




/************************************************************************
 *
 *                          VFD_write_lcd
 *  Description :
 *  -------------
 *  write data to LCD
 *
 *  Parameters :
 *  ------------
 * 
 *  'p_param',   IN,    variable of type, t_vfd_ctrl_descriptor.
 *
 *
 *  Return values :
 *  ---------------
 *
 * 'OK' = 0x00:                         succesfully
 *
 *
 *
 ************************************************************************/
static
UINT32 VFD_write_lcd(t_vfd_ctrl_descriptor *p_param )
{
  
  /* Byte 0: Data setting command, write to display memory, incremental address, normal operation */
  UINT8 byte0 = 0x40;
  /* Byte 1: Address setting command */
  UINT8 byte1 = 0xc0 | p_param->addr;
  
  /* Byte 2: Data */
  UINT8 byte2 = (p_param->data) & 0xff;
  
  REG32(KSEG1(MIS_VFDO)) = ((UINT32) byte0) | ((UINT32)(byte1) << 8) | 
		     ((UINT32)(byte2) << 16); 

  REG32(KSEG1(MIS_VFD_WRCTL)) = 0x00001370;    
   
  /* wait_write_done */
  while((REG32(KSEG1(MIS_VFD_WRCTL)) & 0x02) == 0x00);

 

  /* extra command #1 and command #4 to meet PT6311 recommanded software flowchart */
  REG32(KSEG1(MIS_VFDO)) = 0x00008f0a;
  REG32(KSEG1(MIS_VFD_WRCTL)) = 0x00001330 ; //cy test
    
  /* wait_write_done */
  while((REG32(KSEG1(MIS_VFD_WRCTL)) & 0x02) == 0x00);
  
    
  return OK; 
}

/************************************************************************
 *
 *                          VFD_write_led
 *  Description :
 *  -------------
 *  write data to LED
 *
 *  Parameters :
 *  ------------
 * 
 *  'p_param',   IN,    variable of type, t_vfd_ctrl_descriptor.
 *
 *
 *  Return values :
 *  ---------------
 *
 * 'OK' = 0x00:                         succesfully
 *
 *
 *
 ************************************************************************/
static
UINT32 VFD_write_led(t_vfd_ctrl_descriptor *p_param )
{
  
  REG32(KSEG1(MIS_VFDO)) 	= (((UINT32)p_param->data) << 8) | 0x41;
  REG32(KSEG1(MIS_VFD_WRCTL)) 	= 0x00001130;
  
  /* wait_write_done */
  while((REG32(KSEG1(MIS_VFD_WRCTL)) & 0x02) == 0x00);

  /* extra command #1 and command #4 to meet PT6311 recommanded software flowchart */
  REG32(KSEG1(MIS_VFDO)) = 0x00008f0a;
#if 0
  REG32(KSEG1(MIS_VFD_WRCTL)) = 0x00000330 ;
#else
  REG32(KSEG1(MIS_VFD_WRCTL)) = 0x00001330 ; //cyhuang add bit12 as 1
#endif
    
  /* wait_write_done */
  while((REG32(KSEG1(MIS_VFD_WRCTL)) & 0x02) == 0x00);

  return OK; 
}

static
UINT32 Count(UINT32 w)
{
        // binary technique for 32 bit words
        w = (0x55555555 & w) + (0x55555555 & (w>> 1));
        w = (0x33333333 & w) + (0x33333333 & (w>> 2));
        w = (0x0f0f0f0f & w) + (0x0f0f0f0f & (w>> 4));
        w = (0x00ff00ff & w) + (0x00ff00ff & (w>> 8));
        w = (0x0000ffff & w) + (0x0000ffff & (w>>16));
        return w;
}
/************************************************************************
 *      Implementation : Public functions
 ************************************************************************/


/************************************************************************
 *
 *                          vfd_install
 *  Description :
 *  -------------
 *
 *  Installs the VFD device driver services in the IO system at the 
 *  reserved device slot, found in the 'sysdev.h' file, which defines 
 *  all major device numbers.
 *
 *  Note:
 *  This service is the only public declared interface function; all
 *  provided device driver services are static declared, but this
 *  function installs the function pointers in the io-system to
 *  enable the provided public driver services.
 *
 *  Parameters :
 *  ------------
 *
 *  -
 *
 *  Return values :
 *  ---------------
 *
 *  'OK'(=0)
 *  'ERROR_IO_ILLEGAL_MAJOR':  Illegal major device number
 *  'ERROR_IO_NO_SPACE':       Device slot already allocated
 *
 ************************************************************************/
INT32 
vfd_install( void )
{   
    
    /* Install device services */
    IO_install( SYS_MAJOR_VFD,                  /* major device number */
                    (t_io_service)vfd_init,	/* 'init'  service     */
                    NULL,			/* 'open'  service  na */
                    NULL,			/* 'close' service  na */
                    NULL,			/* 'read'  service  na */
		    NULL,			/* 'write' service  na */
                    (t_io_service)vfd_ctrl ) ;	/* 'ctrl'  service     */
   
    /* call our own 'init' service */
    /* hcy : already done in Bootup */
   IO_init( SYS_MAJOR_VFD, 0, NULL);
 
   return OK;
}
